##############################################################################
#
# Copyright (c) 2002 Zope Corporation and Contributors. All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.0 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
##############################################################################
__doc__="""Verbose version of Zope security policy


$Id: VerboseSecurityPolicy.py,v 1.7 2004/06/17 02:38:44 shane Exp $"""
__version__='$Revision: 1.7 $'[11:-2]


from Acquisition import aq_base, aq_inner, aq_parent
from AccessControl import Unauthorized
from AccessControl.SimpleObjectPolicies import Containers, _noroles
from zLOG import LOG, PROBLEM, BLATHER

from AccessControl.PermissionRole import _what_not_even_god_should_do, \
     rolesForPermissionOn


class VerboseSecurityPolicy:

    def __init__(self, ownerous=1, authenticated=1):
        """Create a Zope security policy.

        Two optional keyword arguments may be provided:

        ownerous -- Untrusted users can create code
                    (e.g. Python scripts or templates),
                    so check that code owners can access resources.
                    The argument must have a truth value.
                    The default is true.

        authenticated -- Allow access to resources based on the
                    privaledges of the authenticated user.
                    The argument must have a truth value.
                    The default is true.

                    This (somewhat experimental) option can be set
                    to false on sites that allow only public
                    (unauthenticated) access. An anticipated
                    scenario is a ZEO configuration in which some
                    clients allow only public access and other
                    clients allow full management.
        """
        self._ownerous = ownerous
        self._authenticated = authenticated

    def validate(self, accessed, container, name, value, context,
                 roles=_noroles, getattr=getattr, _noroles=_noroles,
                 valid_aq_=('aq_parent','aq_inner', 'aq_explicit')):

        ############################################################
        # Provide special rules for the acquisition attributes
        if isinstance(name, str):
            if name.startswith('aq_') and name not in valid_aq_:
                info = setUnauthorized(
                    'aq_* names (other than %s) are not allowed'
                    % ', '.join(valid_aq_),
                    accessed, container, name, value, context)
                raise Unauthorized(info)

        containerbase = aq_base(container)
        accessedbase = aq_base(accessed)
        if accessedbase is accessed:
            # accessed is not a wrapper, so assume that the
            # value could not have been acquired.
            accessedbase = container

        ############################################################
        # If roles weren't passed in, we'll try to get them from the object

        if roles is _noroles:
            roles = getattr(value, '__roles__', roles)

        ############################################################
        # We still might not have any roles

        if roles is _noroles:

            ############################################################
            # We have an object without roles and we didn't get a list
            # of roles passed in. Presumably, the value is some simple
            # object like a string or a list.  We'll try to get roles
            # from its container.
            if container is None:
                # Either container or a list of roles is required
                # for ZopeSecurityPolicy to know whether access is
                # allowable.
                info = setUnauthorized(
                    'No container provided',
                    accessed, container, name, value, context)
                raise Unauthorized(info)

            roles = getattr(container, '__roles__', roles)
            if roles is _noroles:
                if containerbase is container:
                    # Container is not wrapped.
                    if containerbase is not accessedbase:
                        info = setUnauthorized(
                            'Unable to find __roles__ in the container '
                            'and the container is not wrapped',
                            accessed, container, name, value, context)
                        raise Unauthorized(info)
                else:
                    # Try to acquire roles
                    try: roles = container.aq_acquire('__roles__')
                    except AttributeError:
                        if containerbase is not accessedbase:
                            info = setUnauthorized(
                                'Unable to find or acquire __roles__ '
                                'from the container',
                                accessed, container, name, value, context)
                            raise Unauthorized(info)

            # We need to make sure that we are allowed to
            # get unprotected attributes from the container. We are
            # allowed for certain simple containers and if the
            # container says we can. Simple containers
            # may also impose name restrictions.
            p = Containers(type(container), None)
            if p is None:
                p = getattr(container,
                            '__allow_access_to_unprotected_subobjects__',
                            None)

            if p is not None:
                tp = p.__class__
                if tp is not int:
                    if tp is dict:
                        if isinstance(name, basestring):
                            p = p.get(name)
                        else:
                            p = 1
                    else:
                        p = p(name, value)

            if not p:
                info = setUnauthorized(
                    'The container has no security assertions',
                    accessed, container, name, value, context
                    )
                raise Unauthorized(info)

            if roles is _noroles:
                return 1

            # We are going to need a security-aware object to pass
            # to allowed(). We'll use the container.
            value = container

        # Short-circuit tests if we can:
        try:
            if roles is None or 'Anonymous' in roles:
                return 1
        except TypeError:
            # 'roles' isn't a sequence
            LOG('Zope Security Policy', PROBLEM, "'%s' passed as roles"
                " during validation of '%s' is not a sequence." % (
                `roles`, name))
            raise

        # Check executable security
        stack = context.stack
        if stack:
            eo = stack[-1]

            # If the executable had an owner, can it execute?
            if self._ownerous:
                owner = eo.getOwner()
                if (owner is not None) and not owner.allowed(value, roles):
                    # We don't want someone to acquire if they can't
                    # get an unacquired!
                    if len(roles) < 1:
                        info = setUnauthorized(
                            "The object is marked as private",
                            accessed, container, name, value, context)
                    elif userHasRolesButNotInContext(owner, value, roles):
                        info = setUnauthorized(
                            "The owner of the executing script is defined "
                            "outside the context of the object being "
                            "accessed",
                            accessed, container, name, value, context,
                            required_roles=roles, eo_owner=owner, eo=eo)
                    else:
                        info = setUnauthorized(
                            "The owner of the executing script does not "
                            "have the required permission",
                            accessed, container, name, value, context,
                            required_roles=roles, eo_owner=owner, eo=eo,
                            eo_owner_roles=getUserRolesInContext(
                            owner, value))
                    raise Unauthorized(info)

            # Proxy roles, which are a lot safer now.
            proxy_roles = getattr(eo, '_proxy_roles', None)
            if proxy_roles:
                # Verify that the owner actually can state the proxy role
                # in the context of the accessed item; users in subfolders
                # should not be able to use proxy roles to access items
                # above their subfolder!
                owner = eo.getWrappedOwner()

                if owner is not None:
                    if container is not containerbase:
                        # Unwrapped objects don't need checking
                        if not owner._check_context(container):
                            # container is higher up than the owner,
                            # deny access
                            info = setUnauthorized(
                                "The owner of the executing script is defined "
                                "outside the context of the object being "
                                "accessed.  The script has proxy roles, "
                                "but they do not apply in this context.",
                                accessed, container, name, value, context,
                                required_roles=roles, eo_owner=owner, eo=eo)
                            raise Unauthorized(info)

                for r in proxy_roles:
                    if r in roles:
                        return 1

                # Proxy roles actually limit access!
                if len(roles) < 1:
                    info = setUnauthorized(
                        "The object is marked as private",
                        accessed, container, name, value, context)
                else:
                    info = setUnauthorized(
                        "The proxy roles set on the executing script "
                        "do not allow access",
                        accessed, container, name, value, context,
                        eo=eo, eo_proxy_roles=proxy_roles,
                        required_roles=roles)
                raise Unauthorized(info)

        try:
            if self._authenticated and context.user.allowed(value, roles):
                return 1
        except AttributeError:
            pass

        if len(roles) < 1:
            info = setUnauthorized(
                "The object is marked as private",
                accessed, container, name, value, context)
        elif not self._authenticated:
            info = setUnauthorized(
                "Authenticated access is not allowed by this "
                "security policy",
                accessed, container, name, value, context)
        elif userHasRolesButNotInContext(context.user, value, roles):
            info = setUnauthorized(
                "Your user account is defined outside "
                "the context of the object being accessed",
                accessed, container, name, value, context,
                required_roles=roles, user=context.user)
        else:
            info = setUnauthorized(
                "Your user account does not "
                "have the required permission",
                accessed, container, name, value, context,
                required_roles=roles, user=context.user,
                user_roles=getUserRolesInContext(context.user, value))
        raise Unauthorized(info)


    def checkPermission(self, permission, object, context):
        # XXX proxy roles and executable owner are not checked
        roles = rolesForPermissionOn(permission, object)
        if isinstance(roles, basestring):
            roles = [roles]
        return context.user.allowed(object, roles)


def item_repr(ob):
    """Generates a repr without angle brackets (to avoid HTML quoting)"""
    return repr(ob).replace('<', '(').replace('>', ')')


def simplifyRoles(roles):
    """Simplifies and sorts a role list."""
    d = {}
    for r in roles:
        d[r] = 1
    lst = d.keys()
    lst.sort()
    return lst


def setUnauthorized(msg, accessed, container, name, value, context,
                    required_roles=None,
                    user_roles=None,
                    user=None,
                    eo=None,
                    eo_owner=None,
                    eo_owner_roles=None,
                    eo_proxy_roles=None,
                    ):
    """Returns the message with extra info appended.

    Also saves the message in a thread-specific buffer so that
    the error can be recovered later."""
    s = '%s.  Access to %s of %s' % (
        msg, repr(name), item_repr(container))
    if aq_base(container) is not aq_base(accessed):
        s += ', acquired through %s,' % item_repr(accessed)
    info = [s + ' denied.']
    if user is not None:
        try:
            ufolder = '/'.join(aq_parent(aq_inner(user)).getPhysicalPath())
        except:
            ufolder = '(unknown)'
        info.append('Your user account, %s, exists at %s.' % (
            str(user), ufolder))
    if required_roles is not None:
        p = None
        required_roles = list(required_roles)
        for r in required_roles:
            if r.startswith('_') and r.endswith('_Permission'):
                p = r[1:]
                required_roles.remove(r)
                break
        sr = simplifyRoles(required_roles)
        if p:
            info.append('Access requires %s, '
                        'granted to the following roles: %s.' %
                        (p, sr))
        else:
            info.append('Access requires one of the following roles: %s.'
                        % sr)
    if user_roles is not None:
        info.append(
            'Your roles in this context are %s.' % simplifyRoles(user_roles))
    if eo is not None:
        s = 'The executing script is %s' % item_repr(eo)
        if eo_proxy_roles is not None:
            s += ', with proxy roles: %s' % simplifyRoles(eo_proxy_roles)
        if eo_owner is not None:
            s += ', owned by %s' % repr(eo_owner)
        if eo_owner_roles is not None:
            s += ', who has the roles %s' % simplifyRoles(eo_owner_roles)
        info.append(s + '.')
    res = ' '.join(info)
    LOG('VerboseSecurity', BLATHER, 'Unauthorized: %s' % res)
    return res


def getUserRolesInContext(user, context):
    """Returns user roles for a context."""
    if hasattr(aq_base(user), 'getRolesInContext'):
        return user.getRolesInContext(context)
    else:
        return ()


def userHasRolesButNotInContext(user, object, object_roles):
    '''Returns 1 if the user has any of the listed roles but
    is not defined in a context which is not an ancestor of object.
    '''
    if object_roles is None or 'Anonymous' in object_roles:
        return 0
    usr_roles = getUserRolesInContext(user, object)
    for role in object_roles:
        if role in usr_roles:
            # User has the roles.
            return (not verifyAcquisitionContext(
                user, object, object_roles))
    return 0


def verifyAcquisitionContext(self, object, object_roles=None):
    """Mimics the relevant section of User.allowed(). self is a user object.
    """
    ufolder = aq_parent(self)
    ucontext = aq_parent(ufolder)
    if ucontext is not None:
        if object is None:
            # This is a strange rule, though
            # it doesn't cause any security holes. SDH
            return 1
        if not hasattr(object, 'aq_inContextOf'):
            if hasattr(object, 'im_self'):
                # This is a method.  Grab its self.
                object=object.im_self
            if not hasattr(object, 'aq_inContextOf'):
                # object is not wrapped, therefore we
                # can't determine context.
                # Fail the access attempt.  Otherwise
                # this would be a security hole.
                return None
        if not object.aq_inContextOf(ucontext, 1):
            if 'Shared' in object_roles:
                # Damn, old role setting. Waaa
                object_roles=self._shared_roles(object)
                if 'Anonymous' in object_roles: return 1
            return None
    # Note that if self were not wrapped, it would
    # not be possible to determine the user's context
    # and this method would return 1.
    # However, as long as user folders always return
    # wrapped user objects, this is safe.
    return 1
