//#include "config.h"

#include <iostream>

#include <uc++/plugin.h>

namespace uC
{


struct PluginInfo
{
    PluginInfo(const string& n, Plugin *p) : name(n), plugin(p) { }
    string name;
    Plugin *plugin;
};

PluginNode::PluginNode()
{
  node_ = 0;
}

PluginNode::PluginNode(const string& name, Plugin *p = 0)
{
  node_ = g_node_new(new PluginInfo(name, p));
  if (p)
    p->reference();
}

PluginNode::PluginNode(GNode *node)
{
  node_ = node;
}

PluginNode::~PluginNode()
{
  if (node_ && !node_->parent)
    clear();
}

string PluginNode::full_name() const
{
  string fullname;
  
  GNode *node = node_;
  while (node && node->parent) 
  {
    fullname.insert(0, PluginNode(node).name());
    
    if ((node = node->parent)->parent)
      fullname.insert(0, ".");
  }
  return(fullname);
}

string PluginNode::name() const
{
  return(node_ ? ((PluginInfo *)node_->data)->name : string());
}

Plugin *PluginNode::plugin() const
{
  return(node_ ? ((PluginInfo *)node_->data)->plugin : 0);
}

void PluginNode::set_plugin(Plugin *p)
{
  if (node_)
  {
    PluginInfo *info = (PluginInfo *)node_->data;
    
    if (info->plugin)
      info->plugin->unreference();
    info->plugin = p;
    if (p) p->reference();
  }
}

bool PluginNode::is_ancestor_of(const PluginNode& descendant) const
{
  return(g_node_is_ancestor(node_, descendant.node_));
}

PluginNode::iterator PluginNode::insert(const string& name, Plugin *plugin)
{
  string::size_type pos = name.find('.');
  string namebase = name.substr(0, pos);

  GNode *node = g_node_first_child(node_);
  while (true)
  {
    PluginNode tree(node);

    if (node == 0 || tree.name() > namebase)
    {
      if (pos != string::npos)
      {
        GNode *newnode =
          g_node_insert_data_before(node_, node,
                                    new PluginInfo(namebase, 0));
        return(PluginNode(newnode).insert(name.substr(pos + 1), plugin));
      }
      else
      {
        return(PluginNode::iterator(
                g_node_insert_data_before(node_, node,
                                          new PluginInfo(namebase, plugin))));
      }
    }
    if (tree.name() == namebase)
    {
      if (pos != string::npos)
        return(tree.insert(name.substr(pos + 1), plugin));
      else
      {
        tree.set_plugin(plugin);
        return(iterator(node));
      }
    }
      
    node = g_node_next_sibling(node); 
  }
}

void PluginNode::erase(iterator it)
{
  g_node_unlink((*it).node_);
  PluginNode((*it).node_).clear();
}

namespace {

gboolean free_node_data(GNode *node, gpointer data)
{
  PluginInfo *info = (PluginInfo *)(node->data);
  if (info->plugin)
  {
    info->plugin->unreference();
  }
  delete info;
  
  return(true);
}

void do_destroy(GNode *node, gpointer data)
{
  g_node_traverse(node,  G_POST_ORDER, G_TRAVERSE_ALL, -1,
                  free_node_data, data);
  g_node_destroy(node);
}

}

void PluginNode::clear()
{
  g_node_children_foreach(node_, G_TRAVERSE_ALL, do_destroy, 0);
  set_plugin(0);
  if (node_->parent == 0)
    g_node_destroy(node_);
}

namespace { void print_func(GNode *node, void *data) {
  cout << ((PluginInfo *)node->data)->name << endl;
} }

PluginNode::iterator PluginNode::find_(const string& name) const
{
  string::size_type pos = name.find('.');
  
  string namebase = pos == string::npos ? name : name.substr(0, pos);

  GNode *node = g_node_first_child(node_);
  while (true)
  {
    PluginNode tree(node);
    
    if (node == 0 || tree.name() > namebase)
      return(iterator(0));
    if (tree.name() == namebase)
    {
      if (pos == string::npos)
        return(iterator(node));
      else
        return(tree.find_(name.substr(pos + 1)));
    }
    node = g_node_next_sibling(node);
  }
}

void PluginNode::print(int indent) const
{
  for (GNode *node = g_node_first_child(node_); node;
       node = g_node_next_sibling(node))
  {
    PluginNode tree(node);
    
    cout << string(indent, ' ') << tree.name()
         << ": node " << node
         << ", data " << node->data
         << ", plugin " << tree.plugin()
         << endl;
    tree.print(indent + 2);
  }
}

}

