#!perl -w
#
# IdeDev -- probed values for an IDE device as found in /proc/ide
#   Copyright (C) 2005  Erik van Konijnenburg
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
#
#
# The probed values for an IDE device as found in /proc/ide,
# based on the name in /sys/devices/.../ideX/X.X/block.
#
# path - location in sysfs
# name - descriptive string, retrieved from device
# media - type of device: disk, tape, ...
# model - descriptive string, retrieved from device
#
# Background: hotplug is triggered when an IDE controller is
# found on for instance the PCI bus.  The loaded module should
# be good enough to talk to the hardware, but that does not
# mean you can actually use it: you will also need something
# to use the hardware driver to send IDE commands over the
# IDE cable to the controller on the other end of the cable.
# Those commands also have to make sense: a disk controller
# uses a different set of IDE commands than an IDE tape controller.
# The ide-disk, ide-cdrom etc modules are the protocol drivers
# that handle this.
#
use strict;
use warnings;
use Base;
use Conf;
package IdeDev;
use base 'Obj';

sub fill {
	my $self = shift;
	$self->SUPER::fill();
	$self->takeArgs ('path');
	my $path = $self->path;
	my $link = readlink ("$path/block");
	if (! defined ($link)) {
		Base::fatal ("no link to block device in $path");
	}
	if ($link !~ m!.*/([^/]+)!) {
		Base::fatal ("malformed link to block device in $path");
	}
	my $name = $1;
	my $ideDir = Conf::get('procFs') . "/ide";
	$self->{name} = $name;
	$self->{media} = Base::getStringFile ("$ideDir/$name/media");
	$self->{model} = Base::getStringFile ("$ideDir/$name/model");
}

sub path	{ return $_[0]->{path}; }
sub name	{ return $_[0]->{name}; }
sub media	{ return $_[0]->{media}; }
sub model	{ return $_[0]->{model}; }

sub string {
	my $self = shift;
	my $path = $self->path();
	my $name = $self->name();
	my $media = $self->media();
	my $model = $self->model();
	return "$name ($media) = $model at $path";
}


#
# findModuleByIdeDev -- list of suitable IDE drivers;
# you need all of them.
#
sub findModuleByIdeDev ($) {
	my ($ideDev) = @_;
	my $media = $ideDev->media();
	my $result = [];
	
	my $driver;
	$driver = "ide-disk" if ($media eq "disk");
	$driver = "ide-tape" if ($media eq "tape");

	# The CD may also need ide-generic.
	$driver = "ide-cd" if ($media eq "cdrom");

	$driver = "ide-floppy" if ($media eq "floppy");
	if (defined ($driver)) {
		push @{$result}, $driver;
	}

	return $result;
}


1;

