/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// --------------------------------------------------------------------------
// CONFIGURATION FILE EVALUATION
// --------------------------------------------------------------------------

#ifdef XSID_WB_DEBUG
#include <iomanip.h>
#include <iostream.h>
#endif
#include <fstream.h>
#include <stdlib.h>
#include <unistd.h>

#include "ConfigFileOld.h"
#include "ConfigFile.h"
#include "ConfigC.h"
#include "AudioConfig.h"
#include "HVSC_Config.h"

// Keywords.
static const char key_Frequency[] = "Frequency";
static const char key_Channels[] = "Channels";
static const char key_BitsPerSample[] = "BitsPerSample";
static const char key_SampleEncoding[] = "SampleEncoding";
static const char key_MaxFrags[] = "MaxFrags";
static const char key_FragSize[] = "FragSize";
static const char key_BufSize[] = "BufSize";

static const char key_ForceSongSpeed[] = "ForceSongSpeed";
static const char key_ClockSpeed[] = "ClockSpeed";
static const char key_MemoryMode[] = "MemoryMode";
static const char key_MOS8580[] = "MOS8580";
static const char key_MeasuredMasterVolume[] = "MeasuredMasterVolume";
static const char key_UseFilter[] = "UseFilter";
static const char key_FilterFs[] = "FilterFs";
static const char key_FilterFm[] = "FilterFm";
static const char key_FilterFt[] = "FilterFt";

static const char key_Mixer[] = "Mixer";
static const char key_AutoPanning[] = "AutoPanning";
static const char key_Voice1Vol[] = "Voice1Vol";
static const char key_Voice2Vol[] = "Voice2Vol";
static const char key_Voice3Vol[] = "Voice3Vol";
static const char key_Voice4Vol[] = "Voice4Vol";
static const char key_PanPosLeftHQ[] = "PanPosLeftHQ";
static const char key_PanPosRightHQ[] = "PanPosRightHQ";
static const char key_PanPosLeftFP[] = "PanPosLeftFP";
static const char key_PanPosRightFP[] = "PanPosRightFP";

static const char key_Yes[] = "Yes";
static const char key_No[] = "No";
static const char key_PAL[] = "PAL";
static const char key_NTSC[] = "NTSC";
static const char key_FullBankSwitching[] = "FullBankSwitching";
static const char key_TransparentROM[] = "TransparentROM";
static const char key_PlaySID[] = "PlaySID";
static const char key_Normal[] = "Normal";
static const char key_Centered[] = "Centered";
static const char key_StereoSurround[] = "StereoSurround";
static const char key_HQ[] = "HighQuality";
static const char key_FP[] = "FullPanning";

static const char key_Usage[] = "Usage";

static const char key_SongLenDB[] = "SongLenDB";
static const char key_PlaytimeDefault[] = "PlaytimeDefault";
static const char key_FadeOutDefault[] = "FadeOutDefault";

static const char key_Directory[] = "Directory";
static const char key_NameFilter[] = "NameFilter";

static const char key_HVSCroot[] = "HVSCroot";
static const char key_STILshowGlobal[] = "STILshowGlobal";
static const char key_STILshowEntries[] = "STILshowEntries";
static const char key_HVSCbugEntries[] = "HVSCbugEntries";
static const char key_STILallOrSong[] = "STILallOrSong";
static const char key_STILautoResize[] = "STILautoResize";
static const char key_STILmaxHeight[] = "STILmaxHeight";
static const char key_StilOpen[] = "StilOpen";  // case!

static const char key_MainPosX[] = "MainPosX";
static const char key_MainPosY[] = "MainPosY";
static const char key_MainExtX[] = "MainExtX";
static const char key_MainExtY[] = "MainExtY";
static const char key_PlaylistPosX[] = "PlaylistPosX";
static const char key_PlaylistPosY[] = "PlaylistPosY";
static const char key_PlaylistExtX[] = "PlaylistExtX";
static const char key_PlaylistExtY[] = "PlaylistExtY";
static const char key_DirPosX[] = "DirPosX";
static const char key_DirPosY[] = "DirPosY";
static const char key_DirExtX[] = "DirExtX";
static const char key_DirExtY[] = "DirExtY";
static const char key_DirOpen[] = "DirOpen";

ConfigFileOld::ConfigFileOld(const char* inFileName)
: TextFile(), fileName(inFileName)
{
}

// -------------------------------------------------------------------- Input

bool ConfigFileOld::load()
{
    open(fileName);
    while (status && isGood && !isEOF())  // line-by-line loop
    {
        getLine();
        // Skip blank and comment lines.
        while (status && !isEOF() && isBlank() || isComment())
        {
            getLine();
        };
        if ( isEOF() )
            break;
        // Evaluate line.
#ifdef XSID_WB_DEBUG
        cout << "Line " << getLineNum() << ", " << getLineLen() << ": ";
        cout << getLineBuf() << endl;
        cout << "ParseBuf: " << getParseBuf() << endl;
#endif
        
        // Audio settings.
        if (isKey(key_Frequency))
            myAudioConfig.frequency = atoi(getCurParseBuf());
        else if (isKey(key_Channels))
        {
            myAudioConfig.channels = atoi(getCurParseBuf());
        }
        else if (isKey(key_BitsPerSample))
        {
            myAudioConfig.precision = atoi(getCurParseBuf());
        }
        else if (isKey(key_SampleEncoding))
        {
            myAudioConfig.encoding = atoi(getCurParseBuf());
        }
        else if (isKey(key_BufSize))
        {
            myAudioConfig.bufSize = atoi(getCurParseBuf());
        }
        else if (isKey(key_MaxFrags))
        {
            myAudioConfig.maxFrags = atoi(getCurParseBuf());
        }
        else if (isKey(key_FragSize))
        {
            myAudioConfig.fragSize = atoi(getCurParseBuf());
        }
        
        // Emulator settings.
        else if (isKey(key_ClockSpeed))
        {
            if (isKey(key_PAL))
                myEmuConfig.clockSpeed = SIDTUNE_CLOCK_PAL;
            else
                myEmuConfig.clockSpeed = SIDTUNE_CLOCK_NTSC;
        }
        else if (isKey(key_ForceSongSpeed))
        {
            myEmuConfig.forceSongSpeed = isValue(key_Yes);
        }
        else if (isKey(key_MemoryMode))
        {
            if (isKey(key_FullBankSwitching,false))
                myEmuConfig.memoryMode = MPU_BANK_SWITCHING;
            else if (isKey(key_TransparentROM,false))
                myEmuConfig.memoryMode = MPU_TRANSPARENT_ROM;
            else
                myEmuConfig.memoryMode = MPU_PLAYSID_ENVIRONMENT;
        }
        else if (isKey(key_MOS8580))
        {
            myEmuConfig.mos8580 = isValue(key_Yes);
        }
        else if (isKey(key_MeasuredMasterVolume))
        {
            myEmuConfig.measuredVolume = isValue(key_Yes);
        }
        
        // Filter settings.
        else if (isKey(key_UseFilter))
            myEmuConfig.emulateFilter = isValue(key_Yes);
        else if (isKey(key_FilterFs))
            myEmuConfig.filterFs = atof(getCurParseBuf());
        else if (isKey(key_FilterFm))
            myEmuConfig.filterFm = atof(getCurParseBuf());
        else if (isKey(key_FilterFt))
            myEmuConfig.filterFt = atof(getCurParseBuf());
        
        // HVSC/STIL.
        else if (isKey(key_HVSCroot))
            myHVSC_Config.hvscRootPath.setPath(getCurLineBuf());
        else if (isKey(key_STILshowGlobal))
            myHVSC_Config.showGlobalComments = isValue(key_Yes);
        else if (isKey(key_STILshowEntries))
            myHVSC_Config.showStilEntries = isValue(key_Yes);
        else if (isKey(key_HVSCbugEntries))
            myHVSC_Config.showBugListEntries = isValue(key_Yes);
        else if (isKey(key_STILallOrSong))
        {
            myHVSC_Config.showForFile = isValue(key_Yes);
            myHVSC_Config.showForCurrentSong = !myHVSC_Config.showForFile;
        }
        else if (isKey(key_STILautoResize))
            myHVSC_Config.autoResize = isValue(key_Yes);
        else if (isKey(key_STILmaxHeight))
            myHVSC_Config.maxHeight = getCurParseBuf();  // QString copies
        
        else if (isKey(key_StilOpen))
        {
            myConfig.enableStilView = isValue(key_Yes);
        }
        
        // Songlength DB and playlist defaults.
        else if ( isKey(key_SongLenDB) )
            myConfig.enableSongLenDB = isValue(key_Yes);
        else if ( isKey(key_PlaytimeDefault) )
            myConfig.playtimeDef = atoi(getCurParseBuf());
        else if ( isKey(key_FadeOutDefault) )
            myConfig.fadeoutDef = atoi(getCurParseBuf());
        
        // Listening mileage.
        else if (isKey(key_Usage))
            myConfig.usage = atol(getCurParseBuf());
        
        // Mixer settings.
        else if (isKey(key_Mixer))
        {
            if (isKey(key_HQ,false))
                myEmuConfig.volumeControl = SIDEMU_VOLCONTROL;
            else if (isKey(key_FP,false))
                myEmuConfig.volumeControl = SIDEMU_FULLPANNING;
            else if (isKey(key_StereoSurround,false))
                myEmuConfig.volumeControl = SIDEMU_STEREOSURROUND;
            else
                myEmuConfig.volumeControl = SIDEMU_NONE;
        }
        else if (isKey(key_AutoPanning))
        {
            if (isKey(key_Centered))
                myEmuConfig.autoPanning = SIDEMU_CENTEREDAUTOPANNING;
            else
                myEmuConfig.autoPanning = SIDEMU_NONE;
        }
        else if (isKey(key_Voice1Vol))
            myConfig.myMixerConfig.volTotal[0] = atoi(getCurParseBuf());
        else if (isKey(key_Voice2Vol))
            myConfig.myMixerConfig.volTotal[1] = atoi(getCurParseBuf());
        else if (isKey(key_Voice3Vol))
            myConfig.myMixerConfig.volTotal[2] = atoi(getCurParseBuf());
        else if (isKey(key_Voice4Vol))
            myConfig.myMixerConfig.volTotal[3] = atoi(getCurParseBuf());
        
        else if (isKey(key_PanPosLeftHQ))
            myConfig.myMixerConfig.readMergedPanPosLeft(myConfig.myMixerConfig.volHQ,atol(getCurParseBuf()));
        else if (isKey(key_PanPosRightHQ))
            myConfig.myMixerConfig.readMergedPanPosRight(myConfig.myMixerConfig.volHQ,atol(getCurParseBuf()));
        else if (isKey(key_PanPosLeftFP))
            myConfig.myMixerConfig.readMergedPanPosLeft(myConfig.myMixerConfig.volFP,atol(getCurParseBuf()));
        else if (isKey(key_PanPosRightFP))
            myConfig.myMixerConfig.readMergedPanPosRight(myConfig.myMixerConfig.volFP,atol(getCurParseBuf()));
    
        // Window geometries.
        // Main dialog.
        else if (isKey(key_MainPosX))
            myConfig.geomMainDlg.x = atoi(getCurParseBuf());
        else if (isKey(key_MainPosY))
            myConfig.geomMainDlg.y = atoi(getCurParseBuf());
        else if (isKey(key_MainExtX))
            myConfig.geomMainDlg.w = atoi(getCurParseBuf());
        else if (isKey(key_MainExtY))
            myConfig.geomMainDlg.h = atoi(getCurParseBuf());
        // Playlist dialog.
        else if (isKey(key_PlaylistPosX))
            myConfig.geomPlaylistDlg.x = atoi(getCurParseBuf());
        else if (isKey(key_PlaylistPosY))
            myConfig.geomPlaylistDlg.y = atoi(getCurParseBuf());
        else if (isKey(key_PlaylistExtX))
            myConfig.geomPlaylistDlg.w = atoi(getCurParseBuf());
        else if (isKey(key_PlaylistExtY))
            myConfig.geomPlaylistDlg.h = atoi(getCurParseBuf());
        // File dialog.
        else if (isKey(key_DirPosX))
            myConfig.geomDirDlg.x = atoi(getCurParseBuf());
        else if (isKey(key_DirPosY))
            myConfig.geomDirDlg.y = atoi(getCurParseBuf());
        else if (isKey(key_DirExtX))
            myConfig.geomDirDlg.w = atoi(getCurParseBuf());
        else if (isKey(key_DirExtY))
            myConfig.geomDirDlg.h = atoi(getCurParseBuf());
        else if (isKey(key_DirOpen))
            myConfig.showDirDlg = isValue(key_Yes);
        // Sidtunes directory.
        else if (isKey(key_Directory))
            myConfig.currentDir = getCurLineBuf();  // QString copies
        else if (isKey(key_NameFilter))
            myConfig.nameFilter = getCurParseBuf();  // QString copies
        
    };

    (myAudioConfig.channels == 1) ? (myEmuConfig.channels = SIDEMU_MONO)
        : (myEmuConfig.channels = SIDEMU_STEREO);
    (myAudioConfig.precision == 8) ? (myEmuConfig.bitsPerSample = SIDEMU_8BIT)
        : (myEmuConfig.bitsPerSample = SIDEMU_16BIT);
    (myAudioConfig.encoding == 0x7f) ? (myEmuConfig.sampleFormat = SIDEMU_SIGNED_PCM)
        : (myEmuConfig.sampleFormat = SIDEMU_UNSIGNED_PCM);
    
    close();
    return isGood;
}

// ------------------------------------------------------------------- Output

void ConfigFileOld::writeBoolKey(ofstream& toFile, const char* key, const bool flag)
{
    toFile << key << '=';
    (flag) ? (toFile << key_Yes << endl) : (toFile << key_No << endl);
}

bool ConfigFileOld::save()
{
    bool wasSuccess = false;

#ifdef XSID_HAVE_IOS_BIN
    ofstream toFile(fileName,ios::out|ios::bin|ios::trunc);
#else
    ofstream toFile(fileName,ios::out|ios::binary|ios::trunc);
#endif
    if ( !toFile.fail() )
    {
        toFile 
            << "# ~/.sidplayrc" << endl
            << "#" << endl
            << "# SIDPLAY/X11 configuration file." << endl
            << "#" << endl
            << "# DO NOT edit manually." << endl
            << endl;

        // Sidtunes directory.
        writeKey(toFile,key_Directory) << myConfig.currentDir << endl;
        writeKey(toFile,key_NameFilter) << myConfig.nameFilter << endl;
        
        // HVSC/STIL.
        writeKey(toFile,key_HVSCroot) << myHVSC_Config.hvscRootPath.path() << endl;
        writeBoolKey(toFile,key_STILshowGlobal,myHVSC_Config.showGlobalComments);
        writeBoolKey(toFile,key_STILshowEntries,myHVSC_Config.showStilEntries);
        writeBoolKey(toFile,key_HVSCbugEntries,myHVSC_Config.showBugListEntries);
        writeBoolKey(toFile,key_STILallOrSong,myHVSC_Config.showForFile);
        writeBoolKey(toFile,key_STILautoResize,myHVSC_Config.autoResize);
        writeKey(toFile,key_STILmaxHeight) << myHVSC_Config.maxHeight << endl;

        writeBoolKey(toFile,key_StilOpen,myConfig.enableStilView);

        // Listening mileage.
        writeKey(toFile,key_Usage) << myConfig.usage << endl
            << endl;
        
        toFile << "# Audio quality settings." << endl;
        writeKey(toFile,key_Frequency) << myAudioConfig.frequency << endl;
        writeKey(toFile,key_Channels) << myAudioConfig.channels << endl;
        writeKey(toFile,key_BitsPerSample) << myAudioConfig.precision << endl;
        writeKey(toFile,key_SampleEncoding) << myAudioConfig.encoding << endl;
        writeKey(toFile,key_BufSize) << myAudioConfig.bufSize << endl;
        writeKey(toFile,key_MaxFrags) << myAudioConfig.maxFrags << endl;
        writeKey(toFile,key_FragSize) << myAudioConfig.fragSize << endl
            << endl;

        toFile << "# Emulator settings." << endl;

        writeKey(toFile,key_ClockSpeed);
        if (myEmuConfig.clockSpeed == SIDTUNE_CLOCK_PAL)
            toFile << key_PAL << endl;
        else  // if (myEmuConfig.clockSpeed == SIDTUNE_CLOCK_NTSC)
            toFile << key_NTSC << endl;

        writeBoolKey(toFile,key_ForceSongSpeed,myEmuConfig.forceSongSpeed);
        
        writeKey(toFile,key_MemoryMode);
        if (myEmuConfig.memoryMode == MPU_BANK_SWITCHING)
            toFile << key_FullBankSwitching << endl;
        else if (myEmuConfig.memoryMode == MPU_TRANSPARENT_ROM)
            toFile << key_TransparentROM << endl;
        else if (myEmuConfig.memoryMode == MPU_PLAYSID_ENVIRONMENT)
            toFile << key_PlaySID << endl;

        writeBoolKey(toFile,key_MOS8580,myEmuConfig.mos8580);
        writeBoolKey(toFile,key_MeasuredMasterVolume,myEmuConfig.measuredVolume);
        writeBoolKey(toFile,key_UseFilter,myEmuConfig.emulateFilter);
        writeKey(toFile,key_FilterFs) << myEmuConfig.filterFs << endl;
        writeKey(toFile,key_FilterFm) << myEmuConfig.filterFm << endl;
        writeKey(toFile,key_FilterFt) << myEmuConfig.filterFt << endl
            << endl;

        toFile << "# Mixer settings." << endl;

        writeKey(toFile,key_Mixer);
        if (myEmuConfig.volumeControl == SIDEMU_VOLCONTROL)
            toFile << key_HQ << endl;
        else if (myEmuConfig.volumeControl == SIDEMU_FULLPANNING)
            toFile << key_FP << endl;
        else if (myEmuConfig.volumeControl == SIDEMU_STEREOSURROUND)
            toFile << key_StereoSurround << endl;
        else
            toFile << key_No << endl;
            
        writeKey(toFile,key_AutoPanning);
        if (myEmuConfig.autoPanning == SIDEMU_CENTEREDAUTOPANNING)
            toFile << key_Centered;
        else
            toFile << key_No;
        toFile << endl;
        
        writeKey(toFile,key_Voice1Vol) << (int)myConfig.myMixerConfig.volTotal[0] << endl;
        writeKey(toFile,key_Voice2Vol) << (int)myConfig.myMixerConfig.volTotal[1] << endl;
        writeKey(toFile,key_Voice3Vol) << (int)myConfig.myMixerConfig.volTotal[2] << endl;
        writeKey(toFile,key_Voice4Vol) << (int)myConfig.myMixerConfig.volTotal[3] << endl;

        writeKey(toFile,key_PanPosLeftHQ) 
            << myConfig.myMixerConfig.mergePanPosLeft(myConfig.myMixerConfig.volHQ) << endl;
        writeKey(toFile,key_PanPosRightHQ)
            << myConfig.myMixerConfig.mergePanPosRight(myConfig.myMixerConfig.volHQ) << endl;
        writeKey(toFile,key_PanPosLeftFP)
            << myConfig.myMixerConfig.mergePanPosLeft(myConfig.myMixerConfig.volFP) << endl;
        writeKey(toFile,key_PanPosRightFP)
            << myConfig.myMixerConfig.mergePanPosRight(myConfig.myMixerConfig.volFP) << endl;
        toFile << endl;

        toFile << "# Playlist defaults." << endl;
        writeBoolKey(toFile,key_SongLenDB,myConfig.enableSongLenDB);
        writeKey(toFile,key_PlaytimeDefault) << myConfig.playtimeDef << endl;
        writeKey(toFile,key_FadeOutDefault) << myConfig.fadeoutDef << endl
            << endl;
        
        toFile << "# Window geometries." << endl;
        writeKey(toFile,key_MainPosX) << myConfig.geomMainDlg.x << endl;
        writeKey(toFile,key_MainPosY) << myConfig.geomMainDlg.y << endl;
        writeKey(toFile,key_MainExtX) << myConfig.geomMainDlg.w << endl;
        writeKey(toFile,key_MainExtY) << myConfig.geomMainDlg.h << endl;
        writeKey(toFile,key_PlaylistPosX) << myConfig.geomPlaylistDlg.x << endl;
        writeKey(toFile,key_PlaylistPosY) << myConfig.geomPlaylistDlg.y << endl;
        writeKey(toFile,key_PlaylistExtX) << myConfig.geomPlaylistDlg.w << endl;
        writeKey(toFile,key_PlaylistExtY) << myConfig.geomPlaylistDlg.h << endl;
        writeKey(toFile,key_DirPosX) << myConfig.geomDirDlg.x << endl;
        writeKey(toFile,key_DirPosY) << myConfig.geomDirDlg.y << endl;
        writeKey(toFile,key_DirExtX) << myConfig.geomDirDlg.w << endl;
        writeKey(toFile,key_DirExtY) << myConfig.geomDirDlg.h << endl;
        writeBoolKey(toFile,key_DirOpen,myConfig.showDirDlg);
        toFile << endl;
        
        toFile.close();
        wasSuccess = !toFile.fail();
    }
    
    return wasSuccess;
}

// --------------------------------------------------------------- Conversion

void ConfigFileOld::convertToIni(ConfigFile* pi)
{
    // HVSC/STIL
    pi->setHVSC_Config(myHVSC_Config);
    
    // Selector
    pi->setGroup(ConfigC::groups[ConfigC::Selector]);
    pi->setEntry(ConfigC::keys[ConfigC::Dir],myConfig.currentDir);
    pi->setEntry(ConfigC::keys[ConfigC::NameFilter],myConfig.nameFilter);
    pi->setEntry(ConfigC::keys[ConfigC::X],myConfig.geomDirDlg.x);
    pi->setEntry(ConfigC::keys[ConfigC::Y],myConfig.geomDirDlg.y);
    pi->setEntry(ConfigC::keys[ConfigC::W],myConfig.geomDirDlg.w);
    pi->setEntry(ConfigC::keys[ConfigC::H],myConfig.geomDirDlg.h);
    pi->setEntry(ConfigC::keys[ConfigC::Open],myConfig.showDirDlg);
        
    // Misc
    pi->setGroup(ConfigC::groups[ConfigC::Misc]);
    pi->setEntry(ConfigC::keys[ConfigC::ConfigVersion],ConfigC::version);
    pi->setEntry(ConfigC::keys[ConfigC::Usage],myConfig.usage);
    pi->setEntry(ConfigC::keys[ConfigC::STIL_Viewer],myConfig.enableStilView);
        
    // Audio
    pi->setAudioConfig(myAudioConfig);

    // Emulation
    pi->setEmuConfig(myEmuConfig);

    // Mixer
    pi->setGroup(ConfigC::groups[ConfigC::Mixer]);
    pi->setEntry(ConfigC::keys[ConfigC::Mixer],myEmuConfig.volumeControl);
    pi->setEntry(ConfigC::keys[ConfigC::AutoPanning],myEmuConfig.autoPanning);
    pi->setEntry(ConfigC::keys[ConfigC::Voice1Vol],(int)myConfig.myMixerConfig.volTotal[0]);
    pi->setEntry(ConfigC::keys[ConfigC::Voice2Vol],(int)myConfig.myMixerConfig.volTotal[1]);
    pi->setEntry(ConfigC::keys[ConfigC::Voice3Vol],(int)myConfig.myMixerConfig.volTotal[2]);
    pi->setEntry(ConfigC::keys[ConfigC::Voice4Vol],(int)myConfig.myMixerConfig.volTotal[3]);
    pi->setEntry(ConfigC::keys[ConfigC::PanPosLeftHQ],myConfig.myMixerConfig.mergePanPosLeft(myConfig.myMixerConfig.volHQ));
    pi->setEntry(ConfigC::keys[ConfigC::PanPosRightHQ],myConfig.myMixerConfig.mergePanPosRight(myConfig.myMixerConfig.volHQ));
    pi->setEntry(ConfigC::keys[ConfigC::PanPosLeftFP],myConfig.myMixerConfig.mergePanPosLeft(myConfig.myMixerConfig.volFP));
    pi->setEntry(ConfigC::keys[ConfigC::PanPosRightFP],myConfig.myMixerConfig.mergePanPosRight(myConfig.myMixerConfig.volFP));

    // Playlist
    pi->setGroup(ConfigC::groups[ConfigC::Playlist]);
    pi->setEntry(ConfigC::keys[ConfigC::SongLenDB],myConfig.enableSongLenDB);
    pi->setEntry(ConfigC::keys[ConfigC::PlaytimeDefault],myConfig.playtimeDef);
    pi->setEntry(ConfigC::keys[ConfigC::FadeOutDefault],myConfig.fadeoutDef);
    pi->setEntry(ConfigC::keys[ConfigC::X],myConfig.geomPlaylistDlg.x);
    pi->setEntry(ConfigC::keys[ConfigC::Y],myConfig.geomPlaylistDlg.y);
    pi->setEntry(ConfigC::keys[ConfigC::W],myConfig.geomPlaylistDlg.w);
    pi->setEntry(ConfigC::keys[ConfigC::H],myConfig.geomPlaylistDlg.h);

    // Main Window
    pi->setGroup(ConfigC::groups[ConfigC::MainWindow]);
    pi->setEntry(ConfigC::keys[ConfigC::X],myConfig.geomMainDlg.x);
    pi->setEntry(ConfigC::keys[ConfigC::Y],myConfig.geomMainDlg.y);
    pi->setEntry(ConfigC::keys[ConfigC::W],myConfig.geomMainDlg.w);
    pi->setEntry(ConfigC::keys[ConfigC::H],myConfig.geomMainDlg.h);
}

