/* $Header: /fridge/cvs/xscorch/sgame/soptions.c,v 1.5 2001/04/08 01:18:31 justins Exp $ */
/*
   
   xscorch - soptions.c       Copyright(c) 2001,2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/

   Scorched option processing
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <soptions.h>
#include <sconfig.h>
#include <scffile.h>
#include <sweapon.h>
#include <sland.h>
#include <snet/snet.h>
#include <sutil/sstr.h>
#include <sutil/sreg.h>


#define  SC_OPTION_BUFFER        0x100

#define  SC_OPTION_CONFIG_FILE   0x1001
#define  SC_OPTION_INSANITY      0x1002
#define  SC_OPTION_YIELDS        0x1003
#define  SC_OPTION_HQMIXER       0x1004
#define  SC_OPTION_NO_HQMIXER    0x1005

#define  SC_OPTION_NAME          0x2001
#define  SC_OPTION_PORT          0x2002
#define  SC_OPTION_SERVER        0x2003
#define  SC_OPTION_CLIENT        0x2004



static struct option _sc_long_options[] = {
 { "config", 1, 0, SC_OPTION_CONFIG_FILE },
 { "geometry", 1, 0, 'g' },
 { "help", 0, 0, 'h' },
 { "insanity", 0, 0, SC_OPTION_INSANITY },
 { "yields", 0, 0, SC_OPTION_YIELDS },
 { "sound", 0, 0, 'S' },
 { "nosound", 0, 0, 's' },
 { "hqmixer", 0, 0, SC_OPTION_HQMIXER },
 { "nohqmixer", 0, 0, SC_OPTION_NO_HQMIXER },
#if USE_NETWORK
 { "name", 1, 0, SC_OPTION_NAME },
 { "port", 1, 0, SC_OPTION_PORT },
 { "server", 0, 0, SC_OPTION_SERVER },
 { "client", 1, 0, SC_OPTION_CLIENT },
#endif /* Network? */
 { NULL, 0, 0, 0 }
};



static bool _sc_options_x_y(char *arg, int *x, int *y) {

   char *p = arg;

   while(*p != '\0' && *p != 'x' && *p != 'X') ++p;
   if(*p == '\0') return(false);
   
   *p = '\0';
   *x = atoi(arg);
   *y = atoi(p + 1);
   return(true);
   
}



int sc_options_usage(const char *progname) {

   printf("Usage:  %s [options]\n\n", progname);
   
   printf("--config=file       Load an alternate config file\n");
   printf("--geometry=nxm, -g  Set geometry to width n, height m\n");
   printf("--help              Display this help\n");
   printf("--insanity          If you have to ask ...\n");
   printf("--yields            Display weapon yields\n");
   printf("--sound, -S         Enable music and sound effects\n");
   printf("--nosound, -s       Disable music and sound effects\n");
   printf("--hqmixer           Enable use of high-quality mixer\n");
   printf("--nohqmixer         Disable use of high-quality mixer\n");
   #if USE_NETWORK
      printf("--name=playername   Player name to use in network connection\n");
      printf("--port=number       Network port to use (client or server)\n");
      printf("--client=servername Connect as a client to specified server\n");
      printf("--server            Start a new server\n");
   #endif /* Network? */

   printf("\n");
   return(1);

}



int sc_options_parse(sc_config *c, int argc, char **argv) {

  #if USE_NETWORK
   char name[SC_OPTION_BUFFER];
   char server[SC_OPTION_BUFFER];
   bool isserver = false;
   bool isclient = false;
   int port;
  #endif /* Net? */
   int ch;

   /* Set network default options */   
   #if USE_NETWORK
      strncpyn(name, getenv("USER"), sizeof(name));
      strncpyn(server, SC_NET_DEFAULT_SERVER, sizeof(name));
      port = SC_NET_DEFAULT_PORT;
   #endif /* Net? */

   while(EOF != (ch = getopt_long(argc, argv, "gsS:h", _sc_long_options, NULL))) switch(ch) {
      case SC_OPTION_CONFIG_FILE:
         /* New config file to parse in! */
         reg_set_name(c->cfreg, optarg);
         if(!sc_config_file_load(c)) {
            fprintf(stderr, "Failed to load alternate configuration file \"%s\".\n", optarg);
         }
         break;
   
      case SC_OPTION_INSANITY:
         /* If you are reading this, then umm... wow. */
         /* You'll find no help here, why do you want to know? */
         /* This is where mechanical excellence, one-thousand 
            four-hundred horsepower pays off.  */
         fprintf(stderr, "Warning:  The insanity subsystem is enabled.\n\n");
         fprintf(stderr, "   Life... Dreams... Hope...\n");
         fprintf(stderr, "   Where'd they come from...\n");
         fprintf(stderr, "   And where are they headed?\n");
         fprintf(stderr, "   These things... I am going to destroy!\n");
         fprintf(stderr, "    -- Kefka, Fourth Tier, Final Fantasy VI\n\n");
            
         c->insanity = true;
         break;

      case SC_OPTION_YIELDS:
         sc_weapon_print_yields(c->weapons);
         return(1);
        
      case 'g':
         if(_sc_options_x_y(optarg, &c->fieldwidth, &c->fieldheight)) {
            if(c->fieldwidth < SC_MIN_FIELD_WIDTH || c->fieldwidth > SC_MAX_FIELD_WIDTH) {
               fprintf(stderr, "--geometry, width given, %d, out of range [%d, %d]\n", c->fieldwidth, SC_MIN_FIELD_WIDTH, SC_MAX_FIELD_WIDTH);
               return(1);
            } else if(c->fieldheight < SC_MIN_FIELD_HEIGHT || c->fieldheight > SC_MAX_FIELD_HEIGHT) {
               fprintf(stderr, "--geometry, height given, %d, out of range [%d, %d]\n", c->fieldheight, SC_MIN_FIELD_HEIGHT, SC_MAX_FIELD_HEIGHT);
               return(1);
            }
            if(c->fieldheight < c->maxheight) {
               c->maxheight = c->fieldheight - 32;
            }
            sc_land_setup(c->land, c->fieldwidth, c->fieldheight, sc_land_flags(c));
         } else {
            fprintf(stderr, "--geometry requires an option of form <width>x<height>, e.g. \"640x480\"\n");
            return(1);
         }
         break;
                     
      case 'h':
         return(sc_options_usage(argv[0]));
         
      case 'S':   
         c->enablesound = true;
         break;

      case 's':
         c->enablesound = false;
         break;

      case SC_OPTION_HQMIXER:
         c->usehqmixer = true;
         break;

      case SC_OPTION_NO_HQMIXER:
         c->usehqmixer = false;
         break;

      #if USE_NETWORK
         case SC_OPTION_NAME:
            strncpyn(name, optarg, sizeof(name));
            break;
         case SC_OPTION_PORT:
            port = atoi(optarg);
            break;
         case SC_OPTION_SERVER:
            isserver = true;
            break;
         case SC_OPTION_CLIENT:
            isclient = true;
            strncpyn(server, optarg, sizeof(server));
            break;
      #endif /* Network? */

      case ':':
      case '?':
      default:
         return(sc_options_usage(argv[0]));
         
   }
   
   /* Running a server or client connection? */
   #if USE_NETWORK
      if(isserver) {
         c->server = sc_net_server_new(c, port);
         if(c->server != NULL) isclient = true;
      }
      if(isclient) {
         c->client = sc_net_client_new(name, server, port);
      }
   #endif /* Network? */
   
   return(0);

}



