/* $Header: /fridge/cvs/xscorch/sgame/saddconf.c,v 1.21 2001/07/08 22:38:17 jacob Exp $ */
/*

   xscorch - saddconf.c       Copyright(c) 2001 Jacob Luna Lundberg
   jacob(at)chaos2.org        http://chaos2.org/~jacob

   Scorched read/append config (weapons and accessories)


   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <sweapon.h>
#include <sphoenix.h>
#include <saddconf.h>
#include <saccessory.h>
#include <sinventory.h>
#include <sutil/sreg.h>
#include <sutil/sstr.h>
#include <sutil/shash.h>
#include <sutil/sslink.h>



sc_inventory_info *_sc_addconf_append_item(sc_inventory_list *il, sc_inventory_list_item *item) {
/* _sc_addconf_append_item
   Local method to tack a new item def onto the item_list in il. */

   int hashindex;

   if(il == NULL || item == NULL) {
      free(item);
      return(NULL);
   }

   /* Add the def to the dll */
   if(il->defs_head == NULL) {
      /* Empty list */
      il->defs_tail = il->defs_head = item;
      il->defs_head->item->ident = 0;
   } else {
      /* Append to list */
      il->defs_tail->next = item;
      il->defs_tail->next->item->ident = il->defs_tail->item->ident + 1;
      il->defs_tail->next->prev = il->defs_tail;
      il->defs_tail = il->defs_tail->next;
   }

   /* Add the def to the hash chains */
   hashindex = shash(SC_INVENTORY_HASH_BITS, il->defs_tail->item->ident);
   if(il->defs_hash[hashindex] == NULL) {
      /* Nothing on hash chain */
      il->defs_hash[hashindex] = il->defs_tail;
   } else {
      /* Already an item on the hash chain */
      il->defs_tail->next = il->defs_hash[hashindex];
      while(il->defs_tail->next->chain != NULL)
         il->defs_tail->next = il->defs_tail->next->chain;
      il->defs_tail->next->chain = il->defs_tail;
      il->defs_tail->next = NULL;
   }

   /* Only now are we truely the winners */
   return(item->item);

}



static const reg_class_data _reg_weapons_class[] = {
   { "weaponName",              REG_STRING,     NULL },
   { "description",             REG_STRING,     NULL },
   { "armsLevel",               REG_INTEGER,    NULL },
   { "price",                   REG_INTEGER,    NULL },
   { "bundle",                  REG_INTEGER,    NULL },
   { "radius",                  REG_INTEGER,    NULL },
   { "force",                   REG_INTEGER,    NULL },
   { "liquid",                  REG_INTEGER,    NULL },
   { "scatter",                 REG_INTEGER,    NULL },
   { "children",                REG_INTEGER,    NULL },
   { "useless",                 REG_BOOLEAN,    NULL },
   { "indirect",                REG_BOOLEAN,    NULL },
   { "stateFlags",              REG_STRING,     NULL },  /* bit field */
   { "phoenixFlags",            REG_STRING,     NULL },  /* bit field */
   { "phoenixChild",            REG_STRING,     NULL },
   { 0, 0, 0 }
};

static const reg_class_list _reg_class_weapons_parent[] = {
   { "weapons_class",           _reg_weapons_class },
   { 0, 0 }
};



bool sc_weapon_append_file(sc_weapon_config *wc, const char *filename) {
/* sc_weapon_append_file
   Append weapons to the weapons registries.  Returns true on success. */

   int iterator;
   int count = 0;
   sc_weapon_info *info;
   sc_weapon_info *child;
   sc_weapon_list *wl = wc->weaponlist;
   char childname[SC_INVENTORY_MAX_NAME_LEN];
   char desc[SC_INVENTORY_MAX_DESC_LEN];
   char name[SC_INVENTORY_MAX_NAME_LEN];
   reg_var *weapon;
   reg_var *top;
   reg *readr;

   /* Read the file into the registry */
   readr = reg_new(filename);
   if(readr == NULL) return(false);
   reg_class_register_list(readr, _reg_class_weapons_parent);
   if(!reg_load(readr)) {
      /* Failed to read file; signal error */
      printf("saddconf - aborting, weapons def file unreadable\n");
      reg_free(&readr);
      return(false);
   }

   /* Read the variables in order */
   top = reg_get_top(readr);
   weapon = reg_get_block_head(top);
   while(weapon != NULL) {
      /* Check the class name */
      if(strcmpn(reg_get_var_class(weapon), "weapons_class")) {
         /* Get the weapon's name string. */
         if(!reg_get_string(readr, weapon, "weaponName", name, SC_INVENTORY_MAX_NAME_LEN) || name[0] == '\0') {
            printf("saddconf - ignoring attempt to add unnamed weapon\n");
            continue;
         }
         /* Check the name for duplication */
         if(sc_weapon_lookup_by_name(wc, name, SC_WEAPON_LIMIT_NONE)) {
            printf("saddconf - ignoring attempt to add weapon \"%s\" - duplicate by name\n", name);
            continue;
         }
         /* addconf_append will set the weapon's ident for us */
         info = (sc_weapon_info *)_sc_addconf_append_item((sc_inventory_list *)wl,
                (sc_inventory_list_item *)sc_weapon_list_item_new());
         if(info == NULL) {
            printf("saddconf - aborting, weapon list append failed, likely malloc error\n");
            break;
         } else {
            info->name = (char *)malloc((strnlen(name, SC_INVENTORY_MAX_NAME_LEN - 1) + 1) * sizeof(char));
            if(info->name == NULL) {
               printf("saddconf - aborting, failed to allocate memory for weapon name\n");
               /* TEMP PAINFUL - We need list removal here! - JL */
               break;
            }
            strncpyn(info->name, name, strnlen(name, SC_INVENTORY_MAX_NAME_LEN - 1) + 1);
         }

         /* Set in the other fields in the weapon_info struct */
         reg_get_integer(readr, weapon, "armsLevel", &info->armslevel);
         reg_get_integer(readr, weapon, "price",     &info->price);
         reg_get_integer(readr, weapon, "bundle",    &info->bundle);
         reg_get_integer(readr, weapon, "radius",    &info->radius);
         reg_get_integer(readr, weapon, "force",     &info->force);
         reg_get_integer(readr, weapon, "liquid",    &info->liquid);
         reg_get_integer(readr, weapon, "scatter",   &info->scatter);
         reg_get_integer(readr, weapon, "children",  &info->children);

         reg_get_boolean(readr, weapon, "useless",   &info->useless);
         reg_get_boolean(readr, weapon, "indirect",  &info->indirect);

         info->state = reg_get_bitmask_from_values(readr, weapon, "stateFlags",   0,
                                                   sc_weapon_state_bit_names(),
                                                   sc_weapon_state_bit_items());
         info->ph_fl = reg_get_bitmask_from_values(readr, weapon, "phoenixFlags", 0,
                                                   sc_phoenix_flags_bit_names(),
                                                   sc_phoenix_flags_bit_items());

         /* Read in the weapon description if there is one */
         if(!reg_get_string(readr, weapon, "description", desc, SC_INVENTORY_MAX_DESC_LEN) || desc[0] == '\0') {
            info->description = NULL;
         } else {
            info->description = (char *)malloc((strnlen(desc, SC_INVENTORY_MAX_DESC_LEN - 1) + 1) * sizeof(char));
            strncpyn(info->description, desc, strnlen(desc, SC_INVENTORY_MAX_DESC_LEN - 1) + 1);
         }

         /* Set the child if there is one */
         childname[0] = '\0';
         reg_get_string(readr, weapon, "phoenixChild", childname, SC_INVENTORY_MAX_NAME_LEN);
         if(childname[0] != '\0') {
            child = sc_weapon_lookup_by_name(wc, childname, SC_WEAPON_LIMIT_NONE);
            if(child == NULL) {
               printf("saddconf - warning, \"%s\" has missing child \"%s\"\n", name, childname);
            } else {
               info->ph_ch = child->ident;
            }
         }

         /* Test for weapon acceptance */
         if(SC_WEAPON_IS_PHOENIX(info) && !sc_phoenix_verify(wc, info)) {
            printf("saddconf - \"%s\" is an invalid phoenix weapon, killing it\n", name);
            /* TEMP - Well, ok, so we don't kill it yet; there's no list removal - JL */
            info->indirect = true;
            continue;
         }

         /* Clean up minor details such as player inventories. */
         for(iterator = 0; iterator < SC_MAX_PLAYERS; ++iterator) {
            if(SC_WEAPON_IS_INFINITE(info))
               info->inventories[iterator] = SC_INVENTORY_MAX_ITEMS;
            else
               info->inventories[iterator] = 0;
         }

         /* Signal we've completed an addition. */
         ++count;
      } else {
         /* Bad class given; signal the error. */
         printf("saddconf - reading weapon def file, expected weapons_class but didn't get it\n");
      }

      /* Get the next variable */
      weapon = reg_get_next_var(weapon);
   }

   /* Release the registry */   
   reg_free(&readr);

   /* Let the next layer up know whether we found some weapons */
   if(count)
      return(true);
   else
      return(false);

}



static const reg_class_data _reg_accessories_class[] = {
   { "accessoryName",           REG_STRING,     NULL },
   { "description",             REG_STRING,     NULL },
   { "armsLevel",               REG_INTEGER,    NULL },
   { "price",                   REG_INTEGER,    NULL },
   { "bundle",                  REG_INTEGER,    NULL },
   { "shielding",               REG_INTEGER,    NULL },
   { "repulsion",               REG_INTEGER,    NULL },
   { "fuel",                    REG_INTEGER,    NULL },
   { "useless",                 REG_BOOLEAN,    NULL },
   { "indirect",                REG_BOOLEAN,    NULL },
   { "stateFlags",              REG_STRING,     NULL },  /* bit field */
   { 0, 0, 0 }
};

static const reg_class_list _reg_class_accessories_parent[] = {
   { "accessories_class",           _reg_accessories_class },
   { 0, 0 }
};



bool sc_accessory_append_file(sc_accessory_config *ac, const char *filename) {
/* sc_accessory_append_file
   Append accessories to the accessories registries.  Returns true on success. */

   int iterator;
   int count = 0;
   sc_accessory_info *info;
   sc_accessory_list *al = ac->accessorylist;
   char desc[SC_INVENTORY_MAX_DESC_LEN];
   char name[SC_INVENTORY_MAX_NAME_LEN];
   reg_var *accessory;
   reg_var *top;
   reg *readr;

   /* Read the file into the registry */
   readr = reg_new(filename);
   if(readr == NULL) return(false);
   reg_class_register_list(readr, _reg_class_accessories_parent);
   if(!reg_load(readr)) {
      /* Failed to read file; signal error */
      printf("saddconf - aborting, accessories def file unreadable\n");
      reg_free(&readr);
      return(false);
   }

   /* Read the variables in order */
   top = reg_get_top(readr);
   accessory = reg_get_block_head(top);
   while(accessory != NULL) {
      /* Check the class name */
      if(strcmpn(reg_get_var_class(accessory), "accessories_class")) {
         /* Get the accessory's name string. */
         if(!reg_get_string(readr, accessory, "accessoryName", name, SC_INVENTORY_MAX_NAME_LEN) || name[0] == '\0') {
            printf("saddconf - ignoring attempt to add unnamed accessory\n");
            continue;
         }
         /* Check the name for duplication */
         if(sc_accessory_lookup_by_name(ac, name, SC_ACCESSORY_LIMIT_NONE)) {
            printf("saddconf - ignoring attempt to add accessory \"%s\" - duplicate by name\n", name);
            continue;
         }
         /* addconf_append will set the accessory's ident for us */
         info = (sc_accessory_info *)_sc_addconf_append_item((sc_inventory_list *)al,
                (sc_inventory_list_item *)sc_accessory_list_item_new());
         if(info == NULL) {
            printf("saddconf - aborting, accessory list append failed, likely malloc error\n");
            free(name);
            break;
         } else {
            info->name = (char *)malloc((strnlen(name, SC_INVENTORY_MAX_NAME_LEN - 1) + 1) * sizeof(char));
            if(info->name == NULL) {
               printf("saddconf - aborting, failed to allocate memory for accessory name\n");
               /* TEMP PAINFUL - We need list removal here! - JL */
               break;
            }
            strncpyn(info->name, name, strnlen(name, SC_INVENTORY_MAX_NAME_LEN - 1) + 1);
         }

         /* Set in the other fields in the accessory_info struct */
         reg_get_integer(readr, accessory, "armsLevel", &info->armslevel);
         reg_get_integer(readr, accessory, "price",     &info->price);
         reg_get_integer(readr, accessory, "bundle",    &info->bundle);
         reg_get_integer(readr, accessory, "shielding", &info->shield);
         reg_get_integer(readr, accessory, "repulsion", &info->repulsion);
         reg_get_integer(readr, accessory, "fuel",      &info->fuel);

         reg_get_boolean(readr, accessory, "useless",   &info->useless);
         reg_get_boolean(readr, accessory, "indirect",  &info->indirect);

         info->state = reg_get_bitmask_from_values(readr, accessory, "stateFlags",   0,
                                                   sc_accessory_state_bit_names(),
                                                   sc_accessory_state_bit_items());

         /* Read in the accessory description if there is one */
         if(!reg_get_string(readr, accessory, "description", desc, SC_INVENTORY_MAX_DESC_LEN) || desc[0] == '\0') {
            info->description = NULL;
         } else {
            info->description = (char *)malloc((strnlen(desc, SC_INVENTORY_MAX_DESC_LEN - 1) + 1) * sizeof(char));
            strncpyn(info->description, desc, strnlen(desc, SC_INVENTORY_MAX_DESC_LEN - 1) + 1);
         }

         /* Clean up minor details such as player inventories. */
         for(iterator = 0; iterator < SC_MAX_PLAYERS; ++iterator) {
            if(SC_ACCESSORY_IS_INFINITE(info))
               info->inventories[iterator] = SC_INVENTORY_MAX_ITEMS;
            else
               info->inventories[iterator] = 0;
         }

         /* Signal we've completed an addition. */
         ++count;
      } else {
         /* Bad class given; signal the error. */
         printf("saddconf - reading accessory def file, expected accessory_class but didn't get it\n");
      }

      /* Get the next variable */
      accessory = reg_get_next_var(accessory);
   }

   /* Release the registry */   
   reg_free(&readr);

   /* Let the next layer up know whether we found some accessories */
   if(count)
      return(true);
   else
      return(false);

}
