/* $Header: /fridge/cvs/xscorch/sai/saiturn.c,v 1.16 2001/10/30 00:47:20 justins Exp $ */
/*
   
   xscorch - saiturn.c        Copyright(c) 2001,2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/
    
   AI turn code
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <saiint.h>              /* Main internal AI header */
#include <sgame/sinventory.h>    /* Need inv count (weapon selection) */
#include <sgame/stankpro.h>      /* Need tank radius data */
#include <sgame/splayer.h>       /* Need access to player functions */
#include <sgame/sshield.h>       /* Raising/lowering shields */
#include <sgame/sweapon.h>       /* How many weapons are there? */
#include <sgame/sconfig.h>       /* Dereference config for ph ptr */
#include <sgame/swindow.h>       /* Needed for the status display */
#include <sgame/sland.h>         /* Land's line-of-sight function */
#include <sutil/srand.h>         /* Needed for random trajectory */


/*

   Not you again.  
   *sigh*
      
   Ed:     Do you own a video camera?
   Renee:  No, Fred hates them.
   Fred:   I like to remember things my own way.
   Ed:     What do you mean by that?
   Fred:   How I remember them, not necessarily the way they happened.
   
   The whole key is right there.  It's just a matter of applying it

*/



typedef int (*trajfn)(const struct _sc_config *c, sc_player *p, const sc_player *vp);



static void _sc_ai_select_last_weapon(const sc_config *c, sc_player *p) {
/* sc_ai_select_last_weapon
   Selects the last (highest-index) weapon available to the AI.  The naive
   AI's make the assumption that weapons further down the inventory list
   must clearly be better.  */

   sc_weapon_info *info = sc_weapon_last(c->weapons, SC_WEAPON_LIMIT_ALL);
   int count = sc_weapon_count(c->weapons, SC_WEAPON_LIMIT_ALL);

   for(; count > 0; --count) {
      if(info->inventories[p->index] > 0) {
         /* We found a weapon.. */
         #if SC_AI_DEBUG_SELECT
            printf("   AI %s selected weapon \"%s\"\n", sc_ai_name(p->aitype), info->name);
         #endif /* Debug weapon selection? */
         sc_player_set_weapon(c, p, info);
         return;
      } /* Do we have this weapon? */
   } /* Looking for a weapon... */

   /* Well, heh, if we get here we have no weapons -- should be impossible. */
   #if SC_AI_DEBUG_SELECT
      printf("   AI %s has no weapons to fire!\n", sc_ai_name(p->aitype));
   #endif /* Debug weapon selection? */

}



static void _sc_ai_select_weapon_by_score(const sc_config *c, sc_player *p) {
/* sc_ai_select_weapon_by_score
   Selects the highest-ranked weapon available to the AI.  This is slightly
   better, but we might end up selecting a very powerful weapon to kill only
   a single tank.  */

   sc_weapon_info *info;        /* Weapon tracking */
   sc_weapon_info *maxinfo;     /* Weapon index of best weapon */
   sc_weapon_info *nextinfo;    /* Weapon index of second-best */
   int maxscore;     /* Score of best weapon so far */
   int nextscore;    /* Score of second best weapon */
   int score;        /* Score of current weapon */
   int count;        /* Iterator */

   /* Setup and iterate */   
   count = sc_weapon_count(c->weapons, SC_WEAPON_LIMIT_ALL);
   info = sc_weapon_last(c->weapons, SC_WEAPON_LIMIT_ALL);
   maxinfo = NULL;
   nextinfo= NULL;
   maxscore = 0;
   nextscore= -1;

   for(; count > 0; --count) {
      if(info->inventories[p->index] > 0) {
         /* We have some of this weapon; is it ideal? */
         score = sc_weapon_statistic(c->weapons, info, p, SC_WEAPON_STAT_YIELD);
         if(score > maxscore) {
            /* This weapon is ideal (so far) */
            nextinfo = maxinfo;
            nextscore= maxscore;
            maxinfo  = info;
            maxscore = score;
         } else if(score > nextscore) {
            /* weapon is second-best (so far) */
            nextinfo = info;
            nextscore= score;
         } /* Does this weapon have a better score? */
      } /* Do we have this weapon in inventory? */
      info = sc_weapon_prev(c->weapons, info, SC_WEAPON_LIMIT_ALL);
   } /* Loop through all weapons. */

   /* Set new weapon to one of the "best" found. */
   if(nextinfo != NULL && game_drand() < (double)nextscore / (nextscore + maxscore)) {
      /* Probabilistically select "next best" */
      maxinfo = nextinfo;
   }

   /* We have only one option now */
   if(maxinfo != NULL) {
      sc_player_set_weapon(c, p, maxinfo);

   #if SC_AI_DEBUG_SELECT
      printf("   AI %s selected weapon \"%s\"\n", sc_ai_name(p->aitype), maxinfo->name);
   } else {
      printf("   AI %s has no weapons to fire!\n", sc_ai_name(p->aitype));
   #endif /* Debug weapon selection? */

   }

}



static void _sc_ai_random_fire(const sc_config *c, sc_player *p) {
/* sc_ai_random_fire
   Fires a random trajectory and power.  This is the simplest form of AI;
   currently, MORON uses it exclusively, and other AI's may use this
   mechanism as a starter or a fallback targetting system.  */

   /* Mark Anderson corrected this in 20011029015201.A3951@www.muking.org */
   int min_power   = MAX(SC_AI_MORON_MIN_POWER, p->power - SC_AI_POWER_DELTA_MAX);
   int max_power   = MAX(SC_AI_MORON_MIN_POWER, p->power + SC_AI_POWER_DELTA_MAX);
   int power_range = max_power - min_power;
   int power       = min_power + game_lrand(power_range + 1);

   #ifdef __TEMP_OLD_CODE
   int power;        /* New power level */
   do {  /* Find a power that meets our minimum standards. */
      power = p->power + game_lrand(SC_AI_POWER_DELTA_MAX * 2 + 1) - SC_AI_ANGLE_DELTA_MAX;
   } while(power < SC_AI_MORON_MIN_POWER);
   #endif
   
   /* Set a new angle and power level. */
   sc_player_advance_turret(c, p, game_lrand(SC_AI_ANGLE_DELTA_MAX * 2 + 1) - SC_AI_ANGLE_DELTA_MAX);
   sc_player_set_power(c, p, power);

   /* No victim used */
   p->ai->victim = NULL;

   #if SC_AI_DEBUG_TRAJECTORY
      printf("AI_trajectory:   %s, %s random fire.\n", sc_ai_name(p->aitype), p->name);
   #endif /* debug? */

}



static inline int _sc_ai_line_of_sight(const sc_config *c, sc_player *p, const sc_player *vp, trajfn line, trajfn noline) {
/* sc_ai_line_of_sight
   Call the appropriate trajectory function.  */

   if(sc_land_line_of_sight(c,c->land, p->x - p->tank->radius, p->y, vp->x - p->tank->radius, vp->y) ||
     sc_land_line_of_sight(c, c->land, p->x - p->tank->radius, p->y, vp->x + p->tank->radius, vp->y) ||
     sc_land_line_of_sight(c, c->land, p->x + p->tank->radius, p->y, vp->x - p->tank->radius, vp->y) ||
     sc_land_line_of_sight(c, c->land, p->x + p->tank->radius, p->y, vp->x + p->tank->radius, vp->y)) {
      return(line(c, p, vp));
   } else {
      return(noline(c, p, vp));
   }
     
}



static void _sc_ai_target_practice(const sc_config *c, sc_player **playerlist) {
/* sc_ai_target_practice */

   sc_player *tmp;
   int i;
   int j;
   
   if(!c->aicontrol->humantargets) return;

   for(i = 0; i < c->numplayers; ++i) {
      if(playerlist[i]->aitype == SC_AI_HUMAN) {
         tmp = playerlist[i];
         for(j = 0; j < i && tmp != NULL; ++j) {
            if(playerlist[j]->aitype != SC_AI_HUMAN) {
               playerlist[i] = playerlist[j];
               playerlist[j] = tmp;
               tmp = NULL;
            }
         }
      }
   }

}



static void _sc_ai_calculated_fire(const sc_config *c, sc_player *p) {
/* sc_ai_calculated_fire
   See if we can shoot anyone with the above two trajectory mechanisms.  We
   check players in random order, and if we can reach any of them, then fire
   we go.  Otherwise, we fall back on random firing scheme.  This mechanism
   only considers gravity, which is sufficient for up to CHOOSER.  */
   
   const sc_player *playerlist[SC_MAX_PLAYERS]; /* List of players, random order */
   const sc_player *vp; /* Current "victim" */
   int victim;          /* Index to "victim" */

   /* Get a random ordering of players */
   sc_player_random_order((sc_config *)c, (sc_player **)playerlist);
   _sc_ai_target_practice(c, (sc_player **)playerlist);

   /* Try to find a victim. */   
   victim = 0;
   while(victim < c->numplayers) {
      vp = playerlist[victim];
      if(p->index != vp->index && !vp->dead) {
         /* "Victim" isn't ourself (that would suck) */
         if(_sc_ai_line_of_sight(c, p, vp, sc_ai_trajectory_line, sc_ai_trajectory)) {
            /* Victim selected */
            return;
         }
      } /* Make sure victim isn't self */
      ++victim;   /* Next please */
   } /* Looking for a victim ... */

   /* No suitable victim; just fire at random. */
   _sc_ai_random_fire(c, p);

}



static void _sc_ai_calculated_fire_wind(const sc_config *c, sc_player *p) {
/* sc_ai_calculated_fire_wind
   See if we can shoot anyone with the above two trajectory mechanisms.  We
   check players in random order, and if we can reach any of them, then fire
   we go.  Otherwise, we fall back on random firing scheme.  This mechanism
   considers gravity and wind, which CALCULATER needs.  */
   
   const sc_player *playerlist[SC_MAX_PLAYERS]; /* List of players, random order */
   const sc_player *vp; /* Current "victim" */
   int victim;          /* Index to "victim" */

   /* Get a random ordering of players */
   sc_player_random_order((sc_config *)c, (sc_player **)playerlist);
   _sc_ai_target_practice(c, (sc_player **)playerlist);

   /* Try to find a victim. */   
   victim = 0;
   while(victim < c->numplayers) {
      vp = playerlist[victim];
      if(p->index != vp->index && !vp->dead) {
         /* "Victim" isn't ourself (that would suck) */
         if(_sc_ai_line_of_sight(c, p, vp, sc_ai_trajectory_line_wind, sc_ai_trajectory_wind)) {
            /* Victim selected */
            return;
         }
      } /* Make sure victim isn't self */
      ++victim;   /* Next please */
   } /* Looking for a victim ... */

   /* No suitable victim; just fire at random. */
   _sc_ai_random_fire(c, p);

}



static void _sc_ai_fire_at_victim(const sc_config *c, sc_player *p) {
/* sc_ai_fire_at_victim
   Attack victim in AI state.  If NULL or they are dead, then behave like
   calculated_fire.  This is the "big bully" behaviour; once a victim is
   chosen, well, they're pretty much dead.  */
   
   const sc_player *vp;
   
   /* Is the victim still alive? */
   vp = p->ai->victim;
   if(p->ai->victim != NULL && !p->ai->victim->dead) {
      /* "Victim" isn't dead; can we shoot them? */
      if(_sc_ai_line_of_sight(c, p, vp, sc_ai_trajectory_line, sc_ai_trajectory)) return;
   } /* Was victim alive? */
   
   /* If at this point, we have no victim; find another. */
   _sc_ai_calculated_fire(c, p);

}



static void _sc_ai_fire_at_victim_ruthlessly(const sc_config *c, sc_player *p) {
/* sc_ai_fire_at_victim_ruthlessly
   Attack victim in AI state.  If NULL or they are dead, then behave like
   calculated_fire.  This is the "big bully" behaviour; once a victim is
   chosen, well, they're pretty much dead.  This is insane, it compensates
   for wind.  */
   
   const sc_player *vp;
   
   /* Is the victim still alive? */
   vp = p->ai->victim;
   if(p->ai->victim != NULL && !p->ai->victim->dead) {
      /* "Victim" isn't dead; can we shoot them? */
      if(_sc_ai_line_of_sight(c, p, vp, sc_ai_trajectory_line_wind, sc_ai_trajectory_wind)) return;
   } /* Was victim alive? */
   
   /* If at this point, we have no victim; find another. */
   _sc_ai_calculated_fire_wind(c, p);

}



static void _sc_ai_raise_shields(const sc_config *c, sc_player *p) {
/* sc_ai_raise_shields
   If we don't have any shielding, raise them now.  */
   
   if(p->shield == NULL || p->shield->life <= 0) {
      sc_player_activate_best_shield(c, p);
   }
   
}



static void _sc_ai_set_contact_triggers(const sc_config *c, sc_player *p) {
/* sc_ai_set_contact_triggers
   Always try to arm the contact triggers, if we have any.  */
   
   sc_player_set_contact_triggers(c, p, true);
   
}



static void _sc_ai_recharge_tank(const sc_config *c, sc_player *p) {
/* sc_ai_recharge_tank
   Recharge tank to full health, if weakened.  */
   
   while(sc_player_activate_battery(c, p)) /* Just loop */;
   
}



static inline void _sc_ai_turn_status(const sc_config *c, const sc_player *p) {
/* sc_ai_turn_status */

   if(SC_CONFIG_GFX_FAST(c)) return;
   sc_status_player_message(c->window, p, "AI player is taking their turn ...");
   sc_window_idle(c->window);

}



sc_ai_result sc_ai_player_turn(const sc_config *c, sc_player *p) {
/* sc_ai_player_turn
   AI player takes a turn. */

   switch(p->ai->realaitype) {
      case SC_AI_HUMAN:
         return(SC_AI_NO_ACTION);
         break;

      case SC_AI_RANDOM:
      case SC_AI_NETWORK:
         /* No-ops */
         break;

      case SC_AI_MORON:
         _sc_ai_turn_status(c, p);
         _sc_ai_raise_shields(c, p);
         _sc_ai_set_contact_triggers(c, p);
         _sc_ai_recharge_tank(c, p);
         _sc_ai_select_last_weapon(c, p);
         _sc_ai_random_fire(c, p);
         break;

      case SC_AI_SHOOTER:
      case SC_AI_SPREADER:
         _sc_ai_turn_status(c, p);
         _sc_ai_raise_shields(c, p);
         _sc_ai_set_contact_triggers(c, p);
         _sc_ai_recharge_tank(c, p);
         _sc_ai_select_weapon_by_score(c, p);
         _sc_ai_calculated_fire(c, p);
         break;

      case SC_AI_CHOOSER:
         _sc_ai_turn_status(c, p);
         _sc_ai_raise_shields(c, p);
         _sc_ai_set_contact_triggers(c, p);
         _sc_ai_recharge_tank(c, p);
         _sc_ai_select_weapon_by_score(c, p);
         _sc_ai_fire_at_victim(c, p);
         break;
      
      case SC_AI_CALCULATER:
      case SC_AI_ANNIHILATER:
      case SC_AI_INSANITY:
         _sc_ai_turn_status(c, p);
         _sc_ai_raise_shields(c, p);
         _sc_ai_set_contact_triggers(c, p);
         _sc_ai_recharge_tank(c, p);
         _sc_ai_select_weapon_by_score(c, p);
         _sc_ai_fire_at_victim_ruthlessly(c, p);
         if(c->aicontrol->enablescan) {
            if(p->ai->victim != NULL) sc_ai_trajectory_scan(c, p, p->ai->victim);
         } /* scan refinement? */
         break;
      
   }
   
   return(SC_AI_CONTINUE);

}



