// -*- C++ -*-
#ifndef storage_h
#define storage_h

#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <strings.h>

#include <xdfs_cpp.h>
#include <xdfs_comp.h>

extern MESSAGE_TYPE XPROXY_ERROR_MESSAGE;
extern MESSAGE_TYPE LOG_XPROXY_MESSAGE;
extern MESSAGE_TYPE DEBUG_PROXY_MESSAGE;
extern MESSAGE_TYPE DEBUG_STORAGE_MESSAGE;
extern MESSAGE_TYPE DEBUG_POST_MESSAGE;
extern MESSAGE_TYPE DEBUG_PARSE_MESSAGE;
extern MESSAGE_TYPE DEBUG_HELPER_MESSAGE;
extern MESSAGE_TYPE DEBUG_VERBOSE_MESSAGE;
extern MESSAGE_TYPE DEBUG_REQHEAD_MESSAGE;
extern MESSAGE_TYPE INFO_XPROXY_MESSAGE;


#define XPROXY_ERROR \
	if (XPROXY_ERROR_MESSAGE.active) MESSAGE_CLOSURE (XPROXY_ERROR_MESSAGE, __FILE__, __LINE__)
#define XPROXY_LOG \
	if (LOG_XPROXY_MESSAGE.active) MESSAGE_CLOSURE (LOG_XPROXY_MESSAGE, __FILE__, __LINE__)
#define DEBUG_PROXY \
	if (DEBUG_PROXY_MESSAGE.active) MESSAGE_CLOSURE_FL (DEBUG_PROXY_MESSAGE)
#define DEBUG_STORAGE \
	if (DEBUG_STORAGE_MESSAGE.active) MESSAGE_CLOSURE_FL (DEBUG_STORAGE_MESSAGE)
#define DEBUG_POST \
	if (DEBUG_POST_MESSAGE.active) MESSAGE_CLOSURE_FL (DEBUG_POST_MESSAGE)
#define DEBUG_PARSE \
	if (DEBUG_PARSE_MESSAGE.active) MESSAGE_CLOSURE_FL (DEBUG_PARSE_MESSAGE)
#define DEBUG_HELPER \
	if (DEBUG_HELPER_MESSAGE.active) MESSAGE_CLOSURE_FL (DEBUG_HELPER_MESSAGE)
#define DEBUG_VERBOSE \
	if (DEBUG_VERBOSE_MESSAGE.active) MESSAGE_CLOSURE_FL (DEBUG_VERBOSE_MESSAGE)
#define DEBUG_REQHEAD \
	if (DEBUG_REQHEAD_MESSAGE.active) MESSAGE_CLOSURE_FL (DEBUG_REQHEAD_MESSAGE)
#define INFO_XPROXY \
	if (INFO_XPROXY_MESSAGE.active) MESSAGE_CLOSURE_FL (INFO_XPROXY_MESSAGE)

/* Creates a DBFS in the given directory.  If "new" is true, "dir" is
   cleared and a new DBFS is created; otherwise, DBFS is initialized
   with the previous contents in "dir". */

int storage_init    (const char *dir, const char *log, int oflag, DBFS* dbfs);
int storage_recover (const char *dir);

/* Returns the length of prefix used for "clustering" versions.
 */
int storage_url_prefix (const char *url);

/* Sets the attribute of a version, given the inode "ino". "name"
   specifies the name of the attribute, "value" the value of the
   attributes, and "bytes" the length of the attribute in bytes. */

int storage_set_attribute (TXN *txn, MAJORC *ino, XSTCK name, guint8 *value, int bytes);

/* Gets the value of an attribute given the version inode
   "ino". "name" specifies the name of the attributes, "value"
   contains the value of the attribute upon function return, and
   "bytes" indicates the length of the attribute in bytes. */

int storage_get_attribute (TXN *txn, MAJORC *ino, XSTCK name, guint8 *value, int bytes, bool variable_size);

/* Reads content from "inode" and writes bytes of contents to "buf";
   specify a very large number for "bytes" to get entire file out of
   "inode".  Functions returns the number of bytes written to "buf" or
   -1 if an error occurs. */

int storage_read_inode_into_buffer (TXN *txn, MAJORC *inode, char **buf, int *bytes);

enum STORAGE_STATUS {
    STATUS_INIT     = -1,
    STATUS_NOTFOUND = 1,
    STATUS_MULTI    = 3,
};

extern XSTCK   __STORAGE_AREA_STCK;
extern XSTCK   __STORAGE_LAST_STCK;

extern XSTCK   __XPROXY_EXP_STCK;
extern XSTCK   __XPROXY_LAST_STCK;
extern XSTCK   __XPROXY_ETAG_STCK;
extern XSTCK   __XPROXY_SERVER_STCK;
extern XSTCK   __XPROXY_TYPE_STCK;

extern const int STORAGE_AREAS;

inline int
storage_url_prefix (const char *url)
{
    const char *p = url;
    char c;

    for (; (c = *p); p+=1) {
	switch (c) {
	case '?':
	case '&':
	    goto out;
	}
    }

  out:

    if (p[-1] == '/') {
	// Note: the calling string is always prefixed by http://,
	// thus -1 index is always okay
	p -= 1;
    }

    return p-url;
}

#define STORAGE_URL_OFFSET 7 // strlen ("http://")

inline DKEY url_dkey (const char *url)
{
    return DKEY (url + STORAGE_URL_OFFSET, storage_url_prefix (url + STORAGE_URL_OFFSET));
}

class StorageLocation
{
public:

    StorageLocation (DBFS       &dbfs,
		     const char* ctx,
		     const char *url)
	: _dbfs   (dbfs),
	  _ctx    (ctx),
	  _url    (url),
	  _key    (url + STORAGE_URL_OFFSET, storage_url_prefix (url + STORAGE_URL_OFFSET)),
	  _status (STATUS_INIT)
    {
	g_assert (strncmp ("http://", url, STORAGE_URL_OFFSET) == 0);
    }

    /* Opens a transaction on the specified URL, returns 0
     * on success.
     */
    int  open             ();

    /* Searches for the last version of the specified URL and returns
     * a node handle or DBFS_NOTFOUND if there are no versions. */
    int  get_last_version (MAJORC *nodep);

    /* Retrieves the MD5 value of the specified node. */
    int  get_md5          (MAJORC *nodep, guint8 md5[16]);


    /* Finds version given the MD5; result is returned in "ino".*/
    int  get_version_by_md5 (guint8 md5[16], MAJORC* ino);

    /* Inserts version in XDFS. "entry" specifies the URL, "buf"
       contains the data. */
    int insert_version_from_buffer (char *buf, int bytes, MAJORC *ino);

    /* Extracts delta between two version and writes to a file
       "delta". "file" specifies the "URL", "old" inode points to the
       old version and "new" inode points to the new version. */
    int extract_delta_to_buffer (MAJORC *oldIno, MAJORC *newIno, char **buf, int *bytes);

    /* Inserts a delta from file "delta"; "file" specifies the URL,
       "old" inode points to the old version and "new" inode contains
       the inode pointing to the new version upon function return. */
    int insert_delta_from_buffer (MAJORC *oldIno, char *buf, int bytes, MAJORC *newIno);

    /* Commit the transaction */
    int commit () { return _txn.commit (); }


    DBFS           &_dbfs;
    TXN             _txn;

private:

    /* Updates the most recent version of a file. "file" specifies the
       URL and "ino" contains the inode pointing to the most recent
       version. */
    int update_last_version (MAJORC *ino);
    int pick_storage_area   (SAREA &area);

    const char     *_ctx;
    const char     *_url;
    DKEY            _key;
    STORAGE_STATUS  _status;

    MAJORC          _root;
    MAJORC          _node;    // URL's node
    NODEC           _lastlnk; // Last version reflink
    NODEC           _md5idx;  // Shared MD5 IDX
    XdfsLocation    _xdfs;    // XDFS compression archive
};

#endif storage_h
