#include "storage.h"
#include "helper.h"
#include "xdfs_comp.h"
#include <unistd.h>

MESSAGE_TYPE XPROXY_ERROR_MESSAGE ("XPROXY", MCLASS_ERROR);
MESSAGE_TYPE LOG_XPROXY_MESSAGE   ("xproxy", MCLASS_LOG);
MESSAGE_TYPE DEBUG_PROXY_MESSAGE   ("proxy", MCLASS_DEBUG);
MESSAGE_TYPE DEBUG_STORAGE_MESSAGE ("storage", MCLASS_DEBUG);
MESSAGE_TYPE DEBUG_POST_MESSAGE    ("post", MCLASS_DEBUG);
MESSAGE_TYPE DEBUG_PARSE_MESSAGE    ("parse", MCLASS_DEBUG);
MESSAGE_TYPE DEBUG_HELPER_MESSAGE    ("helper", MCLASS_DEBUG);
MESSAGE_TYPE DEBUG_VERBOSE_MESSAGE    ("verbose", MCLASS_DEBUG);
MESSAGE_TYPE DEBUG_REQHEAD_MESSAGE    ("reqhead", MCLASS_DEBUG);
MESSAGE_TYPE INFO_XPROXY_MESSAGE    ("xproxy", MCLASS_INFO);

static XdfsFlags STORAGE_XFLAGS = (XdfsFlags) (XF_MD5Equality | XF_IdxReverse | XF_IdxShared);

// For an fast query of "how many URLs share this archive"
//static bool REVERSE_URLDIR = false;

XSTCK   __STORAGE_AREA_STCK;
XSTCK   __STORAGE_LAST_STCK;

XSTCK   __XPROXY_EXP_STCK;
XSTCK   __XPROXY_LAST_STCK;
XSTCK   __XPROXY_ETAG_STCK;
XSTCK   __XPROXY_SERVER_STCK;
XSTCK   __XPROXY_TYPE_STCK;

const int  STORAGE_AREAS    = 4;
int        STORAGE_TXN_FLAG = DBFS_TXN_SYNC;

int
storage_recover (const char *dir)
{
    int   ret;
    int   argc    = 1;
    char *argv[] = { "<xproxy recovery>" };
    DBFS  dbfs (argc, argv);
    KILLPROC_DATA data;

    if ((ret = dbfs.open (dir, NULL, DBFS_ADMIN_TOOL))) {
	PROP_ERROR (ret) ("dbfs_killproc_open");
	return ret;
    }

    if ((ret = dbfs.admin_killproc (data))) {
	PROP_ERROR (ret) ("admin_killproc");
	return ret;
    }

    if ((ret = dbfs.close ())) {
	PROP_ERROR (ret) ("dbfs_killproc_close");
	return ret;
    }

    if ((ret = dbfs.open (dir, NULL, DBFS_RECOVER))) {
	PROP_ERROR (ret) ("dbfs_recover_open");
	return ret;
    }

    if ((ret = dbfs.close ())) {
	PROP_ERROR (ret) ("dbfs_recover_close");
	return ret;
    }

    return 0;
}

int
storage_init (const char *dir, const char *log, int oflag, DBFS* dbfs)
{
    int    ret;
    TXN    txn;
    MAJORC root;

    if (oflag & DBFS_ASYNC_INIT) {
	STORAGE_TXN_FLAG = DBFS_TXN_ASYNC;
    }

    DEBUG_STORAGE ("storage init: %s", dir);

    if ((ret = xdfs_library_init ())) {
	PROP_ERROR (ret) ("xdfs_library_init");
	return ret;
    }

    if ((ret = dbfs->open (dir, log, oflag |
			   DBFS_MAINT_INIT |
			   DBFS_DEADLOCK_INIT |
			   DBFS_CREATEROOT_HASH))) {
	PROP_ERROR (ret) ("dbfs_open");
	return ret;
    }

    if ((ret = txn.begin_root (*dbfs, DBFS_TXN_SYNC, root))) {
	PROP_ERROR (ret) ("txn_begin");
	return ret;
    }

    if ((ret = dbfs->catalog_stack_id (txn, "xproxy", "storage area",  "area",   __STORAGE_AREA_STCK)) ||
	(ret = dbfs->catalog_stack_id (txn, "xproxy", "last version",  "lastv",  __STORAGE_LAST_STCK)) ||
	(ret = dbfs->catalog_stack_id (txn, "xproxy", "expiration",    "exp",    __XPROXY_EXP_STCK)) ||
	(ret = dbfs->catalog_stack_id (txn, "xproxy", "last modified", "lastm",  __XPROXY_LAST_STCK)) ||
	(ret = dbfs->catalog_stack_id (txn, "xproxy", "etag",          "etag",   __XPROXY_ETAG_STCK)) ||
	(ret = dbfs->catalog_stack_id (txn, "xproxy", "server",        "serv",   __XPROXY_SERVER_STCK)) ||
	(ret = dbfs->catalog_stack_id (txn, "xproxy", "content type",  "type",   __XPROXY_TYPE_STCK))) {
	PROP_ERROR (ret) ("catalog_stack_id");
	return ret;
    }

    if (oflag & DBFS_CREATE) {

	for (guint16 i = 0; i < STORAGE_AREAS; i += 1) {

	    SAREA  singles_area;
	    NODEC  singles;

	    DEBUG_STORAGE ("storage create area: %d", i);

	    if ((ret = txn.create_area (singles_area))) {
		PROP_ERROR (ret) ("create_area");
		return ret;
	    }

	    if ((ret = root.node_mkey (MKEY (i, __STORAGE_AREA_STCK), singles))) {
		PROP_ERROR (ret) ("node_mkey");
		return ret;
	    }

	    if ((ret = singles.mk_arealink (singles_area, DBFS_NOOVERWRITE))) {
		PROP_ERROR (ret) ("mk_arealink");
		return ret;
	    }
	}

    }

    if ((ret = txn.commit ())) {
	PROP_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

int
StorageLocation::pick_storage_area (SAREA &area)
{
    int ret;
    int x = (random () >> 4) % STORAGE_AREAS;
    NODEC alnk;

    //DEBUG_STORAGE ("storage %s pick area: %d", _ctx, x);

    if ((ret = _root.node_mkey (MKEY (x, __STORAGE_AREA_STCK), alnk))) {
	PROP_ERROR (ret) ("node_mkey");
	return ret;
    }

    if ((ret = alnk.read_arealink (area, 0))) {
	PROP_ERROR (ret) ("read_arealink");
	return ret;
    }

    return 0;
}

int
StorageLocation::open ()
{
    int ret;

    if ((ret = _txn.begin_root (_dbfs, STORAGE_TXN_FLAG, _root))) {
	XPROXY_ERROR (ret) ("txn_begin");
	return ret;
    }

    if ((ret = xdfs_shared_index (_txn, STORAGE_XFLAGS, _md5idx))) {
	PROP_ERROR (ret) ("xdfs_shared_index");
	return ret;
    }

    //DEBUG_STORAGE ("storage %s md5idx: ", _ctx) (_md5idx) (" contid ") () << _md5idx.cont_id ();

    if ((ret = _root.dir_link_lookup (_key, _node, DBFS_LINK_RMW))) {

	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("root_dir_link_lookup");
	    return ret;
	}

	//DEBUG_STORAGE ("storage %s open: %s (NOTFOUND)", _ctx, _url);
	_status = STATUS_NOTFOUND;
	return 0;
    }

    if ((ret = _node.node_mkey (MKEY (__STORAGE_LAST_STCK), _lastlnk))) {
	PROP_ERROR (ret) (_node) ("get_lastnlk_node");
	return ret;
    }

    g_assert (_lastlnk.is_reflink ());

    if ((ret = _xdfs.open (_node))) {
	PROP_ERROR (ret) ("xdfs_loc_open");
	return ret;
    }

    //DEBUG_STORAGE ("storage %s open: %s (FOUND ARCHIVE)", _ctx, _url);

//      if (strcmp (_url, "http://images.salon.com/src/shim.gif") == 0 &&
//  	strncmp (_ctx, "client", 6) == 0) {
//  	static int x = 1;
//  	if (x) { DEBUG_STORAGE ("debug me: %d", getpid ()); }
//  	while (x) { sleep (1); }
//      }

    _status = STATUS_MULTI;
    return 0;
}

int
StorageLocation::insert_version_from_buffer (char *buf, int bytes, MAJORC *ino)
{
    int ret;
    FileHandle *fh;

    /* FH is the file handle for reading the new version. */
    if (! (fh = handle_read_mem ((guint8*) buf, (guint) bytes))) {
	PROP_ERROR ("handle_read_mem");
	return -1;
    }

    if (_status == STATUS_NOTFOUND) {

	XdfsParams params;
	SAREA area;

	if ((ret = pick_storage_area (area))) {
	    PROP_ERROR (ret) ("pick_storage_area");
	    return ret;
	}

	memset (&params, 0, sizeof (params));

	params.policy = XP_Forward;
	params.flags  = STORAGE_XFLAGS;

	if ((ret = _xdfs.create_first (area, & params, fh, _node, *ino)) && (ret != DBFS_EXISTS)) {
	    PROP_ERROR (ret) ("xdfs_loc_create");
	    return ret;
	}

	if (ret == 0) {
	    DEBUG_STORAGE ("storage %s create: %s (CREATE ARCHIVE)", _ctx, _url);
	} else {
	    DEBUG_STORAGE ("storage %s create: %s (FOUND EXISTING ARCHIVE)", _ctx, _url);
	}

	if ((ret = _root.dir_link_insert (_key, _node, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("root_dir_link_insert");
	    return ret;
	}

	if ((ret = _node.node_mkey (MKEY (__STORAGE_LAST_STCK), _lastlnk))) {
	    PROP_ERROR (ret) (_node) ("get_lastnlk_node");
	    return ret;
	}

	_status = STATUS_MULTI;

    } else {

	g_assert (_status == STATUS_MULTI);

	DEBUG_STORAGE ("storage %s insert buffer: %s (MULTI)", _ctx, _url);

	/* INO is the inode returned by XDFS that stores the new version. */
	if ((ret = _xdfs.insert_version (fh, *ino)) && (ret != DBFS_EXISTS)) {
	    PROP_ERROR (ret) ("xdfs_insert_version");
	    return ret;
	}
    }

    handle_free (fh);

    if ((ret = update_last_version (ino))) {
	PROP_ERROR (ret) ("storage_update_last");
	return ret;
    }

    return 0;
}

int
StorageLocation::update_last_version (MAJORC *ino)
{
    int ret;

    g_assert (_status == STATUS_MULTI);

    //DEBUG_STORAGE ("storage %s update last: %s ", _ctx, _url) (ino);

    if ((ret = _lastlnk.mk_reflink (*ino, DBFS_OVERWRITE))) {
	PROP_ERROR (ret) (_lastlnk) ("mk_reflink");
	return ret;
    }

    return 0;
}

int
StorageLocation::get_last_version (MAJORC *ino)
{
    int ret;

    if (_status == STATUS_NOTFOUND) {
	//DEBUG_STORAGE ("storage %s get last: %s (NO VERSIONS)", _ctx, _url);
	return DBFS_NOTFOUND;
    }

    g_assert (_status == STATUS_MULTI);

    if ((ret = _lastlnk.read_reflink (*ino, 0))) {
	PROP_ERROR (ret) (_lastlnk) ("read_reflink");
	return ret;
    }

    //DEBUG_STORAGE ("storage %s get last: %s ", _ctx, _url) (ino);
    return 0;
}

int
StorageLocation::get_md5 (MAJORC *ino, guint8 md5[16])
{
    int ret;
    DKEY dkey;

    g_assert (_status == STATUS_MULTI);

    if ((ret = _md5idx.dir_link_invert (*ino, dkey))) {
	//PROP_ERROR (ret) ("version has no MD5 attribute ") (ino) ("in index ") (_md5idx);
	return ret;
    }

    //DEBUG_STORAGE ("storage %s get MD5: %s ", _ctx, _url) (ino);

    g_assert (dkey.size () == 16);
    memcpy (md5, dkey.data (), 16);

    return 0;
}

int
StorageLocation::get_version_by_md5 (guint8 md5[16], MAJORC* ino)
{
    int ret;

    if (_status == STATUS_NOTFOUND) {
	//DEBUG_STORAGE ("storage %s get version by MD5: %s (NO VERSIONS)", _ctx, _url);
	return DBFS_NOTFOUND;
    }

    g_assert (_status == STATUS_MULTI);

    //DEBUG_STORAGE ("storage %s get version by MD5: %s", _ctx, _url);

    if ((ret = _md5idx.dir_link_lookup (DKEY (md5, 16), *ino, DBFS_NOFLAG))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("idx_dir_link_lookup");
	}
	return ret;
    }

    return 0;
}

int
StorageLocation::extract_delta_to_buffer (MAJORC *oldIno, MAJORC *newIno, char **buf, int *bytes)
{
    FileHandle *dh;
    char *result;
    int previousBytes;
    int ret;

    previousBytes = *bytes;

    DEBUG_STORAGE ("storage %s extract delta: %s", _ctx, _url);

    if (!(dh = handle_write_mem ())) {
	PROP_ERROR ("handle_write_mem");
	return -1;
    }

    if ((ret = _xdfs.extract_delta (*oldIno, *newIno, dh))) {
	PROP_ERROR (ret) ("loc_extract_delta");
	return ret;
    }

    handle_close(dh);
    handle_write_mem_result(dh, (const guint8**) &result, (guint*) bytes);
    // Reallocate space if necessary.
    if (*bytes > previousBytes) {
	reallocateBuffer(buf, &previousBytes, *bytes);
    }
    memcpy(*buf, result, *bytes);

    //handle_close(dh);
    handle_free(dh);

    return 0;
}

int
StorageLocation::insert_delta_from_buffer (MAJORC *oldIno, char *buf, int bytes, MAJORC *newIno)
{
    FileHandle *dh;
    int ret;

    // @@@ How are HTTP redirects handled?  Just curious...
    // @@@ WHY ARE NO DELTAS BEING RECEIVED?
    DEBUG_STORAGE ("storage %s insert delta: %s", _ctx, _url);

    if (!(dh = handle_read_mem((guint8*) buf, (guint) bytes))) {
	PROP_ERROR ("handle_read_mem");
	return -1;
    }

    if ((ret = _xdfs.insert_delta (*oldIno, dh, *newIno))) {
	PROP_ERROR (ret) ("xdfs_loc_insert_delta");
	return ret;
    }

    handle_close(dh);
    handle_free(dh);

    if ((ret = update_last_version (newIno))) {
	PROP_ERROR (ret) ("storage_update_last");
	return ret;
    }

    return 0;
}

int
storage_read_inode_into_buffer(TXN *txn, MAJORC *inode, char **buf, int *bytes)
{
    FileHandle *fh;
    int result;

    if ((result = inode->read_segment (& fh, 0))) {
	PROP_ERROR (result) ("read_segment");
	return -1;
    }

    // Allocate more space if necessary.
    if (handle_length(fh) > *bytes) {
	reallocateBuffer(buf, bytes, handle_length(fh));
    }

    if ((result = handle_read(fh, (guint8*) *buf, *bytes)) < 0) {
	PROP_ERROR (result) ("handle_read");
    }

    handle_close(fh);
    handle_free(fh);

    return result;
}

int
storage_set_attribute(TXN *txn, MAJORC *ino, XSTCK stck, guint8 *value, int bytes) {
    FileHandle *fh;
    NODEC m_ino;
    MKEY mkey (stck);
    int ret;

    if ((ret = ino->node_mkey (mkey, m_ino))) {
	PROP_ERROR (ret) ("node_mkey");
	return ret;
    }

    if ((ret = m_ino.repl_segment (& fh, DBFS_OVERWRITE))) {
	PROP_ERROR (ret) ("m_ino_repl_segment");
	return ret;
    }

    if (! handle_write (fh, value, bytes)) {
	PROP_ERROR ("handle_write");
	return -1;
    }

    handle_close(fh);
    handle_free(fh);

    return 0;
}

int
storage_get_attribute (TXN *txn, MAJORC *ino, XSTCK stck, guint8 *value, int bytes, bool variable_size)
{
    FileHandle *fh;
    NODEC m_ino;
    MKEY mkey (stck);
    int ret;

    if ((ret = ino->node_mkey (mkey, m_ino))) {
	PROP_ERROR (ret) ("node_mkey");
	return ret;
    }

    if ((ret = m_ino.read_segment (& fh, 0))) {
	PROP_ERROR (ret) ("m_ino_read_segment");
	return ret;
    }

    int size = m_ino.length ().key ();

    if ((size > bytes) || (! variable_size && size != bytes)) {
	XPROXY_ERROR ("unexpected size in storage_get_attribute: size %d expect %d", size, bytes);
	return DBFS_INVAL;
    }

    if (handle_read (fh, value, size) != size) {
	PROP_ERROR ("handle_read");
	return -1;
    }

    handle_close(fh);
    handle_free(fh);

    return 0;
}
