/* -*-Mode: C++;-*-
 * $Id: testxdfs.cc,v 1.9 2001/05/15 23:34:06 jmacd Exp $
 *
 * Copyright (C) 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "rcswalk.h"
#include "xdfs_cpp.h"
#include "xdfs_comp.h"
#include "edsiostdio.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <dirent.h>

class ArchiveTest
{
public:

    const char *name;
    int  (*initialize_a) (void);
    int  (*initialize_r) (void);
    int  (*close) (void);
    void (*abort) (void);
    int  (*onefile) (RcsFile *rcs, RcsStats* set, void* data);
    int  (*dateorder) (RcsFile* rcs, RcsVersion *v, void* data);
    int  (*retrieve) (RcsFile* rcs, RcsVersion *v, void* data);

    int         copydir;
    XdfsPolicy  policy;
    int         flags;
};

#define AT_FLAG_RCS_LINEAR  (1 << 0)
#define AT_FLAG_RCS_TREE    (1 << 1)

static char *tmp_file;

int          __argc;
char**       __argv;

DBFS        *dbfs;

GTimer      *timer;

BinCounter  *insert_single_times;
BinCounter  *insert_version_size;
BinCounter  *insert_time_size_ratio;

BinCounter  *retrieve_single_times;
BinCounter  *retrieve_version_size;
BinCounter  *retrieve_time_size_ratio;

IntStat     *xdfs_file_sizes;

double       insert_pass_time;
double       retrieve_pass_time;

int          process_versions;
int          process_files;
int          process_write_commits;

guint64    encoded_size;
guint64    unencoded_size;
guint64    disk_blocks;

ArchiveTest *current_test;

gboolean     atest_retrieve_verify = TRUE;
gint32       atest_min_versions    = 0;
gint32       atest_max_versions    = G_MAXINT;
const char*  atest_test            = NULL;
const char*  atest_base_dir        = NULL;
const char*  atest_log_dir         = NULL;
const char*  atest_ci_path         = NULL;
const char*  atest_co_path         = NULL;
const char*  atest_rcs_path        = NULL;

gint32 atest_short_threshold              = 2048;
gint32 atest_cluster_max_versions         = 40;

static ConfigOption options [] = {
    { "min_versions", "min",    CS_Use,    CO_Optional, CD_Int32,   & atest_min_versions },
    { "max_versions", "max",    CS_Use,    CO_Optional, CD_Int32,   & atest_max_versions },
    { "verify",       "vfy",    CS_Use,    CO_None,     CD_Bool,    & atest_retrieve_verify },
    { "method",       "mth",    CS_Use,    CO_Required, CD_String,  & atest_test },

    { "archive_base", "ab",     CS_Ignore, CO_Required, CD_String,  & atest_base_dir },
    { "archive_log",  "al",     CS_Ignore, CO_Optional, CD_String,  & atest_log_dir },

    { "short_threshold", "st",               CS_Use, CO_Optional, CD_Int32,  & atest_short_threshold },
    { "cluster_max_versions", "cmv",         CS_Use, CO_Optional, CD_Int32,  & atest_cluster_max_versions },
};

/* in rcsextract.c */
int rcs_count (const char* filename, guint *encoded_size);

int
disk_usage (const char *file, guint64 *dblocks)
{
    struct stat buf;
    DIR* thisdir = NULL;
    struct dirent* ent = NULL;
    int ret;

    if ((ret = stat (file, & buf)) < 0) {
	ret = errno;
	SYS_ERROR (ret) ("stat: %s", file);
	return ret;
    }

    (* dblocks) += buf.st_blocks;

    if (S_ISDIR (buf.st_mode)) {

	if (! (thisdir = opendir (file))) {
	    ret = errno;
	    SYS_ERROR (ret) ("opendir: %s", file);
	    return ret;
	}

	while ((ent = readdir (thisdir))) {
	    char* name = ent->d_name;

	    if (strcmp (name, ".") == 0) {
		continue;
	    }

	    if (strcmp (name, "..") == 0) {
		continue;
	    }

	    /* Special cases here: I'm NOT counting Berkeley DB logs and shared
	     * memory regions. */
	    if (strncmp (name, "log.", 4) == 0) {
		continue;
	    }

	    if (strncmp (name, "__db", 4) == 0) {
		continue;
	    }

	    string fullname = file;

	    fullname += '/';
	    fullname += name;

	    if ((ret = disk_usage (fullname.c_str (), dblocks))) {
		goto abort;
	    }
	}
    }

    if (thisdir && (ret = closedir (thisdir)) < 0) {
	ret = errno;
	SYS_ERROR (ret) ("closedir: %s", file);
	return ret;
    }

    return 0;

  abort:

    if (thisdir)
	closedir (thisdir);

    return ret;
}

void
report_stats ()
{
    FILE* tf;
    guint64 real_size = disk_blocks * 512;

    if (! (tf = config_output ("Result.data"))) {
	abort ();
    }

    fprintf (tf, "----------------\n");
    fprintf (tf, "Name:               %s\n", current_test->name);
    fprintf (tf, "Insert time:        %0.2f seconds\n", insert_pass_time);
    fprintf (tf, "Unencoded:          %qd bytes\n", unencoded_size);
    fprintf (tf, "Storage:            %qd bytes\n", real_size);
    fprintf (tf, "Ideal:              %qd bytes\n", encoded_size);
    fprintf (tf, "Storage diff:       %qd bytes\n", real_size - encoded_size);
    fprintf (tf, "Storage overhead:   %0.2f%%\n", 100.0 * ((double) (real_size - encoded_size) / (double) encoded_size));
    fprintf (tf, "Actual compression: %0.2f%%\n", 100.0 * (1.0 - ((double) real_size    / (double) unencoded_size)));
    fprintf (tf, "Ideal compression:  %0.2f%%\n", 100.0 * (1.0 - ((double) encoded_size / (double) unencoded_size)));

    fprintf (tf, "Retrieve time:      %0.2f seconds\n", retrieve_pass_time);

    if (fclose (tf) < 0) {
	SYS_ERROR (errno) ("fclose");
	abort ();
    }
}

int
atest_finalize (RcsStats* set, void* data)
{
    int ret;

    if (current_test->close) {
	g_timer_start (timer);

	if ((ret = current_test->close ())) {
	    return ret;
	}

	g_timer_stop (timer);

	insert_pass_time += g_timer_elapsed (timer, NULL);
    }

    rcswalk_report (set);

    if ((ret = disk_usage (atest_base_dir, & disk_blocks))) {
	return ret;
    }

    return 0;
}

int
atest_onefile (RcsFile *rcs, RcsStats* set, void* data)
{
    process_files += 1;

    if (current_test->onefile) {

	int ret;

	if ((ret = current_test->onefile (rcs, set, data))) {
	    PROP_ERROR (ret) ("current_test->onefile");
	    return ret;
	}
    }

    return 0;
}

double
disk_block_size (off_t size)
{
    return (double) (((size % 4096) + 1) * 4096);
}

int
atest_dateorder (RcsFile* rcs, RcsVersion *v, void* data)
{
    int    ret;
    double stime;

    static int progress = 0;

    process_versions += 1;

    if ((++progress % 1000) == 0) {
	fprintf (stdout, "i %d\n", progress);
    }

    unencoded_size += v->size;

    g_timer_start (timer);

    if ((ret = current_test->dateorder (rcs, v, data))) {
	PROP_ERROR (ret) ("current_test->dateorder");
	return ret;
    }

    g_timer_stop (timer);

    stime = g_timer_elapsed (timer, NULL);

    insert_pass_time += stime;

    stat_bincount_add_item (insert_single_times, v->dateseq, stime);
    stat_bincount_add_item (insert_version_size, v->dateseq, v->size);
    stat_bincount_add_item (insert_time_size_ratio, v->dateseq, stime / disk_block_size (v->size));

    return 0;
}

int
xdfs_dateorder (RcsFile* rcs, RcsVersion *v, void* data)
{
    FileHandle  *fh;
    TXN          txn;
    MAJORC       root;
    XdfsLocation loc;
    int          ret;

    if ((ret = txn.begin_root (*dbfs, DBFS_TXN_SYNC, root))) {
	PROP_ERROR (ret) ("txn_begin");
	return ret;
    }

    MAJORC loc_node;
    DKEY   key (rcs->filename);

    if ((ret = root.dir_link_lookup (key, loc_node, DBFS_NOFLAG))) {

	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("root_dir_link_lookup");
	    return ret;
	}

	SAREA      area;
	XdfsParams params;

	if ((ret = txn.create_area (area))) {
	    PROP_ERROR (ret) ("txn_create_area");
	    return ret;
	}

	params.flags                = /*XF_SequenceDir*/0; // @@@ Broken
	params.policy               = current_test->policy;
	params.cluster_max_versions = atest_cluster_max_versions;

	if ((ret = loc.create (area, & params, loc_node))) {
	    PROP_ERROR (ret) ("xdfs_loc_create");
	    return ret;
	}

	if ((ret = root.dir_link_insert (key, loc_node, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("root_dir_link_insert");
	    return ret;
	}
    } else {

	if ((ret = loc.open (loc_node))) {
	    PROP_ERROR (ret) ("xdfs_loc_open");
	    return ret;
	}
    }

    if (! (fh = handle_read_file (v->filename))) {
	PROP_ERROR (errno) ("handle_read_file");
	return -1;
    }

    MAJORC insert_node;

    if ((ret = loc.insert_version (fh, insert_node))) {
	PROP_ERROR (ret) ("xdfs_insert_version");
	return ret;
    }

    handle_free (fh);

    process_write_commits += 1;

    if ((ret = txn.commit ())) {
	PROP_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

int
xdfs_onefile (RcsFile *rcs, RcsStats* set, void* data)
{
    TXN          txn;
    XdfsLocation loc;
    int          ret;
    MAJORC       root;

    if ((ret = txn.begin_root (*dbfs, DBFS_TXN_SYNC, root))) {
	PROP_ERROR (ret) ("txn_begin");
	return ret;
    }

    MAJORC loc_node;
    DKEY   key (rcs->filename);

    if ((ret = root.dir_link_lookup (key, loc_node, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("root_dir_link_lookup");
	return ret;
    }

    if ((ret = loc.open (loc_node))) {
	PROP_ERROR (ret) ("xdfs_loc_open");
	return ret;
    }

    XdfsState xstat;

    if ((ret = loc.stat (xstat))) {
	PROP_ERROR (ret) ("xdfs_stat");
	return ret;
    }

    encoded_size += xstat.literal_size + xstat.control_size + xstat.patch_size;

    if ((ret = txn.commit ())) {
	PROP_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

int
xdfs_init_a ()
{
    int ret;

    if ((ret = xdfs_library_init ())) {
	PROP_ERROR (ret) ("xdfs_library_init");
	return ret;
    }

    dbfs = new DBFS (__argc, __argv);

    if ((ret = dbfs->open (atest_base_dir, atest_log_dir, DBFS_CLEAR | DBFS_CREATE))) {
	PROP_ERROR (ret) ("dbfs_open");
	return ret;
    }

    return 0;
}

int
xdfs_init_r ()
{
    int ret;

    dbfs = new DBFS (__argc, __argv);

    if ((ret = dbfs->open (atest_base_dir, NULL, 0))) {
	PROP_ERROR (ret) ("dbfs_open");
	return ret;
    }

    return 0;
}

int
xdfs_close ()
{
    int   ret;
    DBFS *tmp = dbfs;

    dbfs = NULL;

    if ((ret = tmp->close ())) {
	PROP_ERROR (ret) ("dbfs_close");
    }

    delete tmp;

    return ret;
}

void
xdfs_abort ()
{
    if (dbfs) {
	dbfs->close ();
	delete dbfs;
	dbfs = NULL;
    }
}

/* Retrieval code
 */

int
rtest_onefile (RcsFile *rcs, RcsStats* set, void* data)
{
    return 0;
}

int
xdfs_retrieve (RcsFile* rcs, RcsVersion *v, void* data)
{
    FileHandle *in_fh, *out_fh;
    int          bin;
    TXN          txn;
    XdfsLocation loc;
    int          ret;
    MAJORC       root, loc_node;

    if ((ret = txn.begin_root (*dbfs, DBFS_TXN_ASYNC, root))) {
	PROP_ERROR (ret) ("txn_begin");
	return ret;
    }

    DKEY   key (rcs->filename);

    if ((ret = root.dir_link_lookup (key, loc_node, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("root_dir_link_lookup");
	return ret;
    }

    if ((ret = loc.open (loc_node))) {
	PROP_ERROR (ret) ("xdfs_loc_open");
	return ret;
    }

    // @@@ Broken
    NODEC  seq;// = loc.sequence_dir ();
    MAJORC lookup;

    if ((ret = seq.dir_link_lookup_seqno (XSEQNO (v->dateseq+1), lookup, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("seq_dir_link_lookup_seqno");
	return ret;
    }

    if ((ret = lookup.read_segment (& in_fh, 0))) {
	PROP_ERROR (ret) ("read_segment");
	return ret;
    }

    if (! (out_fh = handle_write_file (tmp_file))) {
	PROP_ERROR ("handle_write_file");
	return -1;
    }

    if (! handle_drain (in_fh, out_fh)) {
	PROP_ERROR ("handle_drain");
	return -1;
    }

    if (! handle_close (out_fh)) {
	PROP_ERROR ("handle_close");
	return -1;
    }

    handle_close (in_fh);
    handle_free (out_fh);
    handle_free (in_fh);

    bin = rcs->version_count - v->dateseq - 1;

    if ((ret = txn.commit ())) {
	PROP_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

int
verify_version (RcsFile* rcs, RcsVersion *v)
{
    FileHandle *orig, *retr;
    guint8 obuf[1024], rbuf[1024];
    int offset = 0;
    int oc, rc;

    if (! (orig = handle_read_file (v->filename))) {
	PROP_ERROR ("handle_read_file");
	return -1;
    }

    if (! (retr = handle_read_file (tmp_file))) {
	PROP_ERROR ("handle_read_file");
	return -1;
    }

    for (;;) {

	oc = handle_read (orig, obuf, 1024);
	rc = handle_read (retr, rbuf, 1024);

	if (oc < 0 || rc < 0) {
	    PROP_ERROR ("read failed");
	    return -1;
	}

	if (oc != rc) {
	    XDFS_ERROR ("verify failed: different lengths: %d != %d\n", oc, rc);
	    return -1;
	}

	if (oc == 0) {
	    break;
	}

	if (memcmp (obuf, rbuf, oc) != 0) {
	    XDFS_ERROR ("verify failed: different content near offset: %d\n", offset);
	    return -1;
	}

	offset += oc;
    }

    handle_close (orig);
    handle_close (retr);

    handle_free (orig);
    handle_free (retr);

    return 0;
}

int
rtest_dateorder (RcsFile* rcs, RcsVersion *v, void* data)
{
    int ret;
    double stime;
    int bin;
    static int progress = 0;

    if ((++progress % 1000) == 0) {
	fprintf (stdout, "r %d\n", progress);
    }

    g_timer_start (timer);

    if ((ret = current_test->retrieve (rcs, v, data))) {
	PROP_ERROR (ret) ("current_test->retrieve");
	return ret;
    }

    g_timer_stop (timer);

    stime = g_timer_elapsed (timer, NULL);

    retrieve_pass_time += stime;

    /* Decide how to place this single time.  For the RCS tree case, use delta
     * chain length, otherwise use sequence number.
     */
    if (current_test->flags & AT_FLAG_RCS_TREE) {
	bin = v->chain_length;
    } else {
	bin = rcs->version_count - v->dateseq - 1;
    }

    stat_bincount_add_item (retrieve_single_times, bin, stime);
    stat_bincount_add_item (retrieve_version_size, bin, v->size);
    stat_bincount_add_item (retrieve_time_size_ratio, bin, stime / disk_block_size (v->size));

    if (atest_retrieve_verify && (ret = verify_version (rcs, v))) {
	PROP_ERROR (ret) ("verify_version");
	return ret;
    }

    return 0;
}

int
rtest_finalize (RcsStats* set, void* data)
{
    int ret;

    if (current_test->close && (ret = current_test->close ())) {
	PROP_ERROR (ret) ("current_test->close");
	return ret;
    }

    report_stats ();

    stat_bincount_report (insert_single_times);
    stat_bincount_report (insert_version_size);
    stat_bincount_report (insert_time_size_ratio);
    stat_bincount_report (retrieve_single_times);
    stat_bincount_report (retrieve_version_size);
    stat_bincount_report (retrieve_time_size_ratio);
    stat_int_report      (xdfs_file_sizes);

    return 0;
}

RcsWalker atest_walker = {
    NULL,
    & atest_finalize,
    & atest_onefile,
    & atest_dateorder,
    NULL,
    NULL,
    0,
    G_MAXINT,
    TRUE
};

RcsWalker rtest_walker = {
    NULL,
    & rtest_finalize,
    & rtest_onefile,
    & rtest_dateorder,
    NULL,
    NULL,
    0,
    G_MAXINT,
    FALSE
};

ArchiveTest all_tests[] = {
    { "XDFS-none",    xdfs_init_a, xdfs_init_r, xdfs_close, xdfs_abort, xdfs_onefile, xdfs_dateorder, xdfs_retrieve, FALSE, XP_NoCompress, 0 },
    { "XDFS-r",      xdfs_init_a, xdfs_init_r, xdfs_close, xdfs_abort, xdfs_onefile, xdfs_dateorder, xdfs_retrieve, FALSE, XP_Reverse, 0 },
    { "XDFS-f",      xdfs_init_a, xdfs_init_r, xdfs_close, xdfs_abort, xdfs_onefile, xdfs_dateorder, xdfs_retrieve, FALSE, XP_Forward, 0 },
    { "XDFS-fs",       xdfs_init_a, xdfs_init_r, xdfs_close, xdfs_abort, xdfs_onefile, xdfs_dateorder, xdfs_retrieve, FALSE, XP_ForwardSingle, 0 },
};

int
main (int argc, char** argv)
{
    int  ret;

    __argc = argc;
    __argv = argv;

    rcswalk_init ();

    timer                    = g_timer_new ();

    insert_single_times      = stat_bincount_new ("InsertTime");
    insert_version_size      = stat_bincount_new ("InsertSize");
    insert_time_size_ratio   = stat_bincount_new ("InsertTimeSizeRatio");
    retrieve_single_times    = stat_bincount_new ("RetrieveTime");
    retrieve_version_size    = stat_bincount_new ("RetrieveSize");
    retrieve_time_size_ratio = stat_bincount_new ("RetrieveTimeSizeRatio");
    xdfs_file_sizes          = stat_int_new      ("XdfsFileSizes");

    tmp_file = g_strdup_printf ("%s/at.%d", g_get_tmp_dir (), (int) getpid ());

    config_register (options, ARRAY_SIZE (options));

    config_set_string ("rcswalk_experiment", "at");

    if (argc < 2) {
	config_help ();
	goto bail;
    }

    for (int i = 1; i < argc; i += 1) {
	if ((ret = config_parse (argv[i]))) {
	    goto bail;
	}
    }

    if ((ret = config_done ())) {
	goto bail;
    }

    for (uint i = 0; i < ARRAY_SIZE (all_tests); i += 1) {
	if (strcmp (atest_test, all_tests[i].name) == 0) {
	    current_test = all_tests + i;
	    break;
	}
    }

    if (! current_test) {
	g_error ("No such test: %s\n", atest_test);
    }

    fprintf (stderr, "archivetest: verification %s; method: %s\n", atest_retrieve_verify ? "on" : "off", current_test->name);

    if ((ret = config_clear_dir (atest_base_dir))) {
	goto bail;
    }

    if ((ret = config_clear_dir (atest_log_dir))) {
	goto bail;
    }

    rtest_walker.min_versions = atest_min_versions;
    rtest_walker.max_versions = atest_max_versions;

    atest_walker.min_versions = atest_min_versions;
    atest_walker.max_versions = atest_max_versions;

    if (atest_retrieve_verify)
	rtest_walker.write_files = TRUE;

    //dbfs_set_fs_short_threshold (atest_short_threshold);

    if (current_test->initialize_a && (ret = current_test->initialize_a ())) {
	goto bail;
    }

    if ((ret = rcswalk (& atest_walker, current_test->copydir ? atest_base_dir : NULL))) {
	goto bail;
    }

    if (current_test->initialize_r && (ret = current_test->initialize_r ())) {
	goto bail;
    }

    if ((ret = rcswalk (& rtest_walker, current_test->copydir ? atest_base_dir : NULL))) {
	goto bail;
    }

    unlink (tmp_file);

    fprintf (stderr, "archivetest: processed %d versions; %d files; %d write/commits\n", process_versions, process_files, process_write_commits);

    return 0;

  bail:

    if (current_test && current_test->abort)
	current_test->abort ();

    unlink (tmp_file);

    return 2;
}
