/* -*-Mode: C++;-*-
 * $Id: xdfs_comp.h 1.13 Tue, 01 May 2001 06:05:33 -0700 jmacd $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald
 * <jmacd@CS.Berkeley.EDU> and The Regents of the University of
 * California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _XDFS_INT_H_
#define _XDFS_INT_H_

//#define DEBUG_XDFS_ASSERTIONS

#include "xdfs_edsio.h"

#define XP_IsEvolving  (1 << 0)
#define XP_IsReverse   (1 << 1)
#define XP_IsForward   (1 << 2)

enum _XdfsPolicy {
    XP_NoCompress    = 0,
    XP_Reverse       = XP_IsEvolving | XP_IsReverse,
    XP_Forward       = XP_IsEvolving | XP_IsForward,
    XP_ForwardSingle =                 XP_IsForward
};

enum _XdfsFlags {
    XF_MD5Equality    = (1 << 0),
    XF_SHAEquality    = (1 << 1),
    XF_IdxReverse     = (1 << 2),
    XF_IdxShared      = (1 << 3),
    XF_Approximate    = (XF_MD5Equality | XF_SHAEquality),
};

typedef enum   _XdfsPolicy              XdfsPolicy;
typedef enum   _XdfsFlags               XdfsFlags;

class XdfsParams {

public:

    XdfsParams ()
	: policy (XP_Forward),
	  flags  (0),
	  cluster_max_versions (0) { }

    XdfsPolicy  policy;
    int         flags;
    size_t      cluster_max_versions;
};

const char* xdfs_flags_to_string (int flags);

typedef struct _SerialXdfsState         XdfsState;
typedef struct _SerialXdfsInstruction   XdfsInstruction;

int   xdfs_library_init (void);
int   xdfs_state_print  (FileHandle *handle,
			 XdfsState  *state);

int   xdfs_shared_index (TXN       &txn,
			 XdfsFlags  flags,
			 NODEC     &idxnode);

class XdfsLocation
{
    friend class XdfsExtraction;

public:

    XdfsLocation ()
	: _md    (NULL),
	  _stateX (NULL)
    {
    }

    ~XdfsLocation ()
    {
	clear ();
    }

    void clear              ()
    {
	if (_stateX) {
	    g_free (_stateX);
	    _stateX = NULL;
	}

	_md = NULL;
    }

    int  create             (SAREA        &sarea,
			     XdfsParams   *params,
			     MAJORC       &node_out);

    int  create_first       (SAREA        &sarea,
			     XdfsParams   *params,
			     FileHandle   *fh_in,
			     MAJORC       &xdfsloc_out,
			     MAJORC       &version_out);

    int  open               (const MAJORC &node);

    int  insert_version     (FileHandle   *in,
			     MAJORC       &node_out);
    int  extract_delta      (const MAJORC &from_node,
			     const MAJORC &to_node,
			     FileHandle   *delta_out);
    int  insert_delta       (const MAJORC &against_node,
			     FileHandle   *delta_in,
			     MAJORC       &node_out);
    int  print              (FileHandle   *out);
    int  stat               (XdfsState    &state);

    bool has_idx            () { return _stateX->flags & XF_Approximate; }

    NODEC&  index_dir       () { g_assert (has_idx ()); return _idx_node; }
    MAJORC& loc_node        () { return _loc_node; }

    const MessageDigest* message_digest () { return _md; }

private:

    int  insert_new_version (const MAJORC &copy_in,
			     MAJORC       &version_out);
    int  create_internal    (SAREA &sarea, XdfsState *state, MAJORC &xdfsloc_out);

    int  source_segment     (XdeltaGenerator *gen, XdeltaSourceType type, const MAJORC &node);

    bool reject_delta       (const MAJORC &copy_curs,
			     const MAJORC &to_curs,
			     const MAJORC &patch_curs,
			     const NODEC  &cont_curs);


    MAJORC               _loc_node;
    NODEC                _state_node;
    NODEC                _idx_node;

    const MessageDigest *_md;

    XdfsState           *_stateX;
};

class XdfsLocalIndex
{
public:

    XdfsLocalIndex ()
	: _literal (false),
	  _inst    (NULL),
	  _inst_len (0),
	  _seg_len  (0),
	  _array    (NULL)
    {
	memset (& _if_lit, 0, sizeof (_if_lit));
    }

    ~XdfsLocalIndex ()
    {
	clean ();
    }

    void clean ()
    {
	//_node = NULL;
	_inst = NULL;

	if (_array) {
	    g_array_free (_array, TRUE);
	    _array = NULL;
	}
    }

    void clone            (XdfsLocalIndex          &clone);
    int  source           (const MAJORC            &node);
    void update           (const MAJORC            &node,
			   GArray                  *array);
    void whole            (const MAJORC            &node);
    void delta            (const MAJORC            &node,
			   XdfsLocalIndex          *src_lis,
			   XdeltaControl           *control);
    void translate_copies (const XdeltaInstruction *translate,
			   GArray                  *append_inst_array);

    const XdfsInstruction* search (size_t offset) const;

#ifdef DEBUG_XDFS_ASSERTIONS
    void assert_length    ();
#else
    void assert_length    () { }
#endif


    MAJORC                 _node;
    bool                   _literal;
    const XdfsInstruction *_inst;
    size_t                 _inst_len;
    size_t                 _seg_len;
    GArray                *_array;

    XdfsInstruction        _if_lit;
};

int xdfs_apply_delta (FileHandle *from_in,
		      FileHandle *delta_in,
		      FileHandle *cons_out);

class XdfsViewImpl
    : public VIEWIMPL
{
public:

    XdfsViewImpl ();
    virtual ~XdfsViewImpl ();

    ViewBegin begin;
    ViewPgin  pgin;

    XdfsLocalIndex  _index;
};

class XdfsExtrElt
{
public:

    XdfsExtrElt (const MAJORC  &node,
		 XdeltaControl *control)
	: _node          (node),
	  _control       (control),
	  _src_nodes     (new MAJORC[control->src_count]),
	  _primary_index (-1) { }

    ~XdfsExtrElt ()
    {
	delete [] _src_nodes;
    }

    MAJORC          _node;
    XdeltaControl  *_control;
    MAJORC         *_src_nodes;
    int             _primary_index;
    MAJORC          _primary;
};

class XdfsExtrHalf
{
public:

    XdfsExtrHalf (const MAJORC &node)
	: chain_len (0),
	  chain     (NULL),
	  head      (node),
	  home      (node) { }

    int compare_to (const XdfsExtrHalf &half) const { return head.compare_to (half.head); }

    ~XdfsExtrHalf ()
    {
	g_slist_foreach (chain, & free_elt, NULL);
	g_slist_free    (chain);
    }

    static void free_elt (gpointer data,
			  gpointer user_data);

    int            chain_len;
    GSList        *chain;
    MAJORC         head;
    const MAJORC  &home;
};

typedef Slp<guint32,guint64,UINT32_MAX,16,0> InvertTable;

class XdfsInvertIter
{
public:

    XdfsInvertIter (InvertTable::iterator iter)
	: _iter (iter) { }

    size_t offset () const
    {
	return _iter.key ();
    }

    size_t length () const
    {
	return (size_t) (_iter.data () >> 32);
    }

    size_t start () const
    {
	return (size_t) (_iter.data () & 0xffffffff);
    }

    bool end () const
    {
	return _iter.end ();
    }

    void next ()
    {
	++_iter;
    }

    // This is used in XdfsExtraction::extract_overlay, used to avoid
    // running off the end of the overlay.
    void set_off (size_t ifend, size_t &off) const
    {
	if (! end ()) {
	    off = offset ();
	} else {
	    off = ifend;
	}
    }

private:

    InvertTable::iterator _iter;
};

class XdfsInvert
{
public:

    typedef XdfsInvertIter iterator;

    void insert (size_t offset,
		 size_t length,
		 size_t cstart)
    {
	if (_table.slp_size () == 0) {

	    // First insert: search_gub would return end ()
	    real_insert (offset, length, cstart);

	} else {

	    iterator it = (_table.search_gub (offset));

	  again:
	    size_t this_end  = offset + length;
	    size_t it_offset = it.offset ();
	    size_t it_end    = it_offset + it.length ();

	    if (offset >= it_offset) {
		// Inserting at or after the current location

		if (it_end >= this_end) {

		    // Current entry is a larger copy

		} else {

		    // Current entry is shorter

		    if (offset < it_end) {
			// If there is any overlap
			length  = this_end - it_end;
			cstart += it_end - offset;
			offset  = it_end;
		    }

		    it.next ();

		    if (it.end ()) {
			real_insert (offset, length, cstart);
		    } else {
			goto again;
		    }
		}
	    } else {
		// Insert before the current location

		if (this_end <= it_offset) {

		    // No overlap
		    real_insert (offset, length, cstart);

		} else {

		    size_t this_len = it_offset - offset;

		    // Insert the non-overlapping portion
		    real_insert (offset, this_len, cstart);

		    // Repeat-insert overlapping portion
		    insert (it_offset, length - this_len, cstart + this_len);
		}
	    }
	}
    }

    iterator invsearch (size_t offset) const
    {
	// Search for the greatest element less than or equal to
	// offset, then check if the range overlaps offset.
	iterator it (_table.search_gub (offset));

	if (! it.end () && (it.offset () + it.length () <= offset)) {
	    it.next ();
	}

	return it;
    }

    iterator begin () const
    {
	return iterator (_table.begin ());
    }

private:

    void real_insert (size_t offset,
		      size_t length,
		      size_t cstart)
    {
	_table.slp_insert (offset, (((guint64) length) << 32) | (guint64) cstart);
    }

    InvertTable _table;
};

class XdfsExtraction
{
public:
    XdfsExtraction (XdfsLocation &loc, const MAJORC &from, const MAJORC &to)
	: _loc    (loc),
	  _broken (false),
	  _from   (from),
	  _to     (to) { }

    int         furthest           ();
    int         advance            (XdfsExtrHalf &half,
				    bool          is_to_chain);

    void        extract_to_chain      (XdfsLocalIndex &middle);
    XdfsInvert* extract_from_chain    ();
    void        extract_overlay       (XdfsLocalIndex &middle,
				       XdfsInvert     *overlay);
    int         output_delta          (XdfsLocalIndex &middle,
				       FileHandle     *delta_out);

    XdfsLocation &_loc;
    bool          _broken;

    XdfsExtrHalf  _from;
    XdfsExtrHalf  _to;
};


#endif /* _XDFS_INT_H_ */
