;;; x-symbol-nomule.el --- XEmacs/no-Mule support for package x-symbol

;; Copyright (C) 1996-1999 Free Software Foundation, Inc.
;;
;; Author: Christoph Wedler <wedler@fmi.uni-passau.de>
;; Maintainer: (Please use `M-x x-symbol-package-bug' to contact the maintainer)
;; Version: $Id: x-symbol-nomule.el,v 3.3 1999/01/18 14:16:50 wedler Exp d029492 $
;; Keywords: WYSIWYG, LaTeX, HTML, wp, math, internationalization
;; X-URL: http://www.fmi.uni-passau.de/~wedler/x-symbol/

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; If you want to use package x-symbol, please visit the URL (use
;; \\[x-symbol-package-web]) and read the info (use \\[x-symbol-package-info]).

;;; Code:

(when (featurep 'mule)
  (error "This file is meant to be used with XEmacs/no-Mule"))
(provide 'x-symbol-nomule)
(require 'x-symbol-hooks)


;;;===========================================================================
;;;  Function aliases and internal variables
;;;===========================================================================

(defalias 'x-symbol-make-cset 'x-symbol-nomule-make-cset)
(defalias 'x-symbol-make-char 'x-symbol-nomule-make-char)
(defalias 'x-symbol-init-charsym-syntax 'ignore)
(defalias 'x-symbol-charsym-after 'x-symbol-nomule-charsym-after)
(defalias 'x-symbol-string-to-charsyms 'x-symbol-nomule-string-to-charsyms)
(defalias 'x-symbol-match-before 'x-symbol-nomule-match-before)
(defalias 'x-symbol-encode-lisp 'x-symbol-nomule-encode-lisp)
(defalias 'x-symbol-pre-command-hook 'x-symbol-nomule-pre-command-hook)
(defalias 'x-symbol-post-command-hook 'x-symbol-nomule-post-command-hook)

(defvar x-symbol-nomule-mouse-yank-function mouse-yank-function
  "Function that is called upon by `x-symbol-nomule-mouse-yank-function'.")

(defvar x-symbol-nomule-mouse-track-function
  (and (boundp 'default-mouse-track-normalize-point-function)
       default-mouse-track-normalize-point-function)
  "Function that is called upon by `x-symbol-nomule-mouse-track-function'.")

(defvar x-symbol-nomule-cstring-regexp "[\232-\237][\041-\176\240-\377]"
  "Internal configuration.  Regexp matching cstrings of length 2.
You should probably change the value when adding additional csets.")
;; should match `x-symbol-nomule-multibyte-char-p'.

(defvar x-symbol-nomule-font-lock-keywords
  '((x-symbol-nomule-match-cstring
     (0 (progn x-symbol-nomule-font-lock-face) prepend)))
  "Internal configuration.  Extra font-lock keywords for XEmacs/no-Mule.")

(defvar x-symbol-nomule-char-table nil
  "Internal.  Map characters to charsyms.")
(defvar x-symbol-nomule-pre-command nil
  "Internal.  Used for pre- and post-command handling.")

(defvar x-symbol-nomule-leading-faces-alist nil
  "Internal.  Alist of leading character with their faces.
Each element looks like (LEADING NORMAL SUBSCRIPT SUPERSCRIPT).")
(defvar x-symbol-nomule-font-lock-face nil
  "Internal.  Face to fontify current font-lock match.")

(defvar x-symbol-nomule-display-table
  ;; display-table via characters table is not implemented in XEmacs yet...
  (let ((table (make-vector 256 nil))
	(i 128))
    (while (< i 160)
      (aset table i "")
      (incf i))
    table)
  "Display table in faces with non-standard charset registry.
It makes the leading characters, range \\200-\\237, invisible.")

(defvar x-symbol-nomule-character-quote-syntax "\\" ; bug in XEmacs
  "Syntax designator for leading characters in cstrings.")


;;;===========================================================================
;;;  Init code
;;;===========================================================================

(defun x-symbol-nomule-init-faces (fonts prefix &optional display-table)
  "Create and return faces for FONTS.
If a font can not be found, return nil for that font.  PREFIX is the
prefix in the name of the new face.  If non-nil, the new faces use
display table DISPLAY-TABLE."
  (let ((suffixes '("-face" "-sub-face" "-sup-face"))
	(docstrings x-symbol-face-docstrings)
	faces font)
    (while suffixes
      (push (if (setq font (x-symbol-try-font-name (car fonts)))
		(let ((face (intern (concat prefix (car suffixes)))))
		  (make-face face (car docstrings))
		  (set-face-font face font)
		  (if display-table (set-face-display-table face display-table))
		  face)
	      (warn "X-Symbol cannot find font in %s" (car fonts))
	      nil)
	    faces)
      (setq fonts (cdr fonts)
	    suffixes (cdr suffixes)
	    docstrings (cdr docstrings)))
    (nreverse faces)))

(defun x-symbol-nomule-make-cset (cset fonts)
  "Define new charsets according to CSET using FONTS.
See `x-symbol-init-cset'.  Return (NORMAL SUBSCRIPT SUPERSCIPT).  Each
element is a face or nil if the corresponding font in FONTS could not be
found.  Return nil, if no default font for that registry could be found."
  (cond ((noninteractive) (list nil))
	((eq (x-symbol-cset-coding cset) x-symbol-default-coding)
	 (or (x-symbol-nomule-init-faces fonts "x-symbol") ; no registry!
	     (list nil)))
	((x-symbol-try-font-name (car fonts))
	 (let* ((faces (x-symbol-nomule-init-faces
			fonts
			(concat "x-symbol-" (x-symbol-cset-registry cset))
			x-symbol-nomule-display-table))
		(leading (x-symbol-cset-leading cset))
		(ass (assq leading x-symbol-nomule-leading-faces-alist)))
	   (if x-symbol-nomule-character-quote-syntax
	       (modify-syntax-entry leading
				    x-symbol-nomule-character-quote-syntax
				    (standard-syntax-table)))
	   (if ass
	       (setcdr ass faces)
	     (push (cons leading faces)
		   x-symbol-nomule-leading-faces-alist))
	   faces))))

(defun x-symbol-nomule-make-char (cset encoding charsym face coding)
  "Define character in CSET with ENCODING, represented by CHARSYM.
The character is considered to be a 8bit character in CODING.  Use FACE
when character is presented in the grid or has a non-standard registry."
  (unless (char-table-p x-symbol-nomule-char-table)
    (setq x-symbol-nomule-char-table (make-char-table 'generic))
    (put-char-table t nil x-symbol-nomule-char-table))
  (let* ((leading (and (null (eq coding x-symbol-default-coding))
		       (cadar cset)))
	 (table (if leading
		    (get-char-table leading x-symbol-nomule-char-table)
		  x-symbol-nomule-char-table))
	 (cstring (if leading
		      (concat (list leading encoding))
		    (char-to-string (int-to-char encoding)))))
    (unless (char-table-p table)
      (setq table (make-char-table 'generic))
      (put-char-table t nil table)
      (put-char-table leading table x-symbol-nomule-char-table))
    (put-char-table encoding charsym table)
    (x-symbol-set-cstrings charsym coding cstring
			   (and coding (>= encoding 160)
				(char-to-string (int-to-char encoding)))
			   face)
    cstring))


;;;===========================================================================
;;;  Character recognition
;;;===========================================================================

(defun x-symbol-nomule-multibyte-char-p (leading octet)
  "Non-nil if LEADING and OCTET are a multibyte character."
  (and leading (>= leading ?\200) (< leading ?\240)
       octet (or (< octet ?\177) (>= octet ?\240)) (>= octet ?\41)))

(defun x-symbol-nomule-charsym-after (&optional pos)
  "Return x-symbol charsym for character at POS.
POS defaults to point.  If POS is out of range, return nil.  Otherwise,
return (POS1 . CHARSYM) where POS1 is POS-1 if the character before POS
is a leading character and POS1 is POS otherwise.  CHARSYM is the
x-symbol charsym for the character at POS1 or nil otherwise."
  (or pos (setq pos (point)))
  (let ((before (char-before pos))
	(after (char-after pos)))
    (and after
	 (if (or (x-symbol-nomule-multibyte-char-p before after)
		 (x-symbol-nomule-multibyte-char-p
		  (setq before after)
		  (setq after (char-after (incf pos)))))
	     (let ((table (get-char-table before x-symbol-nomule-char-table)))
	       (cons (1- pos)
		     (and (char-table-p table) (get-char-table after table))))
	   (cons (1- pos)
		 (and (symbolp (setq after (get-char-table
					    before
					    x-symbol-nomule-char-table)))
		      after))))))

(defun x-symbol-nomule-string-to-charsyms (string)
  "Return list of charsyms for the characters in STRING.
If a character is not represented as a charsym, use the character itself
if is an ascii in the range \\040-\\176, otherwise nil."
  (let ((chars (nreverse (append string nil)))
	result after table)
    (while chars
      (setq after (pop chars))
      (push (if (x-symbol-nomule-multibyte-char-p (car chars) after)
		(and (setq table (get-char-table (pop chars)
						 x-symbol-nomule-char-table))
		     (get-char-table after table))
	      (or (get-char-table after x-symbol-nomule-char-table) after))
	    result))
    result))

(defun x-symbol-nomule-match-before (atree pos &optional case-fn)
  "Return association in ATREE for longest match before POS.
Return (START . VALUE) where the buffer substring between START and
point is the key to the association VALUE in ATREE.  Do not use matches
where the character before START is a leading character.  If optional
CASE-FN is non-nil, convert characters before the current position with
CASE-FN.  See `x-symbol-atree-push'."
  (or pos (setq pos (point)))
  (let ((result nil)
	char)
    (while (setq char (if case-fn
			  (funcall case-fn (char-after (decf pos)))
			(char-after (decf pos)))
		 atree (cdr (assoc char (cdr atree))))
      (and (car atree)
	   (not (x-symbol-nomule-multibyte-char-p (char-before pos) char))
	   (setq result (cons pos (car atree)))))
    result))


;;;===========================================================================
;;;  Point correction
;;;===========================================================================

(and x-symbol-nomule-mouse-yank-function
     (setq mouse-yank-function 'x-symbol-nomule-mouse-yank-function))
(and x-symbol-nomule-mouse-track-function
     (setq default-mouse-track-normalize-point-function
	   'x-symbol-nomule-mouse-track-function))

(defun x-symbol-nomule-goto-leading-char ()
  "If character before point is a leading character, move point left."
  (if (x-symbol-nomule-multibyte-char-p (char-before (point))
					(char-after (point)))
      (backward-char)))

(defun x-symbol-nomule-mouse-yank-function ()
  "Function used as value for `mouse-yank'.
If character under point is a x-symbol character, move point to its
leading character before calling `x-symbol-nomule-mouse-yank-function'."
  (x-symbol-nomule-goto-leading-char)
  (funcall x-symbol-nomule-mouse-yank-function))

(defun x-symbol-nomule-mouse-track-function (type forwardp)
  ;; checkdoc-params: (type forwardp)
  "Function used as value for `default-mouse-track-normalize-point-function'.
After calling `x-symbol-nomule-mouse-track-function', if character under
point is a x-symbol character, move point to its leading character."
  (funcall x-symbol-nomule-mouse-track-function type forwardp)
  (x-symbol-nomule-goto-leading-char))


;;;===========================================================================
;;;  Command hooks
;;;===========================================================================

;; Functions in these hooks are run twice (and more) when pressing a key which
;; runs a keyboard macro, e.g., if [backspace] runs [delete] and [delete] runs
;; `delete-backward-char'.

(defun x-symbol-nomule-pre-command-hook ()
  "Function used in `pre-command-hook' when `x-symbol-mode' is turned on.
Hide revealed characters, see `x-symbol-hide-revealed-at-point'.
Provide input method TOKEN, see `x-symbol-token-input'.  If character
under point is a x-symbol character, move point to its leading character."
  (x-symbol-hide-revealed-at-point)
  (when (and x-symbol-mode (null x-symbol-nomule-pre-command))
    (setq x-symbol-nomule-pre-command
	  (if (x-symbol-nomule-multibyte-char-p (char-before (point))
						(char-after (point)))
	      (prog1 (point) (backward-char))
	    t))
    (x-symbol-token-input)))

(defun x-symbol-nomule-post-command-hook ()
  "Function used in `post-command-hook' when `x-symbol-mode' is turned on.
Provide input method ELECTRIC, see `x-symbol-electric-input'.  Start
idle timer for info in echo area and revealing invisible characters, see
`x-symbol-start-itimer-once'.  Make sure that not only a part of a
length-two cstring has been deleted by the previous command."
  (when (and x-symbol-nomule-pre-command x-symbol-mode)
    (if (stringp (car-safe (car-safe buffer-undo-list)))
	;; i.e., after deleting text (`delete-char',...)
	(let* ((pos (abs (cdar buffer-undo-list)))
	       (str (caar buffer-undo-list))
	       (len (length str))
	       (pre (and (> len 0)
			 (x-symbol-nomule-multibyte-char-p
			  (char-before (point)) (aref str 0))))
	       (post (and (> len 0)
			  (x-symbol-nomule-multibyte-char-p
			   (aref str (1- len)) (char-after pos)))))
	  (if (or pre post)
	      (delete-region (if pre (1- pos) pos) (if post (1+ pos) pos))))
      (and (null (car-safe buffer-undo-list))
	   (integerp x-symbol-nomule-pre-command)
	   (= (point) x-symbol-nomule-pre-command)
	   ;; i.e., after pressing Right
	   (< x-symbol-nomule-pre-command (point-max))
	   (goto-char (1+ x-symbol-nomule-pre-command))))
    (x-symbol-electric-input)
    (if (x-symbol-nomule-multibyte-char-p (char-after (point))
					  (char-after (1+ (point))))
	(forward-char))
    (x-symbol-start-itimer-once))
  (setq x-symbol-nomule-pre-command nil))


;;;===========================================================================
;;;  Font-lock support
;;;===========================================================================

(defun x-symbol-nomule-match-cstring (limit)
  "Match next cstring of length 2 before LIMIT if `x-symbol-mode' is on.
Sets `x-symbol-nomule-font-lock-face' to the face used for this cstring
considering super- and subscripts."
  (when x-symbol-mode
    (let (faces old)
      (block nil
	(while (re-search-forward x-symbol-nomule-cstring-regexp limit t)
	  (setq faces (cdr (assq (char-after (match-beginning 0))
				 x-symbol-nomule-leading-faces-alist))
		old (get-text-property (match-beginning 0) 'face))
	  (or (listp old) (setq old (list old)))
	  (if (setq x-symbol-nomule-font-lock-face
		    (or (and (memq 'x-symbol-sup-face old) (caddr faces))
			(and (memq 'x-symbol-sub-face old) (cadr faces))
			(car faces)))
	      (return t)))))))

(defun x-symbol-nomule-fontify-cstrings ()
  "Fontify all cstrings in buffer even when `x-symbol-mode' is off.
Faces according to the cstrings are prepended to existing face settings.
See also `x-symbol-nomule-match-cstring'."
  (let ((x-symbol-mode t)
	(limit (point-max)))
    (goto-char (point-min))
    (while (x-symbol-nomule-match-cstring limit)
      (font-lock-prepend-text-property (match-beginning 0) (match-end 0)
				       'face
				       x-symbol-nomule-font-lock-face))))


;;;===========================================================================
;;;  Conversion
;;;===========================================================================

(defun x-symbol-nomule-encode-lisp (alist coding language)
  "Encode all characters in buffer to tokens according to ALIST.
Do not encode 8bit characters in CODING.  Characters are converted to
tokens in LANGUAGE.  Character aliases are not encoded!"
  ;; Notation: oN = octet in latin-N, lN = leading byte in latin-N, sN = lN+oN
  ;;   Case 1: standard-coding = latin-1 (o1=s1), file-coding = latin-5
  ;;    1a) o3 = o5:    dangerous += o3 w/ l3, strip += l3
  ;;    1b) o3 <> o5:   s3 -> s5, dangerous += o5 w/ l5, strip += l5
  ;;    1c) o3, no o5:  s3 -> TeX
  ;;    1d) o5:	        dangerous += o5 w/ l5, strip += l5
  ;;    1e) o1 = o5:    -
  ;;    1f) o1 <> o5:   s1 -> s5, dangerous += o5 w/ l5, strip += l5
  ;;    1g) o1, no o5   s1 -> TeX
  ;;   Case 2: standard-coding = latin-1 (o1=s1), file-coding = latin-1
  ;;    2a) o3, no o1:  o3 -> TeX
  ;;    2b) o1:		-
  (let* ((token-shape (cdr (x-symbol-language-value 'x-symbol-token-shape
						    language)))
	 (token-regexp (car token-shape))
	 (letter-regexp (cdr token-shape))
	 dangerous leadings
	 (case-fold-search nil)		; encoding always case sensitive
	 from charsym to normal char prefixes)
    (while alist
      (if (setq from (caar alist)
		charsym (cdar alist)
		alist (cdr alist)
		to (plist-get (get charsym 'x-symbol-file-cstrings) coding))
	  ;; Cases 1a, 1b, 1d, 1e, 1f, 2b
	  (unless (string-equal from to)
	    ;; Cases 1a, 1b, 1d, 1f
	    (unless (string-equal (substring from -1) to)
	      ;; Cases 1b, 1f
	      (setq to (plist-get (get charsym 'x-symbol-buffer-cstrings)
				  coding))
	      (goto-char (point-min))
	      (if (setq prefixes (and dangerous
				      (= (length from) 1)
				      (assq (aref from 0) dangerous)))
		  (while (search-forward from nil 'limit)
		    (unless (memq (char-before (match-beginning 0)) prefixes)
		      (replace-match to t t)))
		(while (search-forward from nil 'limit)
		  (replace-match to t t)))
	      (setq from to))
	    (if (setq prefixes (assq (aref from 1) dangerous))
		(push (aref from 0) (cdr prefixes))
	      (push (list (aref from 1) (aref from 0)) dangerous))
	    (add-to-list 'leadings (substring from 0 1)))
	;; Cases 1c, 1g, 2a
	(goto-char (point-min))
	(if (setq to (plist-get (get charsym 'x-symbol-tokens) language)
		  normal (and token-regexp
			      (string-match token-regexp to))
		  prefixes (and dangerous
				(= (length from) 1)
				(assq (aref from 0) dangerous)))
	    (while (search-forward from nil 'limit)
	      (unless (memq (char-before (match-beginning 0)) prefixes)
		(replace-match to t t)
		(and normal
		     (setq char (char-after (point)))
		     (string-match letter-regexp (char-to-string char))
		     (insert " "))))
	  (while (search-forward from nil 'limit)
	    (replace-match to t t)
	    (and normal
		 (setq char (char-after (point)))
		 (string-match letter-regexp (char-to-string char))
		 (insert " "))))))
    (while leadings
      (setq from (pop leadings))
      (goto-char (point-min))
      (while (search-forward from nil 'limit)
	(replace-match "" t t)))))

;;; Local IspellPersDict: .ispell_xsymb
;;; x-symbol-nomule.el ends here
