;;; x-symbol-exec.el --- create conversion executables for package x-symbol

;; Copyright (C) 1996-1999 Free Software Foundation, Inc.
;;
;; Author: Christoph Wedler <wedler@fmi.uni-passau.de>
;; Maintainer: (Please use `M-x x-symbol-package-bug' to contact the maintainer)
;; Version: $Id: x-symbol-exec.el,v 3.3 1999/01/18 14:14:55 wedler Exp d029492 $
;; Keywords: WYSIWYG, LaTeX, HTML, wp, math, internationalization
;; X-URL: http://www.fmi.uni-passau.de/~wedler/x-symbol/

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; If you want to use package x-symbol, please visit the URL (use
;; \\[x-symbol-package-web]) and read the info (use \\[x-symbol-package-info]).

;; To create executables for decoding and encoding of tokens of all token
;; languages, use:
;;   M-x x-symbol-exec-create

;; This file requires file `x-symbol.el' which does some initialization.  Thus,
;; do not put any `defcustom' commands into this file.  If you think some
;; variables in this files should be customized, move them to file
;; `x-symbol-vars.el'.

;;; Code:

(provide 'x-symbol-exec)
(require 'x-symbol)
(require 'compile)


;;;===========================================================================
;;;  Variables not really useful for customization
;;;===========================================================================

(defvar x-symbol-exec-compile-command (or compile-command "make -k")
  "Shell command used to create the executables.")

(defvar x-symbol-exec-compilation-mode "XS-Compilation"
  "Major mode name in compilation buffer when creating executables.")

(defvar x-symbol-exec-makefile-name "Makefile"
  "Name of the Makefile used to create the executables.")

;; The SUN make utility has the lex library -ll hard-coded in the rule, i.e.,
;; cc is called with "-ll" and "-lfl" there.  I don't care (see file QUESTIONS
;; in the distribution), use GNUs make.
(defvar x-symbol-exec-makefile-pre "\
LEX=flex
LFLAGS=-B -8
LDLIBS=-lfl

all: clean "
;;; or that of v2.5:
;;;RM=rm
;;;LINK=cc
;;;LEX=flex
;;;LFLAGS=-B -8
;;;LDLIBS=-lfl
;;;.l:\t
;;;\t$(RM) $*.c
;;;\t$(LEX) $< > $*.c
;;;\t$(LINK) -o $@ $*.c $(LDLIBS)
;;;\t$(RM) $*.c
;;; before:
;;;.l:\t
;;;\t$(RM) $*.c
;;;\t$(LEX.l) $< > $*.c
;;;\t$(LINK.c) -o $@ $*.c $(LDLIBS)
;;;\t$(RM) $*.c
  "First part of the Makefile used to create the executables.")

(defvar x-symbol-exec-makefile-post "\nclean:\n\trm -f *.c\n"
  "Last part of the Makefile used to create the executables.")

(defvar x-symbol-exec-case-insensitive-spec "%option case-insensitive"
  "Option in declaration part of flex files for case insensitive tokens.")


;;;===========================================================================
;;;  Latin recoding
;;;===========================================================================

(defun x-symbol-exec-latin-tables (array-alist len)
  "Format tables as strings with LEN entries in each line.
Each element in ARRAY-ALIST looks like (CODING . ARRAY) where CODING is
a key in `x-symbol-exec-option-alist' and ARRAY is an array with string
or nil elements."
  (let (result)
    (dolist (coding+array array-alist)
      (push (format "static char *table_%s[] = {\n  "
		    (cdr (assq (car coding+array)
			       x-symbol-exec-option-alist)))
	    result)
      (let ((table (append (cdr coding+array) nil))
	    (pos 0)
	    cstring)
	(while table
	  (setq cstring (pop table))
	  (push (if cstring
		    (concat "\"" (x-symbol-exec-string-octets cstring) "\"")
		  "0")
		result)
	  (push (if table
		    (if (zerop (% (incf pos) len)) ",\n  " ", ")
		  "\n};\n")
		result))))
    (nreverse result)))

(defun x-symbol-exec-latin-specs ()
  "Return latin specifications used for decoding and encoding.
Return (OPTIONS DECODE-TABLES ENCODE-TABLES . ENCODE-INDEXES).  OPTIONS
are main() argument options for codings, the first option is according
to `x-symbol-default-coding'.  DECODE-TABLES define mappings
file-cstring->buffer-cstring for all codings.  ENCODE-TABLES define
mappings index->file-cstrings for all codings and ENCODE-INDEXES define
mappings charsyms->index."
  (let* ((default (assq x-symbol-default-coding
			x-symbol-exec-option-alist))
	 (options (if default
		      (cons default
			    (delete default
				    (copy-sequence
				     x-symbol-exec-option-alist)))
		    x-symbol-exec-option-alist))
	 (decode-arrays (mapcar (lambda (coding)
				  (let ((array (make-vector 96 nil)))
				    (dotimes (i 96)
				      (aset array i (char-to-string
						     (int-to-char (+ i 160)))))
				    (cons (car coding) array)))
				(cdr options)))
	 (encode-index 0)
	 encode-arrays index-alist)
    (unless default
      (error "Option for default coding is not specified!"))
    (dolist (charsym x-symbol-all-charsyms)
      (let ((file-cstrings (get charsym 'x-symbol-file-cstrings))
	    (cstring (get charsym 'x-symbol-cstring)))
	(when file-cstrings
	  (push (cons charsym encode-index) index-alist)
	  (incf encode-index)
	  (x-symbol-do-plist (coding cstring1 file-cstrings)
	    (unless (eq coding x-symbol-default-coding)
	      (aset (cdr (assq coding decode-arrays))
		    (- (char-to-int (aref cstring1 (1- (length cstring1))))
		       160)
		    cstring))))))
    (setq encode-arrays
	  (mapcar (lambda (coding)
		    (cons (car coding) (make-vector encode-index nil)))
		  options))
    (dolist (charsym-index index-alist)
      (x-symbol-do-plist (coding cstring1 (get (car charsym-index)
					       'x-symbol-file-cstrings))
	(aset (cdr (assq coding encode-arrays)) (cdr charsym-index) cstring1)))
    (list* options
	   (x-symbol-exec-latin-tables decode-arrays 4)
	   (x-symbol-exec-latin-tables encode-arrays 8)
	   index-alist)))


;;;===========================================================================
;;;  Sub calls
;;;===========================================================================

(defun x-symbol-exec-string-octets (string)
  "Return STRING with all characters in octal representation."
  (apply #'concat
	 (mapcar (lambda (char) (format "\\%03o" (char-to-int char)))
		 (append string nil))))

(defun x-symbol-exec-create-lex-pre (tables &optional case-insensitive)
  "Create lex buffer for executables.
Insert all lines of TABLES in declaration part, also an option for case
insensitive tokens if optional CASE-INSENSITIVE is non-nil."
  (set-buffer (get-buffer-create " X-Symbol lex file"))
  (if case-insensitive (insert x-symbol-exec-case-insensitive-spec "\n"))
  (erase-buffer)
  (insert "%{\n")
  (while tables (insert (pop tables)))
  (insert "char **table=0;\n%}\n\n%%\n"))

(defun x-symbol-exec-create-lex-post (file options checksum)
  "Finish the creation of a lex file and save it into FILE.
Insert C code for main() with argument options OPTIONS.  When called
with option \"cs\", the executable will output CHECKSUM."
  (insert "\n%%\n
int main(argc, argv)
     int argc;
     char *argv[];
{
  table = 0;
  if (argc==2) {\n    if (strcmp(argv[1],\"cs\")==0) { printf(\""
	  (number-to-string checksum)
	  "\"); exit(0); }\n")
  (while options
    (let ((option (cdr (pop options))))
      (insert (format "    else if (strcmp(argv[1],\"%s\")==0)\
 table = table_%s;\n" option option))))
  (insert "  }\n  while(yylex()); exit (0);\n}\n")
  (if (file-exists-p file) (delete-file file))
  (write-region (point-min) (point-max) file)
  (kill-buffer (current-buffer))
  checksum)


;;;===========================================================================
;;;  Create Makefile, checksum/decode/encode file
;;;===========================================================================

(defun x-symbol-exec-create-decode (language file options latin-tables)
  "Create lex file FILE for LANGUAGE's decode executable.
Executables provides argument options OPTIONS.  Decoding of 8bit
characters uses LATIN-TABLES."
  (let* ((alist (x-symbol-language-value 'x-symbol-decode-alist language))
	 (case-fold-search (x-symbol-language-value 'x-symbol-case-insensitive
						    language))
	 (token-esc (car (x-symbol-language-value 'x-symbol-token-shape
						  language)))
	 (exclude '(?\n))
	 include echo-patterns
	 (from 1))
    (x-symbol-exec-create-lex-pre latin-tables case-fold-search)
    (insert "[\\240-\\377] { fputs(\
table ? table[((unsigned int) (unsigned char) (yytext[0]))-160] : yytext\
, yyout); return 1; }\n")
    (dolist (decode alist)
      (insert (prin1-to-string (car decode))
	      " { fputs(\""
	      (x-symbol-exec-string-octets (get (cdr decode) 'x-symbol-cstring))
	      "\", yyout); return 1; }\n")
      (pushnew (aref (car decode) 0) exclude))
    (setq exclude (sort (cons 128 (mapcar #'char-to-int exclude)) #'<)
	  include (list "[\\200-\\237][\\240-\\377]|.|[")) ; nreverse!
    (dolist (dont exclude)
      (and (<= dont 128)
	   (< from dont)
	   (push (if (= from (1- dont))
		     (format "\\%03o" from)
		   (format "\\%03o-\\%03o" from (1- dont)))
		 include))
      (setq from (1+ dont)))
    (setq echo-patterns
	  (cons (apply #'concat (nreverse (cons "]*\\n?" include)))
		(and token-esc
		     (list (format "\\%03o." (char-to-int token-esc))))))
    (dolist (rule (cdr (x-symbol-language-value 'x-symbol-exec-specs language)))
      (let ((regexp (car rule)))
	(if (or (null regexp)
		(dolist (item alist t)
		  (or (string-match regexp (car item)) (return))))
	    (push (cdr rule) echo-patterns))))
    (insert (mapconcat #'identity (nreverse echo-patterns) "|")
	    " { ECHO; return 1; }\n")
    (x-symbol-exec-create-lex-post
     file options
     (x-symbol-checksum alist
			(lambda (charsym) (get charsym 'x-symbol-cstring))))))

(defun x-symbol-exec-create-encode (language file options latin-spec)
  "Create lex file FILE for LANGUAGE's decode executable.
Executables provides argument options OPTIONS.  Encoding of 8bit chars
uses LATIN-SPEC which looks like (LATIN-TABLES . INDEX-ALIST)."
  (let* ((alist (x-symbol-language-value 'x-symbol-encode-alist language))
	 (index-alist (cdr latin-spec))
	 (case-fold-search nil)
	 (token-regexp (cadr (x-symbol-language-value
			      'x-symbol-token-shape language)))
	 (token-letter (or (car (x-symbol-language-value
				 'x-symbol-exec-specs language))
			   (cddr (x-symbol-language-value
				  'x-symbol-token-shape language)))))
    (x-symbol-exec-create-lex-pre (car latin-spec))
    (dolist (encode alist)
      (let* ((charsym (cdr encode))
	     (cstring (x-symbol-exec-string-octets
		       (get charsym 'x-symbol-cstring)))
	     (token (plist-get (get charsym 'x-symbol-tokens) language))
	     (index (cdr (assq charsym index-alist)))
	     (token-prefix (if index
			       (format "(table && table[%d]) ? table[%d] : "
				       index index)
			     "")))
	(when (and token-regexp
		   (string-match token-regexp token))
	  (insert "\"" cstring "\"" token-letter
		  " { fputs(" token-prefix
		  (prin1-to-string (concat token " "))
		  ", yyout); putc(yytext["
		  (number-to-string (length (get charsym 'x-symbol-cstring)))
		  "], yyout); return 1; }\n"))
	(insert "\"" cstring "\" { fputs(" token-prefix
		(prin1-to-string token)
		", yyout); return 1; }\n")))
    (insert "[\\200-\\237][\\240-\\377]|\
.|[\\001-\\011\\013-\\177]*\\n? { ECHO; return 1; }\n")
    (x-symbol-exec-create-lex-post
     file options
     (x-symbol-checksum alist
			(lambda (charsym)
			  (plist-get (get charsym 'x-symbol-tokens)
				     language))))))

(defun x-symbol-exec-create-checksum (file decode-checksum encode-checksum)
  "Create checksum file FILE for executables.
It stores checksum DECODE-CHECKSUM and ENCODE-CHECKSUM."
  (set-buffer (get-buffer-create " X-Symbol checksum file"))
  (erase-buffer)
  (prin1 x-symbol-default-coding (current-buffer))
  (insert " ")
  (prin1  system-configuration (current-buffer))
  (insert " ")
  (prin1 decode-checksum (current-buffer))
  (insert " ")
  (prin1 encode-checksum (current-buffer))
  (if (file-exists-p file) (delete-file file))
  (write-region (point-min) (point-max) file))

(defun x-symbol-exec-create-makefile (file exec-files)
  "Create Makefile FILE to create executables EXEC-FILES."
  (set-buffer (get-buffer-create " X-Symbol Makefile"))
  (erase-buffer)
  (insert x-symbol-exec-makefile-pre
	  (mapconcat 'identity exec-files " ")
	  x-symbol-exec-makefile-post)
  (if (file-exists-p file) (delete-file file))
  (write-region (point-min) (point-max) file))


;;;===========================================================================
;;;  Create all files
;;;===========================================================================

(defun x-symbol-exec-check-file (file)
  "Return full name of FILE if FILE can be created by you.
Throws `x-symbol-exec-check-file' otherwise."
  (or (file-name-directory file)
      (setq file (expand-file-name file x-symbol-exec-directory)))
  (if (file-writable-p file)
      file
    (warn "X-Symbol executables: cannot write file %S" file)
    (throw 'x-symbol-exec-check-file nil)))

;;;###autoload
(defun x-symbol-exec-create (&optional makefile)
  "Create executables for all registered token languages.
Create lex files and use MAKEFILE to compile them.  MAKEFILE defaults to
`x-symbol-exec-makefile-name'.  Create `x-symbol-exec-directory' when
necessary.  The executables are not used during the current Emacs
session!

Each language should have an access `x-symbol-exec-specs' which looks
like
  (LETTER-LEX-REGEXP ECHO-RULE...) or t
If it is t, no executables are built for that language.

If non-nil, LETTER-LEX-REGEXP is used instead LETTER-REGEXP in language
access `x-symbol-token-shape', this is useful if the lex regexp syntax
requires a different value.  Each ECHO-RULE looks like (REGEXP . ECHO).
The string ECHO is echoed by the decode executable, if REGEXP is nil or
REGEXP matches all tokens in the decode alist of the language.  If
TOKEN-REGEXP/LETTER-REGEXP in access `x-symbol-token-shape' is non-nil,
a unconditional ECHO is likely to be required.

Executables cannot be used under XEmacs/Mule yet."
  (interactive)
  (when (featurep 'mule)
    (error "X-Symbol executables cannot be created under XEmacs/Mule yet"))
  (let ((buffer (get-buffer
		 (concat "*" (downcase x-symbol-exec-compilation-mode) "*"))))
    (and buffer (get-buffer-process buffer)
	 (error "X-Symbol executables are already in creation")))
  (unless (stringp x-symbol-exec-directory)
    (error "X-Symbol executables need `x-symbol-exec-directory'"))
  (if (file-exists-p x-symbol-exec-directory)
      (unless (and (file-accessible-directory-p x-symbol-exec-directory)
		   (file-writable-p x-symbol-exec-directory))
	(error "X-Symbol executables cannot be written into %s"
	       x-symbol-exec-directory))
    (make-directory x-symbol-exec-directory t))
  (or makefile
      (catch 'x-symbol-exec-check-file
	(setq makefile (x-symbol-exec-check-file x-symbol-exec-makefile-name)))
      (error "X-Symbol executables need a makefile"))
  (let ((latin-specs (x-symbol-exec-latin-specs))
	exec-names)
    (save-excursion
      (dolist (language x-symbol-language-alist)
	(catch 'x-symbol-exec-check-file
	  (setq language (car language))
	  (let ((specs (x-symbol-language-value 'x-symbol-exec-specs language))
		;; after this, i.e., after requiring:
		(decode (get language 'x-symbol-decode-exec))
		(encode (get language 'x-symbol-encode-exec)))
	    (if decode (set decode nil))
	    (if encode (set encode nil))
	    (unless (consp specs)
	      (or (eq specs t)
		  (warn "X-Symbol executables for %s: no specification how to build them" language))
	      (throw 'x-symbol-exec-check-file nil))
	    (unless (and decode encode)
	      (warn "X-Symbol executables: no variables for language %s"
		    language)
	      (throw 'x-symbol-exec-check-file nil)))
	  (let* ((checksum-file (x-symbol-exec-check-file
				 (format x-symbol-nomule-checksum-file-format
					 language)))
		 (decode-name (format x-symbol-nomule-decode-exec-format
				      language))
		 (encode-name (format x-symbol-nomule-encode-exec-format
				      language))
		 (decode-exec (x-symbol-exec-check-file decode-name))
		 (encode-exec (x-symbol-exec-check-file encode-name))
		 (decode-lex (x-symbol-exec-check-file
			      (concat decode-exec ".l")))
		 (encode-lex (x-symbol-exec-check-file
			      (concat encode-exec ".l"))))
	    (x-symbol-exec-create-checksum
	     checksum-file
	     (x-symbol-exec-create-decode language decode-lex
					  (cdar latin-specs)
					  (cadr latin-specs))
	     (x-symbol-exec-create-encode language encode-lex
					  (car latin-specs)
					  (cddr latin-specs)))
	    (setq exec-names
		  (nconc exec-names (list decode-name encode-name))))))
      (x-symbol-exec-create-makefile makefile exec-names)))
  (unless noninteractive
    (let ((default-directory (file-name-as-directory x-symbol-exec-directory)))
      (save-some-buffers (not compilation-ask-about-save) nil)
      (compile-internal x-symbol-exec-compile-command
			"No more errors"
			x-symbol-exec-compilation-mode))))

;;; Local IspellPersDict: .ispell_xsymb
;;; x-symbol-exec.el ends here
