/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef FILTERDIALOG_H
#define FILTERDIALOG_H

#include <QDialog>
#include <QList>

#include "attributes/definitioninfo.h"
#include "models/columncondition.h"

class QLabel;
class QComboBox;
class QGridLayout;
class QPushButton;
class QDialogButtonBox;

namespace WebIssues
{

class AbstractValueEditor;

/**
* Dialog for editing a list of column conditions for filtering.
*/
class FilterDialog : public QDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param projectId Optional project identifier for attribute value context.
    *
    * @param parent The parent widget.
    */
    FilterDialog( int projectId, QWidget* parent );

    /**
    * Destructor.
    */
    ~FilterDialog();

public:
    /**
    * Set the prompt text.
    */
    void setPrompt( const QString& text );

    /**
    * Set a list of available columns for filtering.
    */
    void setAvailableColumns( const QList<Column>& columns );

    /**
    * Initialize the dialog with a list of conditions.
    */
    void setConditions( const QList<ColumnCondition>& conditions );

    /**
    * Read the list of conditions from the dialog.
    */
    QList<ColumnCondition> conditions() const;

    /**
    * Enable or disable filter editing mode.
    *
    * In filter editing mode, there is no Apply button and if the name is
    * empty, a dialog asking for filter name is displayed when OK is clicked.
    */
    void setFilterMode( bool mode );

    /**
    * Return @c true if filter editing mode is enabled.
    */
    bool filterMode() const { return m_filterMode; }

    /**
    * Set filter name in filter editing mode.
    */
    void setFilterName( const QString& name );

    /**
    * Return the filter name in filter editing mode.
    */
    const QString& filterName() const { return m_filterName; }

    /**
    * Set the list of existing filter names.
    *
    * A warning will be displayed when an existing filter is overwritten.
    */
    void setExistingFilters( const QStringList& filters );

    /**
    * Return the list of existing filters.
    */
    const QStringList& existingFilters() const { return m_existingFilters; }

signals:
    /**
    * Emitted when the Ok or Apply button is clicked.
    */
    void settingsApplied();

public: // overrides
    void accept();

private slots:
    void addRow();
    void removeRow();
    void clear();

    void columnActivated( int index );
    void conditionActivated( int index );

private:
    struct FilterRow
    {
        QComboBox* m_columnCombo;
        QComboBox* m_conditionCombo;

        DefinitionInfo m_definitionInfo;

        QList<ColumnCondition::Condition> m_conditions;

        AbstractValueEditor* m_editor;
    };

private:
    void clearAllRows();

    void createRow( const ColumnCondition& condition );

    void selectColumn( FilterRow& row, Column column );
    void selectCondition( FilterRow& row, int index, ColumnCondition::Condition condition );

private:
    int m_typeId;
    int m_projectId;

    bool m_filterMode;
    QString m_filterName;

    QStringList m_existingFilters;

    QLabel* m_promptLabel;

    QWidget* m_gridPanel;
    QGridLayout* m_gridLayout;

    QPushButton* m_moreButton;
    QPushButton* m_fewerButton;
    QPushButton* m_clearButton;

    QDialogButtonBox* m_buttonBox;

    QList<Column> m_columns;

    QList<FilterRow> m_rows;
};

}

#endif
