namespace eval gzip {
    variable gzip

    # Define the gzip array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set gzip {
	list.reset ""
	list.clear ""
	entry.file ""
	entry.suffix ""
	entry.use_command ""
	entry.speed_level ""
	dialog ""
	file ""
	ascii ""
	stdout ""
	decompress ""
	force ""
	help ""
	list_contents ""
	license ""
	no_name ""
	name ""
	test ""
	verbose ""
	version ""
	use_command "gzip"
	speed_level 6
	suffix ""
    }
}

# gzip::create --
#
#   Method to create the dialog box for the gzip comgzipd.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your comgzipd, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last comgzipd saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   use_command - The command to use in the CC since this procedure
#                 works for both gzip and gunzip.
#   wm_title    - Use either gzip or gunzip.
#
# Returns
#
#   None.

proc gzip::create { {use_command gzip} {wm_title gzip} } {
    global tkWorld
    variable gzip

    # Do this since we cannot use the gzip array as an argument
    # to the procedure.
    set gzip(use_command) $use_command; unset use_command

    # Put the focus on the gzip dialog if it is already open.
    if [winfo exists $gzip(dialog)] {
	switch -- [wm state $gzip(dialog)] {
	    normal {
		raise $gzip(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $gzip(dialog)
	    }
	}
	focus $gzip(dialog)
	return
    } else {
	set gzip(dialog) [dialog::create .gzip $wm_title]
    }

    # Set the compression speed list with padding for the ddlistbox.
    set speed_list {"1    " "2    " "3    " "4    " "5    " "6    " \
	    "7    " "8    " "9    "}

    # Build the first tab.
    set tab1 [tabnotebook::page [dialog::interior \
	    $gzip(dialog)] "Options"]

    # Use a frame to encapsulate the file selection so that the
    # frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    button $f1.label_file \
	    -text "File" \
	    -width 9 \
	    -command gzip::open
    set gzip(entry.file) [entry $f1.entry_file \
	    -width 35 \
	    -textvariable gzip::gzip(file)]
    label $f1.label_use_command \
	    -text "Use Command" \
	    -width 12 \
	    -anchor e
    set gzip(entry.use_command) [ddlistbox::create $f1 \
	    gzip::gzip(use_command) "gzip gunzip" 7]
    grid $f1.label_file $f1.entry_file \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f1.label_use_command $gzip(entry.use_command) \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]
    checkbutton $f2.ascii \
	    -text "Ascii Text Mode" \
	    -variable gzip::gzip(ascii) \
	    -onvalue "a" \
	    -offvalue ""
    checkbutton $f2.stdout \
	    -text "Standard Output" \
	    -variable gzip::gzip(stdout) \
	    -onvalue "c" \
	    -offvalue ""
    checkbutton $f2.decompress \
	    -text "Decompress File" \
	    -variable gzip::gzip(decompress) \
	    -onvalue "d" \
	    -offvalue ""
    checkbutton $f2.force \
	    -text "Force Compression" \
	    -variable gzip::gzip(force) \
	    -onvalue "f" \
	    -offvalue ""
    checkbutton $f2.help \
	    -text "Display Help" \
	    -variable gzip::gzip(help) \
	    -onvalue "h" \
	    -offvalue ""
    checkbutton $f2.list_contents \
	    -text "List Contents" \
	    -variable gzip::gzip(list_contents) \
	    -onvalue "l" \
	    -offvalue ""
    checkbutton $f2.license \
	    -text "Display License" \
	    -variable gzip::gzip(license) \
	    -onvalue "L" \
	    -offvalue ""
    checkbutton $f2.no_name \
	    -text "No Original Name" \
	    -variable gzip::gzip(no_name) \
	    -onvalue "n" \
	    -offvalue ""
    checkbutton $f2.name \
	    -text "Name" \
	    -variable gzip::gzip(name) \
	    -onvalue "N" \
	    -offvalue ""
    checkbutton $f2.test \
	    -text "Test" \
	    -variable gzip::gzip(test) \
	    -onvalue "t" \
	    -offvalue ""
    checkbutton $f2.verbose \
	    -text "Verbose" \
	    -variable gzip::gzip(verbose) \
	    -onvalue "v" \
	    -offvalue ""
    checkbutton $f2.version \
	    -text "Version" \
	    -variable gzip::gzip(version) \
	    -onvalue "V" \
	    -offvalue ""
    grid $f2.ascii $f2.stdout $f2.decompress \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.force $f2.help $f2.list_contents \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.license $f2.no_name $f2.name \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.test $f2.verbose $f2.version \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Now build the second tab with the suffix and speed entries.
    set tab2 [tabnotebook::page [dialog::interior \
	    $gzip(dialog)] "Advanced"]

    # Build the frame for the tab.
    set f3 [frame $tab2.f3 \
	    -class TabnotebookFrame]

    label $f3.label_speed_level \
	    -text "Speed" \
	    -width 12 \
	    -anchor e
    set gzip(entry.speed_level) [ddlistbox::create $f3 \
	    gzip::gzip(speed_level) $speed_list "5"]
    label $f3.label_suffix \
	    -text "Use Suffix" \
	    -width 12 \
	    -anchor e
    set gzip(entry.suffix) [entry $f3.entry_suffix \
	    -width 30 \
	    -textvariable gzip::gzip(suffix)]
    grid $f3.label_suffix $f3.entry_suffix \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f3.label_speed_level $gzip(entry.speed_level) \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build all of the tabs.
    pack $f1 $f2 $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Focus baby!
    focus $gzip(entry.file)

    # Define the lists for the reset and clear methods
    set gzip(list.reset) "ascii stdout decompress force help \
	    list_contents license no_name name test verbose version"
    set gzip(list.clear) "file use_command speed_level suffix"
}

# gzip::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc gzip::ok { } {
    global tkWorld
    variable gzip

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.
    
    $tkWorld(cmd_center) insert insert "$gzip(use_command) "

    # Build the command line argument.
    set cmd_arg ""
    foreach x $gzip(list.reset) {
	if [string length $gzip($x)] {
	    append cmd_arg $gzip($x)
	}
    }

    # Only insert the compression speed level when using gzip.
    if {$gzip(use_command) == "gzip"} {
	if {$gzip(speed_level) != "6"} {
	    append cmd_arg [string trimright $gzip(speed_level)]
	}
    }

    # Insert the command arguments with right spacing and
    # flag indicator.
    if [string length $cmd_arg] {
	$tkWorld(cmd_center) insert insert "-$cmd_arg "
    }

    # Insert the suffix if the user has entered it.
    if [string length $gzip(suffix)] {
	$tkWorld(cmd_center) insert insert "-S $gzip(suffix) "
    }

    # Insert the users selected files.
    if [string length $gzip(file)] {
	$tkWorld(cmd_center) insert insert "$gzip(file) "
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# gzip::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc gzip::reset { } {
    variable gzip

    # Reset all of the list elements.
    foreach x $gzip(list.reset) {
	set gzip($x) ""
    }
}

# gzip::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc gzip::clear { } {
    variable gzip

    # Reset the data structure elements and bg/fg.
    foreach x $gzip(list.clear) {
	set gzip($x) ""
	switch $x {
	    speed_level {
		set gzip($x) 6
	    }
	    use_command {
		set gzip($x) gzip
	    }
	    default {
		$gzip(entry.$x) configure -bg #ffffff -fg #000000
	    }
	}
    }

    focus $gzip(entry.file)
}

# gzip::help --
#
#   Method to invoke the Gzip Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc gzip::help { } {
    global tkWorld

    help::create "help/gzip.html" "Gzip Command Help"
}

# gzip::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc gzip::close { } {
    variable gzip
    
    balloonhelp::cancel
    destroy $gzip(dialog)
}

# gzip::open --
#
#   Method to open a single target for gzip.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc gzip::open { } {
    variable gzip

    # Delete any existing files in the target entry.
    $gzip(entry.file) delete 0 end

    # Get the gzipfile from the open dialog and move the cursor.
    regsub { } [file::select] {} gzip(file)
    $gzip(entry.file) icursor end
}
