/*
	SRG - Squid Report Generator
	Report by Location
	Copyright 2005 University of Waikato

	This file is part of SRG.

	SRG is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	SRG is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with SRG; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/

#include "UserReport.h"
#include "LocationReport.h"
#include "srg.h"
#include "prototypes.h"

extern int h_errno;

void LocationReport::process_line(const log_line *line) {

	if (srg.debug)
		fprintf(stderr, "In LocationReport::process_line for "
				"location '%s'\n", mName);

	stats.connects++;
	stats.bytesTransferred += line->size;
	stats.timeSpent += line->elapsedTime;
	if (is_cache_hit(line->resultCode)) {
		stats.hits++;
		stats.bytesHit += line->size;
	} else {
		stats.misses++;
		stats.bytesMissed += line->size;
	}
   
	time_t tLine = line->timestamp;
	times.push_back(tLine);

	if (is_cache_denied(line->resultCode)) {
		stats.deniedHits++;
		denied.push_back(true);
	} else {
		denied.push_back(false);
	}

	if (srg.lookupHosts) {
		char *tclient=dnscache->get_name(line->clientAddress);
		if (tclient) {
			host.push_back(tclient);
		} else {
			host.push_back(strdup(inet_ntoa(
							line->clientAddress)));
		}		
	} else {
		host.push_back(strdup(inet_ntoa(line->clientAddress)));
	}

}

void LocationReport::generate_report(const char * basename, const char *user,
					const char *site) {
	
	char *t = NULL;

	/* Generate the HTML file */
	char *defaultname = "^index^";
	char *filename;
	FILE *outfile = NULL;

	if (strcmp(mName, "") == 0) 
		mName = strdup(defaultname);
	t = md5this(mName);
	
	if (srg.outputMode == OUTPUT_PHP) {
		asprintf(&filename, "%s/%s/%s.php", srg.outputDir, basename, t);
	} else {
		asprintf(&filename, "%s/%s/%s.html", srg.outputDir, 
				basename, t);
	}
	free(t);
	
	outfile = fopen(filename, "w");
	if(outfile==NULL) {
		fprintf(stderr,"LocationReport: Cannot open output file: %s\n",
				filename);
		exit(1);
	}
	free(filename);
	
	/* Header & Title */
	if (srg.groupBy > 0) {
		html_header(outfile, "../../../");
	} else {
		html_header(outfile, "../../");
	}
	
	fprintf(outfile, "<!-- SRG %s (%s) Generated LocationReport -->\n", 
			version, HOME_URL);
	
	/* Misc Stats */
	fprintf(outfile, "<center><table cellpadding=2 cellspacing=2>");
	fprintf(outfile, "<tr><td class=\"bodyText\">Period:</td><td class=\""
			"bodyText\">%d %s %d",
			localtime(&srg.startTime)->tm_mday, 
			month_names[localtime(&srg.startTime)->tm_mon], 
			localtime(&srg.startTime)->tm_year+1900);
	fprintf(outfile, " - %d %s %d</td></tr>", 
			localtime(&srg.endTime)->tm_mday, 
			month_names[localtime(&srg.endTime)->tm_mon], 
			localtime(&srg.endTime)->tm_year+1900);
	if (srg.groupBy > 0) {
		fprintf(outfile, "<tr><td class=\"bodyText\">Group:</td><td "
				"class=\"bodyText\">%s</td></tr>", user);
	}
	fprintf(outfile, "<tr><td align=\"right\" class=\"bodyText\">"
			"Site:\n</td><td class=\"bodyText\"><a href=\"http://%s/\">%s"
			"</a></td></tr>", site, site);
	fprintf(outfile, "</table></center>");
	
	/* Notices Row */
	fprintf(outfile, "<div align=\"center\" id=\"srg-message\">"
			"&nbsp;</div>\n");
	
	fprintf(outfile, "<center><table cellpadding=4 cellspacing=0><tr>"
			"<th>LOCATION</th><th>REQUESTS</th><th>BYTES</th>"
			"<th>HIT</th><th>MISS</th>");
	if (srg.showtimes)
		fprintf(outfile, "<th>TIME(ms)</th>");	
	if (srg.showrates)
		fprintf(outfile, "<th>RATE (kb/s)</th>");
	fprintf(outfile, "</tr><tr><td class=\"cellText\"><a href=\""
			"http://%s%s\">", site, 
			strcasecmp(mName, defaultname)==0 ? "/" : mName);
	if (strcasecmp(mName, defaultname)==0) {
		fprintf(outfile, "/");
	} else {
		t = mName;
		unsigned int tlen = strlen(t);
		for (unsigned int i=0;i<tlen;i+=30) {
			if (i>0)
				fprintf(outfile, "<br>");
			char tmp[31];
			strncpy(tmp, t, 30);
			tmp[30] = '\0';
			fprintf(outfile, "%s", tmp);
			t+=30;
		}
	}
	fprintf(outfile, "</a></td>");
	t = FormatOutput(stats.connects);
	fprintf(outfile, "<td class=\"cellNum\">%s</td>", t);
	free(t);
	t = FormatOutput(stats.bytesTransferred);
	fprintf(outfile, "<td class=\"cellNum\">%s</td>", t);
	free(t);
	float percentin=0, percentout=0;
    if (stats.bytesTransferred>0 && stats.connects>0) {
        percentin = ((float)stats.bytesHit/
                (float)stats.bytesTransferred)*100.0;
        percentout = ((float)stats.bytesMissed/
                (float)stats.bytesTransferred)*100.0;
    }
	fprintf(outfile, "<td class=\"cellNum\">%2.2f%%</td>", percentin);
	fprintf(outfile, "<td class=\"cellNum\">%2.2f%%</td>", percentout);
	if (srg.showtimes) {
		t = FormatOutput(stats.timeSpent);
		fprintf(outfile, "<td class=\"cellNum\">%s</td>", t);
		free(t);
	}
	if (srg.showrates) {
		fprintf(outfile, "<td class=\"cellNum\">%.2f</td>", 
			(float) stats.bytesTransferred/stats.timeSpent);
	}
	fprintf(outfile, "</tr></table></center><br>");

	/* Times Accessed Table */
	fprintf(outfile, "\t<center><table border=0 cellspacing=2 "
			"cellpadding=2 id=\"srgtable\">\n");
	fprintf(outfile, "\t\t<thead><tr>\n");
	fprintf(outfile, "\t\t<th>Time(s) Accessed</th>\n");
	fprintf(outfile, "\n\t\t<th>Requested by</th>\n");
	fprintf(outfile, "\t\t<th>Error</th>\n");
	fprintf(outfile, "\t\t</tr></thead>\n");

	list<time_t>::const_iterator iter;
	list<bool>::const_iterator deniediter;
	list<char*>::const_iterator hostiter;
	hostiter=host.begin();
	
	deniediter=denied.begin();
	int rows = 0;

	times.sort();
	for (iter=times.begin(); iter != times.end(); iter++) {
		time_t thistime = (*iter);
		fprintf(outfile, "\t<tr%s><td class=\"bodyText\">%s</td>",
				((rows%2)==0) ? " class=\"highlightRow\"" : "",
				ctime(&thistime));
		
		fprintf(outfile, "<td class=\"bodyText\">%s</td>", 
				(*hostiter));
		
		fprintf(outfile, "<td class=\"bodyText\">%s</td></tr>\n", 
				(*deniediter) ? "DENIED" : "");
		deniediter++;
	
		hostiter++;
		rows++;
	}
	fprintf(outfile, "\n\t</table><br>");

	if (srg.authenticate) {
		fprintf(outfile, "\n\t<?php } else {\n report_error(\""
				"Could not authenticate user\");\n}?>");
	}

	fprintf(outfile, "\n\t<br><a href=\"%s\">Back</a> to locations "
            "page</center>", srg.indexfname);

	/* Finish off the HTML */
	if (srg.groupBy > 0) {
		html_footer(outfile, "../../../");
	} else {
		html_footer(outfile, "../../");
	}
	fclose(outfile);

}

// Get statistics about this report
summary_info LocationReport::getStats() {

	summary_info returnInfo;

	returnInfo = stats;

	return returnInfo;

}
