/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * wsdl-skels.c: Emit skeleton code for servers
 *
 * Authors:
 *	Dick Porter (dick@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "wsdl-parse.h"
#include "wsdl-soap-skels.h"
#include "wsdl-trace.h"
#include "wsdl-soap-emit.h"

static void
wsdl_emit_soap_skels_binding_operation (FILE * out, const guchar * opns,
					const guchar * opnsuri,
					const wsdl_binding_operation * const op)
{
	wsdl_porttype_operation *porttype_op;

	g_assert (op->name != NULL);
	g_assert (op->soap_operation != NULL);

	fprintf (out, "/* BEGIN Binding Operation %s */\n\n", op->name);

	porttype_op = op->thread_soap_porttype_operation;
	g_assert (porttype_op != NULL);
	g_assert (porttype_op->input != NULL);

	if (op->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", op->documentation->str);
	}

	if (porttype_op->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", porttype_op->documentation->str);
	}

	/* Emit the soup server callback */
	fprintf (out,
		 "static void %s_%s_soup_server(SoupMessage *_msg, gpointer _user_data)\n",
		 opns, 
		 op->name);
	fprintf (out, "{\n");
	fprintf (out,
		 "\t_wsdl_skels_callback_data_t *_cb_data = (_wsdl_skels_callback_data_t *) _user_data;\n");
	fprintf (out, 
		 "\t%s_%s_server_callback _callback=NULL;\n", 
		 opns,
		 op->name);

	if (porttype_op->input != NULL) {
		fprintf (out, "\t/* more args here based on input parts */\n");
		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t%t _in_%p;\n");
	}

	if (porttype_op->output != NULL) {
		fprintf (out, "\t/* more args here based on output parts */\n");
		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t%t _out_%p;\n");
	}

	if (porttype_op->input != NULL) {
		fprintf (out, "\twsdl_param _in_params[]={\n");
		fprintf (out, "\t\t/* params here based on input parts */\n");

		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t\t{\"%p\", &_in_%p, &WSDL_TC_%n_%N_struct},\n");

		fprintf (out, "\t\t{NULL, NULL, NULL},\n");
		fprintf (out, "\t};\n");
	}

	if (porttype_op->output != NULL) {
		fprintf (out, "\twsdl_param _out_params[]={\n");
		fprintf (out, "\t\t/* params here based on output parts */\n");

		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t\t{\"%p\", &_out_%p, &WSDL_TC_%n_%N_struct},\n");

		fprintf (out, "\t\t{NULL, NULL, NULL},\n");
		fprintf (out, "\t};\n");
	}

	fprintf (out, "\tSoupEnv *env = soup_env_new ();\n");
	fprintf (out, "\n");

	/* Are request headers interesting to clients? */

	fprintf (out, "\t/* parse msg->request.body into callback params */\n");
	fprintf (out,
		 "\twsdl_soap_parse(_msg->request.body, \"%s\", _in_params, env, WSDL_SOAP_FLAGS_REQUEST);\n",
		 op->name);

	fprintf (out, "\n");
	fprintf (out, "\tif(_cb_data!=NULL) {\n");
	fprintf (out,
		 "\t\t_callback = (%s_%s_server_callback)_cb_data->callback;\n",
		 opns, 
		 op->name);
	fprintf (out, "\t\tif (_callback!=NULL) {\n");

	if (porttype_op->output != NULL) {
		fprintf (out, "\t\t\t/* Zero the output params */\n");
		fprintf (out, "\t\t\twsdl_soap_initialise(_out_params);\n");
	}
	//fprintf(out, "\t\t\t/* Add all headers to SoupEnv */\n");
	//fprintf(out, "\t\t\tg_hash_table_foreach(_msg->request_headers, (GHFunc) add_request_header, env);\n");
	fprintf (out, "\n");
	fprintf (out, "\t\t\t_callback(env\n");

	if (porttype_op->input != NULL) {
		fprintf (out, "\t\t\t\t/* input args go here */\n");
		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t\t\t\t, _in_%p\n");
	}

	if (porttype_op->output != NULL) {
		fprintf (out, "\t\t\t\t/* output args go here */\n");

		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t\t\t\t, &_out_%p\n");
	}

	fprintf (out, "\t\t\t\t, _cb_data->user_data);\n");
	fprintf (out, "\t\t}\n");
	fprintf (out, "\t}\n");

	if (porttype_op->output != NULL) {
		fprintf (out, "\tif(_callback!=NULL) {\n");
		fprintf (out,
			 "\t\t/* Now marshal the output args into a SOAP reply */\n");
		fprintf (out,
			 "\t\twsdl_soap_marshal(\"%s\", \"%s\", \"%s\", _out_params, &_msg->response, env, WSDL_SOAP_FLAGS_RESPONSE);\n",
			 op->name, 
			 opns, 
			 opnsuri);

		if (porttype_op->output != NULL) {
			fprintf (out,
				 "\t\t/* And free the returned memory */\n");
			fprintf (out, "\t\twsdl_soap_free(_out_params);\n");
		}

		fprintf (out, "\t} else {\n");
		fprintf (out,
			 "\t\t/* Now marshal a \"Not implemented\" fault */\n");
		fprintf (out, "\t}\n");
	}

	fprintf (out, "\n");
	fprintf (out, "\t/* Free SoupEnv struct */\n");
	fprintf (out, "\tsoup_env_free(env);\n");

	fprintf (out, "}\n");
	fprintf (out, "\n");

	/* Emit the register functions */
	fprintf (out,
		 "void %s_%s_soup_register(%s_%s_server_callback callback, gpointer user_data)\n",
		 opns, 
		 op->name, 
		 opns, 
		 op->name);
	fprintf (out, "{\n");
	fprintf (out, "\t_wsdl_skels_callback_data_t *cb_data;\n");
	fprintf (out, "\n");
	fprintf (out, "\tcb_data=g_new0(_wsdl_skels_callback_data_t, 1);\n");
	fprintf (out, "\tcb_data->callback=callback;\n");
	fprintf (out, "\tcb_data->user_data=user_data;\n");
	fprintf (out,
		 "\tsoup_server_register(\"%s\", %s_%s_soup_server, cb_data);\n",
		 op->name, 
		 opns, 
		 op->name);
	fprintf (out, "}\n");
	fprintf (out, "\n");
	fprintf (out,
		 "void %s_%s_soup_register_with_auth(%s_%s_server_callback callback, gpointer user_data, gint allow_types, SoupServerAuthorizeFn auth_cb, gpointer auth_user_data)\n",
		 opns, 
		 op->name, 
		 opns, 
		 op->name);
	fprintf (out, "{\n");
	fprintf (out, "\t%s_%s_soup_register(callback, user_data);\n", 
		 opns,
		 op->name);
	fprintf (out,
		 "\tsoup_server_set_method_auth(\"%s\", allow_types, auth_cb, auth_user_data);\n",
		 op->name);
	fprintf (out, "}\n");

	fprintf (out, "\n\n");

	fprintf (out, "/* END Binding Operation %s */\n\n", op->name);
}

static void
wsdl_emit_soap_skels_binding (FILE                      *out, 
			      const guchar              *opns,
			      const guchar              *opnsuri,
			      const wsdl_binding * const binding)
{
	GSList *iter;
	wsdl_porttype *porttype;

	g_assert (binding->name != NULL);

	fprintf (out, "/* BEGIN Binding %s */\n\n", binding->name);

	if (binding->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", binding->documentation->str);
	}

	porttype = binding->thread_soap_porttype;
	g_assert (porttype != NULL);

	/* For each binding operation, output a function */
	iter = binding->operations;
	while (iter != NULL) {
		wsdl_emit_soap_skels_binding_operation (out, 
							opns, 
							opnsuri,
							iter->data);

		iter = iter->next;
	}

	fprintf (out, "/* END Binding %s */\n\n", binding->name);
}

static void
wsdl_emit_soap_skels_service (FILE                      *out, 
			      const guchar              *opns,
			      const guchar              *opnsuri,
			      const wsdl_service * const service)
{
	GSList *iter;

	g_assert (service->name != NULL);

	fprintf (out, "/* BEGIN Service %s */\n\n", service->name);

	if (service->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", service->documentation->str);
	}

	iter = service->thread_soap_ports;
	while (iter != NULL) {
		wsdl_service_port *port = iter->data;

		g_assert (port->thread_soap_binding != NULL);

		wsdl_emit_soap_skels_binding (out, 
					      opns, 
					      opnsuri,
					      port->thread_soap_binding);

		iter = iter->next;
	}

	fprintf (out, "/* END Service %s */\n\n", service->name);
}

/**
 * wsdl_emit_soap_skels:
 * @outdir: a string containing the path to a directory.  This
 * function expects the string to have a trailing '/'.
 * @fileroot: a string containing the root of a filename.  "-skels.c"
 * will be appended to this name.
 * @definitions: a pointer to a #wsdl_definitions structure,
 * containing a set of WSDL elements.
 *
 * Creates the file @outdir/@fileroot-skels.c, and writes C code
 * containing server skeletons.
 */
void
wsdl_emit_soap_skels (const guchar                  *outdir, 
		      const guchar                  *fileroot,
		      const wsdl_definitions * const definitions)
{
	FILE *out;
	GSList *iter;
	guchar *filename;
	const guchar *opns;

	filename = g_strconcat (outdir, fileroot, "-skels.c", NULL);
	wsdl_debug (WSDL_LOG_DOMAIN_SKELS, 
		    G_LOG_LEVEL_DEBUG, 
		    "file: [%s]",
		    filename);

	out = fopen (filename, "w");
	g_free (filename);

	if (out == NULL) {
		g_warning ("Couldn't open %s for writing: %s", 
			   filename,
			   strerror (errno));
		return;
	}

	fprintf (out, "/*\n");
	if (definitions->name != NULL) {
		fprintf (out, " * %s\n", definitions->name);
		opns = definitions->name;
	} else {
		opns = "m";
	}

	fprintf (out, " *\n");
	fprintf (out, " * Automatically generated by soup-wsdl.\n");
	fprintf (out, " */\n");
	fprintf (out, "\n");
	fprintf (out, "#include <glib.h>\n");
	fprintf (out, "#include <string.h>\n");
	fprintf (out, "#include <libsoup/soup.h>\n");
	fprintf (out, "#include <libwsdl/wsdl.h>\n");
	fprintf (out, "#include \"%s.h\"\n\n", fileroot);

	if (definitions->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", definitions->documentation->str);
	}

	/* Declare structure for passing callback data */
	fprintf (out, "typedef struct {\n");
	fprintf (out, "\tgpointer callback;\n");
	fprintf (out, "\tgpointer user_data;\n");
	fprintf (out, "} _wsdl_skels_callback_data_t;\n\n");

	/* Private functions */
	//fprintf(out, "/* g_hash_table_foreach function to add a request header */\n");
	//fprintf(out, "static void add_request_header (gpointer key, gpointer value, gpointer data)\n");
	//fprintf(out, "{\n");
	//fprintf(out, "\tSoupEnv *env = (SoupEnv *) data;\n");
	//fprintf(out, "\n");
	//fprintf(out, "\tsoup_env_set_request_header(env, (const gchar *) key, (const gchar *) data);\n");
	//fprintf(out, "}\n\n");

	iter = definitions->thread_soap_services;
	while (iter != NULL) {
		wsdl_emit_soap_skels_service (out, 
					      opns,
					      definitions->targetNamespace,
					      iter->data);

		iter = iter->next;
	}

	fclose (out);
}
