/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * wsdl-common.c: Emit code common to both stubs and skels
 *
 * Authors:
 *	Dick Porter (dick@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <glib.h>
#include <string.h>
#include <errno.h>

#include <libwsdl/wsdl-typecodes.h>
#include <libwsdl/wsdl-typecodes-c.h>

#include "wsdl-soap-common.h"
#include "wsdl-trace.h"

static void
wsdl_emit_soap_common_typecodes (const wsdl_typecode * const tc,
				 gpointer                    user_data)
{
	FILE *out = (FILE *) user_data;

	wsdl_typecode_write_c_definition (out, tc);
}

static void
wsdl_emit_soap_common_mm (const wsdl_typecode * const tc, gpointer user_data)
{
	FILE *out = (FILE *) user_data;

	wsdl_typecode_write_c_mm (out, tc);
}

/**
 * wsdl_emit_soap_common:
 * @outdir: a string containing the path to a directory.  This
 * function expects the string to have a trailing '/'.
 * @fileroot: a string containing the root of a filename.  "-common.c"
 * will be appended to this name.
 * @definitions: a pointer to a #wsdl_definitions structure,
 * containing a set of WSDL elements.
 *
 * Creates the file @outdir/@fileroot-common.c, and writes C code
 * containing typecode definitions, memory management functions, and
 * any other generated code that is common to both client stubs and
 * server skeletons.
 */
void
wsdl_emit_soap_common (const guchar                  *outdir, 
		       const guchar                  *fileroot,
		       const wsdl_definitions * const definitions)
{
	FILE *out;
	guchar *filename;
	const guchar *namespace;

	filename = g_strconcat (outdir, fileroot, "-common.c", NULL);
	wsdl_debug (WSDL_LOG_DOMAIN_COMMON, G_LOG_LEVEL_DEBUG, "file: [%s]",
		    filename);

	out = fopen (filename, "w");
	g_free (filename);

	if (out == NULL) {
		g_warning ("Couldn't open %s for writing: %s", filename,
			   strerror (errno));
		return;
	}

	fprintf (out, "/*\n");
	if (definitions->name != NULL) {
		fprintf (out, " * %s\n", definitions->name);
		namespace = definitions->name;
	} else {
		namespace = "";
	}

	fprintf (out, " *\n");
	fprintf (out, " * Automatically generated by soup-wsdl.\n");
	fprintf (out, " */\n");
	fprintf (out, "\n");

	fprintf (out, "#include <glib.h>\n");
	fprintf (out, "#include <libsoup/soup.h>\n");
	fprintf (out, "#include <libwsdl/wsdl.h>\n");
	fprintf (out, "#include \"%s.h\"\n\n", fileroot);

	if (definitions->documentation != NULL) {
		fprintf (out, "/* %s */\n", definitions->documentation->str);
		fprintf (out, "\n");
	}

	fprintf (out, "\n");

	/* 
	 * Call wsdl_typecode_write_c_definition() for each non-simple
	 * typecode known
	 */
	wsdl_typecode_foreach (FALSE, wsdl_emit_soap_common_typecodes, out);
	wsdl_typecode_foreach (FALSE, wsdl_emit_soap_common_mm, out);

	fclose (out);
}
