/* SNAC
 * Copyright (C) 1999 the Free Software Foundation
 * Authors : Matias Mutchinick, Jan Struyf          
 *         
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <glib.h>
#include <math.h>
#include <errno.h>
#include "errorstring.h"
#include "ag_defs.h"
#include "ag_math.h"


extern int errno;


gint
ag_math_overflow(gdouble  num)
{
	if(num < 0)
		num = -num;
	
  	if( num > MAX_VALUE )
		return 1;
	else 
		return 0;
}


gint
ag_math_underflow(gdouble  num)
{
	if(num < 0)
		num = -num;
	
  	if( num < MIN_VALUE )
		return 1;
	else 
		return 0;
}



gdouble
to_radians(gdouble number, gint mode)
{
	switch(mode) {
	case RAD:
		break;
	case GRAD:
		number = number * 0.0209439510239;  
		break;
	case DEG:
		number = number * 0.0174532925199;  
	}
	
	return number;
}


gdouble 
from_radians_to(gdouble number, gint mode)
{
	switch(mode) {
	case RAD:
		break;
	case GRAD:
		number = number * 47.7464829276;  
		break;
	case DEG:
		number = number * 57.29577795131;  
	}
	
	return number;
}



gdouble 
ag_cos(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = cos(to_radians(arg,mode));
	
	return res;
}


gdouble 
ag_sin(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = sin(to_radians(arg,mode));
	
	return res;
}


gdouble 
ag_tan(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = tan(to_radians(arg,mode));
	
	if(errno)
		*error = UNDEFINED_RESULT;
	
	return res;
}



gdouble 
ag_cosh(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = cosh(to_radians(arg,mode));

	if(errno)
		*error = UNDEFINED_RESULT;
	
	return res;
}


gdouble 
ag_sinh(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = sinh(to_radians(arg,mode));
	
	if(errno)
		*error = UNDEFINED_RESULT;

	return res;
}


gdouble 
ag_tanh(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = tanh(to_radians(arg,mode));

	if(errno)
		*error = UNDEFINED_RESULT;
	
	return res;
}



gdouble 
ag_acos(gdouble arg, gint mode, gint *error)
{
	gdouble res = 0;
	
	if(arg > 1 || arg < -1)
		*error = UNDEFINED_RESULT;	
	else
		res = from_radians_to(acos(arg),mode);
	
	return res;
}



gdouble 
ag_asin(gdouble arg, gint mode, gint *error)
{
	gdouble res = 0;
	
	if(arg > 1 || arg < -1)
		*error = UNDEFINED_RESULT;	
	else
		res = from_radians_to(asin(arg),mode);
	
	return res;
}



gdouble 
ag_atan(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = from_radians_to(atan(arg),mode);
	
	if(errno)
		*error = UNDEFINED_RESULT;
	
	return res;
}




gdouble 
ag_ln(gdouble arg, gint mode, gint *error)
{
	gdouble res = 0;
	
	if(arg == 0)
		*error = INFINITE_RESULT;
	else if (arg < 0 )
		*error = UNDEFINED_RESULT;
	else
		res = log(arg);
	
	return res;
}

  


gdouble 
ag_log(gdouble arg, gint mode, gint *error)
{
	gdouble res = 0;
	
	if(arg == 0)
		*error = INFINITE_RESULT;
	else if (arg < 0 )
		*error = UNDEFINED_RESULT;
	else
		res = log10(arg);
	
	return res;
}

  


gdouble 
ag_exp(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = exp(arg);
	
	if(res > 1e300)
		*error = AG_OVERFLOW;
	
	return res;
}

  


gdouble 
ag_abs(gdouble arg, gint mode, gint *error)
{
	gdouble res = 0;
	
	res = fabs(arg);
	
	return res;
}

  


gdouble 
ag_neg(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	res = -arg;
	
	return res;
}

  


gdouble 
ag_inv(gdouble arg, gint mode, gint *error)
{
	gdouble res;
	
	if(arg == 0){
		*error = INFINITE_RESULT;
		return 0;
	} else
		res = 1/arg;
	
	return res;
}



gdouble 
ag_sq(gdouble arg, gint mode, gint *error)
{
	gdouble res;
		
	res = pow(arg,2);
	
	if(ag_math_overflow(res)){
		*error = AG_OVERFLOW;
		return 0;
	} else if (ag_math_underflow(res))
		return 0;
	
	return res;
}



gdouble 
ag_sqrt(gdouble arg, gint mode, gint *error)
{
	gdouble res = 0;
	
	if(arg < 0)
		*error = UNDEFINED_RESULT;
	else
		res = sqrt(arg);
	
	return res;
}


gdouble
ag_fact(gdouble arg, gint mode, gint *error)
{
	gdouble res = 0;
	
	if (arg < 0){
		*error = FACTORIAL_NOT_POSITIVE;
		return 0;
	}

	if ( arg - ((gulong) arg) != 0 ){
		*error = FACTORIAL_NOT_INT;
		return 0;
	}
	
	if (arg > 166){
		*error = AG_OVERFLOW;
		return 0;
	}
	
	if ( arg > 0)
		res = arg * ag_fact(arg-1,mode,error);
	else
		res = 1;
	
	return res;
}
	
gdouble
ag_mult(gdouble ar1, gdouble ar2, gint *error) {
	return ar1 * ar2;
}

gdouble
ag_div(gdouble ar1, gdouble ar2, gint *error) {
	if (ar2 == 0) { /* Division by 0 not allowed */
		if (ar1 == 0)
			*error = UNDEFINED_RESULT;
		else
			*error = INFINITE_RESULT;
		return 0;
	} 
	return ar1 / ar2;
}

gdouble
ag_pow(gdouble ar1, gdouble ar2, gint *error) {
	gdouble result;
	
	result = pow(ar1, ar2);
	if (errno) { /* Inf or NaN: ERROR */
		if(errno == 33)
			*error = UNDEFINED_RESULT;
		else 
			*error = AG_OVERFLOW;
		return 0;
	}
	return result;
}

gdouble
ag_plus(gdouble ar1, gdouble ar2, gint *error) {
	return ar1 + ar2;
}

gdouble
ag_minus(gdouble ar1, gdouble ar2, gint *error) {
	return ar1 - ar2;
}

gdouble
ag_logic(gchar op, gdouble ar1, gdouble ar2, gint *error) {
	gulong l1, l2, r;
	
	if (ar1 < 0.0 || ar2 < 0.0) {
		*error = LOGIC_NOT_POSITIVE;
		return 0;
	}
	
	l1 = floor(ar1);
	l2 = floor(ar2);	
	
	switch (op) {
		case OP_OR: 
			r = l1 | l2;
			break;
		case OP_AND: 
			r = l1 & l2;
			break;
		case OP_XOR: 
			r = l1 ^ l2;
			break;
		case OP_SHL:
			r = l1 << l2;
			break;
		case OP_SHR:
			r = l1 >> l2;
			break;						
		default:
			r = 0.0;
			break;			
	}
	return (gdouble)r;
}

gdouble
ag_or(gdouble ar1, gdouble ar2, gint *error) {
	return ag_logic(OP_OR, ar1, ar2, error);
}

gdouble
ag_and(gdouble ar1, gdouble ar2, gint *error) {
	return ag_logic(OP_AND, ar1, ar2, error);}

gdouble
ag_xor(gdouble ar1, gdouble ar2, gint *error) {
	return ag_logic(OP_XOR, ar1, ar2, error);
}

gdouble
ag_shr(gdouble ar1, gdouble ar2, gint *error) {
	return ag_logic(OP_SHR, ar1, ar2, error);
}

gdouble
ag_shl(gdouble ar1, gdouble ar2, gint *error) {
	return ag_logic(OP_SHL, ar1, ar2, error);
}
