/*==================================================================
 * uif_selections.c - Sound font item selection management routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include "uif_selections.h"
#include "uif_sfont.h"
#include "uif_sftree.h"
#include "glade_interface.h"
#include "uif_pianospan.h"
#include "wavetable.h"
#include "sfundo.h"
#include "sfdofunc.h"
#include "smurfcfg.h"
#include "util.h"
#include "i18n.h"

/* WARNING: Ugly code ahead, proceed with caution :) */

enum { UISEL_START, UISEL_CHANGE, UISEL_REPLACE, UISEL_KEEP, UISEL_CANCEL };

/* hashes to group duplicates and "remember" user choices */
static GHashTable *inst_hash = NULL;
static GHashTable *sample_hash = NULL;

/* current item awaiting user input */
static SFNodeType paste_item_type;
static gpointer paste_item_data;

/* dialog widget awaiting user input */
GtkWidget *dup_item_dialog;

/* user requested action for all items of type.. */
static gint preset_all_mode;
static gint inst_all_mode;
static gint sample_all_mode;

static GSList *uisel_clone_preset (GSList *lpset, UISFont *dstuisf,
				    UISFont *srcuisf, gint mode);
static GSList *uisel_clone_inst (GSList *linst, UISFont *dstuisf,
				  UISFont *srcuisf, gint mode);
static GSList *uisel_clone_sample (GSList *lsam, UISFont *dstuisf,
				   UISFont *srcuisf, gint mode);
static GSList *uisel_new_preset_from_inst (SFTreeRef *dstref, SFTreeRef *srcref,
			UISFont *dstuisf, UISFont *srcuisf, gint mode);
static GSList *uisel_new_inst_from_sample (SFTreeRef *dstref, SFTreeRef *srcref,
			UISFont *dstuisf, UISFont *srcuisf, gint mode);
static GSList *uisel_paste_pzone (GtkCTreeNode *pnode, SFTreeRef *srcref,
				UISFont *dstuisf, UISFont *srcuisf, gint mode);
static GSList *uisel_paste_izone (GtkCTreeNode *pnode, SFTreeRef *srcref,
				UISFont *dstuisf, UISFont *srcuisf, gint mode);
static gpointer uisel_vbank_paste (GtkCTreeNode *dstnode, SFPreset *pset,
				   SFData *srcsf);
static void uisel_dup_item_dialog (gchar *lbl1, ...);
static gboolean uisel_cb_dup_item_destroy (GtkWidget *dialog);
static void uisel_cb_preset_change_okay (GtkWidget *cfgdialog);
static void uisel_cb_inst_change_okay (GtkWidget *cfgdialog);
static void uisel_cb_sample_change_okay (GtkWidget *cfgdialog);

static void uisel_cb_dup_zone (gpointer data, GtkWidget *popdog);
static gboolean uisel_cb_dup_zone_destroy (GtkWidget *popdog);

static void uisel_cb_dialog_selection_delete_okay (gpointer data,
						   GtkWidget *popdog);

void
uisel_paste_items_start (void)
{
  if (sftree_rclicked_itemid == SFITEMID_NONE) return;

  uisel_paste_items (sftree_rclicked_itemid, UISEL_START);
}

/* pastes appropriate selected items to a node */
void
uisel_paste_items (SFItemID itemid, gint mode)
{
  static GtkCTreeNode *dstnode;
  static GList *sel;
  static GList *selp;

  UISFont *dstuisf, *srcuisf;
  SFTreeRef *dstref, *srcref;
  GSList *p;
  gint dsttype, srctype;
  GtkCTreeNode *n;
  gpointer retp;

  if (mode == UISEL_START)	/* start? */
    {
      if (!(dstnode = SFTREE_LOOKUP_ITEMID (itemid))) return;

      sel = selp = sftree_get_selection ();

      inst_hash = g_hash_table_new (NULL, NULL);
      sample_hash = g_hash_table_new (NULL, NULL);

      preset_all_mode = inst_all_mode = sample_all_mode = UISEL_START;

      sfdo_group (_("Paste sound font items"));

      SFTREE_FREEZE ();
    }

  dstuisf = SFTREE_UPFIND_UISF (dstnode);

  dstref = SFTREE_NODE_REF (dstnode);
  dsttype = dstref->type;

  while (selp)			/* loop over selection */
    {
      /* get CTree node from item ID */
      n = SFTREE_LOOKUP_ITEMID (GPOINTER_TO_INT (selp->data));
      if (!n) {			/* node been removed? skip */
	mode = UISEL_START;
	selp = g_list_next (selp);
	continue;
      }

      srcuisf = SFTREE_UPFIND_UISF (n);

      srcref = SFTREE_NODE_REF (n);
      p = srcref->dptr;
      srctype = srcref->type;

      if (srctype == NODE_PRESET && (dsttype == NODE_SFONT
          || dsttype == NODE_PRESETROOT || dsttype == NODE_MELODIC
          || dsttype == NODE_PERCUSS)) 	/* clone preset */
	retp = uisel_clone_preset (p, dstuisf, srcuisf, mode);
      else if (srctype == NODE_INST  /* clone instrument */
	       && (dsttype == NODE_SFONT || dsttype == NODE_INSTROOT))
	retp = uisel_clone_inst (p, dstuisf, srcuisf, mode);
      else if (srctype == NODE_SAMPLE && (dsttype == NODE_SFONT
	  || dsttype == NODE_SAMPLEROOT || dsttype == NODE_USER
	  || dsttype == NODE_ROM)) /* clone sample */
	retp = uisel_clone_sample (p, dstuisf, srcuisf, mode);
      else if (dsttype == NODE_PRESET && /* paste preset zone */
	       (srctype == NODE_INST || srctype == NODE_PZONE))
	retp = uisel_paste_pzone (dstnode, srcref, dstuisf, srcuisf, mode);
      else if (dsttype == NODE_INST && /* paste instrument zone */
	       (srctype == NODE_SAMPLE || srctype == NODE_IZONE))
	retp = uisel_paste_izone (dstnode, srcref, dstuisf, srcuisf, mode);
      else if (srctype == NODE_INST && (dsttype == NODE_PRESETROOT
	  || dsttype == NODE_MELODIC
	  || dsttype == NODE_PERCUSS)) /* create preset from instrument */
	retp =
	  uisel_new_preset_from_inst (dstref, srcref, dstuisf, srcuisf, mode);
      else if (srctype == NODE_SAMPLE /* create instrument from sample */
	       && (dsttype == NODE_INSTROOT))
	retp =
	  uisel_new_inst_from_sample (dstref, srcref, dstuisf, srcuisf, mode);
      else if (srctype == NODE_PRESET && (dsttype == NODE_VBANK
	  || dsttype == NODE_VBNK_MAPROOT || dsttype == NODE_VBNK_MAP))
	{			/* paste preset to vbank */
	  uisel_vbank_paste (dstnode, (SFPreset *)(p->data), srcuisf->sf);
	}
      else if (srctype == NODE_SFONT && dsttype == NODE_VBNK_DEFBANK)
	{
	  UIVBank *uivb = SFTREE_UPFIND_UIVB (dstnode);
	  gchar *s = vbank_base_fname (srcuisf->sf->fname);
	  vbank_set_defsf (uivb->vbnk, s);
	  g_free (s);
	}
      else
	retp = GINT_TO_POINTER (1); /* A non NULL value */

      if (mode == UISEL_CANCEL)	/* User requested Cancel? */
	{
	  sfdo_retract ();	/* undo partially copied selection */
	  SFTREE_THAW ();
	  g_hash_table_destroy (sample_hash);
	  g_hash_table_destroy (inst_hash);
	  g_list_free (sel);
	  return;
	}

      if (!retp) return;	/* only NULL if waiting for user input */

      mode = UISEL_START;
      selp = g_list_next (selp);
    }

  sfdo_done ();
  SFTREE_THAW ();
  g_hash_table_destroy (sample_hash);
  g_hash_table_destroy (inst_hash);
  g_list_free (sel);
}

static GSList *
uisel_clone_preset (GSList *lpset, UISFont *dstuisf, UISFont *srcuisf,
		    gint mode)
{
  SFZone *z;
  GSList *linst;
  GtkCTreeNode *node;
  static SFPreset *dup;
  static GSList *p;
  static gboolean check;
  static GSList *duplpset;

  if (mode == UISEL_START)
    {
      dup = sfont_preset_dup ((SFPreset *)(lpset->data), SFDUP_NORMAL);
      p = dup->zone;
      check = FALSE;
    }

  if (!check)
    {
      if (mode == UISEL_START || mode == UISEL_CHANGE)
	/* check for duplicate preset (including itself) */
	if ((duplpset = sfont_find_preset (dstuisf->sf, dup->name,
					   dup->bank, dup->prenum, NULL)))
	  {
	    if (preset_all_mode == UISEL_START || mode != UISEL_START)
	      {		/* if no all mode or mode not UISEL_START */
		SFPreset *dup2 = (SFPreset *)(duplpset->data);
		gchar *s1, *s2, *s3, *s4;

		paste_item_type = NODE_PRESET;
		paste_item_data = dup;

		s1 = g_strdup_printf ("%d", dup->bank);
		s2 = g_strdup_printf ("%d", dup2->bank);
		s3 = g_strdup_printf ("%d", dup->prenum);
		s4 = g_strdup_printf ("%d", dup2->prenum);

		/* Duplicate preset dialog: Change, Replace, Keep, Cancel */
		uisel_dup_item_dialog (_("Type"), _("Preset"), NULL,
				       _("Name"), dup->name, dup2->name,
				       _("Bank"), s1, s2,
				       _("Preset"), s3, s4,
				       _("Sound font"),
				       sfont_get_info (srcuisf->sf, INAM_ID),
				       sfont_get_info (dstuisf->sf, INAM_ID),
				       _("File name"),
				       srcuisf->sf->fname,
				       dstuisf->sf->fname,
				       NULL);

		g_free (s1); g_free (s2); g_free (s3); g_free (s4);

		return (NULL);	/* return to wait for user input */
	      }
	    else mode = preset_all_mode; /* user requested all mode */
	}

      switch (mode)
	{
	case UISEL_REPLACE:	/* Replace duplicate preset (not done yet) */
	  break;
	case UISEL_KEEP:	/* Keep existing duplicate preset */
	  sfont_preset_destroy (dup);
	  return (duplpset);
	case UISEL_CANCEL:	/* Cancel paste operation */
	  sfont_preset_destroy (dup);
	  return (NULL);
	}
      check = TRUE;		/* flag duplicate check as done */
      mode = UISEL_START;	/* Reset mode */
    }

  /* if src and dest sound font is different, copy instruments/samples too */
  if (dstuisf != srcuisf)
    {
      while (p)			/* loop over preset zones */
	{
	  z = (SFZone *)(p->data);
	  linst = NULL;

	  /* if zone has an inst then duplicate it */
	  if (z->instsamp)
	    {
	      linst = uisel_clone_inst (z->instsamp, dstuisf, srcuisf, mode);

	      if (mode == UISEL_CANCEL)	/* catch cancel request */
		sfont_preset_destroy (dup);

	      /* uisel_clone_inst is waiting for user input or canceled */
	      if (!linst) return (NULL);

	      mode = UISEL_START; /* Reset mode */
	    }
	  z->instsamp = linst;	/* assign instrument pointer */
	  p = g_slist_next (p);
	}
    }

  /* add the duplicated preset to the destination sound font and the sftree */
  sfont_add_preset (dstuisf->sf, dup);
  duplpset = g_slist_find (dstuisf->sf->preset, dup);
  node = sftree_add_preset_sorted (duplpset, dstuisf->nodes);

  /* tell sfundo about our new item */
  dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

  return (duplpset);
}

static GSList *
uisel_clone_inst (GSList *linst, UISFont *dstuisf, UISFont *srcuisf, gint mode)
{
  SFZone *z;
  GSList *lsam;
  GtkCTreeNode *node;
  static SFInst *dup;
  static GSList *p;
  static gboolean check;
  static GSList *duplinst;

  if (mode == UISEL_START)
    {
      /* check if this instrument already processed */
      if ((duplinst = g_hash_table_lookup (inst_hash, linst)))
	return (duplinst);

      dup = sfont_inst_dup ((SFInst *)(linst->data), SFDUP_NORMAL);
      p = dup->zone;
      check = FALSE;
    }

  if (!check)
    {
      if (mode == UISEL_START || mode == UISEL_CHANGE)
	/* check for duplicate instrument (including itself) */
	if ((duplinst = sfont_find_inst (dstuisf->sf, dup->name, NULL)))
	  {
	    if (inst_all_mode == UISEL_START || mode != UISEL_START)
	      {		/* if no all mode or mode not UISEL_START */
		paste_item_type = NODE_INST;
		paste_item_data = dup;

		/* Duplicate inst dialog: Change, Replace, Keep, Cancel */
		uisel_dup_item_dialog (_("Type"), _("Instrument"), NULL,
				       _("Name"), dup->name, NULL,
				       _("Sound font"),
				       sfont_get_info (srcuisf->sf, INAM_ID),
				       sfont_get_info (dstuisf->sf, INAM_ID),
				       _("File name"),
				       srcuisf->sf->fname,
				       dstuisf->sf->fname,
				       NULL);
		return (NULL);	/* return to wait for user input */
	      }
	    else mode = inst_all_mode; /* user requested all mode */
	  }

      switch (mode)
	{
	case UISEL_REPLACE:	/* Replace duplicate inst (not done yet) */
	  break;
	case UISEL_KEEP:	/* Keep existing duplicate inst */
	  /* add ptr to inst hash of the destination inst to keep, it will be
	     used for all references to this intrument */
	  g_hash_table_insert (inst_hash, linst, duplinst);
	  sfont_inst_destroy (dup);
	  return (duplinst);
	case UISEL_CANCEL:	/* Cancel paste operation */
	  sfont_inst_destroy (dup);
	  return (NULL);
	}
      check = TRUE;		/* flag duplicate check as done */
      mode = UISEL_START;	/* Reset mode */
    }

  /* if src and dest sound font is different, copy samples too */
  if (dstuisf != srcuisf)
    {
      while (p)
	{			/* loop over inst zones */
	  z = (SFZone *)(p->data);
	  lsam = NULL;

	  /* if zone has a sample duplicate it */
	  if (z->instsamp)
	    {
	      lsam = uisel_clone_sample (z->instsamp, dstuisf, srcuisf, mode);

	      if (mode == UISEL_CANCEL)	/* catch cancel request */
		sfont_inst_destroy (dup);

	      /* uisel_clone_sample is waiting for user input or canceled */
	      if (!lsam) return (NULL);

	      mode = UISEL_START; /* Reset mode */
	    }
	  z->instsamp = lsam;	/* assign sample pointer */
	  p = g_slist_next (p);
	}
    }

  /* add the duplicated inst to the destination sound font and the sftree */
  sfont_add_inst (dstuisf->sf, dup);
  duplinst = g_slist_find (dstuisf->sf->inst, dup);
  node = sftree_add_inst (duplinst, dstuisf->nodes, SFTREE_NODE_APPEND);

  /* add ptr to this duplicated inst in inst hash to be used for all
     references to the old duplicated inst */
  g_hash_table_insert (inst_hash, linst, duplinst);

  /* tell sfundo about our new item */
  dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

  return (duplinst);
}

static GSList *
uisel_clone_sample (GSList *lsam, UISFont *dstuisf, UISFont *srcuisf,
		    gint mode)
{
  GtkCTreeNode *node;
  static SFSample *dup;
  static GSList *duplsam;

  if (mode == UISEL_START)
    {
      /* check if this sample has already been processed */
      if ((duplsam = g_hash_table_lookup (sample_hash, lsam)))
	return (duplsam);

      dup = sfont_sample_dup ((SFSample *)(lsam->data), SFDUP_NORMAL, TRUE);
    }

  if (mode == UISEL_START || mode == UISEL_CHANGE)
    /* check for duplicate sample (including itself) */
    if ((duplsam = sfont_find_sample (dstuisf->sf, dup->name, NULL)))
      {
	if (sample_all_mode == UISEL_START || mode != UISEL_START)
	  {			/* if no all mode or mode not UISEL_START */
	    paste_item_type = NODE_SAMPLE;
	    paste_item_data = dup;

	    /* Duplicate sample dialog: Change, Replace, Keep, Cancel */
	    uisel_dup_item_dialog (_("Type"), _("Sample"), NULL,
				   _("Name"), dup->name, NULL,
				   _("Sound font"),
				   sfont_get_info (srcuisf->sf, INAM_ID),
				   sfont_get_info (dstuisf->sf, INAM_ID),
				   _("File name"),
				   srcuisf->sf->fname,
				   dstuisf->sf->fname,
				   NULL);
	    return (NULL);	/* return to wait for user input */
	  }
	else mode = sample_all_mode; /* user requested all mode */
      }

  switch (mode)
    {
    case UISEL_REPLACE:
      return (NULL);
    case UISEL_KEEP:
      /* add ptr to sample hash of the destination sample to keep, it will be
	 used for all references to this sample */
      g_hash_table_insert (sample_hash, lsam, duplsam);
      sfont_sample_destroy (dup, TRUE);
      return (duplsam);
    case UISEL_CANCEL:		/* Cancel paste operation */
      sfont_sample_destroy (dup, TRUE);
      return (NULL);
    }

  /* add the duplicated sample to the destination sound font and the sftree */
  sfont_add_sample (dstuisf->sf, dup);
  duplsam = g_slist_find (dstuisf->sf->sample, dup);
  node = sftree_add_sample (duplsam, dstuisf->nodes, SFTREE_NODE_APPEND);

  /* add ptr to this duplicated sample in sample hash to be used for all
     references to the old duplicated sample */
  g_hash_table_insert (sample_hash, lsam, duplsam);

  /* tell sfundo about our new item */
  dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

  return (duplsam);
}

static GSList *
uisel_new_preset_from_inst (SFTreeRef *dstref, SFTreeRef *srcref,
			    UISFont *dstuisf, UISFont *srcuisf, gint mode)
{
  static GSList *instlp;
  static SFPreset *pset;
  static gboolean check;

  SFInst *inst;
  SFZone *zone;
  GSList *psetlp, *p;
  GtkCTreeNode *node;

  if (mode == UISEL_START)
    {
      gint bank, prenum;

      instlp = (GSList *)(srcref->dptr);

      pset = sfont_preset_alloc ();
      strcpy (pset->name, ((SFInst *)(instlp->data))->name);

      /* find a free preset */
      sfont_find_free_preset (dstuisf->sf, &bank, &prenum,
			      dstref->type != NODE_PERCUSS);
      pset->bank = bank;
      pset->prenum = prenum;

      check = FALSE;
    }

  inst = (SFInst *)(instlp->data);

  if (!check)
    {
      if (mode == UISEL_START || mode == UISEL_CHANGE)
	{	
	  /* check for duplicate preset */
	  if ((psetlp = sfont_find_preset (dstuisf->sf, pset->name,
					   pset->bank, pset->prenum, NULL)))
	    {
	      if (preset_all_mode == UISEL_START || mode == UISEL_CHANGE)
		{		/* if no all mode or mode not UISEL_START */
		  SFPreset *dup = (SFPreset *)(psetlp->data);
		  gchar *s1, *s2, *s3, *s4;

		  paste_item_type = NODE_PRESET;
		  paste_item_data = pset;

		  s1 = g_strdup_printf ("%d", pset->bank);
		  s2 = g_strdup_printf ("%d", dup->bank);
		  s3 = g_strdup_printf ("%d", pset->prenum);
		  s4 = g_strdup_printf ("%d", dup->prenum);

		  /* Duplicate preset dialog: Change, Replace, Keep, Cancel */
		  uisel_dup_item_dialog (_("Type"), _("Preset"), NULL,
					 _("Name"), pset->name, dup->name,
					 _("Bank"), s1, s2,
					 _("Preset"), s3, s4,
					 _("Sound font"),
					 sfont_get_info (srcuisf->sf, INAM_ID),
					 sfont_get_info (dstuisf->sf, INAM_ID),
					 _("File name"),
					 srcuisf->sf->fname,
					 dstuisf->sf->fname,
					 NULL);

		  g_free (s1); g_free (s2); g_free (s3); g_free (s4);

		  return (NULL);	/* return to wait for user input */
		}
	      else mode = preset_all_mode; /* user requested all mode */
	    }
	}

      switch (mode)
	{
	case UISEL_REPLACE:	/* Replace duplicate preset (not done yet) */
	  break;
	case UISEL_KEEP:	/* Keep existing duplicate preset */
	  sfont_preset_destroy (pset);
	  return (psetlp);
	case UISEL_CANCEL:	/* Cancel paste operation */
	  sfont_preset_destroy (pset);
	  return (NULL);
	}
      check = TRUE;		/* flag duplicate check as done */
      mode = UISEL_START;	/* Reset mode */
    }

  if (dstuisf != srcuisf)	/* not a global zone, same sound font? */
    {				/* duplicate the instrument */
      p = uisel_clone_inst (instlp, dstuisf, srcuisf, mode);

      if (mode == UISEL_CANCEL)
	sfont_preset_destroy (pset);

      if (!p) return (NULL);	/* waiting for user input or cancel? */
    }
  else p = instlp;

  /* add the preset to the destination sound font and the sftree */
  sfont_add_preset (dstuisf->sf, pset);
  psetlp = g_slist_find (dstuisf->sf->preset, pset);
  node = sftree_add_preset_sorted (psetlp, dstuisf->nodes);

  /* create new zone */
  zone = sfont_zone_alloc ();
  zone->instsamp = p;

  /* add preset zone */
  sfont_add_preset_zone (dstuisf->sf, pset, zone);
  p = g_slist_find (pset->zone, zone);
  sftree_add_pzone (p, node, -1);

  /* tell sfundo about our new item */
  dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

  return (psetlp);
}

static GSList *
uisel_new_inst_from_sample (SFTreeRef *dstref, SFTreeRef *srcref,
			    UISFont *dstuisf, UISFont *srcuisf, gint mode)
{
  static GSList *samlp;
  static SFInst *inst;
  static gboolean check;

  SFSample *sam;
  SFZone *zone;
  GSList *instlp, *p;
  GtkCTreeNode *node;

  if (mode == UISEL_START)
    {
      samlp = (GSList *)(srcref->dptr);

      inst = sfont_inst_alloc ();
      strcpy (inst->name, ((SFSample *)(samlp->data))->name);

      check = FALSE;
    }

  sam = (SFSample *)(samlp->data);

  if (!check)
    {
      if (mode == UISEL_START || mode == UISEL_CHANGE)
	{	
	  /* check for duplicate instrument */
	  if ((instlp = sfont_find_inst (dstuisf->sf, inst->name, NULL)))
	    {
	      if (inst_all_mode == UISEL_START || mode == UISEL_CHANGE)
		{		/* if no all mode or mode not UISEL_START */
		  paste_item_type = NODE_INST;
		  paste_item_data = inst;

		  /* Duplicate inst dialog: Change, Replace, Keep, Cancel */
		  uisel_dup_item_dialog (_("Type"), _("Instrument"), NULL,
					 _("Name"), inst->name, NULL,
					 _("Sound font"),
					 sfont_get_info (srcuisf->sf, INAM_ID),
					 sfont_get_info (dstuisf->sf, INAM_ID),
					 _("File name"),
					 srcuisf->sf->fname,
					 dstuisf->sf->fname,
					 NULL);

		  return (NULL);	/* return to wait for user input */
		}
	      else mode = preset_all_mode; /* user requested all mode */
	    }
	}

      switch (mode)
	{
	case UISEL_REPLACE:	/* Replace duplicate inst (not done yet) */
	  break;
	case UISEL_KEEP:	/* Keep existing duplicate inst */
	  sfont_inst_destroy (inst);
	  return (instlp);
	case UISEL_CANCEL:	/* Cancel paste operation */
	  sfont_inst_destroy (inst);
	  return (NULL);
	}
      check = TRUE;		/* flag duplicate check as done */
      mode = UISEL_START;	/* Reset mode */
    }

  if (dstuisf != srcuisf)	/* not a global zone, same sound font? */
    {				/* duplicate the sample */
      p = uisel_clone_sample (samlp, dstuisf, srcuisf, mode);

      if (mode == UISEL_CANCEL)
	sfont_inst_destroy (inst);

      if (!p) return (NULL);	/* waiting for user input or cancel? */
    }
  else p = samlp;

  /* add the instrument to the destination sound font and the sftree */
  sfont_add_inst (dstuisf->sf, inst);
  instlp = g_slist_find (dstuisf->sf->inst, inst);
  node = sftree_add_inst (instlp, dstuisf->nodes, -1);

  /* create new zone */
  zone = sfont_zone_alloc ();
  zone->instsamp = p;

  /* add instrument zone */
  sfont_add_inst_zone (dstuisf->sf, inst, zone);
  p = g_slist_find (inst->zone, zone);
  sftree_add_izone (p, node, -1);

  /* tell sfundo about our new item */
  dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

  return (instlp);
}

static GSList *
uisel_paste_pzone (GtkCTreeNode *pnode, SFTreeRef *srcref, UISFont *dstuisf,
		   UISFont *srcuisf, gint mode)
{
  SFPreset *pset;
  GSList *p, *p2, *duplzone;
  SFZone *z;
  GtkCTreeNode *node;
  GtkWidget *popdog;

  /* get pointer to SFPreset from preset sftree node */
  pset = (SFPreset *)(((GSList *)(SFTREE_NODE_REF (pnode)->dptr))->data);

  /* pointer to instrument or preset zone list item */
  p = (GSList *)(srcref->dptr);

  if (srcref->type == NODE_INST) /* item is an instrument? */
    {
      if (dstuisf != srcuisf)	/* different sound font? */
	{			/* duplicate the instrument */
	  p = uisel_clone_inst (p, dstuisf, srcuisf, mode);
	  if (!p) return (NULL); /* waiting for user input or cancel? */
	}

      if (mode == UISEL_CANCEL) return (NULL); /* catch cancel request */

      z = sfont_new_preset_zone (dstuisf->sf, pset, p);
    }
  else if (srcref->type == NODE_PZONE) /* item is preset zone? */
    {
      z = (SFZone *) (p->data);	/* get preset zone */
      p2 = NULL;

      /* Global Zone? */
      if (z->instsamp)
	{
	  if (dstuisf != srcuisf) /* not a global zone, same sound font? */
	    {			/* duplicate the instrument */
	      p2 = uisel_clone_inst (z->instsamp, dstuisf, srcuisf, mode);
	      if (!p2) return (NULL); /* waiting for user input or cancel? */
	    }
	}
      else			/* its a global zone */
	{
	  switch (mode)
	    {
	    case UISEL_START:
	      /* preset already has Global Zone? */
	      if (pset->zone && !((SFZone *)(pset->zone->data))->instsamp)
		{
		  /* Duplicate global zone dialog: Replace, Keep, Cancel */
		  popdog = 
		    util_quick_popup (_("Duplicate global preset zone"),
				      _("Keep"), uisel_cb_dup_zone,
				      GINT_TO_POINTER (UISEL_KEEP),
				      _("Cancel"), uisel_cb_dup_zone,
				      GINT_TO_POINTER (UISEL_CANCEL),
				      NULL);
		  gtk_signal_connect (GTK_OBJECT (popdog), "destroy",
				      (GtkSignalFunc)uisel_cb_dup_zone_destroy,
				      NULL);
		  return (NULL); /* return to wait for user input */
		}
	      break;
	    case UISEL_REPLACE:	/* not done yet (fall through) */
	    case UISEL_KEEP:
	      return (p);

	    default:
	      break;
	    }
	}

      if (mode == UISEL_CANCEL) return (NULL); /* catch cancel request */

      z = sfont_zone_dup (z, SFDUP_NORMAL); /* duplicate the pzone */

      /* if dstuisf != srcuisf, point to the new instrument */
      if (p2) z->instsamp = p2;

      sfont_add_preset_zone (dstuisf->sf, pset, z);
    }

  /* add zone to sftree */
  duplzone = g_slist_find (pset->zone, z);
  node = sftree_add_pzone (duplzone, pnode, SFTREE_NODE_APPEND);

  /* tell sfundo about our new zone */
  dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

  if (pset == uisf_selected_elem)
    pianospan_update ();	/* update key spans */

  wtbl_sfitem_changed (pset->itemid, WTBL_ITEM_CHANGE);

  return (duplzone);
}

/* Instrument zone paste */
static GSList *
uisel_paste_izone (GtkCTreeNode *pnode, SFTreeRef *srcref, UISFont *dstuisf,
		   UISFont *srcuisf, gint mode)
{
  SFInst *inst;
  GSList *p, *p2, *duplzone;
  SFZone *z;
  GtkCTreeNode *node;
  GtkWidget *popdog;

  /* get pointer to SFInst from instrument sftree node */
  inst = (SFInst *)(((GSList *)(SFTREE_NODE_REF (pnode)->dptr))->data);

  /* pointer to sample or instrument zone list item */
  p = (GSList *)(srcref->dptr);

  if (srcref->type == NODE_SAMPLE) /* item is a sample? */
    {
      if (dstuisf != srcuisf)	/* different sound font? */
	{			/* duplicate the sample */
	  p = uisel_clone_sample (p, dstuisf, srcuisf, mode);
	  if (!p) return (NULL); /* waiting for user input or cancel? */
	}

      if (mode == UISEL_CANCEL) return (NULL); /* catch cancel request */

      z = sfont_new_inst_zone (dstuisf->sf, inst, p);
    }
  else if (srcref->type == NODE_IZONE) /* item is instrument zone? */
    {
      z = (SFZone *) (p->data);	/* get instrument zone */
      p2 = NULL;

      /* Global Zone? */
      if (z->instsamp)
	{
	  if (dstuisf != srcuisf) /* not a global zone, same sound font? */
	    {			/* duplicate the sample */
	      p2 = uisel_clone_sample (z->instsamp, dstuisf, srcuisf, mode);
	      if (!p2) return (NULL); /* waiting for user input or cancel? */
	    }
	}
      else			/* its a global zone */
	{
	  switch (mode)
	    {
	    case UISEL_START:
	      /* instrument already has Global Zone? */
	      if (inst->zone && !((SFZone *)(inst->zone->data))->instsamp)
		{
		  /* Duplicate global zone dialog: Replace, Keep, Cancel */
		  popdog =
		    util_quick_popup (_("Duplicate global instrument zone"),
				      _("Keep"), uisel_cb_dup_zone,
				      GINT_TO_POINTER (UISEL_KEEP),
				      _("Cancel"), uisel_cb_dup_zone,
				      GINT_TO_POINTER (UISEL_CANCEL),
				      NULL);
		  gtk_signal_connect (GTK_OBJECT (popdog), "destroy",
				      (GtkSignalFunc)uisel_cb_dup_zone_destroy,
				      NULL);
		  return (NULL); /* return to wait for user input */
		}
	      break;
	    case UISEL_REPLACE:	/* not done yet (fall through) */
	    case UISEL_KEEP:
	      return (p);

	    default:
	      break;
	    }
	}

      if (mode == UISEL_CANCEL) return (NULL); /* catch cancel request */

      z = sfont_zone_dup (z, SFDUP_NORMAL); /* duplicate the izone */

      /* if dstuisf != srcuisf, point to the new sample */
      if (p2) z->instsamp = p2;

      sfont_add_inst_zone (dstuisf->sf, inst, z);
    }

  /* add zone to sftree */
  duplzone = g_slist_find (inst->zone, z);
  node = sftree_add_izone (duplzone, pnode, SFTREE_NODE_APPEND);

  /* tell sfundo about our new zone */
  dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);

  if (inst == uisf_selected_elem)
    pianospan_update ();	/* update key spans */

  wtbl_sfitem_changed (inst->itemid, WTBL_ITEM_CHANGE);

  return (duplzone);
}

static gpointer
uisel_vbank_paste (GtkCTreeNode *dstnode, SFPreset *pset, SFData *srcsf)
{
  UIVBank *uivb;
  VBnkItem *item;
  SFTreeRef *ref;
  GSList *lmap;
  gint bank = 0, preset = 0;

  ref = SFTREE_NODE_REF (dstnode);
  uivb = SFTREE_UPFIND_UIVB (dstnode);

  /* if preset was pasted to another map, find the next highest free
     map bank:preset */
  if (ref->type == NODE_VBNK_MAP)
    {
      VBnkItem *item = (VBnkItem *)(((GSList *)(ref->dptr))->data);
      bank = item->map.bank;
      preset = item->map.psetnum;
      if (++preset > 127)
	{
	  preset = 0;
	  bank++;
	}
    }

  vbank_find_free_map (&bank, &preset, uivb->vbnk);

  item = vbank_add_map (bank, preset, -1, pset->bank, pset->prenum, -1,
			vbank_base_fname (srcsf->fname), uivb->vbnk);
  lmap = g_slist_find (uivb->vbnk->items, item);
  sftree_add_vbank_map_sorted (lmap, uivb->nodes);

  return (GINT_TO_POINTER (1));
}

/* creates duplicate item dialog and inserts label, src value and dest
   value triplet parameters.
   (gchar *lbl1, gchar *srcval1, gchar *dstval1, gchar *lbl2, ...) */
static void
uisel_dup_item_dialog (gchar *lbl1, ...)
{
  va_list ap;
  GtkText *src, *dst;
  gchar *s, *s2;

  dup_item_dialog = create_uisel_dup_dialog ();
  src = gtk_object_get_data (GTK_OBJECT (dup_item_dialog), "TXTsrc");
  dst = gtk_object_get_data (GTK_OBJECT (dup_item_dialog), "TXTdst");

  s = lbl1;
  va_start (ap, lbl1);

  while (s)
    {
      s = g_strconcat (s, ": ", NULL);
      gtk_text_insert (src, NULL, NULL, NULL, s, -1);
      gtk_text_insert (dst, NULL, NULL, NULL, s, -1);
      g_free (s);

      s = va_arg (ap, gchar *);
      s = g_strconcat (s, "\n", NULL);
      gtk_text_insert (src, NULL, NULL, NULL, s, -1);

      s2 = va_arg (ap, gchar *);
      if (!s2) s2 = s;
      else s2 = g_strconcat (s2, "\n", NULL);
      gtk_text_insert (dst, NULL, NULL, NULL, s2, -1);

      if (s != s2) g_free(s2);
      g_free (s);

      s = va_arg (ap, gchar *);
    }

  /* if dup window gets destroyed, cancel paste operation */
  gtk_signal_connect (GTK_OBJECT (dup_item_dialog), "destroy",
		      (GtkSignalFunc)uisel_cb_dup_item_destroy, NULL);

  gtk_widget_show (dup_item_dialog);
}

static gboolean
uisel_cb_dup_item_destroy (GtkWidget *dialog)
{
  uisel_paste_items (SFITEMID_NONE, UISEL_CANCEL);

  return (FALSE);
}

/* gets user choice from duplicate item dialog and restarts paste routine */
void
uisel_cb_dup_item (GtkWidget *btn, gint num)
{
  gint mode;
  gboolean all = FALSE;
  SFPreset *pset;
  SFInst *inst;
  SFSample *sam;
  GtkWidget *w;

  switch (num)
    {
    case 0:			/* Change item */
      /* pop dialog to change item information */
      switch (paste_item_type)
	{
	case NODE_PRESET:
	  pset = paste_item_data;
	  w = uisf_create_preset_config_dialog (_("Modify Preset"), pset->name,
				pset->bank, pset->prenum,
				(GtkSignalFunc)uisel_cb_preset_change_okay);
	  gtk_window_set_modal (GTK_WINDOW (w), TRUE);
	  gtk_widget_show (w);
	  break;
	case NODE_INST:
	  inst = paste_item_data;
	  w = uisf_create_inst_config_dialog (_("Rename Instrument"),
				inst->name,
				(GtkSignalFunc)uisel_cb_inst_change_okay);
	  gtk_window_set_modal (GTK_WINDOW (w), TRUE);
	  gtk_widget_show (w);
	  break;
	case NODE_SAMPLE:
	  sam = paste_item_data;
	  /* rate = 0 in call, so rate/fine tune settings will not be shown */
	  w = uisf_create_sample_config_dialog (_("Rename Sample"), sam->name,
			  0, 0, (GtkSignalFunc)uisel_cb_sample_change_okay);
	  gtk_window_set_modal (GTK_WINDOW (w), TRUE);
	  gtk_widget_show (w);
	  break;
	default:
	  return;
	}
      return;			/* return,uisel_cb_*_change_okay will notify */

    case 2:			/* Replace All */
      all = TRUE;  /* fall through case */
    case 1:			/* Replace */
      mode = UISEL_REPLACE;
      break;

    case 4:			/* Keep All */
      all = TRUE;  /* fall through case */
    case 3:			/* Keep */
      mode = UISEL_KEEP;
      break;

    case 5:			/* Cancel */
      mode = UISEL_CANCEL;
      break;
    }

  if (all)			/* User requested Keep All/Replace All */
    {
      switch (paste_item_type)
	{
	case NODE_PRESET:
	  preset_all_mode = mode;
	  break;
	case NODE_INST:
	  inst_all_mode = mode;
	  break;
	case NODE_SAMPLE:
	  sample_all_mode = mode;
	  break;
	default:
	  return;
	}
    }

  gtk_signal_disconnect_by_func (GTK_OBJECT (dup_item_dialog),
				 (GtkSignalFunc)uisel_cb_dup_item_destroy,
				 NULL);
  gtk_widget_destroy (dup_item_dialog);

  uisel_paste_items (SFITEMID_NONE, mode);
}

/* callback for preset change */
static void
uisel_cb_preset_change_okay (GtkWidget *cfgdialog)
{
  SFPreset *pset;
  GtkWidget *widg;

  /* change data for duplicated preset in the works */
  pset = paste_item_data;

  widg = gtk_object_get_data (GTK_OBJECT (cfgdialog), "ENname");
  sfont_set_namestr_nosf (pset->name, gtk_entry_get_text (GTK_ENTRY (widg)));

  widg = gtk_object_get_data (GTK_OBJECT (cfgdialog), "SPBbank");
  pset->bank = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));

  widg = gtk_object_get_data (GTK_OBJECT (cfgdialog), "SPBprenum");
  pset->prenum = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));

  gtk_widget_destroy (cfgdialog);

  gtk_signal_disconnect_by_func (GTK_OBJECT (dup_item_dialog),
				 (GtkSignalFunc)uisel_cb_dup_item_destroy,
				 NULL);
  gtk_widget_destroy (dup_item_dialog);

  /* restart paste function */
  uisel_paste_items (SFITEMID_NONE, UISEL_CHANGE);
}

/* callback for instrument change */
static void
uisel_cb_inst_change_okay (GtkWidget *cfgdialog)
{
  SFInst *inst;
  GtkWidget *widg;

  /* change data for duplicated instrument in the works */
  inst = paste_item_data;

  widg = gtk_object_get_data (GTK_OBJECT (cfgdialog), "ENname");
  sfont_set_namestr_nosf (inst->name, gtk_entry_get_text (GTK_ENTRY (widg)));

  gtk_widget_destroy (cfgdialog);

  gtk_signal_disconnect_by_func (GTK_OBJECT (dup_item_dialog),
				 (GtkSignalFunc)uisel_cb_dup_item_destroy,
				 NULL);
  gtk_widget_destroy (dup_item_dialog);

  /* restart paste function */
  uisel_paste_items (SFITEMID_NONE, UISEL_CHANGE);
}

/* callback for sample change */
static void
uisel_cb_sample_change_okay (GtkWidget *cfgdialog)
{
  SFSample *sam;
  GtkWidget *widg;

  /* change data for duplicated sample in the works */
  sam = paste_item_data;

  widg = gtk_object_get_data (GTK_OBJECT (cfgdialog), "ENname");
  sfont_set_namestr_nosf (sam->name, gtk_entry_get_text (GTK_ENTRY (widg)));

  gtk_widget_destroy (cfgdialog);

  gtk_signal_disconnect_by_func (GTK_OBJECT (dup_item_dialog),
				 (GtkSignalFunc)uisel_cb_dup_item_destroy,
				 NULL);
  gtk_widget_destroy (dup_item_dialog);

  /* restart paste function */
  uisel_paste_items (SFITEMID_NONE, UISEL_CHANGE);
}

static void
uisel_cb_dup_zone (gpointer data, GtkWidget *popdog)
{
  gint mode;

  mode = GPOINTER_TO_INT (data);

  gtk_signal_disconnect_by_func (GTK_OBJECT (popdog),
				 (GtkSignalFunc)uisel_cb_dup_zone_destroy,
				 NULL);
  gtk_widget_destroy (popdog);

  uisel_paste_items (SFITEMID_NONE, mode);
}

static gboolean
uisel_cb_dup_zone_destroy (GtkWidget *popdog)
{
  uisel_paste_items (SFITEMID_NONE, UISEL_CANCEL);

  return (FALSE);
}

void
uisel_dialog_selection_delete (void)
{
  GList *sel;
  GtkWidget *popdog;

  if (!(sel = sftree_get_selection_multi ())) return;

  /* if just one item, call single item delete routine */
  if (!sel->next)
    {
      uisf_dialog_sfitem_delete ();
      return;
    }

  popdog =
    util_quick_popup (_("Confirm delete of selected items"),
		      _("OK"), uisel_cb_dialog_selection_delete_okay, NULL,
		      _("Cancel"), NULL, NULL,
		      NULL);

  /* free selection list when popdog is destroyed */
  gtk_object_set_data_full (GTK_OBJECT (popdog), "sel", sel,
			    (GtkDestroyNotify) g_list_free);
}

/* delete selected sound font items and orphaned instruments/samples if
   specified */
static void
uisel_cb_dialog_selection_delete_okay (gpointer data, GtkWidget *popdog)
{
  SFItemID itemid;
  GList *p;

  p = gtk_object_get_data (GTK_OBJECT (popdog), "sel");

  sfdo_group (_("Delete sound font items"));

  SFTREE_FREEZE ();

  while (p)
    {				/* loop over selection */
      itemid = GPOINTER_TO_INT(p->data);

      /* remove node if it hasn't already been removed */
      if (itemid && SFTREE_LOOKUP_ITEMID (itemid))
	  uisf_sfitem_delete (itemid); /* delete it */

      p = g_list_next (p);
    }

  SFTREE_THAW ();

  sfdo_done ();

  gtk_widget_destroy (popdog);
}
