/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for bank management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbankpluginwidget.h"
#include "skgsortfilterproxymodel.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgtransactionmng.h"
#include "skgunitobject.h"
#include "skgunitvalueobject.h"
#include "skgbankobject.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"

#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kfiledialog.h>

#include <QDomDocument>
#include <QFile>
#include <QDir>
#include <QKeyEvent>

SKGBankPluginWidget::SKGBankPluginWidget ( SKGDocumentBank* iDocument )
        : SKGTabPage ( iDocument )
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::SKGBankPluginWidget" );

    ui.setupUi ( this );
    ui.kUnitEdit->setDocument(iDocument);

    timer = new QTimer ( this );
    timer->setSingleShot ( true );
    connect ( timer, SIGNAL ( timeout() ), this, SLOT ( refreshInfoZone() ) );

    //Add Standard KDE Icons to buttons to Accounts
    ui.kAccountCreatorUpdate->setIcon ( KIcon ( "dialog-ok-apply" ) );
    ui.kAccountCreatorAdd->setIcon ( KIcon ( "list-add" ) );

    ui.kStandardBtn->setIcon ( KIcon ( "configure" ) );

    //Add NLS values for type of account
    //C=current D=credit card P=passif (for objects) I=Investment O=other
    ui.kAccountCreatorType->addItem ( i18nc ( "Noun, a type of account", "Current" ) );
    ui.kAccountCreatorType->addItem ( i18nc ( "Noun, a type of account", "Credit card" ) );
    ui.kAccountCreatorType->addItem ( i18nc ( "Noun, a type of account", "Investment" ) );
    ui.kAccountCreatorType->addItem ( i18nc ( "Noun, a type of account", "Assets" ) );
    ui.kAccountCreatorType->addItem ( i18nc ( "Noun, a type of account", "Other" ) );

    //Bind account creation view
    objectModel = new SKGObjectModel ( static_cast<SKGDocumentBank*>(getDocument()), "v_account_display", "1=0", this, "", false );
    SKGSortFilterProxyModel* modelProxy = new SKGSortFilterProxyModel ( this );
    modelProxy->setSourceModel ( objectModel );
    modelProxy->setSortRole ( Qt::UserRole );
    modelProxy->setDynamicSortFilter ( true );

    ui.kFilterEdit->setProxy ( modelProxy );

    ui.kAccountTableViewEdition->setModel ( modelProxy );

    //Add registered global action in contextual menu
    if ( SKGMainPanel::getMainPanel() )
    {
        ui.kAccountTableViewEdition->insertGlobalAction ( "edit_delete" );
        ui.kAccountTableViewEdition->insertGlobalAction ();
        ui.kAccountTableViewEdition->insertGlobalAction ( "edit_switch_highlight" );
        ui.kAccountTableViewEdition->insertGlobalAction ( "edit_reconciliate" );
        ui.kAccountTableViewEdition->insertGlobalAction ();
        ui.kAccountTableViewEdition->insertGlobalAction ( "edit_find" );
        ui.kAccountTableViewEdition->insertGlobalAction ( "open_report" );
    }
    connect ( ui.kAccountTableViewEdition, SIGNAL ( selectionChangedDelayed() ), this, SLOT ( onSelectionChanged() ) );
    connect ( objectModel, SIGNAL ( beforeReset() ), ui.kAccountTableViewEdition, SLOT ( saveSelection() ) );
    connect ( objectModel, SIGNAL ( afterReset() ), ui.kAccountTableViewEdition, SLOT ( resetSelection() ) );

    ui.kAccountTableViewEdition->sortByColumn ( 0, Qt::AscendingOrder );

    //Logo for banks
    ui.kAccountCreatorIcon->addItem ( "" );
    QFile file ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/list_bank.txt" ) ) );
    if ( file.open ( QIODevice::ReadOnly | QIODevice::Text ) )
    {
        QDir dirLogo ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/" ) ) );

        QTextStream in ( &file );
        while ( !in.atEnd() )
        {
            QString line=in.readLine().trimmed();
            if ( !line.isEmpty() )
            {
                QString bankName=line;
                bankName.remove ( ".png" );
                bankName.replace ( '_', ' ' );

                QRegExp rx ( "(.+) {2,}(.+)" );
                if ( rx.indexIn ( bankName ) !=-1 )
                {
                    //Icon is something like <bank name>__<banknumber>.png
                    listBankNumbers.append ( rx.cap ( 2 ) );
                    bankName=rx.cap ( 1 );
                }
                else
                {
                    listBankNumbers.append ( "" );
                }

                ui.kAccountCreatorIcon->addItem ( QIcon ( dirLogo.absoluteFilePath ( line ) ), bankName );
            }
        }
        ui.kAccountCreatorIcon->addItem ( i18nc("Other type of bank account","Other...") );

        file.close();
    }

    //Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter ( this );

    // Activate the edition panel per default
    setCurrentMode ( 0 );

    //Refresh
    connect ( ( const QObject* ) getDocument(), SIGNAL ( tableModified ( QString, int ) ), this, SLOT ( dataModified ( QString, int ) ), Qt::QueuedConnection );
    dataModified ( "", 0 );
}

SKGBankPluginWidget::~SKGBankPluginWidget()
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::~SKGBankPluginWidget" );
    objectModel=NULL;
    timer=NULL;
}

bool SKGBankPluginWidget::eventFilter ( QObject *object, QEvent *event )
{
    if ( event && event->type() == QEvent::KeyPress )
    {
        QKeyEvent *keyEvent = ( QKeyEvent * ) event;
        if ( ( keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter ) && object==this )
        {
            if ( QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAccountCreatorAdd->isEnabled() )
            {
                ui.kAccountCreatorAdd->click();
            }
            else if ( QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kAccountCreatorUpdate->isEnabled() )
            {
                ui.kAccountCreatorUpdate->click();
            }
        }
    }

    return false;
}

void SKGBankPluginWidget::onSelectionChanged()
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::onSelectionChanged" );
    //Mapping
    QItemSelectionModel *selModel=ui.kAccountTableViewEdition->selectionModel();
    if ( selModel )
    {
        QModelIndexList indexes=selModel->selectedRows();
        if ( indexes.count() && objectModel )
        {
            QModelIndex idx=indexes[indexes.count()-1];

            QSortFilterProxyModel* proxyModel= ( QSortFilterProxyModel* ) ui.kAccountTableViewEdition->model();
            QModelIndex idxs=proxyModel->mapToSource ( idx );

            //set the icon
            SKGAccountObject account ( objectModel->getObject ( idxs ) );
            SKGBankObject bank;
            account.getBank ( bank );
            QString fileName=bank.getIcon();
            QString iconName=fileName;
            if ( iconName.length() )
            {
                iconName.remove ( ".png" );
                iconName.replace ( '_', ' ' );

                QRegExp rx ( "(.+) {2,}(.+)" );
                if ( rx.indexIn ( iconName ) !=-1 )
                {
                    iconName=rx.cap ( 1 );
                }
                if (ui.kAccountCreatorIcon->contains(iconName)) ui.kAccountCreatorIcon->setText ( iconName );
                else if (ui.kAccountCreatorIcon->contains(fileName)) ui.kAccountCreatorIcon->setText ( fileName );
                else
                {
                    int c=ui.kAccountCreatorIcon->count()-1;
                    bool b=ui.kAccountCreatorIcon->blockSignals(true);
                    ui.kAccountCreatorIcon->insertItem (c, QIcon ( fileName ), fileName );
                    ui.kAccountCreatorIcon->setCurrentIndex(c);
                    ui.kAccountCreatorIcon->blockSignals(b);
                }
            }
            else ui.kAccountCreatorIcon->setText ( "" );

            ui.kAccountCreatorBank->setText ( account.getAttribute ( "t_BANK" ) );
            ui.kAccountCreatorAccount->setText ( account.getAttribute ( "t_name" ) );
            ui.kAccountCreatorBankNumber->setText ( account.getAttribute ( "t_BANK_NUMBER" ) );
            ui.kAccountCreatorAgencyNumber->setText ( account.getAttribute ( "t_agency_number" ) );
            ui.kAccountCreatorNumber->setText ( account.getAttribute ( "t_number" ) );
            ui.kAccountCreatorType->setText ( account.getAttribute ( "t_TYPENLS" ) );
            ui.kAccountCreatorAddress->setText ( account.getAttribute ( "t_agency_address" ) );
            ui.kAccountCreatorComment->setText ( account.getAttribute ( "t_comment" ) );

            double oBalance=0;
            SKGUnitObject oUnit;
            account.getInitialBalance ( oBalance, oUnit );

            ui.kAmountEdit->setText ( KGlobal::locale()->formatMoney (oBalance, "", 2) );
            ui.kUnitEdit->setUnit ( oUnit );
        }

        onAccountCreatorModified();
        emit selectionChanged();
    }
}

int SKGBankPluginWidget::getCurrentMode()
{
    return mode;
}

void SKGBankPluginWidget::onBtnModeClicked()
{
    QWidget* sender=static_cast<QWidget*> ( this->sender() );

    int currentMode=getCurrentMode();
    int newMode=0;

    if ( sender==ui.kStandardBtn ) newMode=0;

    if ( currentMode==newMode ) newMode=-1;
    setCurrentMode ( newMode );
}

void SKGBankPluginWidget::setCurrentMode ( int iMode )
{
    mode=iMode;

    ui.SKGEditionWidget->hide();
    ui.SKGEditionButtonswidget->setVisible ( mode!=-1 );

    ui.kStandardBtn->setChecked ( false );
    if ( mode==0 )
    {
        ui.SKGEditionWidget->setVisible ( true );
    }

    onAccountCreatorModified();
}

void SKGBankPluginWidget::onIconChanged()
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::onIconChanged" );

    int c=ui.kAccountCreatorIcon->currentIndex();
    if (c && c==ui.kAccountCreatorIcon->count()-1)
    {
        QString fileName=KFileDialog::getOpenFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "image/png image/jpeg image/gif image/tiff" , this );
        if ( !fileName.isEmpty() )
        {
            if (ui.kAccountCreatorIcon->contains(fileName)) ui.kAccountCreatorIcon->setText ( fileName );
            else
            {
                bool b=ui.kAccountCreatorIcon->blockSignals(true);
                ui.kAccountCreatorIcon->insertItem (c, QIcon ( fileName ), fileName );
                ui.kAccountCreatorIcon->setCurrentIndex(c);
                ui.kAccountCreatorIcon->blockSignals(b);
            }
        } else ui.kAccountCreatorIcon->setCurrentIndex(c-1);
    }

    onAccountCreatorModified();
}


void SKGBankPluginWidget::onAccountCreatorModified()
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::onAccountCreatorModified" );

    bool activated=mode!=-1 &&
                   ui.kAccountCreatorBank->text().length() >0 &&
                   ui.kAccountCreatorAccount->text().length() >0 &&
                   ui.kUnitEdit->currentText().length() >0;

    int nbSelect=getNbSelectedObjects();
    ui.kAccountCreatorAdd->setEnabled ( activated );
    ui.kAccountCreatorUpdate->setEnabled ( activated && nbSelect>0 );
    ui.kAccountCreatorAccount->setEnabled ( nbSelect<=1 );
    ui.kAccountCreatorNumber->setEnabled ( nbSelect<=1 );
    ui.kAmountEdit->setEnabled ( nbSelect<=1 );
    ui.kUnitEdit->setEnabled ( nbSelect<=1 );

    //Facilitate bank name
    if ( ui.kAccountCreatorBank->text().length() ==0 )
    {
        ui.kAccountCreatorBank->setText ( ui.kAccountCreatorIcon->currentIndex() !=0 ? ui.kAccountCreatorIcon->currentText() : "" );
    }

    //Facilitate bank number
    if ( ui.kAccountCreatorBankNumber->text().length() ==0 )
    {
        int pos=ui.kAccountCreatorIcon->currentIndex();
        ui.kAccountCreatorBankNumber->setText ( pos>=1 && pos-1<listBankNumbers.count() ? listBankNumbers[pos-1] : "" );
    }
}

SKGError SKGBankPluginWidget::setInitialBalanceFromEditor(SKGAccountObject& iAccount)
{
    return iAccount.setInitialBalance(ui.kAmountEdit->value(), ui.kUnitEdit->getUnit());
}

void SKGBankPluginWidget::onAddAccountClicked()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGBankPluginWidget::onAddAccountClicked",err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    {
        SKGAccountObject accountObj;

        QString bankname=ui.kAccountCreatorBank->text();
        QString accountname=ui.kAccountCreatorAccount->text();
        QString name=bankname+'-'+accountname;
        {
            SKGBEGINTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Account creation '%1'", name ), err );

            //Create bank object in case of missing
            SKGBankObject bankObj ( getDocument() );
            if ( err.isSucceeded() ) err=bankObj.setName ( ui.kAccountCreatorBank->text() );
            if ( err.isSucceeded() )
            {
                //Build icon name
                QString icon=ui.kAccountCreatorIcon->currentText();
                int pos=ui.kAccountCreatorIcon->currentIndex();
                if ( pos!=0 && pos-1<listBankNumbers.count())
                {
                    if ( listBankNumbers[pos-1].length() ) icon+="  "+listBankNumbers[pos-1];

                    icon.replace ( ' ', '_' );
                    icon+=".png";

                }

                err=bankObj.setIcon ( icon );
            }
            if ( err.isSucceeded() ) err=bankObj.setNumber ( ui.kAccountCreatorBankNumber->text() );

            if ( err.isSucceeded() ) err=bankObj.save();
            if ( err.isSucceeded() ) err=bankObj.load();

            //Create account object in case of missing
            if ( err.isSucceeded() ) err=bankObj.addAccount ( accountObj );
            if ( err.isSucceeded() ) err=accountObj.setName ( accountname );
            if ( err.isSucceeded() ) err=accountObj.setAgencyNumber ( ui.kAccountCreatorAgencyNumber->text() );
            if ( err.isSucceeded() ) err=accountObj.setAgencyAddress ( ui.kAccountCreatorAddress->text() );
            if ( err.isSucceeded() ) err=accountObj.setComment ( ui.kAccountCreatorComment->text() );
            if ( err.isSucceeded() ) err=accountObj.setNumber ( ui.kAccountCreatorNumber->text() );
            if ( err.isSucceeded() ) err=accountObj.setType ( ( SKGAccountObject::AccountType ) ( ui.kAccountCreatorType->currentIndex() ) );
            if ( err.isSucceeded() ) err=accountObj.save ( false );
            if ( err.isSucceeded() ) err=setInitialBalanceFromEditor(accountObj);
            if ( err.isSucceeded() ) err=accountObj.save ( );
        }

        //status bar
        if ( err.isSucceeded() )
        {
            err=SKGError ( 0, i18nc ( "Successful message after an user action", "Account '%1' created", name ) );
            ui.kAccountTableViewEdition->selectObject ( accountObj.getUniqueID() );
        }
        else err.addError ( ERR_FAIL, i18nc ( "Error message",  "Account creation failed" ) );
    }
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

SKGObjectBase::SKGListSKGObjectBase SKGBankPluginWidget::getSelectedObjects()
{
    return ui.kAccountTableViewEdition->getSelectedObjects();
}

int SKGBankPluginWidget::getNbSelectedObjects()
{
    return ui.kAccountTableViewEdition->getNbSelectedObjects();
}

void SKGBankPluginWidget::onModifyAccountClicked()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGBankPluginWidget::onModifyAccountClicked",err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    {
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

        int nb=selection.count();
        SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Account update" ), err, nb );
        for ( int i=0; err.isSucceeded() && i<nb; ++i )
        {
            SKGAccountObject accountObj=selection[i];

            //Update account if single selection
            if ( nb==1 )
            {
                err=accountObj.setName ( ui.kAccountCreatorAccount->text() );
                if ( err.isSucceeded() ) err=accountObj.setNumber ( ui.kAccountCreatorNumber->text() );
                if ( err.isSucceeded() ) err=setInitialBalanceFromEditor(accountObj);
            }
            if ( err.isSucceeded() ) err=accountObj.setType ( ( SKGAccountObject::AccountType ) ( ui.kAccountCreatorType->currentIndex() ) );
            if ( err.isSucceeded() ) err=accountObj.setAgencyNumber ( ui.kAccountCreatorAgencyNumber->text() );
            if ( err.isSucceeded() ) err=accountObj.setAgencyAddress ( ui.kAccountCreatorAddress->text() );
            if ( err.isSucceeded() ) err=accountObj.setComment ( ui.kAccountCreatorComment->text() );
            if ( err.isSucceeded() ) err=accountObj.save();
            if ( err.isSucceeded() ) err=accountObj.save();

            //Update bank
            SKGBankObject bankObj;
            if ( SKGNamedObject::getObjectByName ( getDocument(), "bank", ui.kAccountCreatorBank->text(), bankObj ).isSucceeded() )
            {
                //The created bank already exist ==> update parent bank
                if ( err.isSucceeded() ) err=accountObj.setBank ( bankObj );
                if ( err.isSucceeded() ) err=accountObj.save();
            }
            else
            {
                //The bank does not exist
                int code=KMessageBox::Yes;
                SKGBankObject currentBank;
                err=accountObj.getBank ( currentBank );
                if ( err.isSucceeded() )
                {
                    SKGObjectBase::SKGListSKGObjectBase accounts;
                    err=currentBank.getAccounts ( accounts );
                    if ( err.isSucceeded() && accounts.count() >1 )
                    {
                        QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
                        code=KMessageBox::questionYesNo ( this, i18nc ( "Question", "You are trying to modify bank name of account named \"%1\".\nDo you want to do this modification for all accounts of this bank ?",  accountObj.getName() ) );
                        QApplication::restoreOverrideCursor();
                    }
                }

                if ( code==KMessageBox::Yes )
                {
                    //The bank does not exist ==> update this one
                    if ( err.isSucceeded() ) err=accountObj.getBank ( bankObj );
                    if ( err.isSucceeded() ) err=bankObj.setName ( ui.kAccountCreatorBank->text() );
                }
                else
                {
                    //The bank does not exist ==> create a new one
                    bankObj=SKGBankObject ( getDocument() );
                    if ( err.isSucceeded() ) err=bankObj.setName ( ui.kAccountCreatorBank->text() );
                    if ( err.isSucceeded() ) err=bankObj.save();
                    if ( err.isSucceeded() ) err=accountObj.setBank ( bankObj );
                    if ( err.isSucceeded() ) err=accountObj.save();
                }
            }
            if ( err.isSucceeded() )
            {
                //Build icon name
                QString icon=ui.kAccountCreatorIcon->currentText();
                int pos=ui.kAccountCreatorIcon->currentIndex();
                if ( pos!=0 && pos-1<listBankNumbers.count())
                {
                    if ( listBankNumbers[pos-1].length() ) icon+="  "+listBankNumbers[pos-1];

                    icon.replace ( ' ', '_' );
                    icon+=".png";

                }

                err=bankObj.setIcon ( icon );
            }
            if ( err.isSucceeded() ) err=bankObj.setNumber ( ui.kAccountCreatorBankNumber->text() );
            if ( err.isSucceeded() ) err=bankObj.save();

            if ( err.isSucceeded() ) err=getDocument()->stepForward ( i+1 );
        }

        //Remove bank without account
        if ( err.isSucceeded() )
        {
            err=SKGServices::executeSqliteOrder ( getDocument(), "DELETE FROM bank WHERE NOT EXISTS(SELECT 1 FROM account WHERE account.rd_bank_id=bank.id)" );
        }

        //status bar
        if ( err.isSucceeded() ) err=SKGError ( 0, i18nc ( "Successful message after an user action", "Account updated." ) );
        else err.addError ( ERR_FAIL, i18nc ( "Error message",  "Update failed" ) );
    }
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGBankPluginWidget::onDoubleClickedAccount()
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::onDoubleClickedAccount" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Open in operation plugin
    SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
    int nb=selection.count();
    for ( int i=0; i<nb; ++i )
    {
        SKGAccountObject accountObj=selection[i];

        QDomDocument doc ( "SKGML" );
        doc.setContent ( getDocument()->getParameter ( "SKGOPERATION_DEFAULT_PARAMETERS" ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "account", accountObj.getName() );
        SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );

    }
    QApplication::restoreOverrideCursor();
}

void SKGBankPluginWidget::onFilterChanged()
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::onFilterChanged" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Compute where clause
    QString filter;

    if ( ui.kHideUseless->checkState() ==Qt::Checked )
    {
        filter=" t_close='N'";
    }

    //Update model
    if ( objectModel )
    {
        objectModel->setFilter ( filter );
        ui.kAccountTableViewEdition->setState ( ui.kAccountTableViewEdition->getState() );
    }

    QApplication::restoreOverrideCursor();
}

void SKGBankPluginWidget::onComputeRIB()
{
    QString sb=ui.kAccountCreatorBankNumber->text().rightJustified ( 5, '0', true );
    QString sg=ui.kAccountCreatorAgencyNumber->text().rightJustified ( 5, '0', true );
    QString sc=ui.kAccountCreatorNumber->text().rightJustified ( 11, '0', true );

    QString l1="ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    QString l2="12345678912345678923456789";

    for ( int i=0; i<26; ++i )
    {
        sc=sc.replace ( l1[i], l2[i] );
    }

//         int b=SKGServices::stringToInt ( sb );
//         int g=SKGServices::stringToInt ( sg );
//         int d=SKGServices::stringToInt ( sc.left ( 6 ) );
//         int c=SKGServices::stringToInt ( sc.right ( 5 ) );

//         ui.kRib->setText ( SKGServices::intToString ( 97 - ( ( 89 * b + 15 * g + 76 * d + 3 * c ) % 97 ) ) );
}

void SKGBankPluginWidget::cleanEditor()
{
    if ( getNbSelectedObjects() ==0 )
    {
        ui.kAccountCreatorIcon->setText ( "" );
        ui.kAccountCreatorBank->setText ( "" );
        ui.kAccountCreatorAccount->setText ( "" );
        ui.kAccountCreatorBankNumber->setText ( "" );
        ui.kAccountCreatorAgencyNumber->setText ( "" );
        ui.kAccountCreatorNumber->setText ( "" );
        //Not done because of it's a combo ui.kAccountCreatorType->setText("");
        ui.kAccountCreatorAddress->setText ( "" );
        ui.kAccountCreatorComment->setText ( "" );
    }
}

QString SKGBankPluginWidget::getState()
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "hideUseless", ui.kHideUseless->checkState() ==Qt::Checked ? "Y" : "N" );
    root.setAttribute ( "filter", ui.kFilterEdit->lineEdit()->text() );
    root.setAttribute ( "currentPage", SKGServices::intToString ( getCurrentMode() ) );

    //Memorize table settings
    root.setAttribute ( "view", ui.kAccountTableViewEdition->getState() );
    return doc.toString();
}

void SKGBankPluginWidget::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::setState" );
    QDomDocument doc ( "SKGML" );
    doc.setContent ( iState );
    QDomElement root = doc.documentElement();

    QString hideUseless=root.attribute ( "hideUseless" );
    QString filter=root.attribute ( "filter" );
    QString currentPage=root.attribute ( "currentPage" );

    if ( !hideUseless.isEmpty() ) ui.kHideUseless->setCheckState ( hideUseless=="Y" ? Qt::Checked : Qt::Unchecked );
    if ( !filter.isEmpty() ) ui.kFilterEdit->setText ( filter );
    if ( !currentPage.isEmpty() ) setCurrentMode ( SKGServices::stringToInt ( currentPage ) );

    if ( objectModel )
    {
        bool previous=objectModel->blockRefresh ( true );
        onFilterChanged();
        objectModel->blockRefresh ( previous );
    }

    //!!! Must be done here after onFilterChanged
    ui.kAccountTableViewEdition->setState ( root.attribute ( "view" ) );
}

QString SKGBankPluginWidget::getDefaultStateAttribute()
{
    return "SKGBANK_DEFAULT_PARAMETERS";
}

void SKGBankPluginWidget::dataModified ( const QString& iTableName, int iIdTransaction )

{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::dataModified" );
    Q_UNUSED ( iIdTransaction );

    //Refresh widgets
    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled ( db!=NULL );
    if ( db!=NULL && ( iTableName=="account" || iTableName.isEmpty() ) )
    {
        //Correction bug 2299394 vvv
        if ( ui.kAccountTableViewEdition->isAutoResized() ) ui.kAccountTableViewEdition->resizeColumnsToContentsDelayed();
        //Correction bug 2299394 ^^^

        //Correction 215658: vvv because the table is modified, the selection is modified
        onSelectionChanged();
        //Correction 215658: ^^^

        //Refresh info area
        if ( timer ) timer->start ( 300 );
    }
}

void SKGBankPluginWidget::refreshInfoZone()
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::refreshInfoZone" );

    //Refresh info area
    SKGStringListList listTmp;
    SKGServices::executeSelectSqliteOrder ( getDocument(),
                                            "SELECT TOTAL(f_CURRENTAMOUNT), TOTAL(f_CHECKED), TOTAL(f_COMING_SOON) from v_account_display",
                                            listTmp );
    if ( listTmp.count() ==2 )
    {
        SKGDocumentBank* doc= static_cast<SKGDocumentBank*>(getDocument());
        if ( doc )
        {
            SKGServices::SKGUnitInfo primary=doc->getPrimaryUnit();
            double v1=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 0 ) );
            double v2=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 1 ) );
            double v3=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 2 ) );
            QString s1=doc->formatMoney ( v1, primary );
            QString s2=doc->formatMoney ( v2, primary );
            QString s3=doc->formatMoney ( v3, primary );
            ui.kInfo->setText ( i18nc ( "Message", "Balance: %1     Cleared: %2     In Transit: %3", s1, s2, s3 ) );

            SKGServices::SKGUnitInfo secondaryUnit=doc->getSecondaryUnit();
            if ( !secondaryUnit.Symbol.isEmpty() && secondaryUnit.Value )
            {
                s1=doc->formatMoney ( v1, secondaryUnit );
                s2=doc->formatMoney ( v2, secondaryUnit );
                s3=doc->formatMoney ( v3, secondaryUnit );
            }
            ui.kInfo->setToolTip ( i18nc ( "Message", "<p>Balance: %1</p><p>Cleared: %2</p><p>In Transit: %3</p>", s1, s2, s3 ) );
        }
    }
}

QWidget* SKGBankPluginWidget::getWidgetForPrint()
{
    return ui.kAccountTableViewEdition;
}

void SKGBankPluginWidget::activateEditor()
{
    if (getCurrentMode()==-1) setCurrentMode(0);
    ui.kAccountCreatorBank->setFocus();
}

bool SKGBankPluginWidget::isEditor()
{
    return true;
}

QWidget* SKGBankPluginWidget::zoomableWidget() const
{
    return ui.kAccountTableViewEdition;
}

#include "skgbankpluginwidget.moc"
