/*
**  Sinek (Video Player)
**  Copyright (c) 2001 - 2002 the Sinek Team, see the AUTHORS file.
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  subtitles
*/

#include "common.h"

typedef int (loader_t)(FILE *fd);

GtkWidget *subtitle_win = NULL, *combo, *delay;

static subtitle_t *current = NULL, *subtitles = NULL, *last = NULL;
static int frame_per_second = 25;
static int frame_format = 0;
static float delay_time = 0;
static char buffer[1024];

static loader_t *find_loader(FILE *fd);


int subtitle_load(const char *filename)
{
	FILE *fd;
	loader_t *loader;
	int ret;

	fd = fopen(filename, "r");
	if(!fd)
	{
		warning(_("Cannot open subtitle file:\n%s"), filename);
		return 0;
	}
	frame_format = 0;
	loader = find_loader(fd);
	if(loader)
	{
		ret = loader(fd);
	}
	else
	{
		ret = 0;
		warning(_("Unknown subtitle format!"));
	}
	fclose(fd);

	return ret;
}


subtitle_t *subtitle_find(unsigned long msec)
{
	if(frame_format) msec = (msec / 100) * frame_per_second;
	if(msec < (delay_time * 100)) return(NULL);
	msec = msec - (delay_time * 100);

	if(current == NULL) current = subtitles;
	while(current)
	{
		if(msec >= current->start && msec < current->end)
		{
			return current;
		}
		else if(msec < current->start)
		{
			if(current->prev && current->prev->end < msec)
				return NULL;
			current = current->prev;
		}
		else if(msec >= current->end)
		{
			if(current->next && current->next->start > msec)
				return NULL;
			current = current->next;
		}
	}
	return NULL;
}


void subtitle_delete(void)
{
	current = NULL;
	subtitles = NULL;
	last = NULL;
}


static subtitle_t *append(unsigned long start, unsigned long end)
{
	subtitle_t *sub;

	sub = malloc(sizeof(subtitle_t));
	memset(sub, 0, sizeof(subtitle_t));
	sub->start = start;
	sub->end = end;

	sub->prev = last;
	if(!subtitles) subtitles = sub;
	if(last) last->next = sub;
	last = sub;

	if(sub->prev)
	{
		if(sub->prev->end >= sub->start)
			printf("error in %ld - %ld\n", sub->start, sub->end);
	}
	return sub;
}


static void append_text(subtitle_t *sub, char *line)
{
	if(sub->lines < 4)
	{
		sub->text[sub->lines] = g_strdup(line);
		sub->lines++;
	}
}


static int parse_microdvd(FILE *fd)
{
	subtitle_t *sub;
	char buf2[1024];
	unsigned long start, end;
	char *t, *s;

	frame_format = 1;
	while(1)
	{
		if(!fgets(buffer, 1000, fd)) break;
		if(sscanf(buffer, "{%ld}{%ld}%[^\r\n]", &start, &end, buf2) < 3) continue;
		sub = append(start, end);
		t = buf2;
		while(1)
		{
			s = strchr(t, '|');
			if(s)
			{
				*s = '\0';
				append_text(sub, t);
				t = s + 1;
			}
			else
			{
				append_text(sub, t);
				break;
			}
		}
	}

	return 1;
}


static int parse_subrip(FILE *fd)
{
	subtitle_t *sub;
	unsigned long start, end;
	char *t, *s;
	int a1, a2, a3, a4, b1, b2, b3, b4;

	while(1)
	{
		if(!fgets(buffer, 1000, fd)) break;
		if(sscanf(buffer, "%d:%d:%d.%d,%d:%d:%d.%d",&a1,&a2,&a3,&a4,&b1,&b2,&b3,&b4) < 8) continue;
		start = a1*360000 + a2*6000 + a3*100 + a4;
		end   = b1*360000 + b2*6000 + b3*100 + b4;
		if(!fgets(buffer, 1000, fd)) break;
		strtok(buffer, "\r\n");
		sub = append(start, end);
		t = buffer;
		while(1)
		{
			s = strstr(t, "[br]");
			if(s)
			{
				*s = '\0';
				append_text(sub, t);
				t = s + 4;
			}
			else
			{
				append_text(sub, t);
				break;
			}
		}
	}

	return 1;
}


static int parse_srt(FILE *fd)
{
	subtitle_t *sub;
	unsigned long start, end;
	int a1, a2, a3, a4, b1, b2, b3, b4;

	while(1)
	{
		if(!fgets(buffer, 1000, fd)) break;
		if(sscanf(buffer, "%d:%d:%d,%d --> %d:%d:%d,%d",&a1,&a2,&a3,&a4,&b1,&b2,&b3,&b4) < 8) continue;
		start = a1*360000 + a2*6000 + a3*100 + (a4 / 10);
		end   = b1*360000 + b2*6000 + b3*100 + (b4 / 10);
		sub = append(start, end);
		while(1)
		{
			if(!fgets(buffer, 1000, fd)) break;
			if(buffer[0] == '\0' || buffer[0] == '\r' || buffer[0] == '\n') break;
			strtok(buffer, "\r\n");
			append_text(sub, buffer);
		}
	}

	return 1;
}


static loader_t *find_loader(FILE *fd)
{
	loader_t *loader = NULL;
	int i, j;

	for(i = 0; i < 50; i++)
	{
		if(!fgets(buffer, 1000, fd)) break;
		if(sscanf(buffer, "{%d}{%d}", &j, &j) == 2)
		{
			loader = parse_microdvd;
			break;
		}
		if(sscanf(buffer, "%d:%d:%d.%d,%d:%d:%d.%d", &j, &j, &j, &j, &j, &j, &j, &j) == 8)
		{
			loader = parse_subrip;
			break;
		}
		if(sscanf(buffer, "%d:%d:%d,%d --> %d:%d:%d,%d", &j, &j, &j, &j, &j, &j, &j, &j) == 8)
		{
			loader = parse_srt;
			break;
		}
	}
	rewind(fd);
	return loader;
}


static void cb_frame_rate(GtkWidget *w, gpointer data)
{
	char *str;
	int val;

	str = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry));
	if(str)
	{
		val = atoi(str);
		if(val > 0 && val < 80)
			frame_per_second = val;
	}
}


static void cb_delay(GtkWidget *w, gpointer data)
{
	char *str;

	str = gtk_entry_get_text(GTK_ENTRY(delay));
	if(str)
	{
		delay_time = atof(str);
	}
}


GtkWidget *subtitle_create(void)
{
	GtkWidget *vb, *hb, *w;
	GList *items = NULL;

	if(subtitle_win) goto out;

	subtitle_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW (subtitle_win), _("Sinek Subtitle Editor"));
	gtk_window_set_wmclass(GTK_WINDOW (subtitle_win), "Sinek Subtitles", "Sinek");
gtk_window_set_policy(GTK_WINDOW(subtitle_win), FALSE, FALSE, TRUE);

	vb = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(vb);
	gtk_container_add(GTK_CONTAINER(subtitle_win), vb);
	gtk_container_set_border_width(GTK_CONTAINER(vb), 5);

	w = gtk_label_new(_("Frame rate:"));
	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(vb), w, FALSE, FALSE, 0);

	hb = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, FALSE, 0);

	combo = gtk_combo_new();
	gtk_widget_show(combo);
	gtk_box_pack_start(GTK_BOX(hb), combo, TRUE, TRUE, 0);
	items = g_list_append(items, (gpointer) "15");
	items = g_list_append(items, (gpointer) "25");
	items = g_list_append(items, (gpointer) "29.7");
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), items);
	g_list_free(items);
	w = GTK_COMBO(combo)->entry;
	gtk_widget_show(w);
	gtk_entry_set_max_length(GTK_ENTRY(w), 6);
	gtk_entry_set_text(GTK_ENTRY(w), "25");

	w = gtk_label_new(_("fps"));
	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(hb), w, FALSE, FALSE, 0);

	w = gtk_button_new_with_label(_("Change"));
	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(vb), w, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(cb_frame_rate), NULL);

	w = gtk_label_new(_("Delay:"));
	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(vb), w, FALSE, FALSE, 0);

	hb = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, FALSE, 0);

	delay = gtk_entry_new();
	gtk_widget_show(delay);
	gtk_box_pack_start(GTK_BOX(hb), delay, TRUE, TRUE, 0);
	gtk_entry_set_max_length(GTK_ENTRY(delay), 6);
	gtk_entry_set_text(GTK_ENTRY(delay), "0.0");

	w = gtk_label_new(_("sec"));
	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(hb), w, FALSE, FALSE, 0);

	w = gtk_button_new_with_label(_("Change"));
	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(vb), w, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(cb_delay), NULL);

	wm_manage(subtitle_win, WM_NORMAL);
out:
	gtk_widget_show(subtitle_win);
	return subtitle_win;
}
