
/*
#    Sfront, a SAOL to C translator    
#    This file: Network library -- samidi sender journal functions
#    Copyright (C) 1999  Regents of the University of California
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License (Version 2) as
#    published by the Free Software Foundation.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
*/


#ifndef NSYS_NET
#include "net_include.h"
#endif

/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*     high-level functions: sending recovery journals          */
/*______________________________________________________________*/

/****************************************************************/
/*      generate recovery journal for an RTP packet             */
/****************************************************************/

int nsys_netin_journal_create(unsigned char * packet, int len)

{
  unsigned char * p = packet;
  int i, j, channel;
  netout_jsend_state * jsend;

  /********************************/
  /* add journal header to packet */
  /********************************/

  memcpy(p, netout_jsend_header, NSYS_SM_JH_SIZE);
  p += NSYS_SM_JH_SIZE;
  len -= NSYS_SM_JH_SIZE;

  /****************************/
  /* add chapter(s) if needed */
  /****************************/

  for (channel = 0; channel < netout_jsend_channel_size; channel++)
    {
      jsend = &(netout_jsend[netout_jsend_channel[channel]]);

      if (jsend->cheader_seqnum)
	{	  

	  if ((len -= (NSYS_SM_CH_LENMASK & jsend->clen)) < 0)
	    {
	      len += (NSYS_SM_CH_LENMASK & jsend->clen);

	      if (packet[NSYS_SM_JH_LOC_FLAGS] & NSYS_SM_JH_CHANMASK)
		packet[NSYS_SM_JH_LOC_FLAGS]--;
	      else
		packet[NSYS_SM_JH_LOC_FLAGS] |= NSYS_SM_JH_SETA;

	      nsys_warning(NSYS_WARN_UNUSUAL, "Sending an incomplete journal");
	      continue;
	    }

	  memcpy(p, jsend->cheader, NSYS_SM_CH_SIZE);
	  p += NSYS_SM_CH_SIZE;

	  /*************/
	  /* Chapter P */
	  /*************/

	  if (jsend->chapterp_seqnum)
	    {
	      memcpy(p, jsend->chapterp, NSYS_SM_CP_SIZE);
	      p += NSYS_SM_CP_SIZE;
	    }

	  /*************/
	  /* Chapter W */
	  /*************/

	  if (jsend->chapterw_seqnum)
	    {
	      memcpy(p, jsend->chapterw, NSYS_SM_CW_SIZE);
	      p += NSYS_SM_CW_SIZE;
	    }

	  /*************/
	  /* Chapter N */
	  /*************/
	  
	  if (jsend->chaptern_seqnum)
	    {
	      if (jsend->chaptern_timernum)
		{
		  j = jsend->chaptern_timernum;
		  for (i = jsend->chaptern_size - NSYS_SM_CN_HDRSIZE; 
		       i >= NSYS_SM_CN_HDRSIZE; i -= NSYS_SM_CN_LOGSIZE)
		    if (jsend->chaptern[i+1] & NSYS_SM_CN_CHKY)
		      {
			if (jsend->chaptern_timer[NSYS_SM_CLRS & 
						 jsend->chaptern[i]]
			    < nsys_netout_tstamp)
			  {
			    jsend->chaptern[i+1] &= NSYS_SM_CN_CLRY;
			    jsend->chaptern_timernum--;
			  }
			if ((--j) == 0)
			  break;
		      }
		}
	      memcpy(p, jsend->chaptern, jsend->chaptern_size);
	      p += jsend->chaptern_size;
	      if (jsend->chaptern_sset)
		{
		  for (i = NSYS_SM_CN_HDRSIZE; i < jsend->chaptern_size;
		       i += NSYS_SM_CN_LOGSIZE)
		    jsend->chaptern[i] |= NSYS_SM_SETS;
		  jsend->chaptern_sset = 0;
		}
	      if (jsend->chapterb_size)
		{
		  memcpy(p, &(jsend->chapterb[jsend->chapterb_low]),
			 jsend->chapterb_size);
		  p += jsend->chapterb_size;
		}
	    }
	  	  
	  /*************/
	  /* Chapter A */
	  /*************/
	  
	  if (jsend->chaptera_seqnum)
	    {
	      memcpy(p, jsend->chaptera, jsend->chaptera_size);
	      p += jsend->chaptera_size;
	      if (jsend->chaptera_sset)
		{
		  for (i = NSYS_SM_CA_HDRSIZE; i < jsend->chaptera_size;
		       i += NSYS_SM_CA_LOGSIZE)
		    jsend->chaptera[i] |= NSYS_SM_SETS;
		  jsend->chaptera_sset = 0;
		}
	    }

	  /*************/
	  /* Chapter T */
	  /*************/
	  
	  if (jsend->chaptert_seqnum)
	    {
	      memcpy(p, jsend->chaptert, NSYS_SM_CT_SIZE);
	      p += NSYS_SM_CT_SIZE;
	    }
	  
	  /*************/
	  /* Chapter C */
	  /*************/
	  
	  if (jsend->chapterc_seqnum)
	    {
	      memcpy(p, jsend->chapterc, jsend->chapterc_size);
	      p += jsend->chapterc_size;
	      if (jsend->chapterc_sset)
		{
		  for (i = NSYS_SM_CC_HDRSIZE; i < jsend->chapterc_size;
		       i += NSYS_SM_CC_LOGSIZE)
		    jsend->chapterc[i] |= NSYS_SM_SETS;
		  jsend->chapterc_sset = 0;
		}
	    }
	}
    }

  /****************/
  /* set S flags  */
  /****************/

  while (netout_jsend_slist_size > 0)
    *(netout_jsend_slist[--netout_jsend_slist_size]) |= NSYS_SM_SETS; 

  return (p - packet);
}

/****************************************************************/
/*            sender state: add a new MIDI event                */
/****************************************************************/

void nsys_netout_journal_addstate(unsigned char cmd, unsigned char ndata,
				  unsigned char vdata)

{
  unsigned char chan = 0x0F & cmd;
  netout_jsend_state * jsend;

  netout_jsend_guard_send = 0;

  if (netout_jsend_checkpoint_changed)
    {
      netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] &= NSYS_SM_JH_CLRK;
      netout_jsend_checkpoint_changed = 0;
    }
  if ((cmd &= 0xF0) == CSYS_MIDI_NOOP)
    {
      netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] |= NSYS_SM_SETS;
      return;
    }
  
  netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] &= NSYS_SM_CLRS;
  jsend = &(netout_jsend[chan]);

  if (jsend->cheader_seqnum == 0)
    {  
      netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] = ((unsigned char)
						   netout_jsend_channel_size);
      netout_jsend_channel[netout_jsend_channel_size++] = chan;
      jsend->clen = ((((unsigned short)chan) << (NSYS_SM_CH_CHANSHIFT + 8)) |
		     NSYS_SM_CH_SIZE);
    }

  jsend->cheader_seqnum = nsys_netout_seqnum;
  netout_jsend_slist[netout_jsend_slist_size++] = 
    &(jsend->cheader[NSYS_SM_CH_LOC_FLAGS]);

  switch (cmd) {
  case CSYS_MIDI_PROGRAM:
    nsys_netout_journal_addprogram(jsend, ndata);
    return;
  case CSYS_MIDI_WHEEL:
    nsys_netout_journal_addpwheel(jsend, ndata, vdata);
    return;
  case CSYS_MIDI_NOTEON:
    if (vdata)
      {      
	nsys_netout_journal_addnoteon(jsend, ndata, vdata);
	return;
      }
  case CSYS_MIDI_NOTEOFF:
    nsys_netout_journal_addnoteoff(jsend, ndata);
    return;
  case CSYS_MIDI_PTOUCH:
    nsys_netout_journal_addptouch(jsend, ndata, vdata);
    return;
  case CSYS_MIDI_CTOUCH:
    nsys_netout_journal_addctouch(jsend, ndata);
    return;
  case CSYS_MIDI_CC:
    nsys_netout_journal_addcontrol(jsend, ndata, vdata);
    return;
  }
}

/****************************************************************/
/*            sender state: adds history information            */
/****************************************************************/

void nsys_netout_journal_addhistory(unsigned char cmd, unsigned char ndata,
				    unsigned char vdata)

{
  unsigned char chan = 0x0F & cmd;
  netout_jsend_state * jsend;

  cmd &= 0xF0;

  if ((cmd != CSYS_MIDI_PROGRAM) && (cmd != CSYS_MIDI_CC))
    return;
  
  jsend = &(netout_jsend[chan]);

  if (jsend->history_active == 0)
    jsend->history_active = 1;

  switch (cmd & 0xF0) {
  case CSYS_MIDI_PROGRAM:
    jsend->history_program = ndata | NSYS_SM_SETH;
    jsend->history_program_bankc = jsend->history_cc_bankc;
    jsend->history_program_bankf = jsend->history_cc_bankf;
    break;
  case CSYS_MIDI_CC:  
    switch (ndata) {
    case CSYS_MIDI_CC_BANKSELECT_MSB:
      jsend->history_cc_bankc = vdata | NSYS_SM_SETH;
      break;
    case CSYS_MIDI_CC_BANKSELECT_LSB:
      jsend->history_cc_bankf = vdata | NSYS_SM_SETH;
      break;
    case CSYS_MIDI_CC_MODWHEEL_MSB:
      jsend->history_cc_modwheel = vdata | NSYS_SM_SETH;
      break;
    case CSYS_MIDI_CC_CHANVOLUME_MSB:
      jsend->history_cc_volume = vdata | NSYS_SM_SETH;
      break;
    }
    break;
  }
}

/****************************************************************/
/*    sender state:  trim journal, update checkpoint packet     */
/****************************************************************/

void nsys_netin_journal_trimstate(source * lptr)

{ 
  unsigned long minseq;
  source * minsource;
  source * sptr;
  int channel;
  netout_jsend_state * jsend;

  /********************************************/
  /* localize extension to new last_hiseq_ext */
  /********************************************/

  if (lptr)
    {
      lptr->last_hiseq_ext &= NSYS_RTPSEQ_LOMASK;
      lptr->last_hiseq_ext |= (nsys_netout_seqnum & NSYS_RTPSEQ_EXMASK);
      if (lptr->last_hiseq_ext > nsys_netout_seqnum)
	lptr->last_hiseq_ext -= NSYS_RTPSEQ_EXINCR;
    }

  /********************************************/
  /* find source with smallest last_hiseq_ext */
  /********************************************/

  if (nsys_srcroot == NULL)
    return;

  minsource = NULL;
  minseq = NSYS_RTPSEQ_HIGHEST;
  sptr = nsys_srcroot;

  /* later handle 32-bit wraparound correctly */

  do {
    if (minseq > sptr->last_hiseq_ext)
      {
	minsource = sptr;
	minseq = sptr->last_hiseq_ext;
      }
  } while ((sptr = sptr->next) != nsys_srcroot);

  /***************************************/
  /* return if checkpoint is still valid */
  /***************************************/

  if ((minseq == NSYS_RTPSEQ_HIGHEST) || 
      (minseq == netout_jsend_checkpoint_seqnum))
    return;

  /**********************/
  /* update checkpoint  */
  /**********************/

  netout_jsend_checkpoint_seqnum = minseq;
  nsys_netout_journal_changecheck();

  if (netout_jsend_channel_size == 0)
    return;

  for (channel = netout_jsend_channel_size - 1; channel >= 0; channel--)
    {
      jsend = &(netout_jsend[netout_jsend_channel[channel]]);

      /*******************************/
      /* drop entire channel, or ... */
      /*******************************/

      if (jsend->cheader_seqnum <= minseq)
	{
	  nsys_netout_journal_trimchapter(jsend, channel);

	  if (netout_jsend_channel_size)
	    continue;
	  else
	    break;
	}

      /*****************************************/
      /* ... drop chapters from channel header */
      /*****************************************/

      if (jsend->chapterp_seqnum && (jsend->chapterp_seqnum <= minseq))
	nsys_netout_journal_trimprogram(jsend);

      if (jsend->chapterw_seqnum && (jsend->chapterw_seqnum <= minseq))
	nsys_netout_journal_trimpwheel(jsend);

      if (jsend->chaptert_seqnum && (jsend->chaptert_seqnum <= minseq))
	nsys_netout_journal_trimctouch(jsend);

      if (jsend->chapterc_seqnum && (jsend->chapterc_seqnum <= minseq))
	nsys_netout_journal_trimallcontrol(jsend); 

      if (jsend->chapterc_seqnum)
	nsys_netout_journal_trimpartcontrol(jsend, minseq);
      
      if (jsend->chaptera_seqnum && (jsend->chaptera_seqnum <= minseq))
	nsys_netout_journal_trimallptouch(jsend); 

      if (jsend->chaptera_seqnum)
	nsys_netout_journal_trimpartptouch(jsend, minseq);
      
      if (jsend->chaptern_seqnum && (jsend->chaptern_seqnum <= minseq))
	nsys_netout_journal_trimallnote(jsend); 

      if (jsend->chaptern_seqnum)
	nsys_netout_journal_trimpartnote(jsend, minseq); 

      *((unsigned short *) &(jsend->cheader[NSYS_SM_CH_LOC_LEN])) = 
	htons(jsend->clen);
    }

  /****************/
  /* housekeeping */
  /****************/

  if (netout_jsend_channel_size == 0)
    {
      netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] |= NSYS_SM_JH_SETA;
      netout_jsend_slist_size =  0;

      /* uncomment to disable keep-alive empty-journal RTP packets */ 
      /* netout_jsend_guard_time = netout_jsend_guard_send = 0;    */
    }
  else
    {
      netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] &= (~NSYS_SM_JH_CHANMASK);
      netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] |= 
	(unsigned char)(netout_jsend_channel_size - 1);
    }
  
}


/****************************************************************/
/*               ticks down the guard packet timer              */
/****************************************************************/

void nsys_netout_guard_tick(void)

{
  /* enable conditional to disable keep-alive empty-journal RTP packets */

  if (1 || netout_jsend_channel_size)
    {
      netout_jsend_guard_send = 1;
      netout_jsend_guard_time = netout_jsend_guard_next;
      if ((netout_jsend_guard_next <<= 1) > netout_jsend_guard_maxtime)
	netout_jsend_guard_next = netout_jsend_guard_maxtime;
    }
}


/****************************************************************/
/*         add initialization state to recovery journal         */
/****************************************************************/

void nsys_netout_midistate_init(void)

{
  int slist_size = netout_jsend_slist_size;
  unsigned long seqsafe;
  netout_jsend_state * jsend;
  unsigned char i, cmd, ndata, vdata, bankc, bankf;


  /* seqnum updated so an rtcp packet doesn't delete inits */

  seqsafe = nsys_netout_seqnum;      
  nsys_netout_seqnum = ((nsys_netout_seqnum != NSYS_RTPSEQ_HIGHEST) ?
			(nsys_netout_seqnum + 1) : 1); 


  /* load recovery journal with initial values */

  for (i = 0; i < CSYS_MIDI_NUMCHAN; i++)
    {
      jsend = &(netout_jsend[i]);

      if (jsend->history_active == 0)
	continue;

      /***********************/
      /* MIDI Program Change */ 
      /***********************/
      
      bankc = jsend->history_cc_bankc;    /* save before overwrite happens */
      bankf = jsend->history_cc_bankf;

      if (jsend->history_program)
	{
	  if (jsend->history_program_bankc)
	    {
	      cmd = CSYS_MIDI_CC | i;
	      ndata = CSYS_MIDI_CC_BANKSELECT_MSB;
	      vdata = NSYS_SM_CLRH & jsend->history_program_bankc;
	      nsys_netout_journal_addstate(cmd, ndata, vdata);
	    }

	  if (jsend->history_program_bankf)
	    {
	      cmd = CSYS_MIDI_CC | i;
	      ndata = CSYS_MIDI_CC_BANKSELECT_LSB;
	      vdata = NSYS_SM_CLRH & jsend->history_program_bankf;
	      nsys_netout_journal_addstate(cmd, ndata, vdata);
	    }

	  cmd = CSYS_MIDI_PROGRAM | i;
	  ndata = NSYS_SM_CLRH & jsend->history_program;
	  vdata = 0;
	  nsys_netout_journal_addstate(cmd, ndata, vdata);
	}

      /************************/
      /* MIDI Control Changes */ 
      /************************/

      cmd = CSYS_MIDI_CC | i;

      if (bankc)
	{
	  ndata = CSYS_MIDI_CC_BANKSELECT_MSB;
	  vdata = NSYS_SM_CLRH & bankc;
	  nsys_netout_journal_addstate(cmd, ndata, vdata);
	}

      if (bankf)
	{
	  ndata = CSYS_MIDI_CC_BANKSELECT_LSB;
	  vdata = NSYS_SM_CLRH & bankf;
	  nsys_netout_journal_addstate(cmd, ndata, vdata);
	}

      if (jsend->history_cc_modwheel)
	{
	  ndata = CSYS_MIDI_CC_MODWHEEL_MSB;
	  vdata = NSYS_SM_CLRH & jsend->history_cc_modwheel;
	  nsys_netout_journal_addstate(cmd, ndata, vdata);
	}

      if (jsend->history_cc_volume)
	{
	  ndata = CSYS_MIDI_CC_CHANVOLUME_MSB;
	  vdata = NSYS_SM_CLRH & jsend->history_cc_volume;
	  nsys_netout_journal_addstate(cmd, ndata, vdata);
	}

      while (netout_jsend_slist_size > slist_size)
	*(netout_jsend_slist[--netout_jsend_slist_size]) |= NSYS_SM_SETS;

    }

  if (slist_size == 0)
    netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] |= NSYS_SM_SETS;
  
  while (slist_size > 0)
    *(netout_jsend_slist[--slist_size]) &= NSYS_SM_CLRS;

  nsys_netout_seqnum = seqsafe;   /* restore nsys_netout_seqnum */

  netout_jsend_guard_send = 1;    
  netout_jsend_guard_time = netout_jsend_guard_mintime;
  netout_jsend_guard_next = (netout_jsend_guard_mintime << 
			     (nsys_feclevel == NSYS_SM_FEC_MINIMAL));
}


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*      second-level journal functions: addstate chapters       */
/*______________________________________________________________*/

/****************************************************************/
/*               add MIDI Program Change chapter                */
/****************************************************************/

void nsys_netout_journal_addprogram(netout_jsend_state * jsend, 
				    unsigned char ndata)

{
  jsend->chapterp[NSYS_SM_CP_LOC_PROGRAM] = ndata;
  jsend->chapterp[NSYS_SM_CP_LOC_BANKCOARSE] = jsend->history_cc_bankc;
  jsend->chapterp[NSYS_SM_CP_LOC_BANKFINE] = jsend->history_cc_bankf;
    
  nsys_netout_journal_addhistory(CSYS_MIDI_PROGRAM | jsend->chan, ndata, 0);

  netout_jsend_slist[netout_jsend_slist_size++] = 
    &(jsend->chapterp[NSYS_SM_CP_LOC_PROGRAM]);
  
  if (jsend->chapterp_seqnum)
    jsend->cheader[NSYS_SM_CH_LOC_FLAGS] &= NSYS_SM_CLRS;
  else
    {
      jsend->cheader[NSYS_SM_CH_LOC_TOC] |= NSYS_SM_CH_TOC_SETP;
      
      *((unsigned short *) &(jsend->cheader[NSYS_SM_CH_LOC_LEN])) = 
	htons((jsend->clen += NSYS_SM_CP_SIZE));
    }
  jsend->chapterp_seqnum = nsys_netout_seqnum;

  netout_jsend_guard_time = netout_jsend_guard_mintime;
  netout_jsend_guard_next = (netout_jsend_guard_mintime << 
			     (nsys_feclevel == NSYS_SM_FEC_MINIMAL));

}

/****************************************************************/
/*               add MIDI Pitch Wheel chapter                   */
/****************************************************************/

void nsys_netout_journal_addpwheel(netout_jsend_state * jsend, 
				unsigned char ndata, unsigned char vdata)

{
  jsend->chapterw[NSYS_SM_CW_LOC_FIRST] = ndata;
  jsend->chapterw[NSYS_SM_CW_LOC_SECOND] = vdata;

  netout_jsend_slist[netout_jsend_slist_size++] = 
    &(jsend->chapterw[NSYS_SM_CW_LOC_FIRST]);
  
  if (jsend->chapterw_seqnum)
    jsend->cheader[NSYS_SM_CH_LOC_FLAGS] &= NSYS_SM_CLRS;
  else
    {
      jsend->cheader[NSYS_SM_CH_LOC_TOC] |= NSYS_SM_CH_TOC_SETW;
      
      *((unsigned short *) &(jsend->cheader[NSYS_SM_CH_LOC_LEN])) = 
	htons((jsend->clen += NSYS_SM_CW_SIZE));
    }
  jsend->chapterw_seqnum = nsys_netout_seqnum;

  netout_jsend_guard_time = netout_jsend_guard_mintime;
  netout_jsend_guard_next = (netout_jsend_guard_mintime << 
			     (nsys_feclevel == NSYS_SM_FEC_MINIMAL));
}

/****************************************************************/
/*                  add MIDI NoteOff chapter                    */
/****************************************************************/

void nsys_netout_journal_addnoteoff(netout_jsend_state * jsend, 
				 unsigned char ndata)

{
  unsigned char bitfield;
  int i;

  bitfield = (ndata >> NSYS_SM_CN_BFSHIFT);
  
  /******************************/
  /* initialize chapter and TOC */
  /******************************/
  
  if (jsend->chaptern_seqnum == 0)
    {
      jsend->chaptern_size = NSYS_SM_CN_HDRSIZE;
      jsend->chapterb_size = 1;
      jsend->chapterb_low = bitfield;
      jsend->chapterb_high = bitfield;
      
      jsend->chaptern[NSYS_SM_CN_LOC_LENGTH] = 0;
      jsend->chaptern[NSYS_SM_CN_LOC_LOWHIGH] = 
	((jsend->chapterb_low << NSYS_SM_CN_LOWSHIFT) |
	 jsend->chapterb_high);
      jsend->chapterb[bitfield] = 0;
      
      jsend->clen += 1 + NSYS_SM_CN_HDRSIZE;
      jsend->cheader[NSYS_SM_CH_LOC_TOC] |= NSYS_SM_CH_TOC_SETN;
    }
  
  /***************************/
  /* delete expired note log */
  /***************************/
  
  if (jsend->chaptern_seqarray[ndata] && 
      ((i = jsend->chaptern_logptr[ndata]) < NSYS_SM_CN_SIZE))
    {
      if (jsend->chaptern[i+NSYS_SM_CN_LOC_VEL] & NSYS_SM_CN_CHKY)
	jsend->chaptern_timernum--;
      memmove(&(jsend->chaptern[i]), 
	      &(jsend->chaptern[i + NSYS_SM_CN_LOGSIZE]),
	      jsend->chaptern_size - i - NSYS_SM_CN_LOGSIZE);
      jsend->chaptern_size -= NSYS_SM_CN_LOGSIZE;
      jsend->chaptern[NSYS_SM_CN_LOC_LENGTH]--;
      jsend->clen -= NSYS_SM_CN_LOGSIZE;
      while (i < jsend->chaptern_size)
	{
	  jsend->chaptern_logptr[jsend->chaptern[i] & NSYS_SM_CLRS] 
	    -= NSYS_SM_CN_LOGSIZE;
	  i += NSYS_SM_CN_LOGSIZE;
	}
    }
  
  /********************************/
  /* add bitfield bytes if needed */
  /********************************/
  
  if ((jsend->chapterb_low == NSYS_SM_CN_BFMAX) && 
      (jsend->chapterb_high == NSYS_SM_CN_BFMIN))
    {
      jsend->chapterb_size = 1;
      jsend->chapterb_low = bitfield;
      jsend->chapterb_high = bitfield;
      jsend->chaptern[NSYS_SM_CN_LOC_LOWHIGH] = 
	((jsend->chapterb_low << NSYS_SM_CN_LOWSHIFT) |
	 jsend->chapterb_high);
      jsend->chapterb[bitfield] = 0;
      jsend->clen++;
    }
  
  if (bitfield > jsend->chapterb_high)
    {
      memset(&(jsend->chapterb[jsend->chapterb_high + 1]), 0,
	     (bitfield - jsend->chapterb_high));
      jsend->chapterb_size += (bitfield - jsend->chapterb_high);
      jsend->clen += (bitfield - jsend->chapterb_high);
      jsend->chapterb_high = bitfield;	  
      jsend->chaptern[NSYS_SM_CN_LOC_LOWHIGH] = 
	((jsend->chapterb_low << NSYS_SM_CN_LOWSHIFT) |
	 jsend->chapterb_high);
    }
  
  if (bitfield < jsend->chapterb_low)
    {
      memset(&(jsend->chapterb[bitfield]), 0,
	     (jsend->chapterb_low - bitfield));
      jsend->chapterb_size += (jsend->chapterb_low - bitfield);
      jsend->clen += (jsend->chapterb_low - bitfield);
      jsend->chapterb_low = bitfield;	  
      jsend->chaptern[NSYS_SM_CN_LOC_LOWHIGH] = 
	((jsend->chapterb_low << NSYS_SM_CN_LOWSHIFT) |
	 jsend->chapterb_high);
    }
  
  /*****************/
  /* update slists */
  /*****************/
  
  jsend->chaptern[NSYS_SM_CN_LOC_LENGTH] &= NSYS_SM_CLRS; 
  netout_jsend_slist[netout_jsend_slist_size++] = 
    &(jsend->chaptern[NSYS_SM_CN_LOC_LENGTH]);
  
  /******************************************************************/
  /* set correct bit in bitfield byte, update channel/chapter state */
  /******************************************************************/
  
  jsend->chapterb[bitfield] |= (1 << ((~ndata) & NSYS_SM_CN_BFMASK));
  jsend->chaptern_logptr[ndata] = (NSYS_SM_CN_SIZE + bitfield);
  
  *((unsigned short *) &(jsend->cheader[NSYS_SM_CH_LOC_LEN])) = 
    htons(jsend->clen);
  
  jsend->chaptern_seqarray[ndata] = nsys_netout_seqnum;
  jsend->chaptern_seqnum = nsys_netout_seqnum;

  netout_jsend_guard_time = netout_jsend_guard_mintime;
  netout_jsend_guard_next = (netout_jsend_guard_mintime << 
			     (nsys_feclevel == NSYS_SM_FEC_MINIMAL));

}

/****************************************************************/
/*               add MIDI NoteOn chapter                   */
/****************************************************************/

void nsys_netout_journal_addnoteon(netout_jsend_state * jsend, 
				   unsigned char ndata, unsigned char vdata)

{
  int i;

  /******************************/
  /* initialize chapter and TOC */
  /******************************/
  
  if (jsend->chaptern_seqnum == 0)
    {
      jsend->chaptern_size = NSYS_SM_CN_HDRSIZE;
      jsend->chapterb_size = 0;
      jsend->chapterb_low = NSYS_SM_CN_BFMAX;
      jsend->chapterb_high = NSYS_SM_CN_BFMIN;
      
      jsend->chaptern[NSYS_SM_CN_LOC_LENGTH] = NSYS_SM_CN_SETB;
      jsend->chaptern[NSYS_SM_CN_LOC_LOWHIGH] = 
	((jsend->chapterb_low << NSYS_SM_CN_LOWSHIFT) |
	 jsend->chapterb_high);
      
      jsend->clen += NSYS_SM_CN_HDRSIZE;
      jsend->cheader[NSYS_SM_CH_LOC_TOC] |= NSYS_SM_CH_TOC_SETN;
    }
  
  /*********************************/
  /* zero expired NoteOff bitfield */
  /*********************************/
  
  if (jsend->chaptern_seqarray[ndata] && 
      ((i = (jsend->chaptern_logptr[ndata] - NSYS_SM_CN_SIZE)) >= 0))
    {
      jsend->chaptern_seqarray[ndata] = 0;
      jsend->chapterb[i] &= ~(1 << ((~ndata) & NSYS_SM_CN_BFMASK));
      if ((jsend->chapterb[i] == 0) && 
	  ((i == jsend->chapterb_low) || 
	   (i == jsend->chapterb_high)))
	{
	  while (jsend->chapterb_size && 
		 (jsend->chapterb[jsend->chapterb_low] == 0))
	    {
	      jsend->chapterb_low++;
	      jsend->chapterb_size--;
	      jsend->clen--;
	    }
	  while (jsend->chapterb_size && 
		 (jsend->chapterb[jsend->chapterb_high] == 0))
	    {
	      jsend->chapterb_high--;
	      jsend->chapterb_size--;
	      jsend->clen--;
	    }
	  if (jsend->chapterb_size == 0)
	    {	  
	      jsend->chapterb_low = NSYS_SM_CN_BFMAX;
	      jsend->chapterb_high = NSYS_SM_CN_BFMIN;
	    }
	  jsend->chaptern[NSYS_SM_CN_LOC_LOWHIGH] = 
	    ((jsend->chapterb_low << NSYS_SM_CN_LOWSHIFT) |
	     jsend->chapterb_high);
	}
    }
  
  /********************/
  /* add new note log */
  /********************/
  
  if (jsend->chaptern_seqarray[ndata] == 0)
    {
      jsend->chaptern[jsend->chaptern_size +
		     NSYS_SM_CN_LOC_NUM] = ndata;
      jsend->chaptern[jsend->chaptern_size +
		     NSYS_SM_CN_LOC_VEL] = (vdata | NSYS_SM_CN_SETY);
      jsend->chaptern_logptr[ndata] = jsend->chaptern_size;
      jsend->chaptern_timernum++;
      jsend->chaptern_size += NSYS_SM_CN_LOGSIZE;
      jsend->chaptern[NSYS_SM_CN_LOC_LENGTH]++;
      jsend->clen += NSYS_SM_CN_LOGSIZE;
    }
  else
    {
      i = jsend->chaptern_logptr[ndata];
      if (!(jsend->chaptern[i+NSYS_SM_CN_LOC_VEL] & NSYS_SM_CN_CHKY))
	jsend->chaptern_timernum++;
      jsend->chaptern[i+NSYS_SM_CN_LOC_NUM] = ndata;
      jsend->chaptern[i+NSYS_SM_CN_LOC_VEL] = (vdata|NSYS_SM_CN_SETY);
    }
  jsend->chaptern_timer[ndata] = (nsys_netout_tstamp + (int)
				  (ARATE * NSYS_SM_CN_MAXDELAY));
  
  /********************************/
  /* update channel/chapter state */
  /********************************/
  
  *((unsigned short *) &(jsend->cheader[NSYS_SM_CH_LOC_LEN])) = 
    htons(jsend->clen);
  jsend->chaptern_sset = 1;
  
  jsend->chaptern_seqarray[ndata] = nsys_netout_seqnum;
  jsend->chaptern_seqnum = nsys_netout_seqnum;

  if (netout_jsend_guard_ontime)
    {
      netout_jsend_guard_time = netout_jsend_guard_ontime;
      netout_jsend_guard_next = netout_jsend_guard_mintime;
    }
}

/****************************************************************/
/*                  add MIDI Poly Touch chapter                 */
/****************************************************************/

void nsys_netout_journal_addptouch(netout_jsend_state * jsend, 
				    unsigned char ndata,
				    unsigned char vdata)

{
  int i;

  /*****************************/
  /* initialize chapter header */
  /*****************************/
  
  if (jsend->chaptera_seqnum)
    jsend->chaptera[NSYS_SM_CA_LOC_LENGTH] &= NSYS_SM_CLRS;
  else
    {
      jsend->chaptera_size = NSYS_SM_CA_HDRSIZE;
      jsend->chaptera[NSYS_SM_CA_LOC_LENGTH] = 0;
      jsend->cheader[NSYS_SM_CH_LOC_TOC] |= NSYS_SM_CH_TOC_SETA;
    }
  
  /**************************/
  /* add new controller log */
  /**************************/
  
  if (jsend->chaptera_seqarray[ndata] == 0)
    {
      /* put data in chapter a and update state variables */
      
      if (jsend->chaptera_seqnum)
	jsend->chaptera[NSYS_SM_CA_LOC_LENGTH]++;
      
      jsend->chaptera[jsend->chaptera_size + 
		     NSYS_SM_CA_LOC_NUM] = ndata;
      jsend->chaptera[jsend->chaptera_size + 
		     NSYS_SM_CA_LOC_PRESSURE] = vdata;
      
      jsend->chaptera_logptr[ndata] = jsend->chaptera_size;
      jsend->chaptera_size += NSYS_SM_CA_LOGSIZE;
      
      jsend->clen += (NSYS_SM_CA_LOGSIZE + (jsend->chaptera_seqnum ? 0 :
					    NSYS_SM_CA_HDRSIZE));
      
      *((unsigned short *) &(jsend->cheader[NSYS_SM_CH_LOC_LEN])) = 
	htons(jsend->clen);
      
    }
  else
    {
      i = jsend->chaptera_logptr[ndata];
      jsend->chaptera[i + NSYS_SM_CA_LOC_NUM] = ndata;
      jsend->chaptera[i + NSYS_SM_CA_LOC_PRESSURE] = vdata;
      jsend->cheader[NSYS_SM_CH_LOC_FLAGS] &= NSYS_SM_CLRS;
    }
  
  /*****************/
  /* update slists */
  /*****************/
  
  netout_jsend_slist[netout_jsend_slist_size++] = 
    &(jsend->chaptera[NSYS_SM_CA_LOC_LENGTH]);
  jsend->chaptera_sset = 1;
  
  jsend->chaptera_seqarray[ndata] = nsys_netout_seqnum;
  jsend->chaptera_seqnum = nsys_netout_seqnum;

  netout_jsend_guard_time = netout_jsend_guard_mintime;
  netout_jsend_guard_next = (netout_jsend_guard_mintime << 
			     (nsys_feclevel == NSYS_SM_FEC_MINIMAL));
}


/****************************************************************/
/*                  add MIDI Channel Touch chapter              */
/****************************************************************/

void nsys_netout_journal_addctouch(netout_jsend_state * jsend, 
				 unsigned char ndata)

{
  jsend->chaptert[NSYS_SM_CT_LOC_PRESSURE] = ndata;

  netout_jsend_slist[netout_jsend_slist_size++] = 
    &(jsend->chaptert[NSYS_SM_CT_LOC_PRESSURE]);
  
  if (jsend->chaptert_seqnum)
    jsend->cheader[NSYS_SM_CH_LOC_FLAGS] &= NSYS_SM_CLRS;
  else
    {
      jsend->cheader[NSYS_SM_CH_LOC_TOC] |= NSYS_SM_CH_TOC_SETT;
      
      *((unsigned short *) &(jsend->cheader[NSYS_SM_CH_LOC_LEN])) = 
	htons((jsend->clen += NSYS_SM_CT_SIZE));
    }
  
  jsend->chaptert_seqnum = nsys_netout_seqnum;

  netout_jsend_guard_time = netout_jsend_guard_mintime;
  netout_jsend_guard_next = (netout_jsend_guard_mintime << 
			     (nsys_feclevel == NSYS_SM_FEC_MINIMAL));
}


/****************************************************************/
/*                  add MIDI Controller chapter                */
/****************************************************************/

void nsys_netout_journal_addcontrol(netout_jsend_state * jsend, 
				    unsigned char ndata,
				    unsigned char vdata)

{
  int i;

  /*****************************/
  /* initialize chapter header */
  /*****************************/
  
  if (jsend->chapterc_seqnum)
    jsend->chapterc[NSYS_SM_CC_LOC_LENGTH] &= NSYS_SM_CLRS;
  else
    {
      jsend->chapterc_size = NSYS_SM_CC_HDRSIZE;
      jsend->chapterc[NSYS_SM_CC_LOC_LENGTH] = 0;
      jsend->cheader[NSYS_SM_CH_LOC_TOC] |= NSYS_SM_CH_TOC_SETC;
    }
    
  /**************************/
  /* update ancillary state */
  /**************************/

  nsys_netout_journal_addhistory(CSYS_MIDI_CC | jsend->chan, ndata, vdata);

  switch (ndata) {
  case CSYS_MIDI_CC_ALLNOTESOFF:
    if (!(jsend->chapterc_allnotes = ((jsend->chapterc_allnotes + 1)  
				      & NSYS_SM_CC_MODULO)))
      jsend->chapterc_allnotes++;
    vdata = jsend->chapterc_allnotes;
    break;
  case CSYS_MIDI_CC_ALLSOUNDOFF:
    if (!(jsend->chapterc_allsound = ((jsend->chapterc_allsound + 1)
				      & NSYS_SM_CC_MODULO)))
      jsend->chapterc_allsound++;
    vdata = jsend->chapterc_allsound;
    break;
  case CSYS_MIDI_CC_SUSTAIN:
    if (vdata == 0)
      jsend->chapterc_sustain &= NSYS_SM_CC_CLRP;  /* pedal lifted, clear p */
    else
      {
	if (jsend->chapterc_sustain & NSYS_SM_CC_CHKP)
	  vdata = jsend->chapterc_sustain & NSYS_SM_CC_CLRP;  /* still down */
	else
	  {
	    /* pedal was lifted, now depressed --> update ccccccc */

	    if (!(jsend->chapterc_sustain = 
		  (jsend->chapterc_sustain + 1) & NSYS_SM_CC_MODULO))
	      jsend->chapterc_sustain++; 

	    vdata = jsend->chapterc_sustain;
	    jsend->chapterc_sustain |= NSYS_SM_CC_SETP;
	  }
      }
  }

  /**************************/
  /* add new controller log */
  /**************************/
  
  if (jsend->chapterc_seqarray[ndata] == 0)
    {
      /* put data in chapter c and update state variables */
      
      if (jsend->chapterc_seqnum)
	jsend->chapterc[NSYS_SM_CC_LOC_LENGTH]++;
      
      jsend->chapterc[jsend->chapterc_size + 
		     NSYS_SM_CC_LOC_LNUM] = ndata;
      jsend->chapterc[jsend->chapterc_size + 
		     NSYS_SM_CC_LOC_LVAL] = vdata;
      
      jsend->chapterc_logptr[ndata] = jsend->chapterc_size;
      jsend->chapterc_size += NSYS_SM_CC_LOGSIZE;
      
      jsend->clen += (NSYS_SM_CC_LOGSIZE + (jsend->chapterc_seqnum ? 0 :
					    NSYS_SM_CC_HDRSIZE));
      
      *((unsigned short *) &(jsend->cheader[NSYS_SM_CH_LOC_LEN])) = 
	htons(jsend->clen);
      
    }
  else
    {
      i = jsend->chapterc_logptr[ndata];
      jsend->chapterc[i + NSYS_SM_CC_LOC_LNUM] = ndata;
      jsend->chapterc[i + NSYS_SM_CC_LOC_LVAL] = vdata;
      jsend->cheader[NSYS_SM_CH_LOC_FLAGS] &= NSYS_SM_CLRS;
    }
  
  /*****************/
  /* update slists */
  /*****************/
  
  netout_jsend_slist[netout_jsend_slist_size++] = 
    &(jsend->chapterc[NSYS_SM_CC_LOC_LENGTH]);
  jsend->chapterc_sset = 1;
  
  jsend->chapterc_seqarray[ndata] = nsys_netout_seqnum;
  jsend->chapterc_seqnum = nsys_netout_seqnum;

  netout_jsend_guard_time = netout_jsend_guard_mintime;
  netout_jsend_guard_next = (netout_jsend_guard_mintime << 
			     (nsys_feclevel == NSYS_SM_FEC_MINIMAL));
}


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*      second-level journal functions: trimstate chapters      */
/*______________________________________________________________*/

/****************************************************************/
/*                  trim entire chapter                         */
/****************************************************************/

void nsys_netout_journal_trimchapter(netout_jsend_state * jsend, 
				     int channel)

{
  int i, j, k;

  memset(jsend->cheader, 0, NSYS_SM_CH_SIZE);
  jsend->cheader_seqnum = 0;
  jsend->chapterp_seqnum = 0;
  jsend->chapterw_seqnum = 0;
  jsend->chaptert_seqnum = 0;

  jsend->chapterc_seqnum = 0;
  while (jsend->chapterc_size > NSYS_SM_CC_HDRSIZE)
    {
      jsend->chapterc_size -= NSYS_SM_CC_LOGSIZE;
      i = jsend->chapterc[jsend->chapterc_size];
	      jsend->chapterc_seqarray[NSYS_SM_CC_CLRF & i] = 0;
    }

  jsend->chaptera_seqnum = 0;
  while (jsend->chaptera_size > NSYS_SM_CA_HDRSIZE)
    {
      jsend->chaptera_size -= NSYS_SM_CA_LOGSIZE;
      i = jsend->chaptera[jsend->chaptera_size];
	      jsend->chaptera_seqarray[NSYS_SM_CA_CLRF & i] = 0;
    }

  jsend->chaptern_seqnum = 0;
  for (j = NSYS_SM_CN_HDRSIZE; j < jsend->chaptern_size;
       j += NSYS_SM_CN_LOGSIZE)
    {
      i = NSYS_SM_CLRS & jsend->chaptern[j];
      jsend->chaptern_seqarray[i] = 0;
    }
  i = (jsend->chapterb_low << NSYS_SM_CN_BFSHIFT);
  for (j = jsend->chapterb_low; j <= jsend->chapterb_high; j++)
    {
      if (jsend->chapterb[j])
	for (k = 128; k >= 1; k = (k >> 1))
	  {
	    if (jsend->chapterb[j] & k)
	      jsend->chaptern_seqarray[i] = 0;
	    i++;
	  }
      else
	i += 8;
    }

  jsend->chapterc_sset = 0;
  jsend->chaptera_sset = 0;
  jsend->chaptern_sset = 0;
  jsend->chaptern_timernum = 0;

  if (channel < (--netout_jsend_channel_size))
    memmove(&(netout_jsend_channel[channel]), 
	    &(netout_jsend_channel[channel+1]),
	    sizeof(unsigned char)* (netout_jsend_channel_size -
				    channel));

}

/****************************************************************/
/*                 trim program change chapter                  */
/****************************************************************/

void nsys_netout_journal_trimprogram(netout_jsend_state * jsend) 
 
{
  jsend->chapterp_seqnum = 0;
  jsend->cheader[NSYS_SM_CH_LOC_TOC] &= NSYS_SM_CH_TOC_CLRP;
  jsend->clen -= NSYS_SM_CP_SIZE;
}

/****************************************************************/
/*                 trim pitch wheel chapter                  */
/****************************************************************/

void nsys_netout_journal_trimpwheel(netout_jsend_state * jsend) 
 
{
  jsend->chapterw_seqnum = 0;
  jsend->cheader[NSYS_SM_CH_LOC_TOC] &= NSYS_SM_CH_TOC_CLRW;
  jsend->clen -= NSYS_SM_CW_SIZE;
}

/****************************************************************/
/*                 trim channel touch chapter                   */
/****************************************************************/

void nsys_netout_journal_trimctouch(netout_jsend_state * jsend) 
 
{
  jsend->chaptert_seqnum = 0;
  jsend->cheader[NSYS_SM_CH_LOC_TOC] &= NSYS_SM_CH_TOC_CLRT;
  jsend->clen -= NSYS_SM_CT_SIZE;
}

/****************************************************************/
/*                 trim entire controller chapter               */
/****************************************************************/

void nsys_netout_journal_trimallcontrol(netout_jsend_state * jsend) 
 
{
  int i, j;

  jsend->chapterc_seqnum = 0;
  jsend->cheader[NSYS_SM_CH_LOC_TOC] &= NSYS_SM_CH_TOC_CLRC;
  jsend->clen -= jsend->chapterc_size;
  j = NSYS_SM_CC_HDRSIZE;
  while (j < jsend->chapterc_size)
    {
      i = NSYS_SM_CC_CLRF & jsend->chapterc[j];
      jsend->chapterc_seqarray[i] = 0;
      j += NSYS_SM_CC_LOGSIZE;
    }
  jsend->chapterc_sset = 0;
}

/****************************************************************/
/*                 partially trim controller chapter            */
/****************************************************************/

void nsys_netout_journal_trimpartcontrol(netout_jsend_state * jsend, 
					 unsigned long minseq)

{
  int i, j;

  j = NSYS_SM_CC_HDRSIZE;
  while (j < jsend->chapterc_size)
    {
      i = NSYS_SM_CC_CLRF & jsend->chapterc[j];
      if (jsend->chapterc_seqarray[i] <= minseq)
	{
	  jsend->chapterc_seqarray[i] = 0;
	  memmove(&(jsend->chapterc[j]), 
		  &(jsend->chapterc[j + NSYS_SM_CC_LOGSIZE]),
		  jsend->chapterc_size - j - NSYS_SM_CC_LOGSIZE);
	  jsend->clen -= NSYS_SM_CC_LOGSIZE;
	  jsend->chapterc_size -= NSYS_SM_CC_LOGSIZE;
	  jsend->chapterc[NSYS_SM_CC_LOC_LENGTH]--;
	}
      else
	{
	  jsend->chapterc_logptr[i] = j;
	  j += NSYS_SM_CC_LOGSIZE;
	}
    }
}

/****************************************************************/
/*                 trim entire poly touch chapter               */
/****************************************************************/

void nsys_netout_journal_trimallptouch(netout_jsend_state * jsend) 
 
{
  int i, j;

  jsend->chaptera_seqnum = 0;
  jsend->cheader[NSYS_SM_CH_LOC_TOC] &= NSYS_SM_CH_TOC_CLRA;
  jsend->clen -= jsend->chaptera_size;
  j = NSYS_SM_CA_HDRSIZE;
  while (j < jsend->chaptera_size)
    {
      i = NSYS_SM_CA_CLRF & jsend->chaptera[j];
      jsend->chaptera_seqarray[i] = 0;
      j += NSYS_SM_CA_LOGSIZE;
    }
  jsend->chaptera_sset = 0;
}

/****************************************************************/
/*                 partially trim poly touch chapter            */
/****************************************************************/

void nsys_netout_journal_trimpartptouch(netout_jsend_state * jsend, 
					unsigned long minseq)

{
  int i, j;

  j = NSYS_SM_CA_HDRSIZE;
  while (j < jsend->chaptera_size)
    {
      i = NSYS_SM_CA_CLRF & jsend->chaptera[j];
      if (jsend->chaptera_seqarray[i] <= minseq)
	{
	  jsend->chaptera_seqarray[i] = 0;
	  memmove(&(jsend->chaptera[j]), 
		  &(jsend->chaptera[j + NSYS_SM_CA_LOGSIZE]),
		  jsend->chaptera_size - j - NSYS_SM_CA_LOGSIZE);
	  jsend->clen -= NSYS_SM_CA_LOGSIZE;
	  jsend->chaptera_size -= NSYS_SM_CA_LOGSIZE;
	  jsend->chaptera[NSYS_SM_CA_LOC_LENGTH]--;
	}
      else
	{
	  jsend->chaptera_logptr[i] = j;
	  j += NSYS_SM_CA_LOGSIZE;
	}
    }
}

/****************************************************************/
/*                 trim entire note chapter                     */
/****************************************************************/

void nsys_netout_journal_trimallnote(netout_jsend_state * jsend) 
 
{
  int i, j, k;

  jsend->chaptern_seqnum = 0;
  jsend->cheader[NSYS_SM_CH_LOC_TOC] &= NSYS_SM_CH_TOC_CLRN;
  jsend->clen -= (jsend->chaptern_size + jsend->chapterb_size);
  
  for (j = NSYS_SM_CN_HDRSIZE; j < jsend->chaptern_size;
       j += NSYS_SM_CN_LOGSIZE)
    {
      i = NSYS_SM_CLRS & jsend->chaptern[j];
      jsend->chaptern_seqarray[i] = 0;
    }
  i = (jsend->chapterb_low << NSYS_SM_CN_BFSHIFT);
  for (j = jsend->chapterb_low; j <= jsend->chapterb_high; j++)
    {
      if (jsend->chapterb[j])
	for (k = 128; k >= 1; k = (k >> 1))
	  {
	    if (jsend->chapterb[j] & k) 
	      jsend->chaptern_seqarray[i] = 0;
	    i++;
	  }
      else
	i += 8;
    }
  jsend->chaptern_timernum = 0;
  jsend->chaptern_sset = 0;
}

/****************************************************************/
/*                 trim entire note chapter                     */
/****************************************************************/

void nsys_netout_journal_trimpartnote(netout_jsend_state * jsend, 
				      unsigned long minseq)

{
  int i, j, k;

  /* first prune note logs */
  
  j = NSYS_SM_CN_HDRSIZE;
  while (j < jsend->chaptern_size)
    {
      i = NSYS_SM_CLRS & jsend->chaptern[j];
      if (jsend->chaptern_seqarray[i] <= minseq)
	{	  
	  if (jsend->chaptern[i+NSYS_SM_CN_LOC_VEL] & NSYS_SM_CN_CHKY)
	    jsend->chaptern_timernum--;
	  jsend->chaptern_seqarray[i] = 0;
	  memmove(&(jsend->chaptern[j]), 
		  &(jsend->chaptern[j + NSYS_SM_CN_LOGSIZE]),
		  jsend->chaptern_size - j - NSYS_SM_CN_LOGSIZE);
	  jsend->clen -= NSYS_SM_CN_LOGSIZE;
	  jsend->chaptern_size -= NSYS_SM_CN_LOGSIZE;
	  jsend->chaptern[NSYS_SM_CN_LOC_LENGTH]--;
	}
      else
	{
	  jsend->chaptern_logptr[i] = j;
	  j += NSYS_SM_CN_LOGSIZE;
	}
    }
  
  /* then prune bitfields */
  
  i = (jsend->chapterb_low << NSYS_SM_CN_BFSHIFT);
  for (j = jsend->chapterb_low; j <= jsend->chapterb_high; j++)
    {
      if (jsend->chapterb[j])
	for (k = 128; k >= 1; k = (k >> 1))
	  {
	    if ((jsend->chapterb[j] & k) && 
		(jsend->chaptern_seqarray[i] <= minseq))
	      {
		jsend->chaptern_seqarray[i] = 0;
		jsend->chapterb[j] &= ~k;
	      }
	    i++;
	  }
      else
	i += 8;
      if ((j == jsend->chapterb_low) && 
	  (jsend->chapterb[j] == 0))
	{
	  jsend->chapterb_low++;
	  jsend->chapterb_size--;
	  jsend->clen--;
	}
    }
  while (jsend->chapterb_size && 
	 (jsend->chapterb[jsend->chapterb_high] == 0))
    {
      jsend->chapterb_high--;
      jsend->chapterb_size--;
      jsend->clen--;
    }
  if (jsend->chapterb_size == 0)
    {	 
      jsend->chapterb_low = NSYS_SM_CN_BFMAX;
      jsend->chapterb_high = NSYS_SM_CN_BFMIN;
    }
  jsend->chaptern[NSYS_SM_CN_LOC_LOWHIGH] = 
    ((jsend->chapterb_low << NSYS_SM_CN_LOWSHIFT) |
     jsend->chapterb_high);

}


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*      second-level journal functions: miscellaneous           */
/*______________________________________________________________*/

/****************************************************************/
/*           updates state after checkpoint change              */
/****************************************************************/

void nsys_netout_journal_changecheck(void)

{
  unsigned short check_seqnum_net;

  netout_jsend_checkpoint_changed = 1;
  netout_jsend_header[NSYS_SM_JH_LOC_FLAGS] |= NSYS_SM_JH_SETK;
  check_seqnum_net = htons((unsigned short) (NSYS_RTPSEQ_LOMASK &
					     netout_jsend_checkpoint_seqnum));
  memcpy(&(netout_jsend_header[NSYS_SM_JH_LOC_CHECK]),
	 &check_seqnum_net, sizeof(unsigned short));

}


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                  low-level journal functions                 */
/*______________________________________________________________*/


/* end Network library -- sender journal functions */
