/*  Screem:  screem-preview.c
 *
 *  The html preview widget
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-async-ops.h>

#include <glib/gi18n.h>

#include <libgtkhtml/gtkhtml.h>

#ifdef SCREEM_MOZ
#include <gtkmozembed.h>
#endif

#include "screem-view.h"

#include "screem-preview.h"
#include "screem-window.h"

#include "fileops.h"

#include "support.h"

static void screem_preview_display_view( ScreemView *view );
static void screem_preview_page( ScreemPreview *preview, ScreemPage *page,
				 gboolean online );

static void url_requested( HtmlDocument *document, const gchar *url,
			   HtmlStream *stream, ScreemPreview *preview );
static void url_clicked(  HtmlDocument *document, const gchar *url,
			  ScreemPreview *preview );

static void screem_preview_class_init( ScreemPreviewClass *klass );
static void screem_preview_init( ScreemPreview *preview );
static void screem_preview_finalize( GObject *preview );
static void screem_preview_window_set( ScreemView *view );
static void screem_preview_size_request( GtkWidget *widget, GtkRequisition *req );
static void screem_preview_realize( GtkWidget *widget );
static void screem_preview_show( GtkWidget *widget );


static void screem_preview_async_cb( GnomeVFSAsyncHandle *handle,
				     GnomeVFSResult result,
				     gpointer data );
static void screem_preview_async_read_cb( GnomeVFSAsyncHandle *handle,
					  GnomeVFSResult result,
					  gpointer buffer,
					  GnomeVFSFileSize bytes_requested,
					  GnomeVFSFileSize bytes_read,
					  gpointer data );
static void screem_preview_async_close_cb( GnomeVFSAsyncHandle *handle,
					   GnomeVFSResult result,
					   gpointer data );

static gboolean screem_preview_cancel_all( gpointer key,
					   gpointer value,
					   ScreemPreview *preview );

static HtmlDocument *screem_preview_new_document( ScreemPreview *preview );

struct ScreemPreviewPrivate {
	GtkWidget *sw;
	GtkWidget *html;

	gchar *base_uri;

	GHashTable *streams;
	GMutex *streams_lock;	
};

ScreemPreview *screem_preview_new()
{
	ScreemPreview *preview;
	GType type;

	type = screem_preview_get_type();

	preview = SCREEM_PREVIEW( g_object_new( type, NULL ) );

	return preview;
}

ScreemPreview *screem_preview_new_with_window( ScreemWindow *window )
{

	ScreemPreview *preview;
	GType type;

	type = screem_preview_get_type();

	preview = SCREEM_PREVIEW( g_object_new( type, "window", window,
						NULL ) );

	return preview;
}

static void screem_preview_display_view( ScreemView *view )
{
	ScreemPreview *preview;
	ScreemSite *site;
	ScreemPage *page;
	gboolean online;

	const gchar *baseuri;
	const gchar *pagepath;

	gboolean dynamic;

	preview = SCREEM_PREVIEW( view );
	
	g_object_get( G_OBJECT( view ),
		      "site", &site,
		      "page", &page,
		      "online", &online,
		      NULL );

	g_mutex_lock( preview->private->streams_lock );
	g_hash_table_foreach( preview->private->streams,
				(GHFunc)screem_preview_cancel_all, preview );
	g_mutex_unlock( preview->private->streams_lock );
	
	if( site && ! screem_site_get_fake_flag( site ) ) {
		baseuri = screem_site_get_http_url( site );
	} else {
		baseuri = NULL;
	}
	if( page ) {
		dynamic = screem_page_is_dynamic( page );
		pagepath = screem_page_get_pathname( page );
	} else {
		dynamic = FALSE;
		pagepath = NULL;
	}
	if( pagepath && strlen( pagepath ) && baseuri && 
	    ( ( ! dynamic && 
		screem_site_get_preview_static( site ) ) ||
	      ( dynamic && screem_site_get_preview_dynamic( site ) ) ) ) {
		gchar *uri;
		gchar *tmp;
		
		tmp = g_path_get_basename( pagepath );
		uri = g_strconcat( "http://", baseuri, "/", tmp, NULL );
		g_free( tmp );
		
		screem_preview_uri( SCREEM_PREVIEW( view ),
				    uri, online );
		
		g_free( uri );
	} else {
		screem_preview_page( SCREEM_PREVIEW( view ),
				     page, online );
	}

	if( site ) {
		g_object_unref( site );
	}
	if( page ) {
		g_object_unref( page );
	}
}

static void screem_preview_page( ScreemPreview *preview, ScreemPage *page,
				 gboolean online )
{
	g_return_if_fail( SCREEM_IS_PREVIEW( preview ) );

	if( page ) {
		const gchar *pathname;
		const gchar *mime_type;
		gchar *data;
		gchar *base_uri;
		GnomeVFSURI *uri;
		gchar *charset;
		gchar *tmp;
		
		pathname = screem_page_get_pathname( page );
		mime_type = screem_page_get_mime_type( page );

		if( ! mime_type || 
		    ( strcmp( "text/html", mime_type ) &&
		      strcmp( "application/x-php", mime_type ) ) ) {
			data = g_strconcat( "<html><body style=\"background: black; color: white; text-align: center\">", _( "Not an HTML document " ),
					    "</body></html>", NULL );
		} else {
			data = screem_page_get_data( page );
		}

		charset = screem_page_get_charset( page );
		tmp = screem_support_charset_convert_to( data, 
				charset, "UTF-8" );
		if( tmp ) {
			g_free( data );
			data = tmp;
		}
		g_free( charset );

		if( pathname ) {
			uri = gnome_vfs_uri_new( pathname );
			if( uri ) {
				if( ! uri->parent ) {
					tmp = gnome_vfs_uri_to_string( uri, 0 );
					base_uri = g_path_get_dirname( tmp );
					g_free( tmp );
				} else {
					base_uri = gnome_vfs_uri_to_string( uri->parent, 0 );
				}
			} else {
				base_uri = g_strdup( "file:///" );
			}
		} else {
			base_uri = NULL;
			uri = NULL;
		}
		screem_preview_display( preview, data, base_uri, online );
		
		if( uri ) {
			gnome_vfs_uri_unref( uri );
		}
		if( base_uri ) {
			g_free( base_uri );
		}
		if( data ) {
			g_free( data );
		}
	} else {
		/* can't pass an empty string as HtmlStream gets upset */
		screem_preview_display( preview, NULL, NULL, online );
	}
}


void screem_preview_display( ScreemPreview *preview, const gchar *text,
			     const gchar *base_uri, gboolean online )
{
#ifndef SCREEM_MOZ
	HtmlDocument *document;
	HtmlDocument *old;

	g_return_if_fail( SCREEM_IS_PREVIEW( preview ) );

	if( ! text || ( strlen( text ) == 0 )) {
		text ="<html><body></body></html>";
	}
	if( ! base_uri ) {
		base_uri = "file:///";
	}

	document = screem_preview_new_document( preview );
	old = HTML_VIEW( preview->private->html )->document;
	html_view_set_document( HTML_VIEW( preview->private->html ), document );
	if( old ) {
		g_object_unref( old );
	}
	
	if( preview->private->base_uri ) {
		g_free( preview->private->base_uri );
	}
	preview->private->base_uri = g_strdup( base_uri );
	
	html_document_open_stream( document, "text/html" );
	html_document_write_stream( document, text, strlen( text ) );
	html_document_close_stream( document );
#else
	g_return_if_fail( SCREEM_IS_PREVIEW( preview ) );

	if( ! text || ( strlen( text ) == 0 )) {
		text ="<html><body></body></html>";
	}
	if( ! base_uri ) {
		base_uri = "file:///";
	}

	if( preview->private->base_uri ) {
		g_free( preview->private->base_uri );
	}
	preview->private->base_uri = g_strdup( base_uri );

	gtk_moz_embed_render_data( GTK_MOZ_EMBED( preview->private->html ),
				   text, strlen( text ),
				   base_uri, "text/html" );
#endif
}

void screem_preview_uri( ScreemPreview *preview, const gchar *uri, 
			 gboolean online )
{
#ifndef SCREEM_MOZ
	HtmlDocument *document;
	HtmlDocument *old;

	document = screem_preview_new_document( preview );
	old = HTML_VIEW( preview->private->html )->document;
	html_view_set_document( HTML_VIEW( preview->private->html ), document );
	if( old ) {
		g_object_unref( old );
	}
	
	if( preview->private->base_uri ) {
		g_free( preview->private->base_uri );
	}
	preview->private->base_uri = g_path_get_dirname( uri );

	html_document_open_stream( document, "text/html" );

	url_requested( document, uri, document->current_stream, preview );
#endif
}

void screem_preview_print( ScreemView *view, gboolean prview )
{

}

static void url_requested( HtmlDocument *document, const gchar *url,
			   HtmlStream *stream, ScreemPreview *preview )
{
	ScreemSite *site;
	ScreemPage *page;
	GnomeVFSURI *uri;
	GnomeVFSAsyncHandle *handle;
	gboolean online;
	gboolean dynamic;
	gchar *full;
	gchar *msg;
	const gchar *sitebase;
	
	if( document != HTML_VIEW( preview->private->html )->document ) {
		html_stream_close( stream );
		g_object_unref( document );
		return;
	}
	
	g_object_get( G_OBJECT( preview ), 
			"online", &online, 
			"site", &site,
			"page", &page,
			NULL );
	
	dynamic = FALSE;
	if( page ) {	
		dynamic = screem_page_is_dynamic( page );
	}
	full = NULL;
	if( g_path_is_absolute( url ) ) {
		sitebase = NULL;
		if( site && ! screem_site_get_fake_flag( site ) ) {
			if( ( dynamic && screem_site_get_preview_dynamic( site ) ) ||
			    ( ! dynamic && screem_site_get_preview_static( site ) ) ) {
				sitebase = screem_site_get_http_url( site );
			} else {
				sitebase = screem_site_get_pathname( site );
			}
		} else if( ! uri_exists( url, NULL ) ) {
			sitebase = preview->private->base_uri;
		}
		if( sitebase ) {
			full = g_strconcat( sitebase, url, NULL );
		}
	}
	if( ! full ) {
		full = relative_to_full( url, 
				preview->private->base_uri );
	}
	
	if( full ) {
		uri = gnome_vfs_uri_new( full );
	} else {
		uri = NULL;
	}

	if( page ) {
		g_object_unref( page );
	}
	if( site ) {
		g_object_unref( site );
	}
	
	if( uri ) {
		g_free( full );
		full = gnome_vfs_uri_to_string( uri, 0 );
		if( gnome_vfs_uri_is_local( uri ) || online ) {
			g_mutex_lock( preview->private->streams_lock );
			
			/* async open, the callback will wait on
			   preview->private->streams_lock before
			   doing anything */
			g_object_ref( document );
			gnome_vfs_async_open( &handle, full, GNOME_VFS_OPEN_READ,
					      -10, 
					      (GnomeVFSAsyncOpenCallback)
					      screem_preview_async_cb,
					      preview );
			
			/* put the stream in the hash table */
			g_hash_table_insert( preview->private->streams,
						handle, stream );
			g_object_set_data( G_OBJECT( stream ), "document",
					   document );
			
			g_mutex_unlock( preview->private->streams_lock );
		} else {
			msg = g_strconcat( "Not fetching remote uri: ",
						full, NULL );
			screem_view_show_message( SCREEM_VIEW( preview ),
					msg );
			g_free( msg );
			html_stream_close( stream );
		}
	} else {
		
		msg = g_strconcat( _( "Invalid URI in page: " ),
				   full, NULL );
		screem_view_show_error( SCREEM_VIEW( preview ),
					msg );
		g_free( msg );
		html_stream_close( stream );
	}
	
	g_free( full );
}


static void screem_preview_async_close_cb( GnomeVFSAsyncHandle *handle,
					   GnomeVFSResult result,
					   gpointer data )
{
	ScreemPreview *preview;
	ScreemWindow *window;
	
	preview = SCREEM_PREVIEW( data );
	g_object_get( G_OBJECT( preview ), "window", &window, NULL );
	
	g_mutex_lock( preview->private->streams_lock );

	g_hash_table_remove( preview->private->streams, handle );
	
	g_mutex_unlock( preview->private->streams_lock );
}

static void url_clicked( HtmlDocument *document, const gchar *url, 
			 ScreemPreview *preview )
{
}

static void screem_preview_async_cb( GnomeVFSAsyncHandle *handle,
					GnomeVFSResult result,
					gpointer data )
{
	ScreemPreview *preview;
	ScreemWindow *window;
	
	HtmlStream *stream;
	HtmlDocument *document;
	gchar *buffer;
	
	preview = SCREEM_PREVIEW( data );
	g_object_get( G_OBJECT( preview ), "window", &window, NULL );
	
	g_mutex_lock( preview->private->streams_lock );
	
	stream = HTML_STREAM( g_hash_table_lookup( preview->private->streams,
						   handle ) );
	document = NULL;
	if( ! stream ) {
		result = GNOME_VFS_ERROR_GENERIC;
	} else {
		document = g_object_get_data( G_OBJECT( stream ), "document" );
	}

	if( document != HTML_VIEW( preview->private->html )->document ) {
		result = GNOME_VFS_ERROR_GENERIC;
	}
	
	g_mutex_unlock( preview->private->streams_lock );
	
	switch( result ) {
		case GNOME_VFS_OK:
			buffer = g_new( gchar, 8192 );
			gnome_vfs_async_read( handle, buffer, 8192,
					screem_preview_async_read_cb,
					preview );
			break;
		default:
			/* failed to open, close HtmlStream */
			if( stream ) {
				html_stream_close( stream );
				g_object_unref( document );
			}
			if( result != GNOME_VFS_ERROR_NOT_FOUND ) {
				gnome_vfs_async_close( handle, 
						       screem_preview_async_close_cb,
						       preview );
			}
			break;
	}
}

static void screem_preview_async_read_cb( GnomeVFSAsyncHandle *handle,
					GnomeVFSResult result,
					gpointer buffer,
					GnomeVFSFileSize bytes_requested,
					GnomeVFSFileSize bytes_read,
					gpointer data )
{
	ScreemPreview *preview;
	ScreemWindow *window;
	
	HtmlStream *stream;
	HtmlDocument *document;
	
	preview = SCREEM_PREVIEW( data );
	g_object_get( G_OBJECT( preview ), "window", &window, NULL );
	
	g_mutex_lock( preview->private->streams_lock );

	stream = HTML_STREAM( g_hash_table_lookup( preview->private->streams,
						   handle ) );
	document = NULL;
	if( ! stream ) {
		result = GNOME_VFS_ERROR_GENERIC;
	} else {
		document = g_object_get_data( G_OBJECT( stream ), "document" );
	}

	if( document != HTML_VIEW( preview->private->html )->document ) {
		result = GNOME_VFS_ERROR_GENERIC;
	}
	
	g_mutex_unlock( preview->private->streams_lock );
	
	switch( result ) {
		case GNOME_VFS_OK:
			if( bytes_read > 0 && stream ) {
				html_stream_write( stream, buffer,
						bytes_read );
				
				gnome_vfs_async_read( handle, buffer, 8192,
						screem_preview_async_read_cb,
						preview );
			}
			break;
		default:
			if( stream ) {
				html_stream_close( stream );
				g_object_unref( document );
			}
			gnome_vfs_async_close( handle, 
					       screem_preview_async_close_cb,
					       preview );
			break;
	}
}     

static HtmlDocument *screem_preview_new_document( ScreemPreview *preview )
{
	HtmlDocument *document;
	
	document = html_document_new();
	g_signal_connect_data( G_OBJECT( document ), "request_url",
			       G_CALLBACK( url_requested ), preview,
			       NULL, 0 );
	g_signal_connect_data( G_OBJECT( document ), "link_clicked",
			       G_CALLBACK( url_clicked ), preview,
			       NULL, 0 );

	g_object_ref( document );
	
	return document;	
}

/* G Object stuff */

#define PARENT_TYPE SCREEM_TYPE_VIEW

static gpointer parent_class;

static void screem_preview_class_init( ScreemPreviewClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_preview_finalize;

	widget_class->show = screem_preview_show;
	widget_class->realize = screem_preview_realize;
	widget_class->size_request = screem_preview_size_request;
}

static void screem_preview_init( ScreemPreview *preview )
{
	GtkWidget *html;
	GtkWidget *sw;
	gchar *profile;
	
	ScreemPreviewPrivate *private;

	SCREEM_VIEW( preview )->window_set = screem_preview_window_set;
	SCREEM_VIEW( preview )->display = screem_preview_display_view;
	SCREEM_VIEW( preview )->print = screem_preview_print;

	private = preview->private = g_new0( ScreemPreviewPrivate, 1 );
#ifndef SCREEM_MOZ
	html = private->html = html_view_new();

	gtk_widget_show( private->html );

	sw = preview->private->sw = 
		gtk_scrolled_window_new( GTK_LAYOUT( html )->hadjustment,
					 GTK_LAYOUT( html )->vadjustment );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
						
	gtk_container_add( GTK_CONTAINER( sw ), html );

	gtk_container_add( GTK_CONTAINER( preview ), sw );
#else
	gtk_moz_embed_set_profile_path( "/home/david/.gnome2/epiphany/mozilla", "epiphany" );
	gtk_moz_embed_set_comp_path( g_getenv( "MOZILLA_FIVE_HOME" ) );
	html = private->html = gtk_moz_embed_new();

	gtk_moz_embed_set_chrome_mask( html, GTK_MOZ_EMBED_FLAG_ALLCHROME );
	
	gtk_container_add( GTK_CONTAINER( preview ), html );
#endif
	preview->private->streams = g_hash_table_new( NULL, NULL );
	preview->private->streams_lock = g_mutex_new();
}

static void screem_preview_finalize( GObject *object )
{
	ScreemPreview *preview;

	preview = SCREEM_PREVIEW( object );

	g_mutex_lock( preview->private->streams_lock );
	g_hash_table_foreach( preview->private->streams,
				(GHFunc)screem_preview_cancel_all, preview );
	g_mutex_unlock( preview->private->streams_lock );

	/* FIXME: leaking the mutex / hashtable here */
		
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static gboolean screem_preview_cancel_all( gpointer key,
					   gpointer value,
					   ScreemPreview *preview )
{
	gnome_vfs_async_cancel( key );
	
	return FALSE;
}

static void screem_preview_window_set( ScreemView *view )
{

}

static void screem_preview_size_request( GtkWidget *widget, GtkRequisition *req )
{
	GTK_WIDGET_CLASS( parent_class )->size_request( widget, req );
}

static void screem_preview_realize( GtkWidget *widget )
{
	GTK_WIDGET_CLASS( parent_class )->realize( widget );
}

static void screem_preview_show( GtkWidget *widget )
{
	ScreemPreview *preview;

	preview = SCREEM_PREVIEW( widget );

	GTK_WIDGET_CLASS( parent_class )->show( widget );

	gtk_widget_show( preview->private->html );

	if( preview->private->sw ) {
		gtk_widget_show( preview->private->sw );
	}
}

GType screem_preview_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemPreviewClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_preview_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemPreview ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_preview_init
		};
		
		type = g_type_register_static( PARENT_TYPE,
					       "ScreemPreview",
					       &info, 0 );
	}

	return type;
}
