/*  Screem:  screem-dtd_db.c
 *
 *  Copyright (C) 2003 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include <libgnome/gnome-macros.h>

#include <libgnomevfs/gnome-vfs-utils.h>

#include <gconf/gconf-client.h>

#include <glib/ghash.h>
#include <glib/gstring.h>
#include <glib/gunicode.h>

#include <gtk/gtkcombo.h>

#include <libxml/tree.h>

#include <dirent.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#include "screem-dtd-db.h"
#include "fileops.h"
#include "support.h"

static void screem_dtd_db_class_init( ScreemDTDDBClass *klass );
static void screem_dtd_db_instance_init( ScreemDTDDB *dtd_db );
static void screem_dtd_db_finalize( GObject *object );
static void screem_dtd_db_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_dtd_db_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );

static void screem_dtd_load_catalog( ScreemDTDDB *db, 
				     const GString *pathname );
static void screem_dtd_load_catalogs( ScreemDTDDB *db );
static gint screem_dtd_db_sort_compare( const gchar *a, const gchar *b);
static void dtds_insert( gpointer key, gpointer value, GObject *widget);
static void screem_dtd_db_offline_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );

static xmlNodePtr screem_dtd_db_lookup_override( ScreemDTDDB *db,
						const gchar *publicid );
static gchar *screem_dtd_db_override_get_root( ScreemDTDDB *db,
						const gchar *publicid );
static gchar *screem_dtd_db_override_get_system_id( ScreemDTDDB *db,
						const gchar *publicid );
static ScreemDTD *screem_dtd_db_new_dtd( ScreemDTDDB *db,
					 const gchar *publicid,
					 const gchar *systemid );
static gboolean screem_dtd_db_load_internal( ScreemDTDDB *db,
					const gchar *publicid,
					const gchar *systemid );


struct ScreemDTDDBPrivate {
	GHashTable *loaded_dtds;
	GHashTable *dtd_types;

	xmlDocPtr  overrides;
	gchar *overridepath;

	GtkListStore *store;

	gboolean offline;
	guint notify;
};

GNOME_CLASS_BOILERPLATE( ScreemDTDDB, screem_dtd_db,
			  GObject,
			  G_TYPE_OBJECT )

static void screem_dtd_db_class_init( ScreemDTDDBClass *klass )
{
	GObjectClass *obj_class;
	
	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_dtd_db_finalize;
	obj_class->get_property = screem_dtd_db_get_prop;
	obj_class->set_property = screem_dtd_db_set_prop;
}


static void screem_dtd_db_instance_init( ScreemDTDDB *dtd_db )
{
	ScreemDTDDBPrivate *priv;
	GConfClient *client;
	gchar *tmp;
	xmlNodePtr node;
	
	priv = dtd_db->priv = g_new0( ScreemDTDDBPrivate, 1 );

	client = gconf_client_get_default();

	priv->notify = gconf_client_notify_add( client,
					 "/apps/screem/general/work_offline", 
					 screem_dtd_db_offline_notify,
					 dtd_db, NULL, NULL );
	priv->offline = gconf_client_get_bool( client,
					"/apps/screem/general/work_offline",
					NULL );

	priv->dtd_types = g_hash_table_new( g_str_hash, g_str_equal );
	priv->loaded_dtds = g_hash_table_new( g_str_hash, g_str_equal );

	tmp = screem_get_dot_dir();
	priv->overridepath = g_build_filename( tmp, 
				"dtd_override.xml", NULL );
	g_free( tmp );

	priv->overrides = NULL;
	if( uri_exists( priv->overridepath, NULL ) ) {
		priv->overrides = xmlParseFile( priv->overridepath );
	}
	if( ! priv->overrides ) {
		/* no overrides file, create a new doc */
		priv->overrides = xmlNewDoc( XML_DEFAULT_VERSION );
		node = xmlNewDocNode( priv->overrides,
					NULL, /* ns */
					"screemoverride",
					NULL );
		xmlDocSetRootElement( priv->overrides, node );
		xmlSaveFile( priv->overridepath, priv->overrides );
	}

	screem_dtd_load_catalogs( dtd_db );

	priv->store = gtk_list_store_new( 3, 
				G_TYPE_STRING, 
				G_TYPE_STRING,
				G_TYPE_STRING );

	screem_dtd_db_fill_list( dtd_db, priv->store );
}

static gboolean screem_dtd_db_finalize_loaded( gpointer key,
						gpointer value,
						gpointer data )
{
	g_free( key );
	g_object_unref( G_OBJECT( value ) );

	return FALSE;
}

static gboolean screem_dtd_db_finalize_dtd_types( gpointer key,
						gpointer value,
						gpointer data )
{
	g_free( key );
	g_free( value );

	return FALSE;
}

static void screem_dtd_db_finalize( GObject *object )
{
	ScreemDTDDB *dtd_db;
	ScreemDTDDBPrivate *priv;
	GConfClient *client;
	
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_DTD_DB( object ) );

	dtd_db = SCREEM_DTD_DB( object );

	priv = dtd_db->priv;

	g_free( priv->overridepath );
	xmlFreeDoc( priv->overrides );
	g_object_unref( priv->store );

	g_hash_table_foreach( priv->loaded_dtds,
			(GHFunc)screem_dtd_db_finalize_loaded,
			NULL );
	g_hash_table_destroy( priv->loaded_dtds );

	g_hash_table_foreach( priv->dtd_types,
			(GHFunc)screem_dtd_db_finalize_dtd_types,
			NULL );
	g_hash_table_destroy( priv->dtd_types );

	
	client = gconf_client_get_default();
	gconf_client_notify_remove( client, priv->notify );
	g_object_unref( client );

	g_free( priv );
	
	GNOME_CALL_PARENT( G_OBJECT_CLASS, finalize, (object) );
}

static void screem_dtd_db_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemDTDDB *dtd_db;
	ScreemDTDDBPrivate *priv;
	
	dtd_db = SCREEM_DTD_DB( object );
	priv = dtd_db->priv;

	switch( prop_id ) {
		default:
			break;
	}
}

static void screem_dtd_db_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemDTDDB *dtd_db;
	ScreemDTDDBPrivate *priv;
	
	dtd_db = SCREEM_DTD_DB( object );
	priv = dtd_db->priv;

	switch( prop_id ) {
		default:
			break;
	}
}

/* static stuff */

static void screem_dtd_load_catalog( ScreemDTDDB *db, 
				     const GString *pathname )
{
	gchar *buffer;
	FILE *file;

	gchar *id;
	gchar *path;

	gchar *tid;
	gchar *tpath;
	gchar *temp;
	gint ret;
	GString *tpathname;

	buffer = g_new0( gchar, BUFSIZ );
	id = g_new0( gchar, BUFSIZ );
	path = g_new0( gchar, BUFSIZ );

	/* we have a catalog file */
	file = fopen( pathname->str, "r" );
	if( ! file )
		return;

	while( fgets( buffer, BUFSIZ, file ) ) {
		buffer = g_strchug( buffer );
		if( ! strncmp( "CATALOG ", buffer, strlen( "CATALOG " ) ) ) {
			/* another catalog file to check */
			buffer[ strlen( buffer) - 1 ] = '\0';
			tpathname = g_string_new( buffer + strlen( "CATALOG " ) );
			screem_dtd_load_catalog( db, tpathname );
			g_string_free( tpathname, TRUE );
			continue;
		}
		if( strncmp( "PUBLIC ", buffer, strlen( "PUBLIC " ) ) )
			continue;
		ret = sscanf( buffer, "PUBLIC \"%[^\"]\" \"%[^\"]\"", 
			      id, path );
		if( ! ret )
			break;
		else if( ret == 1 ) {
			/* ok try a different match */
			if( sscanf( buffer, "PUBLIC \"%[^\"]\" %s", 
				    id, path ) != 2 )
				break;
		}
		
		/* is id already in the hash table? */
		if( ! g_hash_table_lookup( db->priv->dtd_types, id ) ) {
			/* nope */
			gchar *tmp;
			
			tid = g_strdup( id );
			tpath = gnome_vfs_escape_host_and_path_string( path );
			if( ! g_path_is_absolute( path ) ) {
				
				tmp = g_path_get_dirname( pathname->str );
				temp = g_strconcat( "file://", tmp,
						G_DIR_SEPARATOR_S,
						tpath, NULL );
				g_free( tmp );
			} else {
				temp = g_strconcat( "file://", tpath,
						NULL );
			}
			g_free( tpath );
			tpath = temp;
			g_hash_table_insert( db->priv->dtd_types, tid, tpath );
		}
	}
	fclose( file );

	g_free( buffer );
	g_free( id );
	g_free( path );
}

static void screem_dtd_load_catalogs( ScreemDTDDB *db )
{
	gchar *paths[] = {
		SCREEM_DTD_PATH,
		"/etc/sgml",
		"/usr/lib/sgml",                      /* Debian, RedHat */
		"/usr/share/sgml",                    /* SuSE */
		"/usr/share/lib/sgml/locale/C/dtds",  /* Solaris, use the 
							 current locale at 
							 runtime? */
		NULL,                                 /* replaced with the
							 users .screem dir at
							 runtime */
		NULL
	};

	gint i;
	GString *file;

	DIR *dir;
	struct dirent *entry;

	gchar *tmp;
	
	tmp = screem_get_dot_dir();
	paths[ 4 ] = g_strconcat( tmp, G_DIR_SEPARATOR_S,
				  "dtds", NULL );
	g_free( tmp );

	/* make dtds dir if needed */
	mkdir_recursive( paths[ 4 ], GNOME_VFS_PERM_USER_ALL,
			NULL, NULL );

	for( i = 0; paths[ i ]; ++ i ) {
		if( ! ( dir = opendir( paths[ i ] ) ) ) {
			continue;
		}

		/* get next file */
		while( ( entry = readdir( dir ) ) ) {
			if( ( ! g_strcasecmp( "catalog", entry->d_name ) ) ||
			    ( ! g_strncasecmp( "catalog.", 
					       entry->d_name, 
					       strlen( "catalog." ) ) ) ) {
				file = g_string_new( paths[ i ] );
				g_string_append_c( file, 
						G_DIR_SEPARATOR );
				g_string_append( file, entry->d_name );
				
				screem_dtd_load_catalog( db, file );
				
				g_string_free( file, TRUE );
			}
		}
		closedir( dir );
	}
	
	g_free( paths[ 4 ] );
}

static gint screem_dtd_db_sort_compare( const gchar *a, const gchar *b )
{
	gint ret;
		
	ret = 0;

	if( a && ! b ) {
		ret = -1;
	} else if( b && ! a ) {
		ret = 1;
	} else {
		const gchar *htmla;
		const gchar *htmlb;
		
		htmla = strstr( a, "HTML" );
		htmlb = strstr( b, "HTML" );
		if( ! htmla ) {
			htmla = strstr( a, "html" );
		}
		if( ! htmlb ) {
			htmlb = strstr( b, "html" );
		}
		if( htmla && ! htmlb ) {
			ret = -1;
		} else if( htmlb && ! htmla ) {
			ret = 1;
		} else {
			ret = strcmp( a, b );
		}
	}

	return ret;
}

static void dtds_insert( gpointer key, gpointer value, GObject *widget )
{
	if( ! strstr( (gchar*)key, "ENTITIES" ) ) {
		GList *list;

		list = (GList*)g_object_get_data( G_OBJECT( widget ), "list" );

		list = g_list_prepend( list, key );

		g_object_set_data( G_OBJECT( widget ), "list", list );
	}
}

static void screem_dtd_db_offline_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemDTDDB *db;

	db = SCREEM_DTD_DB( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean state;

		state = gconf_value_get_bool( entry->value );
		db->priv->offline = state;
	}
}

static xmlNodePtr screem_dtd_db_lookup_override( ScreemDTDDB *db,
					  const gchar *publicid )
{
	xmlNodePtr ret;
	xmlChar *val;
	
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );

	ret = xmlDocGetRootElement( db->priv->overrides );

	if( ret ) {
		for( ret = ret->children; ret; ret = ret->next ) {
			val = NULL;
			if( ! strcmp( "dtd", ret->name ) ) {
				val = xmlGetProp( ret, "public" ); 
			}
			if( val && ! strcmp( val, publicid ) ) {
				g_free( val );
				break;
			}
			g_free( val );
		}
	}

	return ret;
}

static gchar *screem_dtd_db_override_get_root( ScreemDTDDB *db,
						const gchar *publicid )
{
	gchar *ret;
	xmlNodePtr override;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	ret = NULL;
	override = screem_dtd_db_lookup_override( db, publicid );
	if( override ) {
		ret = xmlGetProp( override, "root" );
	}

	return ret;
}

static gchar *screem_dtd_db_override_get_system_id( ScreemDTDDB *db,
						const gchar *publicid )
{
	gchar *ret;
	xmlNodePtr override;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	ret = NULL;
	override = screem_dtd_db_lookup_override( db, publicid );
	if( override ) {
		ret = xmlGetProp( override, "systemid" );
	}

	return ret;
}

static ScreemDTD *screem_dtd_db_new_dtd( ScreemDTDDB *db,
					 const gchar *publicid,
					 const gchar *systemid )
{
	ScreemDTDDBPrivate *priv;
	ScreemDTD *dtd;
	gchar *data;
	ScreemDTDParse parse;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL ||
			      systemid != NULL, NULL );

	priv = db->priv;

	/* check we aren't creating a new dtd for one
	   we already have */
	dtd = g_hash_table_lookup( priv->loaded_dtds, systemid );

	if( dtd ) {
		g_warning( "DTD %s already loaded\n", systemid );
		return dtd;
	}

	dtd = NULL;
	data = screem_dtd_db_resolve_entity( db, publicid,
					     systemid );
	if( data ) {
		dtd = screem_dtd_new( publicid, systemid );

		g_hash_table_insert( priv->loaded_dtds,
				     g_strdup( systemid ),
				     dtd );
		/* parse data, adding to dtd */
		parse.dtd = dtd;
		parse.resolve_entity = screem_dtd_db_resolve_entity;
		parse.userdata = db;
		
		screem_dtd_parse( dtd, &parse, data );

		g_free( data );
	}

	return dtd;
}

gchar *screem_dtd_db_resolve_entity( void *db,
				  const gchar *publicid,
				  const gchar *systemid )
{
	ScreemDTDDB *dtddb;
	ScreemDTDDBPrivate *priv;
	const gchar *localuri;
	GString *file;
	gchar *ret;
	GnomeVFSURI *uri;
	gboolean local;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );

	dtddb = SCREEM_DTD_DB( db );

	priv = dtddb->priv;
	file = NULL;
	localuri = NULL;	
	if( publicid ) {
		localuri = g_hash_table_lookup( priv->dtd_types, 
						publicid );

		if( ! localuri ) {
			/* no local copy of publicid, or we haven't
			   loaded a catalog with it in */

			if( screem_dtd_db_load_internal( dtddb, publicid,
			   				systemid ) ) {
				localuri = g_hash_table_lookup( priv->dtd_types,
								publicid );
			}
		}
	} else if( systemid ) {
		/* there was no public identifier, there was
		   a system id, so just load that */
		uri = gnome_vfs_uri_new( systemid );
		if( uri ) {
			local = gnome_vfs_uri_is_local( uri );
			if( ( ! priv->offline ) || local ) {
				localuri = systemid;
			}
			gnome_vfs_uri_unref( uri );
		}
	}

	ret = NULL;

	if( localuri ) {
		file = load_file( localuri, NULL, NULL, NULL );
	}
	if( file ) {
		ret = file->str;
		g_string_free( file, FALSE );
	}

	return ret;
}

static gboolean screem_dtd_db_load_internal( ScreemDTDDB *db,
					const gchar *publicid,
					const gchar *systemid )
{
	ScreemDTDDBPrivate *priv;
	GnomeVFSURI *uri;
	GString *file;
	GString *catfile;
	gboolean local;
	gchar *filename;
	gchar *tmp;
	gchar *path;
	gchar *catalog;
	gchar *sysid;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), FALSE );

	if( ! systemid ) {
		return FALSE;
	}
	
	ret = FALSE;

	priv = db->priv;

	file = NULL;
	local = FALSE;
	uri = gnome_vfs_uri_new( systemid );
	if( uri ) {
		local = gnome_vfs_uri_is_local( uri );
		if( ( ! priv->offline ) || local ) {
			file = load_file( systemid, NULL, NULL, NULL );
		}
		gnome_vfs_uri_unref( uri );
	}
	
	if( file && publicid ) {
		filename = gnome_vfs_escape_slashes( systemid );
		tmp = screem_get_dot_dir();
		path = g_build_filename( tmp, "dtds", NULL );
		g_free( tmp );
		tmp = g_build_filename( path, filename, NULL );
		sysid = filename;

		filename = gnome_vfs_escape_host_and_path_string( tmp );
		g_free( tmp );
		
		if( save_file( filename, file->str,
				GNOME_VFS_PERM_USER_READ |
				GNOME_VFS_PERM_USER_WRITE,
				FALSE,
				NULL ) ) {
			catalog = g_build_filename( path, "catalog",
						    NULL );
			tmp = g_strdup_printf( "PUBLIC \"%s\" \"%s\"\n",
						publicid, sysid );

			catfile = load_file( catalog, NULL, NULL, NULL );
			if( ! catfile ) {
				catfile = g_string_new( NULL );
			}
			g_string_append( catfile, tmp );
			save_file( catalog, catfile->str,
				   GNOME_VFS_PERM_USER_READ |
				   GNOME_VFS_PERM_USER_WRITE, 
				   FALSE,
				   NULL );
			g_string_free( catfile, TRUE );

			g_free( tmp );
			g_free( catalog );
		}
		g_free( path );
		g_free( sysid );

		g_string_free( file, TRUE );
	
		g_hash_table_insert( priv->dtd_types,
				     g_strdup( publicid ),
				     filename );
		ret = TRUE;
	}

	return ret;
}


/* public stuff */

ScreemDTDDB *screem_dtd_db_new( void )
{
	ScreemDTDDB *dtd_db;

	dtd_db = g_object_new( SCREEM_TYPE_DTD_DB, NULL );

	return dtd_db;
}

GtkListStore *screem_dtd_db_get_store( ScreemDTDDB *db )
{
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );

	return db->priv->store;
}

ScreemDTD *screem_dtd_db_get_default_dtd( ScreemDTDDB *db )
{
	ScreemDTD *dtd;
	GConfClient *client;
	gchar *ddtd;

	client = gconf_client_get_default();

	ddtd = gconf_client_get_string( client, 
				"/apps/screem/editor/default_dtd",
				NULL );
	g_object_unref( client );

	if( ! ddtd ) {
		ddtd = g_strdup( "-//W3C//DTD HTML 4.01 Transitional//EN" );
	}
	
	dtd = screem_dtd_db_get_dtd( db, ddtd, NULL ); 

	g_free( ddtd );
	
	return dtd;
}

void screem_dtd_db_fill_list( ScreemDTDDB *db, GtkListStore *store )
{
	GList *list;
	GList *tmp;
	GtkTreeIter it;
	gchar *sysid;

	gtk_list_store_clear( store );
	g_object_set_data( G_OBJECT( store ), "list", NULL );
	g_hash_table_foreach( db->priv->dtd_types, 
			      (GHFunc)dtds_insert, store );
	list = g_object_get_data( G_OBJECT( store ), "list" );
	list = g_list_sort( list, (GCompareFunc)screem_dtd_db_sort_compare );
	for( tmp = list; tmp; tmp = tmp->next ) {
		sysid = screem_dtd_db_get_system_id( db,
				tmp->data, NULL );
		gtk_list_store_append( store, &it );
		gtk_list_store_set( store, &it,
				0, tmp->data,
				1, sysid,
				-1 );

		g_free( sysid );
	}
	g_list_free( list );
}

void screem_dtd_db_override_root( ScreemDTDDB *db,
				  const gchar *publicid,
				  const gchar *root )
{
	xmlNodePtr droot;
	xmlNodePtr override;
	
	g_return_if_fail( SCREEM_IS_DTD_DB( db ) );
	g_return_if_fail( publicid != NULL );

	override = screem_dtd_db_lookup_override( db, publicid );
	if( ! override ) {
		override = xmlNewNode( NULL, "dtd" );
		xmlSetProp( override, "public", publicid );
		droot = xmlDocGetRootElement( db->priv->overrides );
		xmlAddChild( droot, override );
	}
	if( root ) {
		xmlSetProp( override, "root", root );
	} else {
		xmlUnsetProp( override, "root" );
		if( ! xmlHasProp( override, "systemid" ) ) {
			xmlUnlinkNode( override );
			xmlFreeNode( override );
		}
	}

	xmlSaveFile( db->priv->overridepath, db->priv->overrides );
}

void screem_dtd_db_override_systemid( ScreemDTDDB *db,
				 const gchar *publicid,
				 const gchar *uri )
{
	xmlNodePtr droot;
	xmlNodePtr override;
	
	g_return_if_fail( SCREEM_IS_DTD_DB( db ) );
	g_return_if_fail( publicid != NULL );

	override = screem_dtd_db_lookup_override( db, publicid );
	if( ! override ) {
		override = xmlNewNode( NULL, "dtd" );
		xmlSetProp( override, "public", publicid );
		droot = xmlDocGetRootElement( db->priv->overrides );
		xmlAddChild( droot, override );
	}
	if( uri ) {
		xmlSetProp( override, "systemid", uri );
	} else {
		xmlUnsetProp( override, "systemid" );
		if( ! xmlHasProp( override, "root" ) ) {
			xmlUnlinkNode( override );
			xmlFreeNode( override );
		}
	}
	
	xmlSaveFile( db->priv->overridepath, db->priv->overrides );
}

gchar *screem_dtd_db_get_root( ScreemDTDDB *db, const gchar *publicid,
				gboolean *override )
{
	ScreemDTD *dtd;
	gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	dtd = screem_dtd_db_get_dtd( db, publicid, NULL ); 

	ret = screem_dtd_db_override_get_root( db, publicid );
	if( override ) {
		*override = ( ret != NULL );
	}
	if( dtd && ! ret ) {
		ret = (gchar*)screem_dtd_get_root_name( dtd );
		if( ret ) {
			ret = g_strdup( ret );
		}
	}

	return ret;
}

gchar* screem_dtd_db_get_system_id( ScreemDTDDB *db,
				const gchar *publicid,
				gboolean *override )
{
	GnomeVFSURI *uri;
	gchar *systemid;
	gchar *tmp;
	gchar *base;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	systemid = screem_dtd_db_override_get_system_id( db,
						publicid );
	if( override ) {
		*override = ( systemid != NULL );
	}
	if( ! systemid ) {
		systemid = g_hash_table_lookup( db->priv->dtd_types, 
						publicid ); 

		systemid = g_strdup( systemid );
		tmp = g_path_get_basename( systemid );
		base = gnome_vfs_unescape_string( tmp, "" );
		g_free( tmp );
		uri = gnome_vfs_uri_new( base );
		if( uri ) {
			if( ! gnome_vfs_uri_is_local( uri ) ) {
				g_free( systemid );
				systemid = gnome_vfs_unescape_string( base, "" );
			}
			gnome_vfs_uri_unref( uri );
		}
		g_free( base );
	}
	
	return systemid;
}

void screem_dtd_db_parse_doctype( ScreemDTDDB *db,
				 const gchar *doctype,
				 gchar **publicid,
				 gchar **systemid,
				 gchar **root )
{
	GString *tmp;
	gunichar c;
	gunichar term;
	gboolean istag;

	
	g_return_if_fail( SCREEM_IS_DTD_DB( db ) );
	g_return_if_fail( publicid != NULL );
	g_return_if_fail( systemid != NULL );

	istag = ! g_strncasecmp( "<!DOCTYPE ", doctype,
				 strlen( "<!DOCTYPE" ) );

	*publicid = NULL;
	*systemid = NULL;
	
	if( root ) {
		*root = NULL;
	}

	c = '\0';
	if( istag ) {
		doctype += strlen( "<!DOCTYPE " );
		doctype = g_utf8_skip_space( doctype );
		c = g_utf8_get_char( doctype );

		tmp = g_string_new( NULL );
	
		/* get root name */
		while( *doctype && g_unichar_isalnum( c ) ) {
			g_string_append_unichar( tmp, c );
			doctype = g_utf8_next_char( doctype );
			c = g_utf8_get_char( doctype );
		}
		*root = tmp->str;
		g_string_free( tmp, FALSE );
	
		/* incase we didn't hit whitespace, skip to it */
		while( ! g_unichar_isspace( c ) ) {
			doctype = g_utf8_next_char( doctype );
			c = g_utf8_get_char( doctype );
		}
	}
	doctype = g_utf8_skip_space( doctype );
	if( ( ! strncmp( "PUBLIC", doctype, strlen( "PUBLIC" ) ) ) ||
	    ( ! strncmp( "public", doctype, strlen( "public" ) ) ) ) {
		/* public id found */
		doctype += strlen( "PUBLIC" );

		doctype = g_utf8_skip_space( doctype );
	
		term = g_utf8_get_char( doctype );
		if( term == '"' || term == '\'' ) {
			tmp = g_string_new( NULL );
			do {
				c = g_utf8_get_char( doctype );
				if( c != term ) {
					g_string_append_unichar( tmp,
								c );
				}
				doctype = g_utf8_next_char( doctype );
			} while( *doctype && *doctype != term );
		
			/* doctype should be the char after the " */
			*publicid = tmp->str;
			g_string_free( tmp, FALSE );
		
			if( *doctype == term ) {
				doctype = g_utf8_next_char( doctype );
			}
			doctype = g_utf8_skip_space( doctype );
			term = g_utf8_get_char( doctype );
		}
		if( term == '"' || term == '\'' ) {
			tmp = g_string_new( NULL );
			do {
				c = g_utf8_get_char( doctype );
				if( c != term ) {
					g_string_append_unichar( tmp,
								c );
				}
				doctype = g_utf8_next_char( doctype );
			} while( *doctype && *doctype != term );
		
			/* doctype should be the char after the " */
			*systemid = tmp->str;
			g_string_free( tmp, FALSE );

			if( *doctype == term ) {
				doctype = g_utf8_next_char( doctype );
			}			
			doctype = g_utf8_skip_space( doctype );
			term = g_utf8_get_char( doctype );
		}	
	} else if( ( ! strncmp( "SYSTEM", doctype, strlen( "SYSTEM" ) ) ||
		   ( ! strncmp( "system", doctype, strlen( "system" ) ) ) ) ) {
		/* system id only */
	
		doctype += strlen( "SYSTEM" );

		doctype = g_utf8_skip_space( doctype );
	
		term = g_utf8_get_char( doctype );
		if( term == '"' || term == '\'' ) {
			tmp = g_string_new( NULL );
			do {
				c = g_utf8_get_char( doctype );
				if( c != term ) {
					g_string_append_unichar( tmp,
								c );
				}
				doctype = g_utf8_next_char( doctype );
			} while( *doctype && *doctype != term );
		
			/* doctype should be the char after the " */
			*systemid = tmp->str;
			g_string_free( tmp, FALSE );
		}
		
	}
}

ScreemDTD *screem_dtd_db_get_dtd_from_doctype( ScreemDTDDB *db,
						const gchar *doctype )
{
	ScreemDTD *dtd;
	gchar *publicid;
	gchar *systemid;
	gchar *root;

	screem_dtd_db_parse_doctype( db, doctype, 
				     &publicid, &systemid, &root );
	dtd = NULL;

	if( ( publicid || systemid ) && root ) {
		dtd = screem_dtd_db_get_dtd( db, publicid, systemid );
	}

	g_free( publicid );
	g_free( systemid );
	g_free( root );

	return dtd;
}

ScreemDTD *screem_dtd_db_get_dtd( ScreemDTDDB *db, 
				  const gchar *publicid,
				  const gchar *systemid )
{
	ScreemDTDDBPrivate *priv;
	ScreemDTD *dtd;
	const gchar *localid;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL ||
			      systemid != NULL, NULL );

	priv = db->priv;
	localid = NULL;
	if( publicid ) {
		localid = g_hash_table_lookup( priv->dtd_types,
						publicid );
	}
	
	dtd = NULL;
	if( localid ) {
		systemid = localid;
		dtd = g_hash_table_lookup( priv->loaded_dtds,
					   localid );
	}
	if( systemid && ! dtd ) {
		dtd = screem_dtd_db_new_dtd( db,
					     publicid, 
					     systemid );
	}
	
	return dtd;
}

