
// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------

#include "IIRScram.hh"
#include "IIR_Aggregate.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_RecordTypeDefinition.hh"
#include "IIR_ScalarTypeDefinition.hh"
#include "IIR_ElementDeclaration.hh"
#include "IIR_AssociationList.hh"
#include "IIR_TextLiteral.hh"
#include "IIR_IntegerLiteral32.hh"
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_DesignatorList.hh"
#include "error_func.hh"
#include "symbol_table.hh"
#include "published_file.hh"

#include "sstream-wrap.hh"


class IIR_AccessTypeDefinition;

IIRScram_Aggregate::~IIRScram_Aggregate(){
}

void 
IIRScram_Aggregate::_publish_vhdl(ostream &_vhdl_out) {
  ASSERT( _is_resolved() == TRUE );
  _vhdl_out << " ( ";
  element_association_list._publish_vhdl(_vhdl_out);
  _vhdl_out << " ) ";
}

ostream &
IIRScram_Aggregate::_print( ostream &os ){
  os << "(";
  os << element_association_list;
  os << ")";

  return os;
}

IIR_Boolean 
IIRScram_Aggregate::_is_resolved( ){
  IIR_Boolean retval = TRUE;

  if( element_association_list._is_resolved() == FALSE ){
    retval = FALSE;
  }

  if( my_rval == NULL ){
    retval = FALSE;
  }

  return retval;
}

IIR_Boolean 
IIRScram_Aggregate::_is_readable( ){
  IIR_Boolean retval = TRUE;
  IIR_AssociationElement *current_assoc;

  current_assoc = element_association_list.first();
  while( current_assoc != NULL ){
    if( current_assoc->_is_readable() == FALSE ){
      retval = FALSE;
    }
    current_assoc = element_association_list.successor( current_assoc );
  }

  return retval;
}


IIR_Boolean 
IIRScram_Aggregate::_is_writable( ){
  IIR_Boolean retval = TRUE;
  IIR_AssociationElement *current_assoc;

  current_assoc = element_association_list.first();
  while( current_assoc != NULL ){
    if( current_assoc->_is_writable() == FALSE ){
      retval = FALSE;
    }
    current_assoc = element_association_list.successor( current_assoc );
  }

  return retval;
}


IIR_Boolean 
IIRScram_Aggregate::_is_signal( ){
  IIR_Boolean retval = TRUE;
  IIR_AssociationElement *current_assoc;

  ASSERT( _is_resolved() == TRUE );

  current_assoc = element_association_list.first();
  while( current_assoc != NULL ){
    if( current_assoc->_is_signal() == FALSE ){
      retval = FALSE;
    }
    current_assoc = element_association_list.successor( current_assoc );
  }

  return retval;
}

IIR_Boolean 
IIRScram_Aggregate::_is_variable( ){
  IIR_Boolean retval = TRUE;
  IIR_AssociationElement *current_assoc;

  ASSERT( _is_resolved() == TRUE );

  current_assoc = element_association_list.first();
  while( current_assoc != NULL ){
    if( current_assoc->_is_variable() == FALSE ){
      retval = FALSE;
    }
    current_assoc = element_association_list.successor( current_assoc );
  }

  return retval;
}

IIR_Declaration*
IIRScram_Aggregate::_get_prefix_declaration(){
  return NULL;
}

void
IIRScram_Aggregate::_get_signal_source_info( set<IIR> *siginfo ){
  element_association_list._get_signal_source_info(siginfo);
}


set<IIR_TypeDefinition> *
IIRScram_Aggregate::_get_rval_set( IIR_Boolean (IIR::*)() ){
  // This implements lines 390 - 395, page 103 of the LRM
  set<IIR_TypeDefinition> *retval = new set<IIR_TypeDefinition>;

  // Add all in scope array types.
  set<IIR_ArrayTypeDefinition> *array_types = _get_symbol_table()->get_in_scope_array_types();
  IIR_TypeDefinition *current = array_types->get_element();
  while( current != NULL ){
    ASSERT( current->_is_array_type() == TRUE );
    retval->add( current );
    current = array_types->get_next_element();
  }
  
  // Add all in scope record types.
  set<IIR_RecordTypeDefinition> *record_types =  _get_symbol_table()->get_in_scope_record_types();
  current = record_types->get_element();
  while( current != NULL ){
    ASSERT( current->_is_record_type() == TRUE );
    retval->add( current );
    current = record_types->get_next_element();
  }

  return retval;
}

void 
IIRScram_Aggregate::_type_check( set<IIR_TypeDefinition> * ){
  // First set of type checks.
  IIR_Boolean have_seen_named = false;
  IIR_Boolean have_seen_others = false;
  IIR_AssociationElement *current = element_association_list.first();
  while( current != NULL ){
    if( current->_is_named() == TRUE ){
      have_seen_named = TRUE;
    }
    if( current->_is_positional() == TRUE && have_seen_named == TRUE ){
      ostringstream err;
      err << "Positional associations cannot follow named associations in an aggregate.";
      report_error( current, err.str() );
    }

    if( have_seen_others == TRUE ){
      // That means (due to the order of these if statements) that there was something
      // after the others!
      ostringstream err;
      err << "No association can follow an \"others\" association in an aggregate.";
      report_error( current, err.str() );      
    }

    if( current->_is_by_others() == TRUE ){
      have_seen_others = TRUE;
    }
    current = element_association_list.successor( current );
  }
}

void 
IIRScram_Aggregate::_rval_to_decl_process_named_part( IIR_TypeDefinition *my_type, 
						      IIR_AssociationElement *starting_with ){
  ASSERT( my_type->_is_iir_record_type_definition() == TRUE );
  IIR_AssociationElement *current_association = starting_with;
  while( current_association != NULL ){
    if( current_association->get_formal() != NULL ){
      IIR *current_formal = current_association->get_formal();
      
      switch( current_formal->get_kind() ){
      case IIR_SIMPLE_NAME:{
	set<IIR_Declaration> *element_decls = 
	  my_type->_find_declarations( (IIR_Name *)current_formal );
	if( element_decls == NULL ){
	  ostringstream err;
	  err << "Internal error - |" << *current_formal << "| was not found in record type |"
	      << *my_type << "| in IIRScram_Aggregate::_rval_to_decl_process_named_part.";
	  report_error( this, err.str() );
	}
	else{
	  ASSERT( element_decls->num_elements() == 1 );
	  IIR_Declaration *element_decl = element_decls->get_element();
	  delete element_decls;
	  
	  current_association->set_formal( element_decl );
	  IIR_TypeDefinition *actual_type = element_decl->get_subtype();
	  
	  ASSERT( current_association->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION );	
	  current_association->_set_actual( current_association->_get_actual()->_semantic_transform( actual_type ) );
	  current_association->_get_actual()->_type_check( actual_type );
	  current_association->_set_actual( current_association->_get_actual()->_rval_to_decl( actual_type ) );
	  
	}
	break;
      }
      case IIR_OTHERS_INITIALIZATION:{
	ostringstream err;
	err << "Internal error - others not supported in aggregate initialization of records, yet!";
	report_error( this, err.str() );
	break;
      }
      default:{
	if( current_formal->_is_resolved() == TRUE ){
	  // OK, formal is already resolved; we need to deal with the actual.
	  IIR_TypeDefinition *actual_type = current_formal->get_subtype();
	  ASSERT( current_association->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION );	
	  current_association->_set_actual( current_association->_get_actual()->_semantic_transform( actual_type ) );
	  current_association->_get_actual()->_type_check( actual_type );
	  current_association->_set_actual( current_association->_get_actual()->_rval_to_decl( actual_type ) );
	}
	else{
	  ostringstream err;
	  err << "Internal error - unexpected formal node type " << current_formal->get_kind_text()
	      << "(node == |" << *current_formal << "|)"
	      << " in IIRScram_Aggregate::_rval_to_decl_process_named_part.";
	  report_error( this, err.str() );
	}
      }
      }
    }
    else{
      ostringstream err;
      if( current_association->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION &&
	  current_association->_get_actual() != NULL &&
	  current_association->_get_actual()->get_kind() == IIR_OTHERS_INITIALIZATION ){
	err << "Others initialization isn't handled in this context yet.";
	report_error( this, err.str() );
      }
      else{
	err << "Internal error in IIRScram_Aggregate::_rval_to_decl_process_named_part - "
	    << "formal is null and actual is ";
	if( current_association->_get_actual() == NULL ){
	  err << "NULL";
	}
	else{
	  err << "an " << current_association->_get_actual()->get_kind_text()
	      << "(the node is |" << *current_association->_get_actual() << "|),";
	}
      }
    }

    current_association = element_association_list.successor( current_association );
  }
}

IIR *
IIRScram_Aggregate::_rval_to_decl( IIR_TypeDefinition *my_type ){
  ASSERT( my_type->_is_array_type() == TRUE || my_type->_is_record_type() );
  
  enum processing_mode_t { ARRAY, RECORD, ERROR };

  processing_mode_t processing_mode = ERROR;
  if( my_type->_is_array_type() == TRUE ){
    processing_mode = ARRAY;
  }
  else if( my_type->_is_record_type() == TRUE ){
    processing_mode = RECORD;
  }

  ASSERT( processing_mode != ERROR );
  IIR_TypeDefinition *actual_type = NULL;
  IIR_TypeDefinition *formal_type = NULL;

  int i;
  for( i = 0; i < element_association_list.num_elements(); i++ ){
    IIR_AssociationElement *current_association;
    current_association = element_association_list.get_nth_element( i );
    
    if( processing_mode == ARRAY ){
      formal_type = my_type->_get_index_subtype();
      if( current_association->_is_by_others() == TRUE ){
	actual_type = my_type;
      }
      else{
	actual_type = my_type->_get_element_subtype();
      }
    }
    else{
      // This code isn't handling named associations, at the moment!
      IIR_RecordTypeDefinition *my_record_type = (IIR_RecordTypeDefinition *)my_type;
      actual_type = my_record_type->_get_element_subtype( i );
      formal_type = my_record_type->_get_element_subtype( i );
    }
    
    ASSERT( actual_type != NULL );
    ASSERT( formal_type != NULL );

    if( current_association->get_formal() != NULL ){
      if( processing_mode == ARRAY ){
	IIR *old_formal = current_association->get_formal();

	IIR *new_formal = old_formal->_semantic_transform( formal_type );
	new_formal->_type_check( formal_type );
	new_formal = new_formal->_rval_to_decl( formal_type );

	current_association->set_formal( new_formal );
      }
      else{
	_rval_to_decl_process_named_part( my_type, current_association );
	break;
      }
    }
    
    if( current_association->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION ){
      current_association->_set_actual( current_association->_get_actual()->_semantic_transform( actual_type ) );
      current_association->_get_actual()->_type_check( actual_type );
      current_association->_set_actual( current_association->_get_actual()->_rval_to_decl( actual_type ) );
    }
  }
    
  my_rval = my_type;
  set_subtype( my_type );

  ASSERT( _is_resolved() == TRUE );

  return this;
}

IIR_TypeDefinition *
IIRScram_Aggregate::get_subtype(){
  return my_rval;
}

void
IIRScram_Aggregate::_publish_cc_lvalue( published_file &_cc_out ) {
  IIR_TypeDefinition *old_aggreate_type = _get_aggregate_iterator_subtype();
  int old_aggreate_count = _get_aggregate_iterator_counter();

  SCRAM_CC_REF( _cc_out, "IIRScram_Aggregate::_publish_cc" );
  
  ASSERT(get_subtype() != NULL);
  _cc_out << get_subtype()->_get_cc_type_name() << OS("(");
  _publish_cc_constructor_args( _cc_out );
  _cc_out.get_stream() << _get_number_of_element_associations();
  _cc_out << "," << NL();

  if( get_subtype()->_is_unconstrained_array_type() == FALSE ){
    _set_aggregate_iterator_subtype(get_subtype()->_get_index_subtype());
  } 
  else {
    ASSERT(get_subtype()->_get_index_subtype()->_get_base_type()->_is_scalar_type() == TRUE);
    ASSERT(((IIR_ScalarTypeDefinition *) get_subtype()->_get_index_subtype()->_get_base_type())->get_left() != NULL );
    _set_aggregate_iterator_subtype(get_subtype()->_get_index_subtype()->_get_base_type());
  }

  _set_aggregate_iterator_counter( 0 );
  
  element_association_list._publish_cc_state_object_init( _cc_out );
  _cc_out << CS(")");
  
  _set_aggregate_iterator_counter( old_aggreate_count );
  _set_aggregate_iterator_subtype(old_aggreate_type);
}

void
IIRScram_Aggregate::_publish_cc_initialization_value( published_file &_cc_out ) {
  ASSERT(get_subtype() != NULL);
  IIR_TypeDefinition *old_aggreate_type = _get_aggregate_iterator_subtype();
  int  old_aggreate_count = _get_aggregate_iterator_counter();

  SCRAM_CC_REF( _cc_out, "IIRScram_Aggregate::_publish_cc_initialization_value" );

  _cc_out << _get_number_of_element_associations();
  _cc_out << "," << NL();
  
  if (get_subtype()->_is_unconstrained_array_type() == FALSE) {
    _set_aggregate_iterator_subtype(get_subtype()->_get_index_subtype());
    _set_aggregate_iterator_counter(0);
  }
  else {
    ASSERT(get_subtype()->_get_index_subtype()->_get_base_type()->_is_scalar_type() == TRUE);
    ASSERT(((IIR_ScalarTypeDefinition *) get_subtype()->_get_index_subtype()->_get_base_type())->get_left() != NULL );
    _set_aggregate_iterator_subtype(get_subtype()->_get_index_subtype()->_get_base_type());
  }
  
  element_association_list._publish_cc_state_object_init( _cc_out );

  _set_aggregate_iterator_counter(old_aggreate_count);
  _set_aggregate_iterator_subtype(old_aggreate_type);
}


IIR_Int32
IIRScram_Aggregate::_get_number_of_element_associations() {
  return element_association_list._get_number_of_element_associations();
}

void
IIRScram_Aggregate::_get_list_of_input_signals( set<IIR> *new_list ){
  element_association_list._get_list_of_input_signals( new_list );
}

void
IIRScram_Aggregate::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  element_association_list._build_sensitivity_list(sensitivity_list);
}


void
IIRScram_Aggregate::_publish_cc_state_object_init( published_file &_cc_out ) {
  IIR_TypeDefinition *old_aggreate_type = _get_aggregate_iterator_subtype();
  int  old_aggreate_count = _get_aggregate_iterator_counter();

  SCRAM_CC_REF( _cc_out, "IIRScram_Aggregate::_publish_cc_state_object_init" );
  
  ASSERT(get_subtype() != NULL);
  _cc_out << " new " << get_subtype()->_get_cc_type_name() << OS("(");
  _publish_cc_constructor_args( _cc_out );
  _cc_out << _get_number_of_element_associations();
  _cc_out << "," << NL();

  _set_aggregate_iterator_subtype(get_subtype()->_get_index_subtype());
  _set_aggregate_iterator_counter(0);
  
  element_association_list._publish_cc_state_object_init( _cc_out );
  _cc_out << CS(")");

  _set_aggregate_iterator_counter(old_aggreate_count);
  _set_aggregate_iterator_subtype(old_aggreate_type);
}

void
IIRScram_Aggregate::_publish_cc_constructor_args( published_file &_cc_out ) {
  IIR_TypeDefinition* type;

  SCRAM_CC_REF( _cc_out, "IIRScram_Aggregate::_publish_cc_constructors_args" );
  
  ASSERT(get_subtype() != NULL);
  type = get_subtype();

  if ((type->_is_anonymous() == TRUE) && (type->_is_array_type() == TRUE) &&
      (type->_get_base_type() == NULL) && (type->_get_type_mark() != NULL)) {
    ASSERT ( type->_get_type_mark()->_is_resolved() == TRUE );
    ASSERT ( type->_get_type_mark()->_is_array_type() == TRUE );
    ASSERT ( type->_get_type_mark()->_is_unconstrained_array_type() == FALSE );
    
    type = type->_get_type_mark();
  }
  
  _cc_out << "ObjectBase::VARIABLE," << NL();
  if ((type->_is_array_type() == TRUE) || (type->_is_record_type() == TRUE)) {
    type->_publish_cc_object_type_info( _cc_out, FALSE, NULL, TRUE);
    _cc_out << "," << NL();

    type->_publish_cc_resolution_function_id( _cc_out );
    _cc_out << "," << NL();
  }
  
  if(type->_is_array_type() == TRUE)  {
    if (type->_is_unconstrained_array_type() == TRUE) {
      if(type->_is_single_dimensional_array_type() == TRUE) {
	_publish_cc_range( _cc_out );
	_cc_out << "," << NL();
      }
      else {
	_publish_cc_multi_dimensional_range( _cc_out );
      }
    } else {
      if (type->_is_anonymous() == TRUE) {
	if (type->is_element() == TRUE) {
	  type->_publish_cc_range( _cc_out );
	}
	else {
	  // The multi-dimensional case when is_element() return TRUE
	  // Here we don't need to publish ranges for all the dimensions
	  // Need to publish the range for this dimension alone. The inner
	  // dimensions will be be published when the inner aggregates are
	  // published. 
	  _publish_cc_range( _cc_out );
	}
	_cc_out << "," << NL();
      }
    }
  }
}

void
IIRScram_Aggregate::_publish_cc_element_list( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_Aggregate::_publish_cc_element_list" );
  
  element_association_list._publish_cc_lvalue( _cc_out );
}

void
IIRScram_Aggregate::_publish_cc_multi_dimensional_range( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_Aggregate::_publish_cc_multi_dimensional_range" );
  
  this->_publish_cc_range( _cc_out );

  IIR_TypeDefinition* element_subtype = get_subtype()->_get_element_subtype();

  if (element_subtype->is_element() == TRUE) {
    // Do not publish ranges for the multi-dimensional array case.
    return;
  }
  
  IIR* aggregate = element_association_list.first();
  while(aggregate->_is_aggregate_expression() == TRUE) {
    _cc_out << "," << NL();
    aggregate->_publish_cc_range( _cc_out );
    ASSERT(aggregate->get_kind() == IIR_AGGREGATE);
    aggregate = (((IIR_Aggregate*)aggregate)->element_association_list).first();
  }
  if((aggregate->_is_array_type() == TRUE) ||
     (aggregate->_is_record_type() == TRUE)) {
    _cc_out << "," << NL();
    aggregate->_publish_cc_range( _cc_out );
  }
}

// This method publishes a default range to the aggregate.  The default range
// is not specified in the LRM.  I used (1, to, n) -- SK.
void
IIRScram_Aggregate::_publish_cc_range( published_file &_cc_out ) {
  IIR_ScalarTypeDefinition *index_subtype = get_subtype()->_get_index_subtype();
  IIR_ScalarTypeDefinition *baseType;

  SCRAM_CC_REF( _cc_out, "IIRScram_Aggregate::_publish_cc_range" );
  
  ASSERT(index_subtype->_is_scalar_type() == TRUE);

  if (index_subtype->get_left() == NULL) {
    baseType = (IIR_ScalarTypeDefinition *) index_subtype->_get_base_type();
    if (baseType->get_left() == NULL) {
      baseType = (IIR_ScalarTypeDefinition *) index_subtype->_get_bottom_base_type();
    }
    
    ASSERT(baseType->get_left() != NULL);
    
    if (baseType->_is_ascending_range() == TRUE) {
      baseType->_publish_cc_universal_left( _cc_out );
      
      _cc_out << "," << NL()
	      << "ArrayInfo::to," << NL();
      
      baseType->_publish_cc_universal_left( _cc_out );
      _cc_out << " + " << (_get_number_of_element_associations() - 1);
    }
    else {
      baseType->_publish_cc_universal_left( _cc_out );
      _cc_out << " + " << (_get_number_of_element_associations() - 1);
      
      _cc_out << "," << NL()
	      << "ArrayInfo::downto," << NL();
      
      baseType->_publish_cc_universal_left( _cc_out );
    }
  }
  else {
    if (index_subtype->_is_ascending_range() == TRUE) {
      index_subtype->_publish_cc_universal_left( _cc_out );
      
      _cc_out << ","
	      << "ArrayInfo::to," << NL();
      
      index_subtype->_publish_cc_universal_right( _cc_out );
    }
    else {
      index_subtype->_publish_cc_universal_left( _cc_out );
      
      _cc_out << "," << NL()
	      << "ArrayInfo::downto,";
      
      index_subtype->_publish_cc_universal_right( _cc_out );
    }
  }
}


void
IIRScram_Aggregate::_add_decl_into_cgen_symbol_table() {
  element_association_list._add_decl_into_cgen_symbol_table();
}

IIR *
IIRScram_Aggregate::_clone(){
  IIR_Aggregate *retval = new IIR_Aggregate();
  IIR_Expression::_clone( retval );

  IIR_List::_listcopy( retval->element_association_list, element_association_list );

  return retval;
}

IIR_Boolean 
IIRScram_Aggregate::_is_globally_static_primary(){
  IIR_Boolean retval = TRUE;
  ASSERT( _is_resolved() == TRUE );

  if( get_subtype()->_is_array_type() == TRUE ){
    retval = element_association_list._is_globally_static_for_array();
  }
  else{
    ASSERT( get_subtype()->_is_record_type() == TRUE );
    retval = element_association_list._is_globally_static_for_record();
  }

  return retval;
}

visitor_return_type *IIRScram_Aggregate::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
 
  return visitor->visit_IIR_Aggregate(this, arg);
};
