#!/usr/bin/perl
#
# Revision = '$Id: MIMEStream.pm,v 1.54 2002/02/15 16:59:07 bre Exp $';
# Version = 'Anomy 0.0.0 : MIMEStream.pm';
#
##  Copyright (c) 2000-2002 Bjarni R. Einarsson. All rights reserved.
##  This program is free software; you can redistribute it
##  and/or modify it under the same terms as Perl itself.  
#
# This is essentially a state machine, which allows me to parse and 
# rewrite MIME messages a little bit at a time.  Messages may be as 
# complex as you like:  the parser knows how to decode (and encode) 
# the standard MIME encodings (base64, quoted-printable, 7bit and 
# 8bit) and parse (and create) both multipart/* and message/rfc822 
# parts properly.
#
# Not included are decoders for encrypted sub-parts.  No batteries 
# either.
#

# USAGE:
#
#	... set up parsers ...
#
#	$A = MIMEStream->New(*STDIN, *STDOUT, \%parsers );
#	$A->ParseHeader();
#	$A->ParseBody();
#
# BASIC IDEA:
#
#	The parse() function will scan the top-level message header, invoking 
#   the appropriate parser function.  The parser function is expected to
#   look about like this:
#
#   sub ParseSomething()
#   {
#		my ($reader) = shift;
#
#		# Read header from container part (or input file handle)
#		$reader->ReadHeader();
#
#		... manipulate header ...
#
#		$reader->WriteHeader();
#
#		# Parse part body
#		while (my $line = $reader->Read())
#		{
#			... manipulate line ...
#
#			# Print out Improved Line
#			$reader->Write()
#		}
#
#		... append stuff ...
#
#		# Flush buffers
#		$reader->Write(undef)
#   }
#
#	The Read() function will transparently decode the input using the 
#	supplied decoder functions, and will return "undef" when the end of 
#	the current section is reached.
#
#	The Write() function will transparently encode the output, using the
#	supplied encoder function.  Writing "undef" will flush the encoder's
#	buffers, which is necessary when Base64- and UU-encoding.
#
#	Useful parser functions are included near the bottom of this file.
#
#	By creating a new part object, using the Writer() constructor, a part's
#	encoding may be altered.  Search for "sub Writer" and read the comments.
#	See the "ParserForce8bit" function for an example.
#


##[ Package definition ]######################################################

package Anomy::MIMEStream;
use strict;
use MIME::Base64;
use MIME::QuotedPrint;
use Anomy::Log;

BEGIN {
	use Exporter	();
	use vars		qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
	
	$VERSION		= do { my @r = (q$Revision: 1.54 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r };
	@ISA	 		= qw(Exporter);
	@EXPORT			= qw(
							&CheckEmail
						);

	@EXPORT_OK		= qw( );
}
use vars @EXPORT_OK;


##[ Variables ]###############################################################

# Default parser values
my $default_parserhash = 
{
#	"text/html"             => \&ParserDiscard,
	"text/*"                => \&ParserForce8bit,
	"message/rfc822"        => \&ParserRFC822,
	"multipart/*"           => \&ParserMultipart,
	"multipart/signed"      => \&ParserCat,
	"multipart/encrypted"   => \&ParserCat,
	"DEFAULT"               => \&ParserCat,
};

# Default decoder values
my $default_decoderhash = 
{
	"DEFAULT"               => \&Decode8bit,
	"8bit"                  => \&Decode8bit,
	"binary"				=> \&Decode8bit,
	"7bit"                  => \&Decode8bit,
	"quoted-printable"      => \&DecodeQP,
	"base64"                => \&DecodeBase64,
	"uue"                   => \&DecodeUU,
	"forwarded"             => \&DecodeFwd,
};

# Default encoder values
my $default_encoderhash = 
{
	"DEFAULT"               => \&Encode8bit,
	"8bit"                  => \&Encode8bit,
	"binary"				=> \&Encode8bit,
	"7bit"                  => \&Encode7bit,
	"quoted-printable"      => \&EncodeQP,
	"base64"                => \&EncodeBase64,
	"uue"                   => \&EncodeUU,
	"forwarded"             => \&EncodeFwd,
};


##[ Functions ]###############################################################

# Constructor:
#
# Usage:	$parser = Anomy::MailStream->New(*INPUT,
#                                            *OUTPUT,
#                                            \%parsers,
#                                            $parent);
#
# The only required arguments are the file handles, there exist default
# values for the rest.
#
# If defined, $parsers should be a reference to a hash of interpretors
# for different content-types.
#
sub New
{
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my ($IN, $OUT, $parsers, $parent) = @_;
	my $boundary;
	my $log;
	my $p = $parent;
	my $common = undef;

	$IN = ($IN || $p->{"IN"} || \*STDIN);
	$OUT = ($OUT || $p->{"OUT"} || undef);

	$boundary = $p->{"mime"}->{"boundary"};
	$parsers  = ($parsers || $p->{"parsers"} || $default_parserhash);
	
	if ($parent)
	{
        $common = $parent->{"common"};
	    $log = $parent->{"log"}->sublog("Part", SLOG_TRACE, 
	                                    { pos => $parent->{"Read_Bytes"} });
	}
	else
	{
		$common = {
			"unset"     => 1,     # Common data is accumulated in a few 
			                      # stages, this is set to zero when we're
								  # all done.
			"root"      => undef,
		    "log"       => $log,
			"headers"   => undef,
			"errors-to" => undef,
			"reply-to"  => undef,
		};
	    $log = new Anomy::Log;
	}
	
	# Subparts normally don't mess with newlines...
	my $nl = 0;
	$nl = undef if ($parent);

	my $part = {
		"parent"   => $parent,
		"common"   => $common,
		"parsers"  => $parsers,
		"decoders" => $default_decoderhash,
		"encoders" => $default_encoderhash,
		"log"      => $log,

		# This buffer always contains decoded data, which is used by
        # Read() for line buffering within parts.
		"IOBuffer"    => "",
		"Read_Bytes"  => 0,
		"Wrote_Bytes" => 0,
		"Wrote_NL"    => 0,

		# This is our end-of-line character.
		#
		"EOL"         => "\012",
		"newline_in"  => $nl,
		"newline_out" => undef,

		# Files and "coparts"
		"INforce"  => !$parent,
		"IN"       => $IN,
		"OUTforce" => !$parent,
		"OUT"      => $OUT,
		"reader"   => undef,
		"writer"   => undef,

		# Misc. flags		
		"eop"	   => undef,
		"debug"    => 0,
		"uupart"   => 0,

		# This is the boundary that marks the end of this part.  Not
		# to be confused with the mime-boundary, which seperates the
		# sub-parts of multipart/* type part.
		"boundary" => $boundary,

		# These contain this part's header, formatted in a few ways
		# for programming conveniance.
		"headers"      => { },
		"rawheader"    => "Content-Type: text/plain\012Content-Transfer-Encoding: 8bit\012\012",
		"cookedheader" => undef,

		# Our assumptions about incoming non-MIME messages.
		"mime" => {
			"boundary"    => undef,
			"type"        => "text/plain",
			"encoding"    => "8bit",       # Exceeds MIME/RFC822 recommendations.
			"charset"     => "iso-8859-1", # Not quite MIME compliant...
			"disposition" => "inline",
		},
		"mime-headers" => {
			"content-type" => "type charset",
			"content-transfer-encoding" => "encoding",
		},
	};

	$common->{"root"} = $part if ($common->{"unset"});

	# Support multipart/digest semantics.
	$part->{"mime"}->{"type"} = "message/rfc822"
		if (($parent) && 
			(lc($parent->{"mime"}->{"type"}) eq "multipart/digest"));

	$part->{"decoder"} = $part->{"decoders"}->{"DEFAULT"};
	$part->{"encoder"} = $part->{"encoders"}->{"DEFAULT"};

	bless($part, $class);
	return $part;
}


##############################################################################
# Part reader methods

# Top level header parsing routine.
#
# Note:  This routine will stop processing the header after reading 
#        approximately 64k, and pass whatever is left to the body parser.
#        I've never seen a header that long, so I don't expect this to
#        be a serious limitation! :-)
#
sub ParseHeader
{
	my $reader = shift;
	$reader = ($reader->{"reader"} || $reader);

	my $header_log = $reader->{"log"}->sublog("ParseHeader", SLOG_TRACE);

	# Scan part header...
	my $line = $reader->Read();
	$line =~ s/\015$//; # Map CRLF => LF only.

    # NOTE:  Munging CRLF into LF only here is a bit nasty, since we might
	#        be processing an attachment which is DOS formatted and should
	# strictly speaking be left that way.  This might break things. FIXME!!

	# Gobble up Unix from line if present.
	if ($line =~ /^>?From /)
	{
		$reader->{"UNIX-FROM"} = $line;
		$line = $reader->Read();
		$line =~ s/\015$//; # Map CRLF => LF only.
	}

	my $header = '';
	my $headers = { };
	my $rawheader = $line;
	while (($line !~ /^\015?$/) && (length($rawheader) < 64000))
	{
		if ($line =~ /^\s+[^\s]+/)
		{
			$headers->{$header} .= $line;
		}
		elsif ($line =~ /^([^\s]+):\s(.*)$/)
		{
			$header = lc($1);
			$headers->{$header} .= $2 . "\012";
		}

		$line = $reader->Read();
		$line =~ s/\015$//; # Map CRLF => LF only.

		$rawheader .= $line;
	}
	# Export...
	$reader->{"headers"} = $headers;

    if ($reader->{"common"}->{"unset"})
	{
	    my $c = $reader->{"common"};
	    if (!defined $c->{"reply-to"})
		{
		    # Set reply-to information, if possible.
			foreach my $rt ($headers->{"reply-to"},
			                $headers->{"from"},
							$headers->{"resent-from"},
							$headers->{"return-path"},
							$headers->{"sender"})
		    {
			    if (CheckEmail($reader->DecodeHeader($rt)))
				{
				    $c->{"reply-to"} = $rt;
					chomp $c->{"reply-to"};
					last;
				}
			}
			$header_log->entry("reply-to", SLOG_DEBUG, 
			                   { value => $c->{"reply-to"} },
							   "Using %value% as reply-to address.");
	    }
	
	    if (!defined $reader->{"errors-to"})
		{
		    # Set errors-to information, if possible.
			foreach my $et ($headers->{"errors-to"}, 
			   	  	        $headers->{"return-path"},
							$headers->{"sender"},
							$headers->{"resent-from"},
							$headers->{"reply-to"},
							$headers->{"from"})
		    {
			    if (CheckEmail($reader->DecodeHeader($et)))
				{
				    $c->{"errors-to"} = $et;
					chomp $c->{"errors-to"};
				    last;
			    }
		    }
			$header_log->entry("errors-to", SLOG_DEBUG, 
		                       { value => $c->{"errors-to"} },
                               "Using %value% as errors address.");
	    }

		# Set back-references to root headers.
		$c->{"headers"} = $headers;

		# Got common headers, skip this stuff next time.
		$c->{"unset"} = 0;
	}

	# Get MIME info from a few headers.
	$reader->ParseContentHeader("; type=", "content-type");
	$reader->ParseContentHeader("; encoding=", "content-transfer-encoding");
	$reader->ParseContentHeader("; disposition=", "content-disposition");

	# Set content decoder and encoder
	$reader->{"mime"}->{"encoding"} = ($reader->{"mime"}->{"encoding"} || "8bit");
	if (my $e = $reader->{"decoders"}->{ lc($reader->{"mime"}->{"encoding"}) })
	{
		$reader->{"decoder"} = $e;
	}
	if (my $e = $reader->{"encoders"}->{ lc($reader->{"mime"}->{"encoding"}) })
	{
		$reader->{"encoder"} = $e;
	}

	$header_log->entry("mime", SLOG_DEBUG, $reader->{"mime"}, 
	                   "Got MIME info: %ATTRIBUTES%");

	# Get rid of Content-Length: and Lines: headers, because we just /know/ 
	# we are going to FUBAR them.
	$rawheader =~ s/^((Lines|Content-Length):.*?)$/X-FUBAR-$1/mgi;
	undef $headers->{"content-length"};
	undef $headers->{"lines"};

	# Export...
	$reader->{"rawheader"} = $rawheader;

	# Decode IOBuffer contents using info from header.
	if (my $decoder = $reader->{"decoder"})
	{
		$reader->{"IOBuffer"} = &$decoder($reader, $reader->{"IOBuffer"});
	}
}

# This routine parses a MIME Content-* header line, grabbing common tags 
# for latter processing.  This routine assumes that tags are unique even 
# across different header boundaries - that is, charset=xyz means the same
# thing whether it occurs in a Content-Type or a Content-Encoding line.
#
# The order of the attributes within each header line is recorded, so we 
# can rebuild them accurately later.
#
sub ParseContentHeader
{
	my $reader = shift;
	$reader = ($reader->{"reader"} || $reader);
	my $prefix = shift;
	my $header = shift;

	return unless ($reader->{"headers"}->{ $header });
	my $line = $reader->{"headers"}->{ $header };

	# Fix first attribute so we recognize illegal header values
	$line =~ s/^\s*([^\"\;][^\;]*?)\s*$/\"$1\";/;
	$line =~ s/^\s*([^\"\;][^\;]*);/\"$1\";/;

	# Add prefix to help parser
	$line =~ s/^\s*/$prefix/;
	my $tmp;

	$reader->{"mime-headers"}->{$header} = undef;
	while ($line =~ s/(?:\s*;\s+|\s+;\s*)([^\s=\*]+?)(\*\d+|)(\*=|=)(\"[^\"]*?\"|[^\;\"][^\s\;]*)//si)
	{
		my ($field, $seq, $eq, $data) = (lc($1), $2, $3, $4);
		my $charset = undef;
		my $comments = undef;

		$data =~ s/^\"(.*)\"$/$1/s;
		
		if ($field eq "boundary")
		{
		    # We also need the un-decoded boundary string, to handle
     		# boundaries which contain RFC822 comments.
		    $reader->{"mime"}->{"undecoded-boundary"} = $data;
		}

		# RFC2231 MIME Parameter Value and Encoded Word Extensions. 
		if ($eq eq "*=")
		{
		    # Record charset
		    $charset = $1 if ($data =~ s/^(.*?)\'\'//);

			# Decode data...
			$data =~ s/%([A-F0-9][A-F0-9])/ chr(hex($1)) /gie;
			
			$reader->{"mime"}->{lc($field)} = $data;
			$reader->{"mime"}->{"*".lc($field)} = $charset;
		}
		else
		{
	        $reader->{"mime"}->{lc($field)} = $reader->DecodeHeader($data, 1, \$comments);
		    $reader->{"mime"}->{"#".lc($field)} = $comments if ($comments);
		}
		$reader->{"mime-headers"}->{$header} .= $field . " ";
	}
}

# Top level body parsing routine.
#
sub ParseBody
{
	my $reader = shift;
	$reader = ($reader->{"reader"} || $reader);

	my $mimetype = lc($reader->{"mime"}->{"type"});
	my $parser;

	do
	{
		$parser = $reader->{"parsers"}->{$mimetype};
		if (!$parser)
		{
		    $mimetype =~ s/\/\*$//;		# blah/* -> blah
			$mimetype =~ s/[^\/]*$/*/;	# blah/blah -> blah/*
		}
	}
	while ((!$parser) && ($mimetype !~ /^\*$/));

	if (!$parser)
	{
		$parser = ($reader->{"parsers"}->{"DEFAULT"} || \&ParserCat);
		$reader->{"log"}->entry("body", SLOG_TRACE, 
		                        { pos => $reader->{"Read_Bytes"} }, 
                                "Parsing body as DEFAULT.");
	}
	else
	{
		$reader->{"log"}->entry("body", SLOG_TRACE, 
		                        { pos => $reader->{"Read_Bytes"} }, 
                                "Parsing body as $mimetype");
	}

	return &$parser($reader); 
	
	die "Ugh!\n";
}

# Switch to reading from an alternate input source.
#
sub ReadFrom
{
	my $reader = shift;	
	$reader = ($reader->{"reader"} || $reader);
	my $fh = shift;

	$reader->{"IN"} = $fh;
	$reader->{"INforce"} = 1;
	$reader->{"newline_in"} = undef;
	$reader->{"eop"} = undef;
	$reader->{"decoder"} = $reader->{"decoders"}->{"8bit"};
}

# This function will return the next few bytes of data in this part, 
# usually about 8k or less.  It will only read a single line at a time 
# from the parent part, so as not to interfere with boundary checks etc.
#
# "undef" will be returned if an attempt is made to read beyond the end 
# of the part.
#
sub RawRead
{
	my $reader = shift;	
	$reader = ($reader->{"reader"} || $reader);

	return undef if ($reader->{"eop"});
	
	my $line;
	if (!$reader->{"INforce"})
	{
		# Slurp at most 8k of input at a time.
		$line = $reader->{"parent"}->Read(8192);
	}
	else
	{
		# Slurp 8k of input at a time.
		$reader->{"IN"}->read($line, 8192);		

		if (defined $reader->{"newline_in"})
		{
		    # Figure out what our newline convention is...
		    my $nl = $reader->{"newline_in"};
			if ((!$nl) && ($line =~ /(\015?\012)/s))
			{
			    my $writer = $reader->{"writer"} || $reader;
		        $reader->{"newline_in"} = $nl = $1;
				$writer->{"newline_out"} = $nl 
				  unless ($writer->{"newline_out"});
			}

			$line =~ s/(\015?\012)/\012/gs; # if ($nl);
		}
 	}

	# Check for end-of-part.
	my $eop = $reader->{"boundary"};
	if ((defined $eop) && ($line =~ /^--\Q$eop\E(--)?\s*$/))
	{
		$reader->{"postponed"} .= $line;
		$reader->{"eop"} = 1;
		return undef;
	}

	return $line;
}


# This function will return the next available decoded line of text, using 
# (if necessary) multiple calls to $reader->RawRead().
#
# An optional parameter may specify the maximum allowable line length, the
# default is 8k.  A max length of 0 is "unlimited".
#
sub Read
{
	my $reader = shift;
	$reader = ($reader->{"reader"} || $reader);

	my $maxlen = shift;
	$maxlen = 8192 unless (defined $maxlen);

	my $decoder = ($reader->{"decoder"} || $default_decoderhash->{"DEFAULT"});
	my $line = undef;
	my $eol = $reader->{"EOL"};

	# Empty buffer first, if it contains anything.
	my ($l, $d) = split($eol, $reader->{"IOBuffer"}, 2);
	if ($d)
	{
		$reader->{"IOBuffer"} = $d;
		$reader->{"Read_Bytes"} += length($l)+length($eol);
		return $l . $eol;
	}

	# Read next line, fill up our buffer.
	while (
			((!$maxlen) || (length($reader->{"IOBuffer"}) < $maxlen)) &&
			# Here either the decoder or the reader can return undef to 
			# signify 'end of part'.  This is used by the UUE attachment
			# parser.
			($line = &$decoder($reader, $reader->RawRead()))
		  )
	{
		$reader->{"IOBuffer"} .= $line;
		my ($l, $d) = split($eol, $reader->{"IOBuffer"}, 2);
		if ($d)
		{
			$reader->{"IOBuffer"} = $d;
			$reader->{"Read_Bytes"} += length($l)+length($eol);
			return $l . $eol;
		}
	}

	# Max line length exceeded, or EOP reached.
	$line = $reader->{"IOBuffer"};
	$reader->{"IOBuffer"} = "";
	$reader->{"Read_Bytes"} += length($line);
	return $line;
}


# This function will put text back on the input buffer, allowing us to do 
# a bit of read-ahead without messing things up too much.
#
sub UnRead
{
	my $reader = shift;
	my $data = shift;
	
	$reader = ($reader->{"reader"} || $reader);
	$reader->{"IOBuffer"} = $data . $reader->{"IOBuffer"};
}


##############################################################################
# Part writer methods

# This will create a writer part usable for output, possibly "cloned" from
# a part which provides input.  This allows us to recode a part, by cloning
# a writer with different MIME attributes from the original.
#
# Usage: $writer = Anomy::MIMEStream->Writer($reader,
#                    {
#						"type" => "mimetype/subtype", 
#                       "encoding" => "encoding",
#                       ...
#                    },
#				   );
#
sub Writer
{
	my $proto = shift;
	my $reader = shift;
	my $attributes = shift;
	my $placement = shift;
	my $writer;

	if (defined $reader)
	{
		if (!$reader->{"writer"})
		{
		    # New writer
		    $writer = $proto->New();
			my $l = $writer->{"log"};

			# Clone the part.
			if ($reader->{"writer"})
			{
			   # Part already has a writer.  Deal.
			}
			else
			{
			    %{ $writer } = %{ $reader };
			}
			
			# We'll be modifying the headers, so...
			$writer->{"headers"} = { };
			%{ $writer->{"headers"} } = %{ $reader->{"headers"} };
			
			undef $writer->{"headers"}->{"content-type"};
			undef $writer->{"headers"}->{"content-disposition"};
			undef $writer->{"headers"}->{"content-transfer-encoding"};
			undef $writer->{"headers"}->{"content-length"};			

			# Chain logs
			$writer->{"log"} = $l;
			$reader->{"log"}->sublog("Writer", SLOG_TRACE, 
			                         { pos => $reader->{"Read_Bytes"} },
									 $writer->{"log"});

	   	    # The part is now schitzo!
			$reader->{"writer"} = $writer;  # WARNING:  Circular reference, GC
			$writer->{"reader"} = $reader;  #           won't work. Use Amputate()
		}
		else
	    {
		    $writer = $reader->{"writer"};
		}
	}
	else
	{
	    $writer = $proto->New();
	}

    $writer->KillRawMimeHeaders();
	$writer->{"rawheader"} =~ s/\s*$//s;
	$writer->{"rawheader"} .= "\012" if ($writer->{"rawheader"} ne "");
	if (!$writer->{"parent"})
	{
	    $writer->{"rawheader"} .= "MIME-Version: 1.0\012";
		$writer->{"headers"}->{"mime-version"} = "1.0\012";
	}

    my $log = $writer->{"log"};

	if (defined $attributes)
	{
		# Set MIME attributes
		$writer->{"mime"} = $attributes;
        $log->entry("mime", SLOG_TRACE, $attributes, 
		            "Set MIME info to: %ATTRIBUTES%");
	}
	if (defined $placement)
	{
	    $writer->{"mime-headers"} = $placement;
	}
	
	# Set up headers
	#
	my $tmp;

	# Content-Type: text/plain; charset="iso-8859-1"
	# Content-Type: multipart/mixed; boundary="abacabacab"
	my $ct;	
	$ct = ($writer->{"mime"}->{"type"} || "text/plain");
	$tmp = $writer->{"mime"}->{"boundary"};
	if (defined $tmp)
	{
		$ct .= "; boundary=\"". $tmp .'"';
	}
	else
	{
		$ct .= '; charset="'. $tmp .'"' 
			if ($tmp = $writer->{"mime"}->{"charset"});
	}
	$ct .= '; protocol="'. $tmp .'"' if ($tmp = $writer->{"mime"}->{"protocol"});
	$ct .= "\012";

	$writer->{"headers"}->{"content-type"} = $ct;
	$writer->{"rawheader"} .= "Content-Type: ". $ct;

	# Content-Transfer-Encoding: 8bit
	if ($writer->{"mime"}->{"encoding"})
	{
		my $cte;
		$cte = ($writer->{"mime"}->{"encoding"} || "8bit");
		$cte .= "\012";

		$writer->{"headers"}->{"content-transfer-encoding"} = $cte;
		$writer->{"rawheader"} .= "Content-Transfer-Encoding: ". $cte;
	}

	# Content-Disposition: attachment; filename="test.txt"
	if (($writer->{"mime"}->{"disposition"}) ||
	    ($writer->{"mime"}->{"filename"}))
	{
		my $cd;
		$cd = ($writer->{"mime"}->{"disposition"} || "attachment");
		$cd .= "; filename=\"". $writer->EncodeHeader($tmp) .'"'
			if ($tmp = $writer->{"mime"}->{"filename"});
		$cd .= "\012";

		$writer->{"headers"}->{"content-disposition"} = $cd;
		$writer->{"rawheader"} .= "Content-Disposition: ". $cd;
	}

	$writer->{"rawheader"} .= "\012";
	
	# Set up encoder
	$writer->{"encoder"} =
		$writer->{"encoders"}->{ lc($writer->{"mime"}->{"encoding"}) };

	return $writer;
}

# Detaches a part from it's clones, to let the garbage collector collect 
# either or both of the parts.
#
sub Amputate
{
	my $part = shift;
	
	if ($part->{"writer"})
	{
		$part->{"writer"}->{"reader"} = undef;
		$part->{"writer"}->{"writer"} = undef;
		$part->{"writer"} = undef;
	}
	
	if ($part->{"reader"})
	{
		$part->{"reader"}->{"reader"} = undef;
		$part->{"reader"}->{"writer"} = undef;
		$part->{"reader"} = undef;
	}
}


sub WriteHeader
{
	my $writer = shift;
	$writer = ($writer->{"writer"} || $writer);
	my $parent = $writer->{"parent"};
	$parent = ($parent->{"writer"} || $parent) if ($parent);
	
	if (defined $parent)
	{
		# Print leading boundary unless this is an inline UU attachment.
		if ($writer->{"uupart"})
		{
			# Fix rawheader, in case some parser messed it up.
			$writer->{"rawheader"} =~ s/\012+/\012/gs;
			if ($writer->{"rawheader"} !~ /^begin (\d\d\d\d?) (\S+.*\S+)\s*$/is)
			{
				$writer->{"rawheader"} = "begin " .
										 $writer->{"mime"}->{"permissions"} .
										 " " .
										 $writer->{"mime"}->{"filename"} .
										 "\012";
			}
		}
		else
		{
			my $boundary = $parent->{"mime"}->{"boundary"};
		
			$boundary = $parent->{"writer"}->{"mime"}->{"boundary"}
				if ($parent->{"writer"});

            if (defined $boundary)
			{
			    $parent->Write("\012") unless ($parent->{"Wrote_NL"});
			    $parent->Write("--$boundary\012")
			}
		}
		$parent->Write($writer->{"UNIX-FROM"});
		$parent->Write($writer->{"rawheader"});
	}
	else
	{
	    my $data = $writer->{"UNIX-FROM"}.$writer->{"rawheader"};
	    my $nl = $writer->{"newline_out"};
		$data =~ s/\012/$nl/gs if ($nl);
		$writer->{"OUT"}->print($data);
	}
}

# This encodes and writes data to our output, using calls to the parent's
# Write() if necessary.
#
sub Write
{
	my $part = shift;
	my $writer = ($part->{"writer"} || $part);

	my $data = shift;
	my $encoder = ($writer->{"encoder"} || $default_encoderhash->{"DEFAULT"});

	$data = &$encoder($writer, $data);
	return unless ($data);

	if (!$writer->{"OUTforce"})
	{
        $writer->{"Wrote_Bytes"} += length($data);
		$writer->{"parent"}->Write($data);
	}
	elsif (my $OUT = $writer->{"OUT"})
	{
	    my $nl = $writer->{"newline_out"};
		$data =~ s/\012/$nl/gs if ($nl);
        $writer->{"Wrote_Bytes"} += length($data);
	    $OUT->print($data);
	}
	$writer->{"Wrote_NL"} = $part->{"Wrote_NL"} = ($data =~ /\012$/s);
}
sub print 
{ 
    Write(@_);
}

# Close a previously opened sequence of parts in the output stream.
# This should only be called at the end of a multipart/* part.
#
sub Close
{
	my $writer = shift;
	$writer = ($writer->{"writer"} || $writer);
	
	my $boundary = $writer->{"mime"}->{"boundary"};
	if (defined $boundary)
	{
	    $writer->Write("\012") unless $writer->{"Wrote_NL"};
		$writer->Write("--". $boundary ."--\012");
	}
	$writer->{"postamble"} =~ s/\012$//s;
	$writer->Write($writer->{"postamble"});
	$writer->{"postamble"} = undef;
	$writer->Write(undef);
}


##############################################################################
# These are renderers - they can be either decoders or encoders.
#

sub RenderHtmlAsText
{
	my $part = shift;
	my $line = shift;
	my $state;
	
	# Get state.
	unless ($state = $part->{"RenderHtmlAsText"})
	{
		$state = $part->{"RenderHtmlAsText"} = {
			"pre" => 0,
			"indent" => "",
			"leftovers" => "",
		};
	}

	my $leftovers = \$state->{"leftovers"};
	my $indent = \$state->{"indent"};
	my $pre = \$state->{"pre"};
	my $out = "";

	# Flush leftovers if we're at the end of the data stream.
	if (!$line)
	{
		$line = $$leftovers;
		$$leftovers = undef;
		return $line;
	}
	
	$line = $$leftovers . $line;
	$$leftovers = undef;

	if ($line =~ /^(.*)<PRE.*?>(.*)$/)
	{
		my ($a,$b) = ($1,$2);
		$out .= StripHTML($a, $pre, $indent, $leftovers);
		$$pre = 1;
		$line = $$leftovers . $b;
	}
	if ($line =~ /^(.*)<\/PRE.*?>(.*)$/)
	{
		my ($a,$b) = ($1,$2);
		$out .= StripHTML($a, $pre, $indent, $leftovers) ."\012". $$indent;
		$$pre = 0;
		$line = $$leftovers . $b;
	}

	return $out . StripHTML($line, $pre, $indent, $leftovers);
}

# Helper function for RenderHtmlAsText
#
sub StripHTML
{
	my $line = shift;
	my $pre = shift;
	my $indent = shift;
	my $leftovers = shift;

	$line =~ s/\s+/ /sg unless ($$pre);
	
	while ($line =~ s/\s*<[UO]L.*?>\s*/\012/si)
	{
		$$indent .= "\t";
	}
	while ($line =~ s/\s*<\/[OU]L.*?>\s*/\012/si)
	{
		$$indent =~ s/\t$//;
	}
	
	$line =~ s/&nbsp;/ /sgi;
	$line =~ s/\s*<HR.*?>\s*/\012------------------------------------------------------------\012$$indent/sgi;
	$line =~ s/\s*<(P|H\d).*?>\s*/\012\012$$indent/sgi;
	$line =~ s/<A\s+[^>]*HREF=\"([^\"#][^\"]+)\".*?>(.*?)<\/A>/$2 &lt;$1&gt;/sgi;
	$line =~ s/\s*<LI>\s*/\012$$indent * /sgi;
	$line =~ s/\s*<\/?((B|T)R|HEAD).*?>\s*/\012$$indent/sgi;
	$line =~ s/\s*<\/TD.*?>\s*/\t/sgi;
	$line =~ s/<.+?>//sg;
#	$line =~ s/([^\s][^\012]{40,60}.*?)\s+/$1\012$indent/sg unless ($pre);

	if ($line =~ s/(<.+)$//si)
	{
		$$leftovers = $1;
		if (length($$leftovers) > 2048) 
		{
			$$leftovers .= "><";
		}
		$line .= " ";
	}

	$line =~ s/&lt;/</sgi;
	$line =~ s/&gt;/>/sgi;
	return $line;
}


##############################################################################
# Useful decoders for common data encodings

sub Decode8bit
{
	my $reader = shift;
	my $line = shift;

	return $line;
}

sub DecodeQP
{
	my $reader = shift;
	my $line = shift;

	return decode_qp($line);
}

sub DecodeBase64
{
	my $reader = shift;
	my $line = shift;

	# This hacks the decoder to handle mangled Base64 text properly, by
	# properly ignoring white space etc.  Note that this will lose the 
	# last 1-3 bytes of data if it isn't properly padded.  We also record
	# the encoded line-length, so we can re-encode stuff using the same 
	# length.
	#
	if (!$reader->{"DecodeBase64llen"}) 
	{
		$line =~ s/[^A-Za-z0-9\/+\012=]+//gs;
		
		my $nlpos = int((3*(index($line, "\012") + 1)) / 4);
		$line =~ s/\012//gs;

		my $llen = int((3*length($line)) / 4);
		my $t = $llen;
		$t = $nlpos if (($nlpos < $llen) && ($nlpos > 0));

		$reader->{"DecodeBase64llen"} = $t;
	}
	else
	{
		$line =~ s/[^A-Za-z0-9\/+=]+//gs;
	}
	$line = $reader->{"DecodeBase64"} . $line;
	$line =~ s/^((?:....)*)(.*?)$/$1/s;
	$reader->{"DecodeBase64"} = $2;

	return decode_base64($line);
}

sub DecodeUU
{
	my $reader = shift;
	my $line = shift;

	if ($line =~ /^end/i)
	{
		$reader->{"postponed"} .= $line;
		$reader->{"eop"} = 1;
		return undef;
	}

	# Sanitiy check - is this really a uuencoded part?
	if ($line =~ /^(.)(.*)\s*$/)
	{
		my $bytes = (ord($1) - ord(' ')) % 64;
		my $len = length($2);

		use integer;
		if ( (($bytes + 2) / 3) != ($len / 4) )
		{
			$reader->{"postponed"} .= $line;
			$reader->{"eop"} = 1;
			return undef;
		}
	}

	return unpack("u", $line);
}

# This will decode a forwarded message so it (probably) looks like the
# original.
#
sub DecodeFwd
{
	my $writer = shift;
	my $line = shift;

	if ($line =~ /^---+\s+End.*?forward/i)
	{
		$writer->{"postponed"} .= $line;
		$writer->{"eop"} = 1;
		return undef;
	}

	$line =~ s/^>(From\s+)/$1/gsm;
	$line =~ s/^- (--)/$1/gsm;

	return $line;
}

# This routine will decode any header-encoded substrings in a string.
# It's generally a good idea not to use this on a header until after 
# you have picked the header apart, since the encoded string might
# contain stuff that would confuse the parser.
#
sub DecodeHeader
{
	my $writer = shift;
	my $string = shift;
	my $comments = shift;
	my $t;
	my $commentref = shift || \$t;

    return "" unless $string;

	# Remove header-continuation characters.  EXPIRAMENTAL!
	$string =~ s/(\?=)\s*\012\s+(=\?)/$1$2/gs;
	$string =~ s/\s*\012(\s+)/$1/gs;

	if ($comments)
	{
	    # Remove RFC822 comments and return them seperately
	    $string = " ". $string;
	    $$commentref = "";
	    while ($string =~ s/([^\\])\((.*[^\\])\)/$1/)
	    {
	        $$commentref .= $1;
	    }
	    $string =~ s/^ //;
	}

	# Decode MIME stuff
	$string =~ s/=\?([^\?]+)\?[Bb]\?([^\?]+)\?=/ $t=$2; $_=decode_base64($t) /ge;
	$string =~ s/=\?([^\?]+)\?[Qq]\?([^\?]+)\?=/ $t=$2; $t=~s|_|=20|g; $_=decode_qp($t) /ge;
 
	return $string;
}


##############################################################################
# Useful encoders for common data encodings
#
# These all either return the encoded data or <undef> if the supplied
# data wasn't sufficient to complete encoding.  If too much data is supplied,
# remainders will be saved for later.  If the data is <undef>, then the
# remainder in the buffer will be padded, encoded and returned.

sub Encode8bit
{
	my $writer = shift;
	my $line = shift;

	return $line;
}

sub Encode7bit
{
	my $writer = shift;
	my $line = shift;

	# This performs some "nice" iso-8859-1 -> US-ASCII munging.
	# Probably not be a good idea (shouldn't this be in the "charset" mapping
	# code I haven't written?) ... but hopefully it'll never even be used, so 
	# that's OK.
	#
	$line =~ tr//aAoOynNcCaeiouyodAEIOUY0D/;
	$line =~ s//th/g;
	$line =~ s//Th/g;
	$line =~ s//ae/g;
	$line =~ s//Ae/g;
	# Give up...
	$line =~ s/[\x80-\xFF]/\?/g;

	return $line;
}

sub EncodeQP
{
	my $writer = shift;
	my $line = shift;

	return encode_qp($line);
}

sub EncodeBase64
{
	my $writer = shift;
	my $line = shift;

	# Flush buffer on undef
	if (!defined $line)
	{
		return undef unless ($writer->{"EncodeBase64"} ne undef);
		$line = $writer->{"EncodeBase64"};
		undef $writer->{"EncodeBase64"};
		return encode_base64($line)."\012";
	}

	# Get old stuff from buffer.
	$line = $writer->{"EncodeBase64"} . $line;

	# Chop data up, as recommended by MIME::Base64 pod or using the
	# line-size of the original message if possible.
	my $out;
	my $llen = $writer->{"DecodeBase64llen"} || 57;
	while ($line =~ s/^(.{$llen,$llen})//s)
	{
		my $chunk = $1;
		$out .= encode_base64($chunk);
	}
	
	# Save remainder.
	$writer->{"EncodeBase64"} = $line;
	
	# Return encoded data or undef.
	return $out;
}

sub EncodeUU
{
	my $writer = shift;
	my $line = shift;
	my $data;
	
	if (!defined $line)
	{
		return "" unless (defined $writer->{"EncodeUU"});

		$data = pack('u', $writer->{"EncodeUU"});
		undef $writer->{"EncodeUU"};
		return $data . "`\012";
	}
	else
	{
		$line = $writer->{"EncodeUU"} . $line;

		use integer;
		my $chunks = length($line) / 45;
		
		if ($chunks < 1)
		{
			# Need... more... data...
			$writer->{"EncodeUU"} = $line;
			return undef;
		}
		else
		{
			$data = substr($line, 0, $chunks * 45);
			$writer->{"EncodeUU"} = substr($line, $chunks * 45);
			return pack('u', $data);
		}
	}
	# Not reached
}

# This will encode a message for forwarding.
#
sub EncodeFwd
{
	my $writer = shift;
	my $line = shift;

	$line =~ s/^(From\s+)/>$1/gsm;
	$line =~ s/^(--)/- $1/gsm;
	
	return $line;
}

# This routine will encode a string so it is safe for 
# inclusion within a message header, that is, if it contains
# any non-7bit characters or other things which might get
# misunderstood, it will be Base64-encoded.
#
sub EncodeHeader
{
	my $writer = shift;
	my $string = shift;
	
	# FIXME: RFC compliance, check it!
	if ($string =~ /[^A-Za-z0-9,_!\?\.\\\/\@\$\+\=: -]/)
	{
	        my $prefix = '=?'.$writer->{"mime"}->{"charset"}.'?Q?';
		
		$string = encode_qp($string);
		$string =~ s/_/=5F/g;
		$string =~ s/ /_/g;

		chomp $string;
		$string =~ s/=\015?\012//gs;
		$string = $prefix.$string.'?=';
	}
	return $string;
}


##############################################################################
# Useful handlers for common MIME types.
#
# These all assume the library is being used by a stream editor, that is
# that our output will be a new MIME message.
#

# This parser deletes a part from the stream.
#
sub ParserDiscard
{
 	my $reader = shift;

    $reader->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserDiscard");
	$reader->{"decoder"} = \&Decode8bit;
	while (my $l = $reader->Read())
	{
		# discard
	}
}

# This parser doesn't modify the part at all.
#
sub ParserCat
{
 	my $part = shift;
    $part->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserCat");

	# Input -> Output !
	$part->WriteHeader();
	while (my $l = $part->Read())
	{
	   	$part->Write($l);
	}

	# Flush
	$part->Write(undef);
}

# This parser is for plain text you expect might contain inline uuencoded
# attachments (e.g. mail from Outlook).  This will also check for forwarded
# messages, and scan them as embedded message/rfc822 parts.
#
sub ParserUUText
{
 	my $part = shift;
    $part->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserUUText");
	
	# Input -> Output !
	$part->WriteHeader();
	while (my $l = $part->Read())
	{
		if ($l =~ /^begin (\d\d\d\d?) \S+/)
		{
			$part->UnRead($l);
			$l = $part->ParserUUAttachment();
		}
		if ($l =~ /^---+.*?Forward.*?---+$/i)
		{
		   	$part->Write($l);
			$l = $part->ParserForwardedMessage();
		}
	   	$part->Write($l);
	}

	# Flush
	$part->Write(undef);
}

# This parser recodes the part to 8-bits, no matter what it contains.
#
sub ParserForce8bit
{
 	my $reader = shift;
	my $writer = $reader->{"writer"};
    $reader->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserForce8bit");

	# Map current encoding to 8-bit encoding, cloning a new writer
	# if necessary.
	if ($writer)
	{
		$writer->{"mime"}->{"encoding"} = "8bit";
	}
	if ($reader->{"mime"}->{"encoding"} =~ /8bit/i)
	{
		$writer = $reader;
	}
	else
	{
		my $recode = { };
		%{ $recode } = %{ $reader->{"mime"} };
		$recode->{"encoding"} = "8bit";
		$writer = Anomy::MIMEStream->Writer($reader, $recode);
	}

	# Input -> Output !
	$writer->WriteHeader();
	while (my $l = $reader->Read())
	{
	   	$writer->Write($l);
	}

	# Need this, since the part contents might not end in a newline,
	# thus messing up following boundaries.
   	$writer->Write("\012");
	
	# Flush
	$writer->Write(undef);

	# Detach writer from reader (for garbage collection).
	$writer->Amputate();
}

# This part will recursively parse the attached message.
#
sub ParserRFC822
{
 	my $part = shift;
	$part->WriteHeader();
    $part->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserRFC822");

	my $subpart = Anomy::MIMEStream->New(undef, undef, undef, $part);
	
	# Check for newline conventions within the embedded message.
    $subpart->{"newline_in"} = 0;

	# We have to juggle boundary strings a little bit here, since we
	# share boundary strings with the encapsulated message.
	#
	$subpart->{"boundary"} = $part->{"boundary"};
	$part->{"mime"}->{"boundary"} = undef;

	$subpart->ParseHeader();
	$subpart->ParseBody();
	
	# And more juggling...
	$part->{"postponed"} = $subpart->{"postponed"} . $part->{"postponed"};
}

# This part will recursively parse each of the sub-parts in a 
# multipart-part.
#
sub ParserUnclosedMultipart
{
 	my $part = shift;
	my $writer = $part->{"writer"} || $part;
	my $reader = $part->{"reader"} || $part;
	$part->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserUnclosedMultipart");

    my $sep_re = undef;
	my $end_re = undef;

	$part->{"mime"}->{"boundary"} = $part->GuessBoundary() 
	  unless (defined ($part->{"mime"}->{"boundary"}));

    if (defined $part->{"mime"}->{"boundary"})
	{
	    my $b1 = $part->{"mime"}->{"boundary"} || "";
		my $b2 = $part->{"mime"}->{"undecoded-boundary"} || "";
	    $sep_re = "^--(\Q$b1\E|\Q$b2\E)\\s*\$";
	    $end_re = "^--(\Q$b1\E|\Q$b2\E)--\\s*\$";
	}

	$part->WriteHeader();

	my $postponed = undef;
	my $line = undef;
	my $postamble = 0;
	$part->{"postponed"} = "";
    while (($line = $postponed) || ($line = $part->Read()))
	{
		$postponed = undef;

		if ((defined $sep_re) && ($line =~ $sep_re)) 
		{
			# Update regular expressions & boundary info
			my $b = $1;
			$reader->{"mime"}->{"boundary"} = $b;
			$sep_re = "^--(\Q$b\E)\\s*\$";
			$end_re = "^--(\Q$b\E)--\\s*\$";

			my $subpart = Anomy::MIMEStream->New(undef, undef, undef, $part);
			$subpart->ParseHeader();
			$subpart->ParseBody();
			$postponed = $subpart->{"postponed"};
			$postponed =~ s/\012$//s;
		}
		elsif ((defined $end_re) && ($line =~ $end_re))
		{
			# Do nothing much - we may want to append to this part.
			$sep_re = undef;
			$end_re = undef;
			$postamble = 1;
		}
		elsif ($postamble)
		{
		    $part->{"postamble"} .= $line;
		}
		else
		{
		    $part->Write($line);
		}
	}
}
sub ParserMultipart
{
 	my $part = shift;
    $part->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserMultipart");
	$part->ParserUnclosedMultipart();
	$part->Close();
}

# UU encoded attachment parser.
#
# Essentially we treat a UU-encoded block of text as a seperate part
# with a rather unusual header.
#
sub ParserUUAttachment
{
	my $reader = shift;
	$reader = ($reader->{"reader"} || $reader);
	my $subpart;
    $reader->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserUUAttachment");

	# Read the "begin 664 blah.ext" line, use it as our raw header.
	#
	my $begin = $reader->Read();
	if ($begin =~ /^begin (\d*) (\S+.*\S+)\s*$/i)
	{
		$subpart = Anomy::MIMEStream->New(undef, undef, undef, $reader);
		$subpart->{"mime"}->{"permissions"} = $1;
		$subpart->{"mime"}->{"filename"} = $2;
	}
	else
	{
		return;
	}

	# Set up the header fields used by other parsers.
	#
	$subpart->{"rawheader"}	= $begin;
	$subpart->{"uupart"}	= 1;
	$subpart->{"decoder"}	= \&DecodeUU;
	$subpart->{"encoder"}	= \&EncodeUU;

	$subpart->{"mime"}->{"boundary"} = undef;
	$subpart->{"mime"}->{"encoding"} = "uue";
	$subpart->{"mime-headers"} = { };
	$subpart->{"mime"}->{"type"} = "INLINE/". $subpart->GuessMimeType();

	$subpart->ParseBody();
	$subpart->Write(undef);

	return $subpart->{"postponed"};
}

# Forwarded message parser.
#
# Attempt to recognize a forwarded message as an embedded message/rfc822 
# part with a weird encoding.
#
sub ParserForwardedMessage
{
 	my $part = shift;
	my $preamble;
    $part->{"log"}->entry("parser", SLOG_TRACE, undef, "ParserForwardedMessage");

	# Ignore leading empty lines.
	while (($preamble = $part->Read()) && ($preamble =~ /^\s*$/))
	{
		$part->Write($preamble);
	}
	return unless (defined $preamble);
	$part->UnRead($preamble);

	my $subpart = Anomy::MIMEStream->New(undef, undef, undef, $part);
	
	# We have to juggle encoders and decoders here to make the info 
	# in the forwarded message legible.  We wrap the encoders/decoders
	# for the current part, because the subpart may need it's own 
	# routines to handle it's own MIME encoding.
	my $reader = ($part->{"reader"} || $part);
	my $writer = ($part->{"writer"} || $part);
	my ($ode, $oen) = ($reader->{"decoder"}, $writer->{"encoder"});
	$reader->{"decoder"} = $reader->NestRenderers(\&DecodeFwd, $ode);
	$writer->{"encoder"} = $writer->NestRenderers($oen, \&EncodeFwd);

	$subpart->ParseHeader();
	$subpart->ParseBody();
	$subpart->Write(undef);

    # Fix encoders/decoders.
	$reader->{"decoder"} = $ode;
	$writer->{"encoder"} = $oen;

    return $subpart->{"postponed"};
}


##############################################################################
# Miscellania

# This function allows us to do cool things.  The following example will 
# make ParserCat automagically convert a QP-encoded HTML part to a QP-encoded 
# text part.  This will probably come in handy if we decide to to character
# set conversions or some such later on.
#
#   # Some stuff may already have been decoded:
#	$part->{"IOBuffer"} = $part->RendarHtmlAsText($part->{"IOBuffer"});
#   $part->{"decoder"} = $part->NestRenderers(\&RenderHtmlAsText, \&DecodeQP);
#
# Or, if we prefer to work with HTML within the parser:
#
#   $part->{"encoder"} = $part->NestRenderers(\&EncodeQP, \&RenderHtmlAsText);
#
# This is the first time I've used closures for anything useful!  Yay!
#
sub NestRenderers
{
	my $part = shift;
	my $outer = shift;
	my $inner = shift;

	return sub {
		my $p = shift; 
		return &$outer( $p, &$inner( $p, shift ) ); 
	};
}

# This will return a guess as to what the mime type is for the given
# part, determined by the file name.  This is a primitive 'magic'-like
# function.
#
# FIXME:  This function should be more easily configurable/extendable.
#
sub GuessMimeType
{
	my $part = shift;
	my $fn = $part->{"mime"}->{"filename"};
	
	$fn = $part->{"mime"}->{"name"} unless ($fn);

	return "text/plain" if ($fn =~ /\.(txt|[ch](\+\+|pp))$/i);
	return "text/html" if ($fn =~ /\.html?$/i);
	
	# Take a peek at our contents...
	#
	my $eol = $part->{"EOL"};
	$part->{"EOL"} = "etta er smilega undarlegur newline stafur!"; # FIXME
	my $readahead = $part->Read(512);
	$part->UnRead($readahead);
	$part->{"EOL"} = $eol;

	## Magic follows ##

	# HTML
	return "text/html" if ($readahead =~ /<(HTML|BODY)/i);

	# message/rfc822
	return "message/rfc822" if ($readahead =~ /^(Subject|From|To|Sender|Received):/im);

	# Shell scripts, C, C++ code
	return "text/plain" if ($readahead =~ /^#(!|include|define|ifn?def)/im);

	# Java code
	return "text/plain" if ($readahead =~ /^(import |class |public )/im);

	# Default.
	return "application/octet-stream";
}

# This will return a guess as to what the boundary is for a given message
# part.
#
sub GuessBoundary
{
	my $part = shift;
	my $fn = $part->{"mime"}->{"filename"};

	# Take a peek at our contents...
	#
	my $eol = $part->{"EOL"};
	$part->{"EOL"} = "etta er smilega undarlegur newline stafur!"; # FIXME
	my $readahead = $part->Read(512);
	$part->UnRead($readahead);
	$part->{"EOL"} = $eol;

    if ($readahead =~ /^--(\S+)\s*$/im)
	{
	    my $bound = $1;
		$bound =~ s/--$//;
		return $bound if ($bound ne "");
	}

	return undef;
}

# This will remove all MIME-related headers (except for the MIME-Version)
# from the part's "rawheader" variable, making way for new ones.
#
sub KillRawMimeHeaders
{
	my $part = shift;
	my $killed = "";
	
	my $rh = "\012". $part->{"rawheader"};
	while ($rh =~ s/(\012(MIME-Version:|Content-)[^\012]*(\012[ \t]+[^\012]*)*)//si)
	{
	    $killed .= $1;
	};
	while ($rh =~ s/(\012begin \d\d\d\d? \S+[^\012]*)//si)
	{
	    $killed .= $1;
	};
	$rh =~ s/^\012\s*//s;
	$part->{"rawheader"} = $rh;

	return $killed;
}

# This routine generates a nice randomish boundary string, which is 
# guaranteed never to repeat itself within the same process.
#
my $MakeBoundarySeq = 0;
sub MakeBoundary
{
	my $boundary = 'MIMEStream=_'. $MakeBoundarySeq++ .'+'. 
			(300476 * rand()) . (rand() * $$);  # hopefully, overkill. :)

	$boundary =~ tr/./_/;
	
	return $boundary;
}

# Returns the "important" part of an email address, or nothing if the
# address is invalid.
#
sub CheckEmail
{
	my ($address, $nameref) = @_;
	my $t;
	$nameref = \$t unless ($nameref);	

	# FIXME:  Not RFC822 compliant!!

	# Remove comments, interpret them as names.
	$address = " ". $address;
	$$nameref = "";
	while ($address =~ s/(\"(.*?)\"|[^\\]\((.*?[^\\])\))//)
	{
		$$nameref .= $2 . $3;
	}

	# Find <email> in address
	if ($address =~ s/([^\\])<(.*?[^\\])>/$1/)	
	{
		$$nameref .= $address;
		$address = $2;
	}
	
	# Remove excess whitespace
	$address  =~ s/^\s*(.*?)\s*$/$1/;
	$$nameref =~ s/^\s*(.*?)\s*$/$1/;

	# By now $address should only be one word...
	return 0 if ($address =~ /\s/);

# Use this to require a FQDN.
#
#	if ($address =~ /^[a-zA-Z0-9_\=\/\!\.\%-]+\@[a-zA-Z0-9\.-]+\.[a-zA-Z0-9-]{2,4}$/)
	if ($address =~ /^[a-zA-Z0-9_\=\/\!\.\%-]+\@[a-zA-Z0-9\.-]+$/)
	{
		# It's okay!
		return $address;
	}

	return 0;
}


##############################################################################
									   
END { };
1;
#EOF#
