#ifndef RUMBA_MANIFOLD_FILE_H
#define RUMBA_MANIFOLD_FILE_H

#include <rumba_config.h>
#include <rumba/baseManifold.h>
#include <rumba/log.h>
#include <rumba/factory.h>
#include <rumba/exemplar.h>
#include <string>
#include <map>
#include <iostream>
#include <list>


/**
  * \file manifoldFile.h
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */





namespace RUMBA { 
	
	class ManifoldFile;

	/** 
	  * Prototype (in the sense of "Design Patterns") . 
	  * If you don't know what a prototype is, you don't need to. 
	  * For those who are interested, the prototype is used by the 
	  * static construct method.
	  */
	extern ManifoldFile* ManifoldFileExemplar;
	extern ManifoldFile ManifoldFileInstance;
	
	/** cache subvolumes of size width*height*depth */
	extern const int CACHE_VOL; 

	/** cache subvolumes of size width*height*1*timepoints */	
	extern const int CACHE_TIME; 

	/** don't cache */
	extern const int CACHE_NONE;

 }


/**
  * The ManifoldFile class is an abstraction whose purpose is to conveniently
  * access data in files, and treat the data  as if it was in memory. In
  * particular, it is designed to facilitate convenient and transparent 
  * partial loading of data, to make it easy for developers to perform 
  * computations over very large data sets without having to store them in
  * memory. 
  *
  * In particular, the class facilitates implict partial loading (or caching).
  * There are three possible caching strategies: CACHE_NONE,  CACHE_TIME
  * and CACHE_VOL. The non-trivial caching strategies cache volumes. The
  * decision to take this approach is based on the reasoning that typical
  * fMRI volumes are reasonably small, for example 2^17 voxels for
  * a volume, which is 0.5mb-2mb depending on data size is fairly typical.
  * Experimenting, it seemed that this strategy was a good compromise between
  * memory consumption and disk access. The CACHE_TIME strategy works well
  * when the data is being viewed as isolated time series, which CACHE_VOLUME
  * works well when volumes are analyzed seperately.
  *
  * The ManifoldFile class also abstracts handling files of different formats.
  * Each file format has its own plug-in which is created by deriving from
  * ManifoldFile. The plugins autonomously decide whether they can handle
  * a file. The result is that client code does no need to concern itself
  * with specifics of different formats.
  *
  * 
  *
  */
namespace RUMBA
{
class IOHandlerBase;
}

namespace RUMBA
{

class ManifoldFile 
: public BaseManifold
{
protected: 

	IOHandlerBase* Data;

public:
	ManifoldFile(std::string);
	ManifoldFile(Exemplar, std::string);
	virtual ~ManifoldFile();

	static ManifoldFile* 
		construct(
				std::string filename, 
				std::ios_base::openmode mode = std::ios::in, 
				const BaseManifold* = 0);
	static ManifoldFile* 
		construct(
				std::string filename, 
				std::string datatype_request, 
				const RUMBA::intPoint& e);

	virtual BaseManifold* get(int x, int y, int z, int t, int dx, int dy, int dz, int dt) const;
	virtual BaseManifold* get(const intPoint& o, const intPoint& e) const;
	virtual void get(int x, int y, int z, int t, int dx, int dy, int dz, int dt, BaseManifold* res) const;
	virtual void get(const intPoint& o, const intPoint& e, BaseManifold* res) const;

	virtual void put ( BaseManifold*, int x, int y, int z, int t );
	virtual void put ( BaseManifold*, const intPoint&);


	virtual void loadHeader(const char* ){}
	virtual void saveHeader(const char* ){}

//	template<class TYPE> void loadData(Manifold<TYPE>);

	virtual int getDataType() const { return -1; }

	virtual void print(std::ostream& out) const;

	virtual double getElementDouble(int pos) const;
	virtual double getElementDouble(int x, int y, int z, int t) const;
	virtual double getElementDouble(const intPoint&) const;

	virtual int getElementInt(int pos) const;
	virtual int getElementInt(int x, int y, int z, int t) const;
	virtual int getElementInt(const intPoint&) const;

	virtual void setElementDouble(int pos, double val);
	virtual void setElementDouble(int x, int y, int z, int t, double val);
	virtual void setElementDouble(const intPoint&, double val);

	virtual void setElementInt(int pos, int val);
	virtual void setElementInt(int x, int y, int z, int t, int val);
	virtual void setElementInt(const intPoint&, int val);

	virtual void loadData(ManifoldFile*);
	virtual void saveData(ManifoldFile* f) const;

	// Forwarded to IOHandler
	void seekp (int pos);
	void seekg (int pos);

		

	void cGet (char*   dest, int nelts);
	void cGet (short*  dest, int nelts);
	void cGet (int*    dest, int nelts);
	void cGet (float*  dest, int nelts);
	void cGet (double* dest, int nelts);

	void cPut (char*   src, int nelts);
	void cPut (short*  src, int nelts);
	void cPut (int*    src, int nelts);
	void cPut (float*  src, int nelts);
	void cPut (double* src, int nelts);

	void setCacheStrategy(int x);
	int cacheStrategy() const
	{
		return CacheStrategy;
	}



protected:
	virtual ManifoldFile* make( std::string filename, std::ios_base::openmode mode = std::ios::in,  const BaseManifold* = 0);
	virtual ManifoldFile* make( std::string filename, std::string datatype_request, const intPoint&);
	virtual ManifoldFile* getNew();
	virtual bool isMine ( std::string filename );
	virtual void initFile ( std::ios_base::openmode mode );
	virtual std::string normalizeDataType ( std::string);

	bool cached (int p) const;
	bool cached (const intPoint& p) const;
	void setCachePoint(int p) const;
	void setCachePoint(const intPoint& p) const;
	void writeCache() const;


	static std::list<ManifoldFile*>& TypeList();

	std::map<std::string, std::string> types;

	std::string HeaderFile;
	std::string DataFile;
	// strings that identify the filetype
	std::string header_extension;
	std::string data_extension;
	std::string classname; // for debugging
	bool RandomAccess; // true if data is in a random access file

	mutable BaseManifold* Cache;
	int CacheStrategy;
	mutable int CachePoint;
};

} /* namespace RUMBA */

#endif
