#include <rumba/manifold.h>
#include <rumba/util.hpp>
#include <math.h>

#include <algorithm>

double RUMBA::maximum ( const BaseManifold* M  )
{
	double max = M->getElementDouble ( 0 );
	int i;
	for ( i = 0; i < M->size(); ++i )
		if ( M->getElementDouble(i) > max )
			max = M->getElementDouble(i);

	return max;
}

double RUMBA::minimum ( const BaseManifold* M )
{
	double min = M->getElementDouble ( 0 );
	int i;
	for ( i = 0; i < M->size(); ++i )
		if ( M->getElementDouble(i) < min )
			min = M->getElementDouble(i);

	return min;
}

double RUMBA::mean ( const BaseManifold* M )
{
	int i;
	double m = 0;
	for ( i = 0; i < M->size(); ++i )
		m += M->getElementDouble(i)  ;
	m /= M->size();
	return m;
}

double RUMBA::standard_deviation  ( const BaseManifold* M )
{
	int i;
	double sum = 0;
	double average = mean ( M );
	for ( i = 0; i < M->size(); ++i )
		sum += pow ( ( M->getElementDouble(i) - average ), 2 );

	return sqrt ( sum / M->size() );
}

RUMBA::Manifold<double> 
RUMBA::z_normalize ( const BaseManifold* M )
{
	Manifold<double> R  ( M->extent() );
	double m = mean(M);
	double sd = standard_deviation(M);

	for ( int i =0; i < M->size(); ++i )
		R[i] = ( M->getElementDouble ( i ) - m ) / sd;
	
	return R;
}

RUMBA::Manifold<double> RUMBA::cycle_vector(const std::vector<double>& v, bool transpose  )
{
	int n = v.size();
	RUMBA::Manifold<double> result ( intPoint ( n,1,1,n ));

	for ( int i = 0; i < n; ++i )
	{
		for ( int j = 0; j < n; ++ j )
		{
			if (!transpose)
				result.setElement ( i,0,0,j,  v[ (j-i+n) % n]);
			else
				result.setElement ( j,0,0,i, v[ (j-i+n) % n]);

		}	
	}

	return result;
}


std::vector<int> RUMBA::histogram( int n, const RUMBA::BaseManifold* M, 
		double* min, double* max )
{
	const double epsilon = 1e-8;
	int i;
	double lower = minimum(M);
	double upper = maximum(M);
	if (min)
		*min=lower;
	if (max)
		*max = upper;
	int slotNumber;
	std::vector<int> v ((size_t)(n+1),0);	// Initialize and allocate
	int size=M->size();	
	if (lower == upper)
	{
		v[0]=size; 
		return v;
	}
	for ( i = 0; i<size; ++i )
	{
		slotNumber = static_cast<int>  (
			( n * ( static_cast<double>((*M)[i]) - lower +epsilon)) 
			/ ( upper - lower )
		);
		if ( slotNumber >= 0 && slotNumber <= n  ) 
			v[slotNumber]++;	
		#ifdef DEBUG
		cerr << "Number: " << (*M)[i] << " in slot: " << slotNumber << endl;
		#endif
	}
	return v;
}

std::vector<int> 
RUMBA::histogram( int n, const RUMBA::BaseManifold* M, double lower, double upper )
{
	const double epsilon = 1e-8;
	int i;
	int slotNumber;
	std::vector<int> v ((size_t)(n+1),0);	// Initialize and allocate
	int size=M->size();	
	if (lower >= upper )
	{
		throw RUMBA::Exception ("histogram: lower bound can't exceed upper bound");
	}

	for ( i = 0; i<size; ++i )
	{
		slotNumber = static_cast<int>  (
				( n * ( M->getElementDouble(i) - lower +epsilon )) 
				/ ( upper - lower )
			);
		if ( slotNumber >= 0 && slotNumber <= n) 
			v[slotNumber]++;	
		#ifdef DEBUG
		cerr << "Number: " << (*M)[i] << " in slot: " << slotNumber << endl;
		#endif
	}
	return v;
}

void RUMBA::downsample ( 
		const RUMBA::intPoint& step, 
		const RUMBA::BaseManifold* in,
		RUMBA::BaseManifold* out)
{
	RUMBA::intPoint ex;

	for (int i = 0; i < 4; ++i )
	{
		if (!
			   ( out->extent()[i] * step[i] == in->extent()[i] )
		   )
		{
			throw RUMBA::Exception("Arguments out of range");
		}
	}

	ex.x() = in->width() / step.x();
	ex.y() = in->height() / step.y();
	ex.z() = in->depth() / step.z();
	ex.t() = in->timepoints() / step.t();

	intPoint count;
	count.x() = count.y() = count.z() = count.t() = 0;

	for ( count.t() = 0; count.t() < ex.t(); ++count.t() )
		for ( count.z() = 0; count.z() < ex.z(); ++ count.z() )
			for ( count.y() = 0; count.y() < ex.y(); ++count.y() )
				for ( count.x() = 0; count.x() < ex.x(); ++count.x() )
					out->setElementDouble ( count, in->getElementDouble 
							( count.x() * step.x(), count.y() * step.y(), count.z() * step.z(), count.t() * step.t() ) );

}






namespace RUMBA {

template void normalize<char> ( char low, char high, BaseManifold* M );
template void normalize<short> ( short low, short high, BaseManifold* M );
template void normalize<int> ( int low, int high, BaseManifold* M );
template void normalize<float> ( float low, float high, BaseManifold* M );
template void normalize<double> ( double low, double high, BaseManifold* M );

template char maximum<char> ( const Manifold<char>& M  );
template short maximum<short> ( const Manifold<short>& M  );
template int maximum<int> ( const Manifold<int>& M  );
template float maximum<float> ( const Manifold<float>& M  );
template double maximum<double> ( const Manifold<double>& M  );


template char minimum<char> ( const Manifold<char>& M );
template short minimum<short> ( const Manifold<short>& M );
template int minimum<int> ( const Manifold<int>& M );
template float minimum<float> ( const Manifold<float>& M );
template double minimum<double> ( const Manifold<double>& M );

template void normalize<char>( char low, char high, Manifold<char>& M );
template void normalize<short>( short low, short high, Manifold<short>& M );
template void normalize<int>( int low, int high, Manifold<int>& M );
template void normalize<float>( float low, float high, Manifold<float>& M );
template void normalize<double>( double low, double high, Manifold<double>& M );


}
