/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.resource;

import java.io.File;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import jp.sourceforge.mergedoc.pleiades.Pleiades;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.AspectMapping;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.PointCut;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.resource.AbstractTranslationDictionary;
import jp.sourceforge.mergedoc.pleiades.resource.Mnemonics;
import jp.sourceforge.mergedoc.pleiades.resource.Property;
import jp.sourceforge.mergedoc.pleiades.resource.PropertySet;
import jp.sourceforge.mergedoc.pleiades.resource.TranslationString;

/**
 * キャッシュ機構および実行時除外機構を持つ翻訳辞書クラスです。
 * Eclipse 実行時の動的翻訳に利用されるクラスです。
 * <p>
 * @author cypher256
 */
public class DynamicTranslationDictionary extends AbstractTranslationDictionary implements CacheFileNames {

	/** ロガー */
	private static final Logger log = Logger.getLogger(DynamicTranslationDictionary.class);

	/** 翻訳キャッシュ・プロパティー・ファイル */
	private static final File cacheFile = Pleiades.getResourceFile(TRANS_CACHE_PROP);

	/** 呼び出しトレースを検査する最大階層 */
	private static final int TRACE_MAX = 30;

	/** このクラスのシングルトン・インスタンス */
	private static final DynamicTranslationDictionary singleton = new DynamicTranslationDictionary();

	/**
	 * 翻訳辞書インスタンスを取得します。
	 * <p>
	 * @return 翻訳辞書インスタンス
	 */
	public static DynamicTranslationDictionary getInstance() {
		return singleton;
	}

	/**
	 * キャッシュ・プロパティー・セット・クラス。
	 * Map の containsValue メソッドを高速化するために拡張されています。
	 */
	@SuppressWarnings("serial")
	private static class CachePropertySet extends PropertySet {

		Set<String> values = new HashSet<String>();

		@Override
		public boolean containsValue(Object value) {
			return values.contains(value);
		}

		@Override
		public String put(String key, String value) {
			values.add(value);
			return super.put(key, value);
		}
	}

	// -------------------------------------------------------------------------

	/** 翻訳キャッシュ・プロパティー */
	private PropertySet cacheProp;

	/**
	 * 辞書ファイルをロードします。
	 */
	@Override
	protected boolean load() {

		cacheProp = new CachePropertySet();

		if (Pleiades.getPleiadesOption().isClean() || !cacheFile.exists()) {

			// -clean の場合、翻訳辞書をロード
			super.load();
			cacheProp.load(validateExists(TRANS_MULTIBYTES_KEY_PROP));
		} else {
			// -clean でない場合、翻訳辞書キャッシュをロード
			try {
				cacheProp.load(cacheFile);
			} catch (RuntimeException e) {

				// キャッシュ破損、自己復元
				log.warn("翻訳キャッシュ %s の破損を検出。復元中... - %s", cacheFile, e.toString());
				cacheFile.delete();
				cacheProp.clear();
				cacheProp.load(validateExists(TRANS_MULTIBYTES_KEY_PROP));
			}
		}
		return true;
	}

	/**
	 * シャットダウンします。
	 * 翻訳プロパティーはキャッシュとして永続化されます。
	 */
	public void shutdown() {

		isLoadedDefault = true;

		if (Pleiades.getPleiadesOption().isClean() || !cacheFile.exists()) {

			cacheProp.store(cacheFile, "翻訳キャッシュ・プロパティー");

		} else {

			PropertySet cachePropOld = new PropertySet(cacheFile);

			if (cacheProp.size() > cachePropOld.size()) {
				cacheProp.store(cacheFile, "翻訳キャッシュ・プロパティー");
				log.info("翻訳キャッシュ・プロパティーを更新しました。");
			} else {
				log.info("翻訳キャッシュ・プロパティーの更新はありません。");
			}
		}
	}

	/**
	 * 指定した英語リソース文字列から日本語リソースを探します。
	 * ニーモニックは日本用に変換されます。
	 * <p>
	 * @param enWithMnemonic 英語リソース文字列 (ニーモニック含む)
	 * @param jointPoint ジョイント・ポイント
	 * @return 日本語リソース文字列 (ニーモニック含む)。
	 */
	public String lookup(String enWithMnemonic, JointPoint jointPoint) {

		// 翻訳済みの場合は何もしない (リソース・バンドルおよび UI で呼び出されるため)
		if (enWithMnemonic.equals("") || enWithMnemonic.contains("・") || Mnemonics.hasJaMnemonic(enWithMnemonic)) {

			return enWithMnemonic;
		}

		// 英語ニーモニックを除去
		String en = Mnemonics.removeEnMnemonic(enWithMnemonic);

		// 翻訳プロパティーの value に合致 (翻訳済み) する場合は何もしない
		if (cacheProp.containsValue(en.trim())) {
			return enWithMnemonic;
		}

		// 翻訳不要な場合
		if (isNoTranslation(en, jointPoint)) {

			// Javadoc、JsDoc 生成ウィザードの Package（除外指定されている）
			// ニーモニック変換が必要 2009.02.05
			// -----
			// ただし、ニーモニック変換が不要な場合は除外 (例: Debugの構成...)。
			// 今のところハードコード。数が多いようであればプロパティー化。 2009.02.11
			if (en.equals("Debug")) {
				return enWithMnemonic;
			}
			return convertMnemonicEnToJa(enWithMnemonic, en, en);
		}

		// 翻訳プロパティーから日本語訳を取得
		String ja = lookupInternal(en, jointPoint);

		// ニーモニックを日本用に変換。この変換は訳語が見つからなかった場合も行う。
		ja = convertMnemonicEnToJa(enWithMnemonic, en, ja);

		return ja;
	}

	/**
	 * 指定した英語リソース文字列から日本語リソースを探します。
	 * ニーモニックは処理されません。
	 * <p>
	 * @param en 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 日本語リソース文字列。翻訳できない場合は en をそのまま返す。
	 */
	public String lookupIgnoreMnemonic(String en, JointPoint jointPoint) {

		// ほとんど呼び出されないため、高速化たのための翻訳済み判定はしない

		// 翻訳不要な場合は、そのまま返す
		if (isNoTranslation(en, jointPoint)) {
			return en;
		}
		// 翻訳プロパティーから日本語訳を取得
		String ja = lookupInternal(en, jointPoint);

		return ja;
	}

	/**
	 * 指定した英語ヘルプ・リソース文字列から日本語ヘルプ・リソースを探します。
	 * <p>
	 * @param en 英語ヘルプ・リソース文字列
	 * @param jointPoint ジョイント・ポイント (現在未使用)
	 * @return 日本語ヘルプ・リソース文字列。翻訳できない場合は en をそのまま返す。
	 */
	public String lookupHelp(String en, JointPoint jointPoint) {
		return getValueForHelp(en);
	}

	/**
	 * 日本語訳を取得します。
	 * <p>
	 * @param en 英語リソース文字列（ニーモニック無し）
	 * @param jointPoint ジョイント・ポイント
	 * @return 日本語リソース文字列（ニーモニック無し）。翻訳できない場合は en をそのまま返す。
	 */
	protected String lookupInternal(String en, JointPoint jointPoint) {

		// 翻訳キャッシュ・プロパティーから取得
		String ja = cacheProp.get(en);
		if (ja != null) {
			if (ja.equals("")) {
				ja = en;
			}
		} else {

			// 正規表現翻訳プロパティーから取得
			ja = super.getValueByRegex(new TranslationString(en));
			if (ja == null) {

				// 翻訳済み (マルチ・バイトが含まれる) の場合
				if (en.length() != en.getBytes().length) {
					ja = en;

					// デフォルトの翻訳プロパティーから取得
				} else {
					if (super.load()) {
						log.debug("キャッシュにないため翻訳プロパティーをロード: " + Property.escapeKey(en));
					}
					ja = getValue(en);
				}
				// キャッシュする
				cacheProp.put(en, en.equals(ja) ? "" : ja);
			}
		}

		// DEBUG
		// if (en.endsWith("test")) {
		// String s = System.currentTimeMillis() + "「" + en + "」→「" + ja + "」";
		// log.debug(new Exception(s), "デバッグ翻訳追跡スタックトレース jointPoint:" +
		// jointPoint);
		// return s;
		// }

		return ja;
	}

	/**
	 * 正規表現辞書から日本語訳を取得します。
	 * super.getValue から呼び出されますが、this.getValue で実行済みのため、
	 * 空オーバーライドしています。
	 * <p>
	 * @param enTs 英語翻訳文字列
	 * @return 日本語リソース文字列（ニーモニック無し）。翻訳できない場合は null。
	 */
	@Override
	protected String getValueByRegex(TranslationString enTs) {
		return null;
	}

	/**
	 * ニーモニックを英語用から日本用に変換します。<br>
	 * 例）&x → (&X)
	 * <p>
	 * @param enWithMnemonic	英語リソース文字列  （ニーモニック有り）
	 * @param en				英語リソース文字列  （ニーモニック無し）
	 * @param ja				日本語リソース文字列（ニーモニック無し）
	 * @return					日本語リソース文字列（ニーモニック有り）
	 */
	protected String convertMnemonicEnToJa(String enWithMnemonic, String en, String ja) {

		if (Pleiades.getPleiadesOption().isNoMnemonic()) {
			return ja;
		}
		String jaWithMnemonic = Mnemonics.convertMnemonicEnToJa(enWithMnemonic, en, ja);
		return jaWithMnemonic;
	}

	/**
	 * 翻訳が不要か判定します。
	 * 呼び出し元のパッケージやクラスを元に判定されます。
	 * <p>
	 * @param en 英語リソース文字列（ニーモニック無し）
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳が不要な場合は true
	 */
	protected boolean isNoTranslation(String en, JointPoint jointPoint) {

		StackTraceElement[] stes = null;

		// properties [%EXCULUDE%] 呼び出し元による除外（訳語とパッケージ単位）
		// xml 定義のアドバイスに ?{JOINT_POINT} 指定は不要。
		Set<String> noTransPathEntries = ExcludePackageProperties.getInstance().getPathEntries(en);

		if (noTransPathEntries != null) {
			stes = Thread.currentThread().getStackTrace();
			for (int i = 0; i < TRACE_MAX && i < stes.length; i++) {

				StackTraceElement ste = stes[i];
				String className = ste.getClassName();

				for (String noTransPath : noTransPathEntries) {
					if (className.startsWith(noTransPath)) {
						// パフォーマンス優先のためログはコメントアウト
						// log.debug("translation-exclude.properties 翻訳除外 [" +
						// en + "] " + ste);
						return true;
					}
				}
			}
		}

		// xml [*clludeTrace] 呼び出し元トレースによる除外と限定（再帰的に遡る）。
		// xml 定義のアドバイスに ?{JOINT_POINT} 指定が必要。
		if (jointPoint != null) {

			PointCut pointCut = AspectMapping.getInstance().getPointCut(jointPoint);
			if (pointCut != null) {

				List<JointPoint> excludeTrace = pointCut.getExcludeTrace();
				List<JointPoint> includeTrace = pointCut.getIncludeTrace();

				// 除外指定に一致する場合は翻訳不要
				if (excludeTrace.size() > 0) {
					if (stes == null) {
						stes = Thread.currentThread().getStackTrace();
					}
					if (containsTrace(excludeTrace, stes)) {
						// パフォーマンス優先のためログはコメントアウト
						// log.debug("excludeTrace 翻訳除外 [" + en + "] " +
						// jointPoint.getClassName() + "#" +
						// jointPoint.getMethodName());
						return true;
					}
				}

				// 限定指定に一致しない場合は翻訳不要
				if (includeTrace.size() > 0) {
					if (stes == null) {
						stes = Thread.currentThread().getStackTrace();
					}
					if (!containsTrace(includeTrace, stes)) {
						// パフォーマンス優先のためログはコメントアウト
						// log.debug("includeTrace 翻訳限定 [" + en + "] " +
						// jointPoint.getClassName() + "#" +
						// jointPoint.getMethodName());
						return true;
					}
				}
			}
		}

		return false;
	}

	/**
	 * スタックトレースにトレース・リストのジョイント・ポイントが含まれるか判定します。
	 * <p>
	 * @param trace トレース・リスト
	 * @param stes スタックトレース
	 * @return 含まれる場合は true
	 */
	protected boolean containsTrace(List<JointPoint> trace, StackTraceElement[] stes) {

		for (JointPoint jp : trace) {

			for (int i = 0; i < TRACE_MAX && i < stes.length; i++) {

				StackTraceElement ste = stes[i];

				// クラス名は前方一致
				if (ste.getClassName().startsWith(jp.getClassName())) {

					// メソッド名は完全一致
					String methodName = jp.getMethodName();
					if (methodName == null || ste.getMethodName().equals(methodName)) {
						// パフォーマンス優先のためログはコメントアウト
						// log.debug(" *cludeTrace 一致: " + ste);
						return true;
					}
				}
			}
		}
		return false;
	}
}
