/***********************************/ 
/* oroborus (c) Ken Lynch Jan 2001 */ 
/* Distributed under the GPL       */ 
/* See LICENSE for more details    */ 
/***********************************/ 
 
#include "oroborus.h" 
 
/* 
 * 
 * Handle different window gravities 
 * 
 */ 

void gravitate_client(Client *c, int grav) 
{ 
  int dx=0, dy=0; 
 
#ifdef DEBUG 
  printf("gravitate_client\n"); 
#endif 
 
  switch(GRAVITY) 
  { 
  case NorthWestGravity: 
    dx=LW; 
    dy=TH; 
    break; 
  case NorthGravity: 
    dx=(LW+RW)/2; 
    dy=TH; 
    break; 
  case NorthEastGravity: 
    dx=-RW; 
    dy=TH; 
    break; 
  case WestGravity: 
    dx = LW; 
    dy=(TH+BH)/2; 
    break; 
  case EastGravity: 
    dx=-RW; 
    dy=(TH+BH)/2; 
    break;  
  case SouthWestGravity: 
    dx=LW; 
    dy=-BH; 
    break; 
  case SouthGravity: 
    dx=(LW+RW)/2; 
    dy=-BH; 
    break; 
  case SouthEastGravity: 
    dx=-RW; 
    dy=-BH; 
    break; 
  case CenterGravity: 
    dx=(LW+RW)/2; 
    dy=(TH+BH)/2; 
    break; 
  } 
  c->x=c->x+(dx*grav); 
  c->y=c->y+(dy*grav); 
} 
 
/* 
 * 
 * Create a new frame part 
 * 
 */ 

Window create_frame_part(Window win, int x, int y, int w, int h) 
{ 
  Window new;   
 
#ifdef DEBUG 
  printf("create_frame_part\n"); 
  printf("  creating frame part for client (%#lx)\n", win); 
#endif 
 
  new=XCreateSimpleWindow(dpy, win, x, y, w, h, 0, 0, 0); 
  XSelectInput(dpy, new, ButtonPressMask|ButtonReleaseMask); 
  XMapWindow(dpy, new); 
  return new; 
} 
 
/* 
 * 
 * Create a new client frame 
 * 
 */ 

void create_new_client(Window w, int reposition) 
{ 
  Client *c=NULL; 
  XWindowAttributes attr; 
  XSetWindowAttributes wattr; 
  PropMwmHints *mhints=NULL; 
  XWMHints *hints=NULL; 
  long dummy; 
  XWindowChanges wc; 
  int x, y, xmax, ymax, xmin, ymin, head=0; 
 
#ifdef DEBUG 
  printf("create_new_client\n"); 
  printf("  creating frame for client (%#lx)\n", w); 
#endif 
 
  if((c=malloc(sizeof *c))==NULL)
  {
    fprintf(stderr, "%s: Unable to allocate memory for new client.\n", PROGNAME);
    exit(1);
  }
 
  /* Initialize client values */ 
  c->window=w; 
  c->ignore_unmap=0; 
  c->hide_state=BUTTON_UNPRESSED; 
  c->close_state=BUTTON_UNPRESSED; 
  c->can_max=True; 
  c->has_frame=True; 
  c->has_focus=False; 
 
#ifdef I18N_SUPPORT 
  if(use_fontset)
    /* call wrapper function for I18N */ 
    XFetchNameWrapper(dpy, c->window, &c->name); 
  else
    XFetchName(dpy, c->window, &c->name);
#else 
  XFetchName(dpy, c->window, &c->name); 
#endif /* I18N */ 
 
  c->size=XAllocSizeHints(); 
  XGetWMNormalHints(dpy, w, c->size, &dummy); 
 
  /* Get size and position of client */ 
  XGetWindowAttributes(dpy, c->window, &attr); 
   
  /* Handle clients with border set > 0 and adjust position accordingly */ 
  if(attr.border_width>0) 
  { 
    switch(GRAVITY) 
    { 
    case SouthEastGravity: 
      attr.x=attr.x+attr.border_width*2; 
      attr.y=attr.y+attr.border_width*2; 
      break; 
    case SouthWestGravity: 
    case SouthGravity: 
      attr.y=attr.y+attr.border_width*2; 
      break; 
    case NorthEastGravity: 
    case EastGravity: 
      attr.x=attr.x+attr.border_width*2; 
      break; 
    } 
  } 
  c->x=attr.x; 
  c->y=attr.y; 
  c->width=attr.width; 
  c->height=attr.height; 
 
  /* Set initial client state */ 
  if(attr.map_state!=IsViewable) 
  { 
    /* Default state NormalState */ 
    set_wm_state(c->window, NormalState); 
    hints=XGetWMHints(dpy, c->window); 
    if(hints) 
    { 
      /* Sets the initial window state if set by client */ 
      if(hints->flags&StateHint) 
        set_wm_state(c->window, hints->initial_state); 
 
      XFree(hints); 
    } 
  } 
 
  /* Get MWM hints */ 
  mhints=get_mwm_hints(c->window); 
  if(mhints) 
  { 
    if(mhints->flags&MWM_HINTS_DECORATIONS && !(mhints->decorations&MWM_DECOR_ALL)) 
    { 
      /* Window has no border if MWM_DECOR_BORDER or MWM_DECOR_TITLE are not set */ 
      c->has_frame=mhints->decorations&(MWM_DECOR_BORDER|MWM_DECOR_TITLE); 
 
      /* Set if client can be maximized or not */ 
      c->can_max=mhints->decorations&MWM_DECOR_MAXIMIZE; 
    } 
    XFree(mhints); 
  } 
 
  /* Set event mask for client */ 
  XSelectInput(dpy, c->window, PropertyChangeMask|FocusChangeMask); 
 
#ifdef SHAPE_SUPPORT 
  /* Select shape events */ 
  if(shape) 
    XShapeSelectInput(dpy, c->window, ShapeNotifyMask); 
#endif /* SHAPE */ 
 
  /* Create frame window */ 
  wattr.override_redirect=True; 
  wattr.event_mask=SubstructureNotifyMask|SubstructureRedirectMask|ButtonPressMask|ButtonReleaseMask|EnterWindowMask; 
  c->frame=XCreateWindow(dpy, root, FX, FY, FW, FH, 0, DefaultDepth(dpy, screen), CopyFromParent, DefaultVisual(dpy, screen), CWOverrideRedirect|CWEventMask, &wattr); 
   
  /* This ignores the unmap event when reparenting */ 
  if(attr.map_state==IsViewable) 
    c->ignore_unmap++; 
 
  /* Reparent client window */ 
  XReparentWindow(dpy, c->window, c->frame, LW, TH); 
    
  /* Create frame parts if has border and/or title */ 
  if(c->has_frame) 
  { 
    /* Side frame parts */ 
    c->left=create_frame_part(c->frame, LX, LY, LW, LH); 
    c->right=create_frame_part(c->frame, RX, RY, RW, RH); 
     
    /* Bottom frame parts */ 
    c->bottom=create_frame_part(c->frame, BX, BY, BW, BH); 
    c->bottom_left=create_frame_part(c->frame, BLX, BLY, BLW, BLH); 
    c->resize=create_frame_part(c->frame, RSX, RSY, RSW, RSH); 
   
    /* Top frame parts */ 
    c->title=create_frame_part(c->frame, TX, TY, TW<1?1:TW, TH); 
    c->top_left=create_frame_part(c->frame, TLX, TLY, TLW, TLH); 
    c->top_right=create_frame_part(c->frame, TRX, TRY, TRW, TRH);  
   
    /* Buttons */ 
    c->hide=create_frame_part(c->frame, HX, HY, HW, HH); 
    c->close=create_frame_part(c->frame, CX, CY, CW, CH);   
  } 
 
  /* Move the client to position set by hints */ 
  if((c->size->flags & (PPosition|USPosition)) || !reposition) 
  {
#ifdef XINERAMA_SUPPORT
    if(xinerama)
    {
      /* If window positioned within 5 pixels of center of display, consider it to be centered on the current head */
      if(abs((c->x+(c->width/2))-(DisplayWidth(dpy, screen)/2))<5 && abs((c->y+(c->height/2))-(DisplayHeight(dpy, screen)/2))<5)
      {
        get_mouse_xy(&x, &y);
        head=find_head(x, y);
        c->x=screen_x(head)+(screen_width(head)/2)-(c->width/2);
        c->y=screen_y(head)+(screen_height(head)/2)-(c->height/2);
      }
      head=find_head(c->x, c->y);
    }
#endif /* XINERAMA */
 
    xmin=screen_x(head)+LW; 
    ymin=screen_y(head)+TH; 
    xmax=(screen_x(head)+screen_width(head))-c->width-RW; 
    ymax=(screen_y(head)+screen_height(head))-c->height-BH; 
   
    gravitate_client(c, APPLY_GRAVITY); 
    wc.x=c->x; 
    wc.y=c->y; 
 
    /* Keep window fully inside desktop */ 
    if(wc.x>xmax) wc.x=xmax; 
    if(wc.y>ymax) wc.y=ymax; 
    if(wc.x<xmin) wc.x=xmin; 
    if(wc.y<ymin) wc.y=ymin; 
  } 
 
  /* Otherwise follow window placement policy */ 
  else 
  { 
#ifdef XINERAMA_SUPPORT
    if(xinerama)
    {
      get_mouse_xy(&x, &y);
      head=find_head(x, y);
    }
#endif /* XINERAMA */

    xmin=screen_x(head)+LW+config.left_margin; 
    ymin=screen_y(head)+TH+config.top_margin; 
    xmax=(screen_x(head)+screen_width(head))-c->width-RW-config.right_margin; 
    ymax=(screen_y(head)+screen_height(head))-c->height-BH-config.bottom_margin; 
    if(config.window_placement==CENTER_ROOT) 
    { 
      x=screen_x(head)+(screen_width(head)/2); 
      y=screen_y(head)+(screen_height(head)/2); 
    } 
    else 
      get_mouse_xy(&x, &y); 
 
    wc.x=x-(c->width/2); 
    wc.y=y-(c->height/2); 
 
    /* Keep window fully inside desktop */ 
    if(wc.x>xmax) wc.x=xmax; 
    if(wc.y>ymax) wc.y=ymax; 
    if(wc.x<xmin) wc.x=xmin; 
    if(wc.y<ymin) wc.y=ymin; 
  } 
  wc.height=c->height; 
  wc.width=c->width; 
  wc.border_width=0; 
  configure_client(c, CWHeight|CWWidth|CWX|CWY|CWBorderWidth, &wc, False); 
  redraw_client_frame(c); 
 
  /* Set client to correct state */   
  if(attr.map_state==IsViewable) 
  { 
    if(get_wm_state(c->window)==IconicState) 
      hide_client(c); 
    else 
      raise_client(c); 
  } 
  else 
  { 
    if(get_wm_state(c->window)==NormalState) 
      raise_client(c); 
  } 
 
  /* Button grab for click-to-focus */ 
  XGrabButton(dpy, AnyButton, AnyModifier, c->frame, False, ButtonPressMask|ButtonReleaseMask, GrabModeSync, GrabModeAsync, None, None); 
 
#ifdef KEYBOARD_SUPPORT 
  grab_keys(c->window); 
#endif /* KEYBOARD */ 
 
#ifdef WORKSPACE_SUPPORT 
  set_gnome_hint(c->window, WIN_WORKSPACE, workspace); 
#endif /* WORKSPACE */ 
 
  /* Add client to client list */ 
  if(client_list==NULL) 
  { 
    client_list=c; 
    c->prev=c; 
    c->next=c; 
  } 
  else 
  { 
    c->prev=client_list->prev; 
    c->next=client_list; 
    client_list->prev->next=c; 
    client_list->prev=c; 
  } 
  client_count++; 
  update_gnome_client_list(); 
 
  /* Check if we want to focus new clients */ 
  if(config.focus_new && !(get_gnome_hint(c->window, WIN_HINTS)&WIN_HINTS_SKIP_FOCUS)) 
    set_focus_client(c); 

#ifdef DEBUG
  printf("  exit create_new_client\n");
#endif
} 
 
/*  
 *  
 * Get client frame from window ID 
 * 
 */ 

Client *get_client_from_window(Window w, int win) 
{ 
  Client *c=NULL; 
  long i; 
 
#ifdef DEBUG 
  printf("get_client_from_window\n"); 
#endif 
 
  c=client_list; 
  for(i=0;i<client_count;i++) 
  { 
    if(win==TITLE) 
    { 
      if(c->title==w) 
        return c; 
    } 
    else if(win==FRAME) 
    { 
      if(c->frame==w) 
        return c;  
    }  
    else 
    { 
      if(c->window==w) 
        return c; 
    } 
    c=c->next; 
  } 
  return NULL; 
} 
 
/* 
 * 
 * Get client with focus 
 * 
 */ 

Client *get_focus_client() 
{ 
  Client *c; 
  Window focus_win; 
  int dummy; 
 
#ifdef DEBUG 
  printf("get_focus_client\n"); 
#endif 
 
  XGetInputFocus(dpy, &focus_win, &dummy); 
  c=get_client_from_window(focus_win, WINDOW); 
  return c;   
} 
 
/* 
 * 
 * Get next client in client list 
 * 
 */ 

Client *get_next_client(Client *c) 
{ 
  long i; 
 
#ifdef DEBUG 
  printf("get_next_client\n"); 
#endif 
 
  if(c!=NULL) 
  { 
    for(i=0;i<client_count;i++) 
    { 
      c=c->next; 
#ifdef WORKSPACE_SUPPORT 
      if(!get_gnome_hint(c->window, WIN_HINTS)&WIN_HINTS_SKIP_FOCUS && get_gnome_hint(c->window, WIN_WORKSPACE)==workspace) 
#else 
      if(!get_gnome_hint(c->window, WIN_HINTS)&WIN_HINTS_SKIP_FOCUS) 
#endif /* WORKSPACE */ 
        return c; 
    } 
  } 
  return NULL; 
} 

/* 
 * 
 * Get previous client in client list 
 * 
 */ 

Client *get_prev_client(Client *c) 
{ 
  long i; 
 
#ifdef DEBUG 
  printf("get_prev_client\n"); 
#endif 
 
  if(c!=NULL) 
  { 
    for(i=0;i<client_count;i++) 
    { 
      c=c->prev; 
#ifdef WORKSPACE_SUPPORT 
      if(!get_gnome_hint(c->window, WIN_HINTS)&WIN_HINTS_SKIP_FOCUS && get_gnome_hint(c->window, WIN_WORKSPACE)==workspace) 
#else 
      if(!get_gnome_hint(c->window, WIN_HINTS)&WIN_HINTS_SKIP_FOCUS) 
#endif /* WORKSPACE */ 
        return c; 
    } 
  } 
  return NULL;
}
 
/* 
 * 
 * Get top-most client 
 * 
 */ 

Client *get_top_most_client(int layer) 
{ 
  Window *windows, w1, w2; 
  int count, i; 
  Client *top=NULL; 
#ifdef LAYER_SUPPORT 
  Client *c=NULL; 
#endif /* LAYER */ 
  XWindowAttributes attr; 
 
#ifdef DEBUG 
  printf("get_top_most_client\n"); 
#endif 
 
  XQueryTree(dpy, root, &w1, &w2, &windows, &count); 
#ifdef LAYER_SUPPORT 
  for(i=0;i<count;i++) 
  { 
    c=get_client_from_window(windows[i], FRAME); 
    if(c!=NULL) 
    { 
      XGetWindowAttributes(dpy, c->window, &attr); 
      if(attr.map_state==IsViewable) 
      { 
        if(get_gnome_hint(c->window, WIN_LAYER)<=layer) 
          top=c; 
        else 
          break; 
      } 
    } 
  } 
#else 
  for(i=count;i<=0;i--) 
  { 
    top=get_client_from_window(windows[i], FRAME); 
    if(top!=NULL) 
    { 
      XGetWindowAttributes(dpy, top->window, &attr); 
      if(attr.map_state==IsViewable) 
        break; 
    } 
  } 
#endif /* LAYER */ 
  if(windows) XFree(windows); 
  return top; 
} 
 
/* 
 * 
 * Get bottom-most client 
 * 
 */ 

Client *get_bottom_most_client(int layer) 
{ 
  Window *windows=NULL, w1, w2; 
  int count, i; 
  Client *c=NULL; 
  XWindowAttributes attr; 
 
#ifdef DEBUG 
  printf("get_bottom_most_client\n"); 
#endif 
 
  XQueryTree(dpy, root, &w1, &w2, &windows, &count); 
  for(i=0;i<count;i++) 
  { 
    c=get_client_from_window(windows[i], FRAME); 
    if(c!=NULL) 
    { 
      XGetWindowAttributes(dpy, c->window, &attr); 
      if(attr.map_state==IsViewable) 
      { 
#ifdef LAYER_SUPPORT 
        if(get_gnome_hint(c->window, WIN_LAYER)>=layer) 
          break; 
#else 
        break; 
#endif /* LAYER */ 
      } 
    } 
  } 
  if(windows) XFree(windows);  
  return c; 
} 
 
/* 
 *  
 * Raise client window 
 * 
 */ 

void raise_client(Client *c) 
{ 
  XWindowChanges wc; 
 
#ifdef DEBUG 
  printf("raise_client\n"); 
#endif 
 
  if(c==NULL) 
  { 
#ifdef DEBUG 
    printf("  client is NULL\n"); 
#endif 
  } 
  else 
  { 
#ifdef DEBUG 
    printf("  raising client (%#lx)\n", c->window); 
#endif 
 
    show_client(c); 
    wc.stack_mode=Above; 
    configure_client(c, CWStackMode, &wc, False); 
  } 
} 
 
/* 
 *  
 * Give client window keyboard focus 
 * 
 */ 

void set_focus_client(Client *c) 
{ 
#ifdef DEBUG 
  printf("set_focus_client\n"); 
#endif 
 
  if(c==NULL) 
  { 
#ifdef DEBUG 
    printf("  client is NULL\n"); 
#endif 

    XSetInputFocus(dpy, check_win, RevertToNone, CurrentTime); 
  } 
  else 
  { 
#ifdef DEBUG 
    printf("  focusing client (%#lx)\n", c->window); 
#endif 

    XSetInputFocus(dpy, c->window, RevertToNone, CurrentTime); 
  } 
} 
 
/* 
 * 
 * Lower client window 
 * 
 */ 

void lower_client(Client *c) 
{ 
  XWindowChanges wc; 
 
#ifdef DEBUG 
  printf("lower_client\n"); 
  printf("  lowering client (%#lx)\n", c->window); 
#endif 
 
  wc.stack_mode=Below; 
  configure_client(c, CWStackMode, &wc, False); 
} 
 
/* 
 *  
 * Show client window 
 * 
 */ 
void show_client(Client *c) 
{ 
#ifdef DEBUG 
  printf("show_client\n"); 
  printf("  showing client (%#lx)\n", c->window); 
#endif 
 
  XMapWindow(dpy, c->frame); 
  XMapWindow(dpy, c->window); 
  set_wm_state(c->window, NormalState); 
} 
 
/* 
 * 
 * Hide client window 
 * 
 */ 

void hide_client(Client *c) 
{ 
#ifdef DEBUG 
  printf("hide_client\n"); 
  printf("  hiding client (%#lx)\n", c->window); 
#endif 
 
  /* We need to ignore one of the unmaps, otherwise our client gets deleted */ 
  c->ignore_unmap++; 
  XUnmapWindow(dpy, c->frame); 
  XUnmapWindow(dpy, c->window); 
  set_wm_state(c->window, IconicState); 
} 
 
/*  
 * 
 * Configure client - the main resizing, raising, lowering function 
 * 
 */ 

void configure_client(Client *c, unsigned int mask, XWindowChanges *wc, int frame_only) 
{ 
  XConfigureEvent ce; 
  int w, w1, h, h1;
 
#ifdef DEBUG 
  printf("configure_client\n"); 
  printf("  configuring client (%#lx)\n", c->window); 
#endif 

  if(mask&CWX) c->x=wc->x; 
  if(mask&CWY) c->y=wc->y; 
  if(mask&CWWidth) 
  { 
    w=wc->width; 
 
    /* Get resize increments */ 
    if(c->size->flags & PResizeInc) 
    { 
      w1=(w-c->size->min_width)/c->size->width_inc; 
      w=c->size->min_width+(w1*c->size->width_inc); 
    } 
    /* Get min size */ 
    if(c->size->flags & PMinSize) 
      if(w<c->size->min_width) w=c->size->min_width; 
    /* Get max size */ 
    if(c->size->flags & PMaxSize) 
      if(w>c->size->max_width) w=c->size->max_width; 
 
    /* Make sure we never set width to <1 */ 
    if(w<1) w=1; 
     
    c->width=w; 
  } 
  if(mask&CWHeight) 
  { 
    h=wc->height; 
 
    /* Get resize increments */ 
    if(c->size->flags & PResizeInc) 
    { 
      h1=(h-c->size->min_height)/c->size->height_inc; 
      h=c->size->min_height+(h1*c->size->height_inc); 
    } 
    /* Get min size */ 
    if(c->size->flags & PMinSize) 
      if(h<c->size->min_height) h=c->size->min_height; 
    /* Get max size */ 
    if(c->size->flags & PMaxSize) 
      if(h>c->size->max_height) h=c->size->max_height; 
 
    /* Make sure we never set height to <1 */ 
    if(h<1) h=1; 
     
    c->height=h; 
  } 
 
#ifdef LAYER_SUPPORT 
  /* Deal with window layers */ 
  if(mask&CWStackMode) 
  { 
    Client *sibling=NULL; 
 
    /* Takes care of layers */ 
    switch(wc->stack_mode) 
    { 
    case Above: 
      /* Window wants to be raised to top of its layer */ 
      sibling=get_top_most_client(get_gnome_hint(c->window, WIN_LAYER)); 
      if(sibling==NULL) 
        wc->stack_mode=Below; 
      break; 
    case Below: 
      /* Window wants to be lowered to bottom of its layer */ 
      sibling=get_bottom_most_client(get_gnome_hint(c->window, WIN_LAYER)); 
      break; 
    } 
    if(sibling==NULL) 
    { 
      if(mask&CWSibling) mask=mask^CWSibling; 
    } 
    else if(sibling==c) 
    { 
      if(mask&CWStackMode) mask=mask^CWStackMode; 
      if(mask&CWSibling) mask=mask^CWSibling; 
    } 
    else 
    { 
      mask=mask|CWSibling; 
      wc->sibling=sibling->frame; 
    } 
#ifdef WORKSPACE_SUPPORT 
    set_gnome_hint(c->window, WIN_WORKSPACE, workspace); 
#endif /* WORKSPACE */ 
  } 
#endif /* LAYER */ 
 
  wc->x=FX; 
  wc->y=FY; 
  wc->width=FW; 
  wc->height=FH; 
  XConfigureWindow(dpy, c->frame, mask, wc); 

  if(!frame_only)
  {
    wc->x=LW;
    wc->y=TH;
    wc->width=c->width;
    wc->height=c->height;

    /* We don't want any stacking order changes on our actual client */ 
    if(mask&CWStackMode) mask=mask^CWStackMode; 
    if(mask&CWSibling) mask=mask^CWSibling;

    XConfigureWindow(dpy, c->window, mask, wc);
  }

  /* If size has changed */ 
  if(mask&(CWHeight|CWWidth)) 
  { 
    if(c->has_frame) 
    { 
      /* Reposition and resize frame parts */ 
      XMoveResizeWindow(dpy, c->title, TX, TY, TW<1?1:TW, TH); 
      XMoveResizeWindow(dpy, c->top_left, TLX, TLY, TLW, TLH); 
 
      XMoveResizeWindow(dpy, c->top_right, TRX, TRY, TRW, TRH);
      XMoveResizeWindow(dpy, c->left, LX, LY, LW, LH);
      XMoveResizeWindow(dpy, c->right, RX, RY, RW, RH);
      XMoveResizeWindow(dpy, c->bottom_left, BLX, BLY, BLW, BLH);
      XMoveResizeWindow(dpy, c->bottom, BX, BY, BW, BH);
      XMoveResizeWindow(dpy, c->resize, RSX, RSY, RSW, RSH);
      XMoveResizeWindow(dpy, c->close, CX, CY, CW, CH);
      XMoveResizeWindow(dpy, c->hide, HX, HY, HW, HH);
      redraw_client_title(c);
    }

#ifdef SHAPE_SUPPORT 
    set_client_shape(c);
#endif /* SHAPE */ 
  }
 
  /* Notify client of window changes if changes to position or size */ 
  if(mask&(CWX|CWY|CWHeight|CWWidth) && !frame_only) 
  { 
    ce.type=ConfigureNotify; 
    ce.event=c->window; 
    ce.window=c->window; 
    ce.x=c->x; 
    ce.y=c->y; 
    ce.width=c->width; 
    ce.height=c->height; 
    ce.border_width=0; 
    ce.above=None; 
    ce.override_redirect=0; 
    XSendEvent(dpy, c->window, False, StructureNotifyMask, (XEvent *)&ce); 
  } 
} 
 
/* 
 * 
 * Move client to new position 
 * 
 */ 

void move_client(Client *c, int x, int y) 
{ 
  XWindowChanges wc; 
 
#ifdef DEBUG 
  printf("move_client\n"); 
  printf("  moving client (%#lx)\n", c->window); 
#endif 
 
  wc.x=x; 
  wc.y=y; 
  configure_client(c, CWX|CWY, &wc, False); 
} 
 
/* 
 * 
 * Resize client 
 * 
 */ 

void resize_client(Client *c, int w, int h, int frame_only) 
{ 
  XWindowChanges wc; 
 
#ifdef DEBUG 
  printf("resize_client\n"); 
  printf("  resizing client (%#lx)\n", c->window); 
#endif 
 
  wc.width=w; 
  wc.height=h; 
  configure_client(c, CWWidth|CWHeight, &wc, frame_only); 
} 
 
/* 
 * 
 * Toggle client maximized 
 * 
 */ 

void toggle_client_maximized(Client *c) 
{ 
  XWindowChanges wc; 
 
#ifdef DEBUG 
  printf("toggle_client_maximized\n"); 
#endif 
 
  if(c->can_max) 
  { 
    if(get_gnome_hint(c->window, WIN_STATE)&WIN_STATE_MAXIMIZED) 
    { 
      c->x=c->old_x; 
      c->y=c->old_y; 
      c->width=c->old_width; 
      c->height=c->old_height; 
    } 
    else 
    { 
      int head=0; 
 
#ifdef XINERAMA_SUPPORT 
      if(xinerama)
        head=find_head(c->x+c->width/2, c->y+c->height/2); 
#endif /* XINERAMA */ 
 
      c->old_x=c->x; 
      c->old_y=c->y; 
      c->old_width=c->width; 
      c->old_height=c->height; 
      c->x=screen_x(head)+LW+config.left_margin; 
      c->y=screen_y(head)+TH+config.top_margin; 
      c->width=screen_width(head)-LW-RW-config.left_margin-config.right_margin; 
      c->height=screen_height(head)-TH-BH-config.top_margin-config.bottom_margin;
    } 
    wc.x=c->x; 
    wc.y=c->y; 
    wc.height=c->height; 
    wc.width=c->width; 
    configure_client(c, CWHeight|CWWidth|CWX|CWY, &wc, False); 
    set_gnome_hint(c->window, WIN_STATE, get_gnome_hint(c->window, WIN_STATE)^WIN_STATE_MAXIMIZED); 
  } 
} 
 
/* 
 *  
 * Toggle client shaded 
 * 
 */ 

void toggle_client_shaded(Client *c) 
{ 
#ifdef DEBUG 
  printf("toggle_client_shaded\n"); 
#endif 
 
  if(c->has_frame) 
  { 
    set_gnome_hint(c->window, WIN_STATE, get_gnome_hint(c->window, WIN_STATE)^WIN_STATE_SHADED); 
    resize_client(c, c->width, c->height, True);
    redraw_client_frame(c); 
  } 
} 
 
#ifdef LAYER_SUPPORT 
/* 
 * 
 * Set client layer 
 * 
 */ 

void set_client_layer(Client *c, int layer) 
{ 
#ifdef DEBUG 
  printf("set_client_layer\n"); 
#endif 
 
  set_gnome_hint(c->window, WIN_LAYER, layer); 
  if(get_wm_state(c->window)==NormalState) 
    raise_client(c); 
} 
 
/* 
 * 
 * Get client layer 
 * 
 */ 

int get_client_layer(Client *c) 
{ 
#ifdef DEBUG 
  printf("get client layer\n"); 
#endif 
 
  return get_gnome_hint(c->window, WIN_LAYER); 
} 
#endif /* LAYER */ 
 
/* 
 * 
 * Delete client 
 * 
 */ 

void delete_client(Client *c, int remap) 
{ 
#ifdef DEBUG 
  printf("delete_client\n"); 
  printf("  deleting client (%#lx)\n", c->window); 
#endif 
 
  /* Check if frame is still the clients parent */ 
  if(get_window_parent(c->window)==c->frame) 
  { 
    /* Reparent our client back to root if it still exists */ 
    gravitate_client(c, REMOVE_GRAVITY); 
    XReparentWindow(dpy, c->window, root, c->x, c->y); 
    set_wm_state(c->window, WithdrawnState); 

#ifdef KEYBOARD_SUPPORT
    /* Remove any key grabs on this window */
    ungrab_keys(c->window);
#endif
 
    /* Remap is true if we are quitting Oroborus */ 
    if(remap) 
      XMapWindow(dpy, c->window); 
    else 
      XUnmapWindow(dpy, c->window); 
  } 

  /* Get rid of our frame window */ 
  XDestroyWindow(dpy, c->frame); 
 
  /* Remove reference from our client list */ 
  if(client_count==1) 
    client_list=NULL; 
  else 
  { 
    c->next->prev = c->prev; 
    c->prev->next = c->next; 
    if(c==client_list) 
      client_list=client_list->next; 
  } 
 
  if(c->name) XFree(c->name); 
  if(c->size) XFree(c->size); 
  free(c); 
 
  client_count--; 
  update_gnome_client_list();

#ifdef DEBUG
  printf("  client removed\n");
#endif
} 
 
/* 
 * 
 * Send close message to client window 
 * 
 */ 

void close_client(Client *c) 
{ 
  int i, n, found=False; 
  Atom *protocols=NULL; 
  XEvent e; 
 
#ifdef DEBUG 
  printf("close_client\n"); 
  printf("  sending close message to client (%#lx)\n", c->window); 
#endif 
 
  if(XGetWMProtocols(dpy, c->window, &protocols, &n)) 
  { 
    for(i=0; i<n; i++) 
      if(protocols[i]==icccm[WM_DELETE]) found=True; 
    if(protocols) XFree(protocols);  
  } 
  if(found) 
  { 
    e.type=ClientMessage; 
    e.xclient.window=c->window; 
    e.xclient.message_type=icccm[WM_PROTOCOLS]; 
    e.xclient.format=32; 
    e.xclient.data.l[0]=icccm[WM_DELETE]; 
    e.xclient.data.l[1]=CurrentTime; 
    XSendEvent(dpy, c->window, False, NoEventMask, &e); 
  } 
  else 
    XKillClient(dpy, c->window); 
} 
 
/* 
 * 
 * Redraw client borders 
 * 
 */ 

void redraw_client_frame(Client *c) 
{ 
  int state=FRAME_INACTIVE, shaded=0; 
 
#ifdef DEBUG 
  printf("redraw_client_frame\n"); 
#endif 
 
  if(!c->has_frame) 
  { 
#ifdef DEBUG 
    printf("  client (%#lx) has no frame\n", c->window); 
#endif 
    return; 
  } 

#ifdef DEBUG 
  printf("  redrawing frame for client (%#lx)\n", c->window); 
#endif 
 
  if(c==get_focus_client()) 
    state=FRAME_ACTIVE;

  if(get_gnome_hint(c->window, WIN_STATE)&WIN_STATE_SHADED)
    shaded=2; 
 
  XSetWindowBackgroundPixmap(dpy, c->top_left, config.top_left.pixmap[state+shaded]); 
  XClearWindow(dpy, c->top_left); 
  XSetWindowBackgroundPixmap(dpy, c->top_right, config.top_right.pixmap[state+shaded]); 
  XClearWindow(dpy, c->top_right); 
  XSetWindowBackgroundPixmap(dpy, c->left, config.left.pixmap[state]); 
  XClearWindow(dpy, c->left); 
  XSetWindowBackgroundPixmap(dpy, c->right, config.right.pixmap[state]); 
  XClearWindow(dpy, c->right); 
  XSetWindowBackgroundPixmap(dpy, c->bottom_left, config.bottom_left.pixmap[state]); 
  XClearWindow(dpy, c->bottom_left); 
  XSetWindowBackgroundPixmap(dpy, c->bottom, config.bottom.pixmap[state]); 
  XClearWindow(dpy, c->bottom); 
  XSetWindowBackgroundPixmap(dpy, c->resize, config.resize.pixmap[state]); 
  XClearWindow(dpy, c->resize); 
 
  redraw_client_buttons(c); 
  redraw_client_title(c); 
} 
 
/* 
 *  
 * Redraw client buttons 
 * 
 */ 

void redraw_client_buttons(Client *c) 
{ 
  int state=FRAME_INACTIVE; 
 
#ifdef DEBUG 
  printf("redraw_client_buttons\n"); 
#endif 
 
  if(!c->has_frame) 
  { 
#ifdef DEBUG 
    printf("  client (%#lx) has no frame\n", c->window); 
#endif 
    return; 
  } 
 
#ifdef DEBUG 
  printf("  redrawing buttons for client (%#lx)\n", c->window); 
#endif 
 
  if(c==get_focus_client()) 
    state=FRAME_ACTIVE; 
 
  XSetWindowBackgroundPixmap(dpy, c->hide, config.hide.pixmap[c->hide_state==BUTTON_PRESSED?BUTTON_PRESSED:state]); 
  XClearWindow(dpy, c->hide); 
  XSetWindowBackgroundPixmap(dpy, c->close, config.close.pixmap[c->close_state==BUTTON_PRESSED?BUTTON_PRESSED:state]); 
  XClearWindow(dpy, c->close); 
} 
 
/* 
 * 
 * Redraw client name in the title bar 
 * 
 */ 

void redraw_client_title(Client *c) 
{ 
  int state=FRAME_INACTIVE, shaded=0; 
  Pixmap temp; 
  GC gc; 
  XGCValues gv; 
#ifdef I18N_SUPPORT 
  XRectangle ink, logical; 
  int ascent=0, descent=0; 
#endif /* I18N */ 
  int width; 
 
#ifdef DEBUG 
  printf("redraw_client_title\n"); 
#endif 
 
  if(!c->has_frame) 
  { 
#ifdef DEBUG 
    printf("  client (%#lx) has no frame\n", c->window); 
#endif 
    return; 
  }
 
#ifdef DEBUG 
  printf("  redrawing title for client (%#lx)\n", c->window); 
#endif 
 
  if(c==get_focus_client()) 
    state=FRAME_ACTIVE; 

  if(get_gnome_hint(c->window, WIN_STATE)&WIN_STATE_SHADED)
    shaded=2;
 
  gv.function=GXcopy; 
  gv.tile=config.title.pixmap[state+shaded]; 
  gv.fill_style=FillTiled; 
  gc=XCreateGC(dpy, root, GCFunction|GCFillStyle|GCTile, &gv); 
  temp=XCreatePixmap(dpy, root, TW, TH, DefaultDepth(dpy, screen)); 
  XFillRectangle(dpy, temp, gc, 0, 0, TW, TH); 
 
  if(c->name) 
  { 
    int pos=0; 
#ifdef I18N_SUPPORT
    if(use_fontset)
    {
      XmbTextExtents(config.fontset, c->name, strlen(c->name), &ink, &logical); 
      ascent=-logical.y; 
      descent=logical.height+logical.y; 
      width=logical.width; 
    }
    else
      width=XTextWidth(config.font, c->name, strlen(c->name));
#else  
    width=XTextWidth(config.font, c->name, strlen(c->name)); 
#endif /* I18N */ 
 
    switch(config.text_align) 
    { 
    case ALIGNMENT_MIDDLE: 
      pos=width>TW-4?2:TW/2-width/2; 
      break; 
    case ALIGNMENT_RIGHT: 
      pos=width>TW-4?2:TW-width-2; 
      break; 
    case ALIGNMENT_LEFT: 
      pos=2; 
      break; 
    } 

#ifdef I18N_SUPPORT
    if(use_fontset)
      XmbDrawString(dpy, temp, config.fontset, config.font_gc[state], pos, (TH-descent)-((TH-ascent-descent)/2), c->name, strlen(c->name));
    else
      XDrawString(dpy, temp, config.font_gc[state], pos, (TH-config.font->descent)-((TH-config.font->ascent-config.font->descent)/2), c->name, strlen(c->name));
#else 
    XDrawString(dpy, temp, config.font_gc[state], pos, (TH-config.font->descent)-((TH-config.font->ascent-config.font->descent)/2), c->name, strlen(c->name)); 
#endif /* I18N */ 
 
  }
  XSetWindowBackgroundPixmap(dpy, c->title, temp); 
  XClearWindow(dpy, c->title); 
 
  XFreeGC(dpy, gc); 
  XFreePixmap(dpy, temp); 
} 
 
#ifdef SHAPE_SUPPORT 
/* 
 * 
 * Set client window shape 
 * 
 */ 

void set_client_shape(Client *c) 
{ 
  int shaded=0;
  Window temp; 
 
#ifdef DEBUG 
  printf("set_client_shape\n"); 
  printf("  setting shape for client (%#lx)\n", c->window); 
#endif 
 
  if(shape) 
  { 
    temp=XCreateSimpleWindow(dpy, root, 0, 0, FW, FH, 0, 0, 0);
    XShapeCombineShape(dpy, temp, ShapeBounding, LW, TH, c->window, ShapeBounding, ShapeSet);

    if(c->has_frame) 
    {
      if(get_gnome_hint(c->window, WIN_STATE)&WIN_STATE_SHADED)
        shaded=1;

      XShapeCombineMask(dpy, c->top_left, ShapeBounding, 0, 0, config.top_left.mask[shaded], ShapeSet);
      XShapeCombineMask(dpy, c->top_right, ShapeBounding, 0, 0, config.top_right.mask[shaded], ShapeSet);
      XShapeCombineMask(dpy, c->bottom_left, ShapeBounding, 0, 0, config.bottom_left.mask[0], ShapeSet);
      XShapeCombineMask(dpy, c->resize, ShapeBounding, 0, 0, config.resize.mask[0], ShapeSet);
      XShapeCombineMask(dpy, c->hide, ShapeBounding, 0, 0, config.hide.mask[0], ShapeSet);
      XShapeCombineMask(dpy, c->close, ShapeBounding, 0, 0, config.close.mask[0], ShapeSet);

      XShapeCombineMask(dpy, temp, ShapeBounding, TLX, TLY, config.top_left.mask[shaded], ShapeUnion);
      XShapeCombineMask(dpy, temp, ShapeBounding, TRX, TRY, config.top_right.mask[shaded], ShapeUnion);
      XShapeCombineMask(dpy, temp, ShapeBounding, RSX, RSY, config.resize.mask[0], ShapeUnion); 
      XShapeCombineMask(dpy, temp, ShapeBounding, BLX, BLY, config.bottom_left.mask[0], ShapeUnion);
      XShapeCombineShape(dpy, temp, ShapeBounding, TX, TY, c->title, ShapeBounding, ShapeUnion);  
      XShapeCombineShape(dpy, temp, ShapeBounding, LX, LY, c->left, ShapeBounding, ShapeUnion); 
      XShapeCombineShape(dpy, temp, ShapeBounding, RX, RY, c->right, ShapeBounding, ShapeUnion); 
      XShapeCombineShape(dpy, temp, ShapeBounding, BX, BY, c->bottom, ShapeBounding, ShapeUnion);
    } 
    XShapeCombineShape(dpy, c->frame, ShapeBounding, 0, 0, temp, ShapeBounding, ShapeSet); 
    XDestroyWindow(dpy, temp);
  } 
} 
#endif /* SHAPE */ 
 
#ifdef WORKSPACE_SUPPORT 
/* 
 * 
 * Toggle sticky/unsticky 
 * 
 */ 

void toggle_client_sticky(Client *c) 
{ 
#ifdef DEBUG 
  printf("toggle_client_sticky\n"); 
  printf("  toggling sticky flag for client (%#lx)\n", c->window); 
#endif 
 
  set_gnome_hint(c->window, WIN_STATE, get_gnome_hint(c->window, WIN_STATE)^WIN_STATE_STICKY); 
} 
 
/* 
 * 
 * Move client to new workspace 
 * 
 */ 

void move_client_to_workspace(Client *c, int ws) 
{ 
#ifdef DEBUG 
  printf("move_client_to_workspace\n"); 
#endif 
 
  if(ws==workspace || ws<0 || ws>=config.workspace_count) return; 
  set_gnome_hint(c->window, WIN_WORKSPACE, ws); 
  change_to_workspace(ws); 
  set_focus_client(c); 
  raise_client(c); 
} 
#endif /* WORKSPACE */ 

