//                                               -*- C++ -*-
/**
 *  @file  t_SORM_std.cxx
 *  @brief The test file of SORM class
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: t_SORM_std.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <iostream>
#include <iomanip>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Event.hxx"
#include "RandomVector.hxx"
#include "Less.hxx"
#include "NumericalMathFunction.hxx"
#include "Simulation.hxx"
#include "SORM.hxx"
#include "IdentityMatrix.hxx"
#include "Normal.hxx"
#include "AbdoRackwitz.hxx"
#include "Pointer.hxx"

using namespace OT;
using namespace OT::Test;

String printNumericalPoint(const NumericalPoint & point, const UnsignedLong digits)
{
  OSS oss;
  oss << "[";
  NumericalScalar eps(pow(0.1, digits));
  for (UnsignedLong i = 0; i < point.getDimension(); i++)
    {
      oss << std::fixed << std::setprecision(digits) << (i==0 ? "" : ",") << Bulk<double>((fabs(point[i]) < eps) ? fabs(point[i]) : point[i]);
    }
  oss << "]";
  return oss;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try {

    /* We create a numerical math function */
    NumericalMathFunction myFunction("poutre");

    UnsignedLong dim(myFunction.getInputDimension());
    /* We create a normal distribution point of dimension 1 */
    NumericalPoint mean(dim, 0.0);
    mean[0] = 50.0; // E
    mean[1] =  1.0; // F
    mean[2] = 10.0; // L
    mean[3] =  5.0; // I
    NumericalPoint sigma(dim, 1.0);
    IdentityMatrix R(dim);
    Normal myDistribution(mean, sigma, R);

    /* We create a 'usual' RandomVector from the Distribution */
    RandomVector vect(myDistribution);

    /* We create a composite random vector */
    RandomVector output(myFunction, vect);

    /* We create an Event from this RandomVector */
    Event myEvent(output, Less(), -3.0);

    /* We create a NearestPoint algorithm */
    AbdoRackwitz myAbdoRackwitz;
    myAbdoRackwitz.setMaximumIterationsNumber(100);
    myAbdoRackwitz.setMaximumAbsoluteError(1.0e-10);
    myAbdoRackwitz.setMaximumRelativeError(1.0e-10);
    myAbdoRackwitz.setMaximumResidualError(1.0e-10);
    myAbdoRackwitz.setMaximumConstraintError(1.0e-10);
    myAbdoRackwitz.setSpecificParameters(AbdoRackwitzSpecificParameters());
    fullprint << "myAbdoRackwitz=" << myAbdoRackwitz << std::endl;

    /* We create a SORM algorithm */
    /* The first parameter is a NearestPointAlgorithm */
    /* The second parameter is an event */
    /* The third parameter is a starting point for the design point research */
    SORM myAlgo(myAbdoRackwitz, myEvent, mean);

    /* Perform the simulation */
    myAlgo.run();

    /* Stream out the result */
    SORMResult result(myAlgo.getResult());
    UnsignedLong digits(4);
    fullprint << "Breitung event probability=" << result.getEventProbabilityBreitung() << std::endl;
    fullprint << "Breitung generalized reliability index=" << std::setprecision(digits) << result.getGeneralisedReliabilityIndexBreitung() << std::endl;
    fullprint << "HohenBichler event probability=" << std::setprecision(digits) << result.getEventProbabilityHohenBichler() << std::endl;
    fullprint << "HohenBichler generalized reliability index=" << std::setprecision(digits) << result.getGeneralisedReliabilityIndexHohenBichler() << std::endl;
    fullprint << "Tvedt event probability=" << std::setprecision(digits) << result.getEventProbabilityTvedt() << std::endl;
    fullprint << "Tvedt generalized reliability index=" << std::setprecision(digits) << result.getGeneralisedReliabilityIndexTvedt() << std::endl;
    fullprint << "sorted curvatures=" << printNumericalPoint(result.getSortedCurvatures(), digits) << std::endl;
    fullprint << "standard space design point=" << printNumericalPoint(result.getStandardSpaceDesignPoint(), digits) << std::endl;
    fullprint << "physical space design point=" << printNumericalPoint(result.getPhysicalSpaceDesignPoint(), digits) << std::endl;
    fullprint << "is standard point origin in failure space? " << (result.getIsStandardPointOriginInFailureSpace() ? "true" : "false") << std::endl;
    fullprint << "importance factors=" << printNumericalPoint(result.getImportanceFactors(), digits) << std::endl;
    fullprint << "importance factors (classical)=" << printNumericalPoint(result.getImportanceFactors(true), digits) << std::endl;
    fullprint << "Hasofer reliability index=" << std::setprecision(digits) << result.getHasoferReliabilityIndex() << std::endl;

    Study myStudy;
    myStudy.setStorageManager(XMLStorageManager("testStudy.ot"));
    myStudy.add("myAlgo", myAlgo);
    myStudy.save();

    Study myStudy2;
    myStudy2.setStorageManager(XMLStorageManager("testStudy.ot"));
    myStudy2.load();
    SORM sorm;
    myStudy2.fillObject("myAlgo", sorm);
    // Reduce the precision in order to take into account the limited precision of the hessian
    PlatformInfo::SetNumericalPrecision(4);
    fullprint << "loaded SORM=" << sorm << std::endl;

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
