//                                               -*- C++ -*-
/**
 *  @file  RosenblattEvaluation.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-07-03 08:31:34 +0200 (jeu, 03 jui 2008) $
 *  Id:      $Id: RosenblattEvaluation.cxx 867 2008-07-03 06:31:34Z lebrun $
 */
#include "RosenblattEvaluation.hxx"
#include "DistFunc.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {



      CLASSNAMEINIT(RosenblattEvaluation);

      typedef Uncertainty::Distribution::DistFunc      DistFunc;
      typedef Base::Common::NotYetImplementedException NotYetImplementedException;

      /* Parameter constructor */
      RosenblattEvaluation::RosenblattEvaluation(const Distribution & distribution):
	NumericalMathEvaluationImplementation(),
	distribution_(distribution)
      {
	Description description(distribution.getDescription());
	const UnsignedLong size(description.getSize());
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    OSS oss;
	    oss << "y" << i;
	    description.add(oss);
	  }
	setDescription(description);
      }

      /* Virtual constructor */
      RosenblattEvaluation * RosenblattEvaluation::clone() const
      {
	return new RosenblattEvaluation(*this);
      }

      /* Evaluation */
      RosenblattEvaluation::NumericalPoint RosenblattEvaluation::operator () (const NumericalPoint & in) const
	/* throw (InvalidArgumentException, InternalException) */
      {
	const UnsignedLong dimension(getOutputDimension());
	NumericalPoint result(dimension);
	NumericalPoint y(0);
	// Apply Phi^{-1} o conditional CDF over the components
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    result[i] = DistFunc::qNormal(distribution_.computeConditionalCDF(in[i], y));
	    y.add(in[i]);
	  }
	return result;
      }

      /* Gradient according to the marginal parameters. */
      RosenblattEvaluation::Matrix RosenblattEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	throw NotYetImplementedException(HERE);
      }

      /* Accessor for input point dimension */
      UnsignedLong RosenblattEvaluation::getInputDimension() const
	/* throw(InternalException) */
      {
	return distribution_.getDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong RosenblattEvaluation::getOutputDimension() const
	/* throw(InternalException) */
      {
	return distribution_.getDimension();
      }

      /* String converter */
      String RosenblattEvaluation::__repr__() const
      {
	OSS oss;
	oss << "class=" << RosenblattEvaluation::GetClassName()
	    << " description=" << getDescription()
	    << " distribution=" << distribution_;                   
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

