//                                               -*- C++ -*-
/**
 *  @file  CenteredFiniteDifferenceHessian.hxx
 *  @brief Class for the creation of a numerical math hessian implementation
 *         form a numerical math evaluation implementation by using centered
 *         finite difference formula.
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: CenteredFiniteDifferenceHessian.hxx 1473 2010-02-04 15:44:49Z dutka $
 */

#ifndef OPENTURNS_CENTEREDFINITEDIFFERENCEHESSIAN_HXX
#define OPENTURNS_CENTEREDFINITEDIFFERENCEHESSIAN_HXX

#define OPENTURNS_CENTEREDFINITEDIFFERENCEHESSIAN_HXX

#include "NumericalMathHessianImplementation.hxx"
#include "NumericalMathEvaluationImplementation.hxx"
#include "Exception.hxx"
#include "NumericalPoint.hxx"
#include "SymmetricTensor.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Diff {


      /**
       * @class CenteredFiniteDifferenceHessian
       * @brief This class is for the creation of a numerical math hessian implementation
       * @ingroup Function
       *
       * This class is for the creation of a numerical math hessian implementation
       * form a numerical math evaluation implementation by using centered
       * finite difference formula
       */
      class CenteredFiniteDifferenceHessian
	: public Func::NumericalMathHessianImplementation
      {
	CLASSNAME;
      public:

	static const NumericalScalar DefaultEpsilon;

	typedef Func::NumericalMathHessianImplementation::Description              Description;
	typedef Func::NumericalMathHessianImplementation::SymmetricTensor          SymmetricTensor;
	typedef Func::NumericalMathHessianImplementation::NumericalPoint           NumericalPoint;
	typedef Func::NumericalMathHessianImplementation::InvalidArgumentException InvalidArgumentException;
	typedef Func::NumericalMathHessianImplementation::InternalException        InternalException;
	typedef Func::NumericalMathHessianImplementation::StorageManager           StorageManager;
	typedef Common::InvalidDimensionException                                  InvalidDimensionException;
	typedef Common::Pointer<Func::NumericalMathEvaluationImplementation>         EvaluationImplementation;

	/** Default constructor */
	CenteredFiniteDifferenceHessian();
	/** Parameter constructor */
	CenteredFiniteDifferenceHessian(const NumericalPoint & epsilon,
					const EvaluationImplementation & p_evaluation)
	  /* throw(InvalidDimensionException) */;

	/** Second parameter constructor */
	CenteredFiniteDifferenceHessian(const NumericalScalar epsilon,
					const EvaluationImplementation & p_evaluation)
	  /* throw(InvalidDimensionException) */;


	/* Virtual constructor */
	virtual CenteredFiniteDifferenceHessian * clone() const;

	/* Comparison operator */
	Bool operator ==(const CenteredFiniteDifferenceHessian & other) const;

	/* String converter */
	virtual String __repr__() const;



	/* Here is the interface that all derived class must implement */
	
        /** This method computes the hessian at some point
	 * @param in The point where the hessian is computed
	 * @result A tensor
	 */
	virtual SymmetricTensor hessian(const NumericalPoint & in) const 
          /* throw(InvalidArgumentException, InternalException) */;

        /** Accessor for input point dimension
	 * @return The size of the point passed to the gradient method
	 */
	virtual UnsignedLong getInputDimension() const
          /* throw(InternalException) */;

        /** Accessor for output point dimension
	 * @return The size of the point returned by the function whose hessian is computed
	 */
	virtual UnsignedLong getOutputDimension() const
	  /* throw(InternalException) */;

	/** Accessor for the epsilon */
	NumericalPoint getEpsilon() const;

	/** Accessor for the evaluation */
	EvaluationImplementation getEvaluation() const;

 	/* Method save() stores the object through the StorageManager */
	void save(StorageManager::Advocate & adv) const;

	/* Method load() reloads the object from the StorageManager */
	void load(StorageManager::Advocate & adv);

      protected:

    
      private:
	NumericalPoint epsilon_;
	EvaluationImplementation p_evaluation_;
      }; /* class CenteredFiniteDifferenceHessian */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_CENTEREDFINITEDIFFERENCEHESSIAN_HXX */
