/* NVClock 0.5.0 - Linux overclocker for NVIDIA cards
 *
 * site: http://nvclock.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "backend.h"

static int base_freq;
int root = 0;

static struct pci_ids ids[] = 
{ { 0x20, "nVidia Riva TnT", 75, 135, 75, 135, 1 },
{ 0x28, "nVidia Riva TnT 2 Pro", 125, 200, 125, 175, 1 },
{ 0x2a, "nVidia Riva TnT 2", 100, 175, 100, 150, 1 },
{ 0x2b, "nVidia Riva TnT 2", 100, 175, 100, 175, 1 },
{ 0xa0, "nVidia Riva TnT 2 Aladdin (Integrated)", 100, 175, 100, 175, 0 },
{ 0x2c, "nVidia Riva TnT 2 VANTA", 100, 175, 100, 175, 1 },
{ 0x2d, "nVidia Riva TnT 2 M64", 100, 175, 100, 175, 1 },
{ 0x2e, "nVidia Riva TnT 2 VANTA", 100, 175, 100, 175, 1 },
{ 0x2f, "nVidia Riva TnT 2 VANTA", 100, 175, 100, 175, 1 },	    
{ 0x29, "nVidia Riva TnT 2 Ultra", 150, 225, 150, 225, 1 },
{ 0x100, "nVidia Geforce 256 SDR", 150, 200, 100, 150, 1 },
{ 0x101, "nVidia Geforce 256 DDR", 250, 350, 100, 150, 1 },
{ 0x103, "nVidia Quadro", 250, 350, 100, 150, 1 },
{ 0x110, "nVidia Geforce 2 MX/MX400", 150, 250, 150, 250, 1 },
{ 0x111, "nVidia Geforce 2 MX 100/200", 250, 350, 125, 250, 1 },
{ 0x112, "nVidia Geforce 2 GO", 0, 0, 0, 0, 0 },
{ 0x113, "nVidia Quadro 2 GO (Mobile)", 0, 0, 0, 0, 0 },
{ 0x1a0, "nVidia intergrated Geforce 2 MX (nForce)", 0, 0, 0, 0, 0 },
{ 0x150, "nVidia Geforce 2 GTS/PRO", 250, 450, 175, 250, 1 },
{ 0x151, "nVidia Geforce 2 Ti", 300, 500, 175, 275, 1 },
{ 0x152, "nVidia Geforce 2 Ultra", 400, 550, 200, 275, 1 },
{ 0x153, "nVidia Quadro 2 Pro", 250, 400, 150, 250, 1 },
{ 0x170, "nVidia Geforce 4 MX460", 450, 600, 275, 350, 1 },
{ 0x171, "nVidia Geforce 4 MX440", 350, 500, 250, 325, 1 },
{ 0x172, "nVidia Geforce 4 MX420", 150, 250, 225, 300, 1 },
{ 0x174, "nVidia Geforce 4 440 Go", 0, 0, 0, 0, 0 },
{ 0x175, "nVidia Geforce 4 420 Go", 0, 0, 0, 0, 0 },
{ 0x176, "nVidia Geforce 4 420 Go 32M", 0, 0, 0, 0, 0 },
{ 0x178, "nVidia Quadro 4 500 XGL", 150, 600, 150, 350, 1 }, // needs to be adjusted when the real speeds are known
{ 0x179, "nVidia Geforce 4 440 Go 64M", 0, 0, 0, 0, 0 },
{ 0x17a, "nVidia Quadro 4 200/400 NVS", 150, 600, 150, 350, 1 }, // needs to be adjusted when the real speeds are known
{ 0x17b, "nVidia Quadro 4 550 XGL", 150, 600, 150, 350, 1 }, // needs to be adjusted when the real speeds are known
{ 0x17c, "nVidia Quadro 4 GoGL", 0, 0, 0, 0, 0 },
{ 0x200, "nVidia Geforce 3", 350, 550, 175, 275, 1 },
{ 0x201, "nVidia Geforce 3 Titanium 200", 350, 500, 175, 275, 1 },
{ 0x202, "nVidia Geforce 3 Titanium 500", 450, 600, 200, 300, 1 },
{ 0x203, "nVidia Quadro DCC", 350, 550, 175, 275, 1 },
{ 0x250, "nVidia Geforce 4 Ti 4600", 625, 750, 300, 400, 1 },
{ 0x251, "nVidia Geforce 4 Ti 4400", 450, 600, 275, 375, 1 },
{ 0x253, "nVidia Geforce 4 Ti 4200", 350, 550, 225, 325, 1 },
{ 0x258, "nVidia Quadro 4 900 XGL", 450, 750, 200, 400, 1 }, // needs to be adjusted when the real speeds are known
{ 0x259, "nVidia Quadro 4 750 XGL", 450, 650, 200, 400, 1 }, // needs to be adjusted when the real speeds are known
{ 0x25b, "nVidia Quadro 4 700 XGL", 450, 600, 200, 400, 1 }, // needs to be adjusted when the real speeds are known
{ 0x2a0, "nVidia Xbox GPU", 0, 0, 0, 0, 0 },
{ 0, NULL, 0, 0, 0, 0, 0 }
};


/* Get the name of the card */
void GetCardName(int device_id, int number)
{
    struct pci_ids *nv_ids;
    
    nv_ids = ids;
    while(nv_ids->id != 0)
    {
	if(nv_ids->id == device_id)
	{
	    card[number].device_string = (char*)strdup(nv_ids->name);
	    card[number].memclk_min = nv_ids->mem_min;
	    card[number].memclk_max = nv_ids->mem_max;
	    card[number].nvclk_min = nv_ids->nv_min;
	    card[number].nvclk_max = nv_ids->nv_max;
	    card[number].supported = nv_ids->supported;
	    return;
	}

	nv_ids++;
    }	

    card[number].memclk_min = 0;
    card[number].memclk_max = 0;
    card[number].nvclk_min = 0;
    card[number].nvclk_max = 0;
    card[number].supported = 0;
    card[number].device_string = (char*)strdup("Unknown nVidia device");
}

/* Check if we are using the closed source Nvidia drivers */
int CheckDriver()
{
    FILE *proc;
    char buffer[80];
    int i = 0;
    
    proc = fopen("/proc/modules", "r");
    
    if(proc  == NULL) 
    {
	printf("Can't open /proc/modules\n");
    }

    else
    {
        while(fgets(buffer, 80, proc) != NULL)
	{	
	    if((strstr(buffer, "NVdriver") != 0))
	    {
		return 1;
	    }
	}	
    }
    fclose(proc);

    return 0;
}

/* Detect all nvidia cards using ioctls. Requires the closed source nVidia drivers for Linux or FreeBSD.*/
int FindAllCards_ioctl()
{
    int	i, fd, error, test = 0;

    root = 0;

    fd = open("/dev/nvidiactl", O_RDWR, 0);
    if (fd < 0) {
    printf("error: %d\n", error);
    	err(1, "%s", "/dev/nvidiactl"); 
	exit(1);
    }

    error = ioctl(fd, NVIDIA_CARD_INFO, card);
    if (error) {
    printf("error: %d\n", error);
    	err(1, "%s", "/dev/nvidiactl"); 
	exit(1);
    }

    close(fd);

    for (i = 0; i < MAX_CARDS; i++)
    {
        if ((card[i].flags & NVIDIA_IOCTL_CARD_INFO_FLAG_PRESENT) == 0)
        {
	    i--;
	    break;
	}

	card[i].number = i;
	card[i].mem = 0;
	GetCardName(card[i].device_id, i);	
    }

    return(i);
}

/* Detect all nvidia cards using /proc/bus/pci/devices. This only works on Linux. */
int FindAllCards_proc()
{
    int dev, dum, mem, irq, i = -1;
    char buf[0x1000];

    FILE *proc;
    root = 1;

    proc = fopen("/proc/bus/pci/devices", "r");
    if (!proc) return 0;


    while (fgets(buf, sizeof(buf)-1, proc)) 
    {
	if (sscanf(buf,"%x %x %x %x",&dum, &dev, &irq, &mem) != 4) continue;

	if( (dev>>16) == 0x10de)
        {
	    i++;
	    card[i].device_id = (0x0000ffff & dev);
	    card[i].number = i;
	    card[i].vendor_id = ((0xffff0000 & dev)>>16);
	    card[i].mem = mem;
	    card[i].irq = irq;
		    
	    /*  Get the name of the card and the min and max speeds.*/
	    GetCardName(card[i].device_id, i);	
    	}
    }

    fclose(proc);

    return(i);
}

//*****************************************************************************

const struct pci_ids *GetCardInfo(int number)
{
    int id;
    struct pci_ids *nv_ids;
    nv_ids = ids;
    
    if(number < 0 || number > MAX_CARDS) return(NULL);

    id = card[number].device_id;
    while(nv_ids->id != 0)
    {
        if(id == nv_ids->id) return(nv_ids);
        nv_ids++;
    }
    return(NULL);
}
