#!/usr/bin/perl -w

package CEInfo;

use File::Basename;
use Getopt::Long;
use XML::Simple;
use Sys::Hostname;
use Data::Dumper;
use Cwd;

use strict;

# Some optional features that may be unavailable in older Perl versions.
# Should work with Perl v5.8.0 and up.
BEGIN {
    # Makes sure the GLUE document is valid UTF8
    eval {binmode(STDOUT, ":utf8")};
    # Used for reading UTF8 encoded grid-mapfile
    eval {require Encode; import Encode "decode"};
    # Fall back to whole-second precision if not avaiable
    eval {require Time::HiRes; import Time::HiRes "time"};
}

BEGIN {
    my $pkglibdir = dirname($0).'/../../lib/arc';
    unshift @INC, $pkglibdir;
}

########################################################
# Driver for information collection
# Reads old style arc.config and prints out XML
########################################################

use ConfigCentral;
use LogUtils; 

use HostInfo;
use RTEInfo;
use GMJobsInfo;
use LRMSInfo;

use ARC0ClusterInfo;
use ARC1ClusterInfo;

use InfoChecker;

our $log = LogUtils->getLogger(__PACKAGE__);
our $configfile;

sub timed {
    my ($fname, $func, @args) = @_;
    my $t0 = time();
    my $result = &$func(@args);
    my $dt = sprintf('%.3f', time() - $t0);
    $log->debug("Time spent in $fname: ${dt}s");
    return $result;
}


sub main {

    LogUtils::level('INFO');

    # Config defaults
    
    my %config_defaults = (
                   arcversion     => '1.1.0rc6',
                   gm_mount_point => "/jobs",
                   gm_port        => 2811,
                   ttl            =>  60,
                   defaultttl     => "604800",
                   x509_user_cert => "/etc/grid-security/hostcert.pem",
                   x509_cert_dir  => "/etc/grid-security/certificates/",
                   gridmap        => "/etc/grid-security/grid-mapfile",
                   processes      => [qw(arched)]
    
    );


    # Parse command line options
    
    my $print_help;
    GetOptions("config:s" => \$configfile,
               "help|h"   => \$print_help ); 
        
    if ($print_help) { 
        print "Usage: $0 <options>
        --config   - location of arc.conf
        --help     - this help\n";
        exit 1;
    }
    
    unless ( $configfile ) {
        $log->error("a command line argument is missing, see --help ");
    }
    
    # Read ARC configuration

    my $config = timed 'ConfigCentral', \&ConfigCentral::parseConfig, $configfile;
    $config = { %config_defaults, %$config};

    # Change level for root logger (affects all loggers from now on)
    LogUtils::level($config->{debugLevel}) if defined $config->{debugLevel};

    fix_config($config);
    check_config($config);

    # Collect information & print XML

    my $ceinfo = timed 'all info collectors', \&CEInfo::collect, $config;

    my $xml = new XML::Simple(NoAttr => 0, ForceArray => 1, RootName => 'InfoRoot', KeyAttr => ['name']);
    print timed('printing XML', \&XML::Simple::XMLout, $xml, $ceinfo);
}
    

##################################################
# Unified information collector for NG and GLUE2
##################################################

sub collect($) {
    my ($config) = @_;

    # get all local users from grid-map. Sort unique
    my %saw = ();
    my $usermap = read_grid_mapfile($config->{gridmap});
    my @localusers = grep !$saw{$_}++, values %$usermap;


    my $gmjobs_info = get_gmjobs_info($config);

    my @jobids;
    for my $job (values %$gmjobs_info) {
        next unless $job->{status} and $job->{status} eq 'INLRMS';
        next unless defined $job->{localid} and length $job->{localid};
        push @jobids, $job->{localid};
    }

    my $data = {};
    $data->{config} = $config;
    $data->{usermap} = $usermap;
    $data->{host_info} = get_host_info($config,\@localusers);
    $data->{rte_info} = get_rte_info($config);
    $data->{gmjobs_info} = $gmjobs_info;
    $data->{lrms_info} = get_lrms_info($config,\@localusers,\@jobids);

    fix_adotf($config, $data->{host_info});
    fix_adotf($_, $data->{host_info}) for values %{$config->{xenvs}};

    my $ceinfo;

    $ceinfo->{'Domains'}->{'xmlns'} = "http://schemas.ogf.org/glue/2008/05/spec_2.0_d41_r01";
    $ceinfo->{'Domains'}->{'xmlns:xsi'} = "http://www.w3.org/2001/XMLSchema-instance";
    $ceinfo->{'Domains'}->{'xsi:schemaLocation'} = "http://schemas.ogf.org/glue/2008/05/spec_2.0_d41_r01 pathto/GLUE2.xsd";
    $ceinfo->{'Domains'}->{'AdminDomain'}->{'xmlns'} = "http://schemas.ogf.org/glue/2008/05/spec_2.0_d41_r01";
    $ceinfo->{'Domains'}->{'AdminDomain'}->{'Services'}->{'xmlns'} = "http://schemas.ogf.org/glue/2008/05/spec_2.0_d41_r01";
    $ceinfo->{'Domains'}->{'AdminDomain'}->{'Services'}->{'ComputingService'} = timed 'ARC1ClusterInfo', \&ARC1ClusterInfo::collect, $data;

    my $png = $config->{PublishNordugrid};
    unless (defined $png and $png eq 'false') {
        $ceinfo->{'n:nordugrid'} = timed 'ARC0ClusterInfo', \&ARC0ClusterInfo::collect, $data;
        $ceinfo->{'n:nordugrid'}{'xmlns:n'} = "urn:nordugrid-cluster";
    }

    return $ceinfo;
}


##################################################
# Calling other information collectors
##################################################

sub get_host_info($$) {
    my ($config,$localusers) = @_;

    my $host_opts = Storable::dclone($config);
    $host_opts->{localusers} = $localusers;
    $host_opts->{configfile} = $configfile;

    return timed 'HostInfo', \&HostInfo::collect, $host_opts;
}

sub get_rte_info($) {
    my ($config) = @_;

    my $rte_opts;
    $rte_opts->{configfile} = $configfile;
    $rte_opts->{runtimedir} = $config->{runtimedir} if $config->{runtimedir};
    $rte_opts->{JanitorEnabled} = $config->{JanitorEnabled} if $config->{JanitorEnabled};
    $rte_opts->{pkgdatadir} = dirname($0).'/../../share/arc';

    return timed 'RTEInfo', \&RTEInfo::collect, $rte_opts;
}

sub get_lrms_info($$$) {
    my ($config,$localusers,$jobids) = @_;

    # possibly any options from config are needed, so just clone it all
    my $lrms_opts = Storable::dclone($config);
    delete $lrms_opts->{$_} for qw(control xenvs shares);
    $lrms_opts->{jobs} = $jobids;
    for my $share ( keys %{$config->{shares}} ) {
        $lrms_opts->{queues}{$share} = $config->{shares}{$share};
        $lrms_opts->{queues}{$share}{users} = $localusers;
    }

    return timed 'LRMSInfo', \&LRMSInfo::collect, $lrms_opts;
}

sub get_gmjobs_info($) {
    my $config = shift;
    $log->error("controldir option missing") unless $config->{controldir};
    my $gmjobs_info = timed 'GMJobsInfo', \&GMJobsInfo::collect, $config->{controldir};

    my ($lrms, $defaultshare) = split /\s+/, $config->{lrms} || '';
    for my $jobid (keys %$gmjobs_info) {
        my $job = $gmjobs_info->{$jobid};
        my $share = $job->{share};

        # If A-REX has not choosen a share for the job, default to one.
        if (not $share) {
            my $msg = "A-REX has not choosen a share for job $jobid";
            if ($defaultshare) {
                $log->info($msg.". Assuming default: ".$defaultshare);
                $share = $defaultshare;
            } else {
                my @shares = keys %{$config->{shares}};
                if (@shares == 1) {
                    $log->info($msg.". Assuming: ".$shares[0]);
                    $share = $shares[0];
                } else {
                    $log->warning($msg." and no default share is defined.");
                }
            }
        }

        # Set correct queue
        if ($share) {
            my $sconfig = $config->{shares}{$share};
            if ($sconfig) {
                $job->{queue} = $sconfig->{MappingQueue} || $share;
            } else {
                $log->warning("Job $jobid belongs to an invalid share '$share'");
                $share = undef;
            }
        }

		# Group jobs not belonging to any known share into a catch-all share named ''
        $job->{share} = $share || '';
    }
    return $gmjobs_info;
}

#### reading grid-mapfile #####

sub read_grid_mapfile($) {
    my $gridmapfile = shift;
    my $usermap = {};

    unless (open MAPFILE, "<$gridmapfile") {
        $log->warning("can't open gridmapfile at $gridmapfile");
        return;
    }
    while (my $line = <MAPFILE>) {
        chomp($line);
        if ( $line =~ m/\"([^\"]+)\"\s+(\S+)/ ) {
            my $subject = $1;
            eval {
                $subject = decode("utf8", $subject, 1);
            };
            $usermap->{$subject} = $2;
        }
    }
    close MAPFILE;

    return $usermap;
}


sub fix_config {
    my ($config) = @_;

    $config->{control} ||= {};
    $config->{service} ||= {};
    $config->{shares} ||= {};
    $config->{xenvs} ||= {};

    delete $config->{location} unless $config->{location} and %{$config->{location}};
    delete $config->{contacts} unless $config->{contacts} and @{$config->{contacts}};

    my $hostname = $config->{hostname} || hostname();
    {  
        my @dns = split /\./, $hostname;
        my $shorthost = shift @dns;
        my $dnsdomain = join ".", @dns;

        $log->info("AdminDomain config option is missing. Defaulting to: $dnsdomain") unless $config->{AdminDomain};
        $log->info("ClusterName config option is missing. Defaulting to: $shorthost") unless $config->{service}{ClusterName};

        chomp ($config->{AdminDomain} ||= $dnsdomain);
        chomp ($config->{service}{ClusterName} ||= $shorthost);
    }

    if ($config->{endpoint} and $config->{endpoint} =~ m{^(https?)://([^:/]+)(?::(\d+))?}) {
        my ($proto,$host,$port) = ($1,$2,$3);
        $port ||= 80 if $proto eq "http";
        $port ||= 443 if $proto eq "https";
        $config->{arexhostport} = "$host:$port";
    } else {
        $config->{endpoint} = "https://$hostname/arex";
        $config->{arexhostport} = "$hostname:443";
        $log->warning("Config option endpoint (or arex_mount_point) is missing or not a valid URL. Assuming: ".$config->{endpoint});
    }

    # deal with GM per-user options: select one user, ignore others
    my @gmusers = keys %{$config->{control}};
    $log->fatal("No control directory configured") unless @gmusers;
    my $control = $config->{control}{'.'}
               || $config->{control}{$gmusers[0]};
    $log->fatal("No control directory configured") unless $control->{controldir};
    $log->fatal("No session directory configured") unless $control->{sessiondir};
    %$config = (%$config, %$control);
}


sub check_config {
    my ($config) = @_;

    # Cross-check ExecutionEnvironment references
    for my $s (values %{$config->{shares}}) {
        next unless $s->{ExecEnvName};
        for my $group (@{$s->{ExecEnvName}}) {
            $log->error("No such ExecutionEnvironment: $group")
                unless $config->{xenvs}{$group};
        }
    }
    for my $s (values %{$config->{xenvs}}) {
        delete $s->{NodeSelection} unless %{$s->{NodeSelection}};
    }

    my ($lrms, $defaultshare) = split /\s+/, $config->{lrms} || '';
    $log->error("defaultShare set to nonexistent ComputingShare")
        if $defaultshare and not $config->{shares}{$defaultshare};

    if ($config->{contacts}) {
        for (@{$config->{contacts}}) {
            $log->warning("URI expected in Contact Detal: ".$_->{Detail}) unless $_->{Detail} =~ m/^\w+:/;
        }
    }
}


sub fix_adotf {
    my ($h, $hostinfo) = @_;
    if ($h->{nodecpu}) {
        if ($h->{nodecpu} =~ m/(.*?)(?:\s+stepping\s+(\d+))?\s+@\s+([.\d]+)\s*(M|G)Hz$/i) {
            $h->{CPUModel} ||= $1;
            $h->{CPUVersion} ||= $2;
            $h->{CPUClockSpeed} ||= ($4 eq 'G') ? int($3 * 1000) : int($3);
        } elsif ($h->{nodecpu} eq 'adotf') {
            $h->{CPUVendor} ||= 'adotf';
            $h->{CPUModel} ||= 'adotf';
            $h->{CPUClockSpeed} ||= 'adotf';
        } else {
            $log->warning("Invalid value for nodecpu option: ".$h->{nodecpu});
        }
        delete $h->{nodecpu};
    }
    if ($h->{OpSys} and grep {$_ eq 'adotf'} @{$h->{OpSys}}) {
        $h->{OpSys} = [ grep {$_ ne 'adotf'} @{$h->{OpSys}} ];
        $log->error("Failed to autodetect value for 'OSName'. Enter correct value in config file")
            unless defined $hostinfo->{osname};
        $h->{OSName} ||= 'adotf';
        $h->{OSVersion} ||= 'adotf';
        $h->{OSFamily} ||= 'adotf';
    }
    my %hostkey = (Platform => 'machine',
                   PhysicalCPUs => 'cpusocketcount',
                   LogicalCPUs => 'cputhreadcount',
                   CPUVendor => 'cpuvendor',
                   CPUModel => 'cpumodel',
                   CPUClockSpeed => 'cpufreq',
                   MainMemorySize => 'pmem',
                   VirtualMemorySize => 'vmem',
                   OSFamily => 'sysname',
                   OSName => 'osname',
                   OSVersion => 'osversion'
    );
    for my $key (keys %hostkey) {
        if (exists $h->{$key} and $h->{$key} eq 'adotf') {
            $log->error("Failed to autodetect value for '$key'. Enter correct value in config file")
                unless defined $hostinfo->{$hostkey{$key}};
            $h->{$key} = $hostinfo->{$hostkey{$key}};
        }
    }
}

main();
