/***************************************************************************
                          gnunode.cpp  -  description
                             -------------------
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

// the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/ioctl.h>
#include <fcntl.h>

#include <ctype.h>

#include "mutella.h"

#include "preferences.h"
#include "gnushare.h"

#include "gnuupload.h"

#include "gnucache.h"
#include "gnuhash.h"
#include "gnudirector.h"
#include "gnunode.h"
#include "event.h"

#ifdef THIS_FILE
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// MGnuNode

MGnuNode::MGnuNode(MGnuDirector* pComm, CString Host, UINT Port, bool bInbound, bool bForceV4)
{
	m_pDirector  = pComm;
	m_pPrefs = m_pDirector->GetPrefs();
	m_pCache = m_pDirector->GetCache();
	m_pShare = m_pDirector->GetShare();

	m_dwExtraLength  = 0;

	// Node Info
	m_nStatus	= SOCK_CONNECTING;
	m_nSecsTrying   = 0;
	m_nSecsDead		= 0;

	m_sHost		= Host;
	m_ipHost	= StrtoIP(Host);
	m_nPort		= Port;
	m_bInbound  = bInbound;
	m_dwVersion = 4;
	m_bHaveReceivedPongs = false;
	m_bHaveReceivedPings = false;
	m_dwUptime	= xtime();
	
	m_dwID = 0;
	
	m_bForceV4 = bForceV4;
	
	int i;
	for(i = 0; i < MAX_TTL_SELF; i++)
	{
		m_dwFriends[i]		= 0;
		m_dwLibrary[i]		= 0;
		m_dwFiles[i]		= 0;
	}

	m_dwFriendsTotal = 0;
	m_llLibraryTotal = 0;
	m_dwFilesTotal = 0;
	m_dwSharingTotal = 0;
	// bandwidth and other stuff
	// Bandwidth, [0] is Received, [1] is Sent
	m_nTotalPackets[0] = 0;
	m_nTotalPackets[1] = 0;
	m_nllTotalBytes[0] = 0;
	m_nllTotalBytes[1] = 0;
		
	for(i = 0; i < 2; i++)
		for(int j = 0; j < 60; j++)
		{
			m_dwAvgPackets[i][j] = 0;
			m_dwAvgBytes[i][j] = 0;
		}
		
	m_nSecPackets[0] = 0;
	m_nSecPackets[1] = 0;
	m_nSecBytes[0] = 0;
	m_nSecBytes[1] = 0;
	m_dRate[0] = 0;
	m_dRate[1] = 0;
	
	m_nBadPackets = 0;
	m_nRoutingErrors = 0;
	m_nLoopBackErrors = 0;
	m_nDuplicatePackets = 0;
	m_nDroppedPackets = 0;
	m_nSendQueueSize = 0;
	
	m_nSecCounter = 0;	
	
	// buffers
	m_pReceiveBuffer = new BYTE[65536];
	m_pExtraBuffer = new BYTE[65536];
	
	m_pDirector->NodeMessage(SOCK_UPDATE, 0);
}

MGnuNode::~MGnuNode()
{
	ASSERT(m_pDirector);
	ASSERT(!m_pDirector->IsInTheList(this));
	MLock lock(m_sendQueueMutex);
	while (m_sendQueue.size())
	{
		delete [] m_sendQueue.front().pData;
		m_sendQueue.pop();
	}
	delete [] m_pReceiveBuffer;
	delete [] m_pExtraBuffer;
}

int MGnuNode::Send(const void* lpBuf, int nBufLen, int nFlags = 0)
{
	ASSERT(nFlags == 0);
	MLock lock(m_sendQueueMutex);
	// dont send packets, only enqueue them
	if (m_sendQueue.size()>=256)
	{
		// drop
		m_nDroppedPackets++;
		return nBufLen;
	}
	QueuedPacket qp;
	qp.size = nBufLen;
	qp.pData = new char[nBufLen];
	memcpy(qp.pData, lpBuf, nBufLen);
	m_sendQueue.push(qp);
	// ensure that OnSend event will be polled
	ModifySelectFlags(FD_WRITE,0);
	return nBufLen;
}

void MGnuNode::OnSend(int nErrorCode)
{
	MLock lock(m_sendQueueMutex);
	// how to measure ammount of bytes to send?
	int nRes;
	int olderrno = 0;
	errno = 0;
	int nError;
	socklen_t nSize;

#ifdef _DEBUG
	int nFlags = fcntl(m_hSocket,F_GETFL,0);
	ASSERT(nFlags == -1 || (O_NONBLOCK & nFlags));
#endif
	
	while (m_sendQueue.size())
	{
		nRes = MAsyncSocket::Send(m_sendQueue.front().pData, m_sendQueue.front().size);
		
		if (olderrno!=errno)
		{
			//printf("OnSend error changed: %d, was %d\n",errno,olderrno);
			Close();
			return;
		}
		olderrno = errno;
		// addidional checks
		nSize = sizeof(nError);	
		getsockopt(m_hSocket, SOL_SOCKET,SO_ERROR,(char*)&nError,&nSize);
		ASSERT(nSize == sizeof(int));
		if (nRes != m_sendQueue.front().size || nError)
			if (nError==EAGAIN || nError==EWOULDBLOCK)
			{
				TRACE4("wouldblock, nRes=",nRes," size=",m_sendQueue.front().size);
				return;
			}
			else
			{
				//printf("OnSend: nError=%d, errno=%d\n", nError, errno);
				Close();
				return;
			}
		delete [] m_sendQueue.front().pData;
		m_pDirector->m_dwConnBytesSend += m_sendQueue.front().size;
		m_nSecBytes[1] += m_sendQueue.front().size;
		m_nSecPackets[1]++;

		m_sendQueue.pop();
	}
	// save cpu on callbacks when we have nothing to send
	ModifySelectFlags(0, FD_WRITE);
}

/////////////////////////////////////////////////////////////////////////////
// MGnuNode member functions

void MGnuNode::OnConnect(int nErrorCode)
{
	//if(nErrorCode)
	//	return;
	// Get localhost
	UINT    Port;
	CString strHost;
	GetSockName(strHost, Port);
	IP Host = StrtoIP(strHost);
	m_pPrefs->m_ipLocalHost = Host;
	

	// Get Remote host
	CString HostIP;
	GetPeerName(HostIP, m_nPort);
	m_sHost = HostIP;
	m_ipHost = StrtoIP(HostIP);
	// Send handshake
	Send_Hello(!m_bForceV4);
	//
	m_nSecsTrying = 0;
	m_nStatus = SOCK_NEGOTIATING;

	MAsyncSocket::OnConnect(nErrorCode);
}

void MGnuNode::OnReceive(int nErrorCode)
{
	int dwOffset = 0, dwBuffLength = 0;
	BYTE* pDataToSplit;

	// Add any unprocessed BYTEs from last receive
	if(m_dwExtraLength)
	{
		ASSERT(m_dwExtraLength>0);
		ASSERT(m_dwExtraLength<=65536);
		//memcpy(m_pReceiveBuffer, m_pExtra, m_dwExtraLength);
		swap(m_pReceiveBuffer, m_pExtraBuffer);
		dwOffset = m_dwExtraLength;
		m_dwExtraLength = 0;
	}

	dwBuffLength = Receive(m_pReceiveBuffer+dwOffset, 65536 - dwOffset);
    //printf("node recieved %d bytes\n", dwBuffLength);
	// Handle Errors
	switch (dwBuffLength)
	{
	case 0:
		//printf("on receive: no data. errno=%d\n", errno);
		ForceDisconnect();
		return;
	case SOCKET_ERROR:
		ForceDisconnect();
		return;
	}
	dwBuffLength += dwOffset;
	pDataToSplit = m_pReceiveBuffer;

	if(m_nStatus == SOCK_NEGOTIATING)
	{
		// we are in handshake mode
		CString Data((char*) m_pReceiveBuffer, dwBuffLength);
		if(!ParseHandshake(Data, m_pReceiveBuffer, dwBuffLength))
		{
			ASSERT(m_nStatus != SOCK_CONNECTED);
			if( dwBuffLength > 4096)
			{
				ForceDisconnect();
				return;
			}
			else
			{
				swap(m_pReceiveBuffer, m_pExtraBuffer);
				m_dwExtraLength = dwBuffLength;
			}
		}
	} else if(m_nStatus == SOCK_CONNECTED)
	{
		m_dwExtraLength = SplitBundle(pDataToSplit, dwBuffLength);
		ASSERT(m_dwExtraLength <= dwBuffLength);
		if (0 != m_dwExtraLength)
		{
			if(65536 > m_dwExtraLength)
			{
				if (dwBuffLength == m_dwExtraLength && pDataToSplit == m_pReceiveBuffer)
					swap(m_pReceiveBuffer, m_pExtraBuffer);
				else
				{
					ASSERT(dwBuffLength<=65536);
					memcpy(m_pExtraBuffer, pDataToSplit + (dwBuffLength - m_dwExtraLength), m_dwExtraLength); //TODO: check bounds!!
				}
			}
			else
			{
				TRACE("OnReceive: failed to split the bundle");
				ForceDisconnect();
				return;
			}
		}
	}

	// Bandwidth stats
	m_nSecBytes[0] += dwBuffLength;
	m_pDirector->m_dwConnBytesRecv += dwBuffLength;
	
	MAsyncSocket::OnReceive(nErrorCode);
}

void MGnuNode::OnClose(int nErrorCode)
{
	m_nStatus = SOCK_CLOSED;
	m_pDirector->NodeMessage(SOCK_UPDATE, 0);

	m_pDirector->RemoveNode(this);

	MAsyncSocket::OnClose(nErrorCode);
}

/////////////////////////////////////////////////////////////////////////////
// New connections

#define NEW_CONNECT_STRING "GNUTELLA CONNECT/0."
#define OLD_CONNECT_STRING "GNUTELLA CONNECT/0.4"
#define NEW_HANDSHAKE_TERM "\r\n\r\n"
#define OLD_HANDSHAKE_TERM "\n\n"
#define NEW_CONNECT_REPLY1 "GNUTELLA/0."
#define NEW_CONNECT_REPLY2 " 200 OK\r\n"
#define OLD_CONNECT_REPLY  "GNUTELLA OK"
#define SELF_VERSION       "6"

bool MGnuNode::ParseHandshake(const CString& Data, BYTE* pStream, int nStreamLength)
{
	int i,n;
	CString sHandshake;
	CString s;

	// Version 6 or later
	if((i=Data.find(NEW_HANDSHAKE_TERM)) != -1)
	{
		i += strlen(NEW_HANDSHAKE_TERM);
		sHandshake = Data.substr(0, i);
		m_sHandshake += sHandshake;
		// Connect string, GNUTELLA CONNECT/0.x\r\n
		if(m_bInbound && ((n=sHandshake.find(NEW_CONNECT_STRING)) != -1))
		{
			m_dwVersion = sHandshake[n+strlen(NEW_CONNECT_STRING)]-'0';
			//TRACE4("incomming connection from version ", m_dwVersion, " IP:",m_sHost);
			Send_ConnectOK(true, false);
			return true;
		}
		// Ok string, GNUTELLA/0.6 200 OK\r\n
		else if((n=sHandshake.find(NEW_CONNECT_REPLY1)) != -1)
		{
			n += strlen(NEW_CONNECT_REPLY1);
			m_dwVersion = sHandshake[n]-'0';
			//TRACE4("accepting connection with version ", m_dwVersion, " IP:",m_sHost);
			s = sHandshake.substr(n+1);
			if (s.find(NEW_CONNECT_REPLY2) != -1)
			{
				if (m_bInbound)
				{
					// Stream begins
					ASSERT(i <= nStreamLength);
					m_dwExtraLength = nStreamLength - i;
					ASSERT(65536>=m_dwExtraLength);
					memcpy(m_pExtraBuffer, pStream + i, m_dwExtraLength);
					//
					//TRACE("connection accepted");
				}
				else
				{
					Send_ConnectOK(true, true);
				}
				SetConnected();
				return true;
			}
		}
	}
	// Version 4 ?
	else if((i=Data.find(OLD_HANDSHAKE_TERM)) != -1)
	{
		i += strlen(OLD_HANDSHAKE_TERM);
		sHandshake = Data.substr(0, i);
		m_sHandshake += sHandshake;
		if(m_bInbound && (sHandshake.find(OLD_CONNECT_STRING) != -1))
		{
			Send_ConnectOK(false, false);
			SetConnected();
			return true;
		}
		else if((!m_bInbound) && (sHandshake.find(OLD_CONNECT_REPLY) != -1))
		{
			// Stream begins
			ASSERT(i <= nStreamLength);
			m_dwExtraLength = nStreamLength - i;
			ASSERT(65536>=m_dwExtraLength);
			memcpy(m_pExtraBuffer, pStream + i, m_dwExtraLength);
			SetConnected();
			return true;
		}
	}
	return false;
}

void MGnuNode::SetConnected()
{
	// Get Local hostname
	UINT    Port;
	CString strHost;
	GetSockName(strHost, Port);
	IP Host = StrtoIP(strHost);
	m_pPrefs->m_ipLocalHost = Host;

	// Get Remote host
	GetPeerName(m_sHost, Port);
	m_ipHost = StrtoIP(m_sHost);
	
	if (m_bInbound)
	{
		// If too may nodes are connected, send list of pongs
		if(m_pDirector->CountConnections() > m_pPrefs->m_nMaxConnects ||
		   (m_pPrefs->m_nMaxConnPerSubnetA > 0 && m_pDirector->CountConnPerSubnet(8, m_ipHost) > m_pPrefs->m_nMaxConnPerSubnetA) ||
		   (m_pPrefs->m_nMaxConnPerSubnetB > 0 && m_pDirector->CountConnPerSubnet(16,m_ipHost) > m_pPrefs->m_nMaxConnPerSubnetB) ||
		   (m_pPrefs->m_nMaxConnPerSubnetC > 0 && m_pDirector->CountConnPerSubnet(24,m_ipHost) > m_pPrefs->m_nMaxConnPerSubnetC) )
		{
			Send_HostCache(); // this also disconnects the node
			return;
		}
	}

	m_nStatus = SOCK_CONNECTED;
	//m_pComm->NodeMessage(SOCK_UPDATE, NULL);

	Send_Ping(MAX_TTL_SELF);
}

void MGnuNode::Send_Hello(bool bVersion6)
{
	CString sHandshake;
	if (bVersion6)
	{
		CString sMuteVersion = VERSION;
		//
		sHandshake = NEW_CONNECT_STRING SELF_VERSION NEW_CONNECT_REPLY2;
		sHandshake += CString("Node: ") +
		              IPtoStr(m_pPrefs->m_ipForcedHost.S_addr ? m_pPrefs->m_ipForcedHost : m_pPrefs->m_ipLocalHost) + ":" +
		              DWrdtoStr(m_pPrefs->m_dwForcedPort ? m_pPrefs->m_dwForcedPort : m_pPrefs->m_dwLocalPort) + "\r\n";
		sHandshake += CString("User-Agent: Mutella-") + sMuteVersion + "\r\n";
		sHandshake += "\r\n";
	}
	else
	{
		sHandshake += OLD_CONNECT_STRING OLD_HANDSHAKE_TERM;
	}
	Send(sHandshake.c_str(), sHandshake.length());
	m_sHandshake += sHandshake;
}

//"Node: %s\r\n"
//"User-Agent: %s\r\n"
//"Pong-Caching: 0.1\r\n"
//"X-Live-Since: %s\r\n"

void MGnuNode::Send_ConnectOK(bool bVersion6, bool bReply)
{
	CString sHandshake;
	CString sMuteVersion = VERSION;

	if(bVersion6 && bReply)
	{
		sHandshake = NEW_CONNECT_REPLY1 SELF_VERSION NEW_CONNECT_REPLY2;
		sHandshake += "\r\n";
	}
	else if(bVersion6 && !bReply)
	{
		sHandshake = NEW_CONNECT_REPLY1 SELF_VERSION NEW_CONNECT_REPLY2;
		sHandshake += CString("User-Agent: Mutella-") + sMuteVersion + "\r\n";
		sHandshake += "\r\n";
	}
	else
	{
		sHandshake = OLD_CONNECT_REPLY OLD_HANDSHAKE_TERM;
	}
	Send(sHandshake.c_str(), sHandshake.length());

	m_sHandshake += sHandshake;
}

/*void MGnuNode::Recieve_ConnectOK()
{
	m_nStatus = SOCK_CONNECTED;
	m_pDirector->NodeMessage(SOCK_UPDATE, 0);
	Send_Ping(MAX_TTL_SELF);
	//m_pDirector->SendAllLocalQueries(this);
}*/


////////////////////////////
// packet.handlers

DWORD MGnuNode::SplitBundle(BYTE* bundle, DWORD length)
{
	DWORD Payload = 0;
	DWORD nextPos = 0;

	packet_Header* packet;

	enum status { status_DONE,       status_CONTINUE,
				  status_BAD_PACKET, status_INCOMPLETE_PACKET };

	status theStatus = status_CONTINUE;
	
	do
	{
		if (nextPos + sizeof(packet_Header) > length)
			theStatus = status_INCOMPLETE_PACKET;
		else
		{
			packet = (packet_Header*) (bundle + nextPos);
			Payload = packet->le_Payload;
		
			if((packet->Function == 0x00 && Payload ==  0) ||
			   (packet->Function == 0x01 && Payload == 14) ||
			   (packet->Function == 0x40 && Payload == 26) ||
			   (packet->Function == 0x80 && Payload >  2 && Payload < 256) ||
			   (packet->Function == 0x81 && Payload > 26 && Payload < 65536))
			{
				if (nextPos + sizeof(packet_Header) + Payload <= length)
				{
					HandlePacket(packet, sizeof(packet_Header) + Payload);
					
					nextPos += sizeof(packet_Header) + Payload;
					if (nextPos == length)
						theStatus = status_DONE;
				}
				else
				{
					theStatus = status_INCOMPLETE_PACKET;
					if (sizeof(packet_Header) + Payload > 65536)
					{
						TRACE("SplitBundle: too big packet has arrived");
						ForceDisconnect();
					}
				}
			}
			else
			{
		        /*if (nextPos < length - sizeof(packet_Header))
					nextPos++;
		        else*/
				theStatus = status_BAD_PACKET;
				ForceDisconnect();
				//TRACE("SplitBundle: sync lost");
			}
		}
	} while(status_CONTINUE == theStatus);


	return length - nextPos;
}

void MGnuNode::HandlePacket(packet_Header* packet, DWORD length)
{
	m_nSecPackets[0]++;

	switch(packet->Function)
	{
	case 0x00:
		Recieve_Ping((packet_Ping*) packet, length);
		break;

	case 0x01:
		Recieve_Pong((packet_Pong*) packet, length);
		break;

	case 0x40:
		Recieve_Push((packet_Push*) packet, length);
		break;

	case 0x80:
		Recieve_Query((packet_Query*) packet, length);
		break;

	case 0x81:
		Recieve_QueryHit((packet_QueryHit*) packet, length);
		break;

	default:
		Recieve_Unknown((BYTE *) packet, length);
		break;
	}
}

bool MGnuNode::InspectPacket(packet_Header* packet)
{
	// check if the packet should be dead by now
	if(packet->TTL == 0)
		return false;
	if (packet->Hops > MAX_TTL_ACCEPTED)
    	return false;
	// Increment hops
	packet->Hops++;
	packet->TTL--;
	// If packet has hopped more than MAX_TTL_SELF times kill it
	if(packet->Hops > MAX_TTL_SELF)
		return false;	
    // Bad TTL or turned over TTL
    if (packet->TTL > MAX_TTL_ACCEPTED)
    	return false;
	// Reset TTL of packet if it needs to be done
	if(packet->TTL > MAX_TTL_SELF - packet->Hops)
		packet->TTL = MAX_TTL_SELF - packet->Hops;
	return true;
}


/////////////////////////////////////////////////////////////////////////////
// Receiving packets


void MGnuNode::Recieve_Ping(packet_Ping* Ping, int nLength)
{
	//printf("Receive Ping\n");
	// part of the hanbshake procedure
	if (!m_bHaveReceivedPings && Ping->Header.Hops == 0)
	{
		m_bHaveReceivedPings = true;
		if ( Ping->Header.Guid.a[8] == 0xFF)
		{
			m_dwVersion = 5+Ping->Header.Guid.a[15];
		}
	}

	// Inspect TTL & Hops
	if(!InspectPacket((packet_Header*) Ping))
	{
		m_nBadPackets++;
		return;
	}

	key_Value* key = m_pDirector->m_TablePing.FindValue(&Ping->Header.Guid);
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&Ping->Header.Guid);

	if(LocalKey)
	{
		m_nLoopBackErrors++;
		return;
	}

	// Fresh Ping?
	if(key == NULL)
	{
		m_pDirector->m_TablePing.Insert(&Ping->Header.Guid, this);
		if (m_pPrefs->m_nMaxConnects < 0 || m_pDirector->GetLastConnCount() <= m_pPrefs->m_nMaxConnects * 0.8)
			Send_Pong(Ping->Header.Guid, Ping->Header.Hops);
		// Broadcast if still alive
		if(Ping->Header.TTL)
		{
			m_pDirector->Broadcast_Ping(Ping, nLength, this);
			return;
		}
	}
	else
	{
		if(key->Origin == this)
		{
			m_nDuplicatePackets++;
			return;
		}
		else
		{
			m_nRoutingErrors++;
			return;
		}
	}
}


void MGnuNode::Recieve_Pong(packet_Pong* Pong, int nLength)
{
	//printf("Receive Pong\n");
	// Host Cache
	Node NetNode;
	NetNode.Host		= Pong->Host;
	NetNode.Port		= Pong->le_Port;
	NetNode.Ping		= 0;
	NetNode.Speed		= 0;
	NetNode.Distance	= Pong->Header.Hops;
	NetNode.ShareCount  = Pong->le_FileCount;
	NetNode.ShareSize	= Pong->le_FileSize;
	m_pCache->UpdateCache(NetNode);
	// part of the hanbshake procedure
	if (!m_bHaveReceivedPongs && Pong->Header.Hops == 0 && Pong->Host.S_addr == m_ipHost.S_addr)
	{
		m_bHaveReceivedPongs = true;
		m_nPort = Pong->le_Port;
	}

	// Inspect
	if(!InspectPacket((packet_Header*) Pong))
	{
		m_nBadPackets++;
		return;
	}

	key_Value* key = m_pDirector->m_TablePing.FindValue(&Pong->Header.Guid);
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&Pong->Header.Guid);

	if(LocalKey)
	{
		// If Pong GUID matches a GUID of a Ping we sent out, its for us
		// Else, its a loop back error
		
		m_dwFriendsTotal++;
		ASSERT(Pong->Header.Hops <= MAX_TTL_SELF);
		m_dwFriends[Pong->Header.Hops - 1]++;

		DWORD dwLibrary = Pong->le_FileSize;
		DWORD dwFiles = Pong->le_FileCount;
		if (dwLibrary>MAX_SHARE)
			dwLibrary = MAX_SHARE;
		if(dwLibrary)
		{
			m_dwLibrary[Pong->Header.Hops - 1] += dwLibrary;
			m_dwFiles[Pong->Header.Hops - 1] += dwFiles;
			m_llLibraryTotal += dwLibrary;
			m_dwFilesTotal += dwFiles;
			m_dwSharingTotal++;
		}
		return;
	}
	
	if(key)
	{	
		// Send it out
		if(Pong->Header.TTL)
			m_pDirector->Route_Pong(Pong, nLength, key);
		return;
	}
	else
	{
		m_nRoutingErrors++;
		return;
	}
}

void MGnuNode::Recieve_Push(packet_Push* Push, int nLength)
{
	// Host Cache
	Node NetNode;
	NetNode.Host		= Push->Host;
	NetNode.Port		= Push->le_Port;
	NetNode.Ping		= 0;
	NetNode.Speed		= 0;
	NetNode.Distance	= Push->Header.Hops;
	NetNode.ShareCount  = 0;
	NetNode.ShareSize	= 0;
	m_pCache->UpdateCache(NetNode);

	// Inspect
	if(!InspectPacket((packet_Header*) Push))
	{
		m_nBadPackets++;
		return;
	}

	// Find packet in hash tables
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&Push->Header.Guid);
	key_Value* PushKey  = m_pDirector->m_TablePush.FindValue(&Push->ServerID);

	if(LocalKey)
	{
		m_nBadPackets++;
		return;
	}

	// Check ServerID of Push with ClientID of the client
	if(*m_pDirector->GetClientID() == Push->ServerID)
	{
		// Make sure not already pushing file
		if (m_pDirector->CheckIfPushung(Push))
		{
			m_nDuplicatePackets++; // not sure if it is appropriate
			return;
		}
		
		if (m_pPrefs->m_bQuietMode)
			return;
        TRACE("processing push request");

		// Create upload
		MGnuUpload* Upload = new MGnuUpload(m_pDirector);
		Upload->m_ipHost     = Push->Host;
		Upload->m_nPort      = Push->le_Port;
		Upload->m_nFileIndex = Push->le_Index;
		
		if(!Upload->Create(0, SOCK_STREAM, FD_READ | FD_CONNECT | FD_CLOSE)) // FD_WRITE shouldn't harm here
		{
			POST_ERROR(ES_UNIMPORTANT, CString("Upload Create Error: ") + DWrdtoStr(Upload->GetLastError()));
		}

		// Add the node to the linked list
		m_pDirector->AddUpload(Upload);
		Upload->PushFile();
		return;
	}

	if(PushKey)
	{
		if(Push->Header.TTL)
			m_pDirector->Route_Push(Push, nLength, PushKey);
		return;
	}
	else
	{
		m_nRoutingErrors++;
		return;
	}
}

void MGnuNode::Recieve_Query(packet_Query* Query, int nLength)
{
	// Inspect
	if(!InspectPacket((packet_Header*) Query))
	{
		m_nBadPackets++;
		return;
	}
	
	key_Value* key = m_pDirector->m_TableQuery.FindValue(&Query->Header.Guid);
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&Query->Header.Guid);

	if(LocalKey)
	{
		m_nLoopBackErrors++;
		return;
	}

	// Fresh Query?
	if(key == NULL)
	{
		m_pDirector->m_TableQuery.Insert(&Query->Header.Guid, this);
		m_pDirector->m_nQGood++;
		// spam protection
		int nAlpha = 0;
		int nLen = MIN(Query->Header.le_Payload - 2, MAX_QUERY_LEN);
		char* pSearch = (char*) Query+25;
		for (int i=0;i<nLen;++i)
			if (isalnum(pSearch[i]))
				++nAlpha;
		if (nAlpha<4 || nAlpha < nLen/2)
		{
			// spam
			return;
		}
		// Broadcast if still alive
		if(Query->Header.TTL)
			m_pDirector->Broadcast_Query(Query, nLength, this);

		// Queue to be compared with local files
		if(Query->le_Speed <= GetOurSpeed())
		{
			// WATCH OUT!!! IT CRASHED HERE ONCE
			QueryComp* pCompare = new QueryComp;
			pCompare->Origin    = this;
			pCompare->nHops     = Query->Header.Hops;
			pCompare->QueryGuid = Query->Header.Guid;
			
			memcpy(pCompare->Query,(char*)pSearch,nLen);
			pCompare->Query[nLen]=0;
			
			if( ((!m_pDirector->GetPrefs()->m_bSendOnlyAvail) || // dont check quieries against the file list if busy
			     m_pDirector->GetPrefs()->m_nMaxUploads < 0   ||
			     m_pDirector->CountUploading() < m_pDirector->GetPrefs()->m_nMaxUploads ) )
			{
				m_pShare->AppendQuery(pCompare);
			}
			else
				delete pCompare;
		}
		return;
	}
	else
	{
		if(key->Origin == this)
		{
			//printf("R_Q: duplicate packet, conflict with a packet from %d sec before\n   query=`%s'\n", time(NULL)-key->time,((char*)Query)+25);
			m_nDuplicatePackets++;
			m_pDirector->m_nQRepeat++;
			return;
		}
		else
		{
			//printf("R_Q: error routing, conflict with a packet from %d sec before\n   query=`%s'\n", time(NULL)-key->time,((char*)Query)+25);
			m_nRoutingErrors++;
			m_pDirector->m_nQError++;
			return;
		}
	}
}


void MGnuNode::Recieve_QueryHit(packet_QueryHit* QueryHit, DWORD dwLength)
{
	// Host Cache
	Node NetNode;
	NetNode.Host		= QueryHit->Host;
	NetNode.Port		= QueryHit->le_Port;
	NetNode.Ping		= 0;
	NetNode.Speed       = QueryHit->le_Speed;
	NetNode.Distance	= QueryHit->Header.Hops;
	NetNode.ShareCount  = 0;
	NetNode.ShareSize   = 0;

	m_pCache->UpdateCache(NetNode);
	
	// Inspect
	if(!InspectPacket((packet_Header*) QueryHit))
	{
		m_nBadPackets++;
		//
		// still makes sence to check what has been found
		m_pDirector->OnQueryHit(QueryHit, this, false);
		return;
	}

	key_Value* key = m_pDirector->m_TableQuery.FindValue(&QueryHit->Header.Guid);
	key_Value* LocalKey = m_pDirector->m_TableLocal.FindValue(&QueryHit->Header.Guid);

	if(LocalKey)
	{
		// If QueryHit GUID matches a GUID of a Query we sent out its for us
		m_pDirector->OnQueryHit(QueryHit, this, true);
		return;
	}
	else
	{
		// but we will be little bit nosy and check what the others find
		m_pDirector->OnQueryHit(QueryHit, this, false);
	}

	if(key)
	{	
		// Send it out
		if(QueryHit->Header.TTL)
		{
			// Add ClientID of packet to push table
			m_pDirector->m_TablePush.Insert( (GUID*) ((BYTE*)QueryHit + (dwLength - 16)), this);
			m_pDirector->Route_QueryHit(QueryHit, dwLength, key);
			return;
		}
		else
		{
			m_nBadPackets++;
			return;
		}
	}
	else
	{
		m_nRoutingErrors++;
		return;
	}
}


void MGnuNode::Recieve_Unknown(BYTE*, DWORD dwLength)
{
	m_nBadPackets++;
}


/////////////////////////////////////////////////////////////////////////////
// Sending packets

void MGnuNode::Send_Ping(int TTL)
{
	//printf("Send Ping, TTL=%d\n", TTL);
	
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
	{
		POST_ERROR(ES_UNIMPORTANT, "Failed to create a GUID to send a ping");
		return;
	}

	packet_Ping Ping;
	
	Ping.Header.Guid = Guid;
	Ping.Header.Function = 0;
	Ping.Header.Hops = 0;
	Ping.Header.TTL = TTL;
	Ping.Header.le_Payload = 0;

	// TTL of 1 means we are checking if node is responding only
	if(TTL != 1)
		m_pDirector->m_TableLocal.Insert(&Guid, NULL);

	Send(&Ping, 23);
}

void MGnuNode::Send_Pong(GUID Guid, int nHops)
{
    //printf("Send Pong\n");
	// Build the packet
	packet_Pong Pong;

	Pong.Header.Guid		= Guid;
	Pong.Header.Function	= 0x01;
	Pong.Header.TTL			= nHops;
	Pong.Header.Hops		= 0;
	Pong.Header.le_Payload	= 14;

	Pong.le_Port			= m_pPrefs->m_dwForcedPort ? m_pPrefs->m_dwForcedPort : m_pPrefs->m_dwLocalPort;
	Pong.Host				= m_pPrefs->m_ipForcedHost.S_addr ? m_pPrefs->m_ipForcedHost : m_pPrefs->m_ipLocalHost;

	Pong.le_FileCount		= m_pShare->m_dwTotalFiles;
	Pong.le_FileSize		= m_pShare->m_dwTotalSize;

	Send(&Pong, 37);
}

void MGnuNode::Send_QueryHit(QueryComp* pQuery, BYTE* pQueryHit, DWORD &ReplyLength, BYTE &ReplyCount)
{
	int packetLength = 34 + ReplyLength + sizeof(packet_QueryHitEx) + 16;

	packet_QueryHit*  QueryHit = (packet_QueryHit*) pQueryHit;
	packet_QueryHitEx QHD;

	// Build Query Packet
	QueryHit->Header.Guid	    = pQuery->QueryGuid;
	QueryHit->Header.Function   = 0x81;
	QueryHit->Header.TTL	    = pQuery->nHops;
	QueryHit->Header.Hops	    = 0;
	QueryHit->Header.le_Payload = packetLength - 23;

	QueryHit->TotalHits		    = ReplyCount;
	QueryHit->le_Port		    = (WORD) m_pPrefs->m_dwForcedPort ? m_pPrefs->m_dwForcedPort : m_pDirector->GetLocalPort();
	QueryHit->Host			    = m_pPrefs->m_ipForcedHost.S_addr ? m_pPrefs->m_ipForcedHost : m_pPrefs->m_ipLocalHost;
	QueryHit->le_Speed		    = GetOurSpeed();

	// Add Query Hit Descriptor
	bool Busy = false;
	if(m_pPrefs->m_nMaxUploads >= 0)
		if(m_pDirector->CountUploading() >= m_pPrefs->m_nMaxUploads)
			Busy = true;

	strcpy((char*) QHD.VendorID, "MUTE");
	QHD.Length		= 2;
	QHD.FlagPush	= 0;//true;
	QHD.FlagBad		= 1;//true;
	QHD.FlagBusy	= 1;//true;
	QHD.FlagStable	= 1;//true;
	QHD.FlagSpeed	= 1;//true;
	QHD.FlagTrash   = 0;
	QHD.Push		= m_pDirector->BehindFirewall();
	QHD.Bad			= 0;//false;
	QHD.Busy		= 0;//false;//Busy;
	QHD.Stable		= 1;//false;//true;//m_pDirector->HaveUploaded();
	QHD.Speed		= 0;//false;//true;//m_pDirector->GetRealSpeed() ? true : false;
	QHD.Trash		= 0;
	//QHD.bug1 = 0xcc;
	//QHD.bug2= 0xcc;

	int pos = 34 + ReplyLength;
	pQueryHit += pos;

	memcpy(pQueryHit, &QHD, sizeof(packet_QueryHitEx)); //TODO: check bounds!!!
	pQueryHit -= pos;

	// Add ID of this node
	pos += sizeof(packet_QueryHitEx);
	pQueryHit += pos;
	memcpy(pQueryHit, m_pDirector->GetClientID(), 16);
	pQueryHit -= pos;

	// Send the packet	
	Send(pQueryHit,packetLength);
}

void MGnuNode::Send_HostCache()
{
	m_pCache->SendCache(this, 20);
	//ForceDisconnect();
	m_nStatus = SOCK_CLOSED; // this delays disconnection to the next OnTimer event
}

void MGnuNode::Send_Host(Node Host)
{
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
		return;

	// Build the packet
	packet_Pong Pong;

	Pong.Header.Guid		= Guid;
	Pong.Header.Function	= 0x01;
	Pong.Header.TTL			= 1;
	Pong.Header.Hops		= Host.Distance;
	Pong.Header.le_Payload	= 14;

	Pong.le_Port			= Host.Port;
	Pong.Host				= Host.Host;

	Pong.le_FileCount		= Host.ShareCount;
	Pong.le_FileSize		= Host.ShareSize;

	Send(&Pong, 37);
}

/////////////////////////////////////////////////////////////////////////////
// Misc functions


DWORD MGnuNode::GetOurSpeed()
{	
	if(m_pPrefs->m_dwSpeedStat)
		return m_pPrefs->m_dwSpeedStat;
	else
		return m_pPrefs->m_dwSpeedDyn;
}

void MGnuNode::ForceDisconnect()
{
	//TRACE4("closing connection with version ", m_dwVersion, " IP:",m_sHost);
	if(m_hSocket != INVALID_SOCKET)
	{
		AsyncSelect(FD_CLOSE);
		ShutDown(2);
	}
	Close();
	if (!m_bForceV4 && !m_bInbound && SOCK_NEGOTIATING == m_nStatus)
		m_pDirector->AddNode(m_sHost, m_nPort, true);
	m_nStatus = SOCK_CLOSED;
	m_pDirector->NodeMessage(SOCK_UPDATE, 0);
}

void MGnuNode::OnTimer()
{
	m_nSecCounter++;
	int nHistCounter = m_nSecCounter % 60;
	int nLastSecInPackets = m_nSecPackets[0];
	// When i = 0 its receive stats
    //      i = 1 its send stats
	// Easier to read without the [i]'s, but oh well
	for(int i = 0; i < 2; i++)
	{
		// Packets
		m_nTotalPackets[i] += m_nSecPackets[i];
		m_dwAvgPackets[i][nHistCounter] = m_nSecPackets[i];
		// Bytes
		m_nllTotalBytes[i] += m_nSecBytes[i];
		//m_dRate[i] += 1.0/60.0*(m_nSecBytes[i] - m_dwAvgBytes[i][nHistCounter]); //didnt work
		m_dwAvgBytes[i][nHistCounter] = m_nSecBytes[i];
		//m_nSecBytes[i] = 0;
		m_dRate[i] = (1.0*m_nllTotalBytes[i])/m_nSecCounter;
		// Packets
		m_nSecPackets[i] = 0;
		// Bytes
		m_nSecBytes[i] = 0;

	}
	
	// calculate avg. SendQueueSize by weithting prev value with the current one
	m_nSendQueueSize *= 0x0f;
	m_nSendQueueSize += m_sendQueue.size();
	m_nSendQueueSize >>= 4;	
	
	if(SOCK_CLOSED == m_nStatus)
		ForceDisconnect();

	NodeManagement(nLastSecInPackets);
}

void MGnuNode::NodeManagement(int nLastSecPackets)
{
	if(SOCK_CONNECTING == m_nStatus || SOCK_NEGOTIATING == m_nStatus)
	{
		m_nSecsTrying++;
		
		if(m_nSecsTrying > CONNECT_TIMEOUT)
		{
			m_pDirector->RemoveNode(this);
			return;
		}
	}
	else if(SOCK_CONNECTED == m_nStatus)
	{
		// Drop if not transfering anything
		if(nLastSecPackets == 0)
		{
			m_nSecsDead++;

			if(m_nSecsDead == 30)
				Send_Ping(1);

			if(m_nSecsDead > 120)
			{
				m_pDirector->RemoveNode(this);
				return;
			}
		}
		else
			m_nSecsDead = 0;

		// Drop if Minimum friends isnt met
		if(m_pPrefs->m_nFriendsMin > 0)
		{
			if(m_nSecCounter>60 && m_dwFriendsTotal < m_pPrefs->m_nFriendsMin)
			{
				m_pDirector->RemoveNode(this);
				return;
			}
		}
		// once a second send a host
		Node nd;
		if (m_pCache->GetRandomNode(nd))
			Send_Host(nd);
		// once a minute send a ping to estimate the horizon
		if (m_nSecCounter % 60 == 30) // tricky offset of 30 secs. see min.friends condition above
			Send_Ping(MAX_TTL_SELF);
	}
	else
	{
		m_pDirector->RemoveNode(this);
		return;
	}
}

