# copyright (C) 1997-2001 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: memstats.tcl,v 1.28 2001/01/27 15:08:19 jfontain Exp $}


package provide memstats [lindex {$Revision: 1.28 $} 1]
package require network 1

namespace eval memstats {

    array set data {
        updates 0
        0,label type 0,type ascii 0,message {type} 0,0 memory 1,0 swap
        1,label available 1,type integer 1,message {total available (in kilobytes)}
        2,label used 2,type integer 2,message "used (in kilobytes)\n(= total - free - buffers - cached)"
        3,label free 3,type integer 3,message {free (in kilobytes)}
        4,label shared 4,type integer 4,message {shared memory (in kilobytes)}
        5,label buffers 5,type integer 5,message {memory used for buffers (in kilobytes)}
        6,label cached 6,type integer 6,message {cached memory (in kilobytes)}
        views {{indices {0 1 2 3 4 5 6} swap 1}}
        switches {-r 1 --remote 1}
    }
    array set data {0,6 {} 1,4 {} 1,5 {}}                                                                 ;# initialize static cells
    set file [open memstats.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar $optionsName options
        variable remote
        variable data
        variable memoryFile
        variable extended

        if {![catch {set locator $options(--remote)}]||![catch {set locator $options(-r)}]} {                   ;# remote monitoring
            set data(pollTimes) {20 10 30 60 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(protocol) $remote(user) $remote(host)
            set data(identifier) memstats($remote(host))
            set file [open "| /usr/bin/$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/version"]
            fileevent $file readable {set ::memstats::remote(busy) 0}
            vwait ::memstats::remote(busy)                                                                  ;# allow GUI interaction
        } else {
            set data(pollTimes) {10 5 20 30 60 120 300 600}
            set file [open /proc/version]
            set memoryFile [open /proc/meminfo]                                       ;# keep local file open for better performance
        }
        regexp {^[\d\.]+} [lindex [gets $file] 2] version                          ;# ignore extra characters, such as in 2.2.0-pre1
        if {[info exists remote]} {
            # detect errors early (but avoid write on pipe with no readers errors by reading whole data)
            if {[catch {read $file} message]||[catch {close $file} message]} {
                error "on remote host $remote(host) as user $remote(user): $message"
            }
        } else {
            close $file
        }
        if {[package vcompare $version 2.4]>=0} {                                                                      ;# kernel 2.4
            array set data {
                7,label active 7,type integer 7,message {active memory (in kilobytes)} 1,7 {}
                8,label dirty 8,type integer 8,message {inactive dirty memory (in kilobytes)} 1,8 {}
                9,label clean 9,type integer 9,message {inactive clean memory (in kilobytes)} 1,9 {}
                10,label target 10,type integer 10,message {inactive target memory (in kilobytes)} 1,10 {}
                11,label {high total} 11,type integer 11,message {(in kilobytes)} 1,11 {}
                12,label {high free} 12,type integer 12,message {(in kilobytes)} 1,12 {}
                13,label {low total} 13,type integer 13,message {(in kilobytes)} 1,13 {}
                14,label {low free} 14,type integer 14,message {(in kilobytes)} 1,14 {}
                views {{indices {0 1 2 3 4 5 6 7 8 9 10 11 12 13 14} swap 1}}
            }
            set extended {}
        } else {                                                                                                ;# previous versions
            set data(views) {{indices {0 1 2 3 4 5 6} swap 1}}
        }
    }

    proc update {} {                                               ;# gather cpu statistics (based on the proc man page information)
        variable remote
        variable memoryFile
        variable data
        variable extended

        if {[info exists remote]} {
            if {![info exists memoryFile]} {                               ;# start data gathering process in a non blocking fashion
                if {$remote(busy)} return                                           ;# core invocation while waiting for remote data
                set remote(busy) 1
                set file [open "| /usr/bin/$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/meminfo"]
                # do not hang GUI, allow other modules updates
                fileevent $file readable "set ::memstats::memoryFile $file; ::memstats::update"
                return                                                                                       ;# wait for remote data
            }                                                                                 ;# else continue below to process data
        } else {
            seek $memoryFile 0                                                                      ;# rewind before retrieving data
        }
        gets $memoryFile; gets $memoryFile; gets $memoryFile                                         ;# skip redundant first 3 lines
        while {[gets $memoryFile line]>=0} {
            foreach {string number} $line break                                              ;# format example: "MemTotal: 62668 kB"
            set value($string) $number
        }
        if {[info exists remote]} {                                                 ;# closing is necessary since seek does not work
            read $memoryFile                                 ;# avoid write on pipe with no readers errors by reading remaining data
            if {[catch {close $memoryFile} message]} {                                   ;# communication error can be detected here
                flashMessage "memstats error: $message"
                set error {}
            }
            unset memoryFile
            set remote(busy) 0
        }
        if {[info exists error]} {                                                                    ;# collected data is corrupted
            array set data {0,1 ? 0,2 ? 0,3 ? 0,4 ? 0,5 ? 1,1 ? 1,2 ? 1,3 ? 1,6 ?}
            if {[info exists extended]} {
                array set data {0,7 ? 0,8 ? 0,9 ? 0,10 ? 0,11 ? 0,12 ? 0,13 ? 0,14 ?}
            }
        } else {                                                                                          ;# whole data is available
            set data(0,1) $value(MemTotal:)                                                                          ;# total memory
            set data(0,3) $value(MemFree:)                                                                            ;# free memory
            set data(0,4) $value(MemShared:)                                                                        ;# shared memory
            set data(0,5) $value(Buffers:)                                                                                ;# buffers
            set data(1,6) $value(Cached:)                                                                           ;# cached memory
            set data(1,1) $value(SwapTotal:)                                                                           ;# total swap
            set data(1,3) $value(SwapFree:)                                                                             ;# free swap
            set data(0,2) [expr {$data(0,1)-$data(0,3)-$data(0,5)-$data(1,6)}]      ;# used memory = total - free - buffers - cached
            set data(1,2) [expr {$data(1,1)-$data(1,3)}]                                                                ;# used swap
            if {[info exists extended]} {
                set data(0,7) $value(Active:)
                set data(0,8) $value(Inact_dirty:)
                set data(0,9) $value(Inact_clean:)
                set data(0,10) $value(Inact_target:)
                set data(0,11) $value(HighTotal:)
                set data(0,12) $value(HighFree:)
                set data(0,13) $value(LowTotal:)
                set data(0,14) $value(LowFree:)
            }
        }
        incr data(updates)
    }
}
