# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Document window.

# load libs
import os
import wx

# load modules
from nucleus.ids import *
from modules.mdoc.main import docData
from nucleus.dlg_tolerance import dlgSetTolerance

from modules.mpeak.main import mPeak
from modules.mspec.main import mSpec
from modules.mseq.main import mSeq
from modules.mcut.main import mCut
from modules.mfrag.main import mFrag
from modules.mdiff.main import mDiff
from modules.mcomp.main import mComp
from modules.mproc.main import mProc

from modules.mdocinfo.main import mDocInfo
from modules.mmatchinfo.main import mMatchInfo
from modules.mmascot.main import mMascotSearch
from modules.mconv.main import mConv

from modules.mprint.spectrum import mSpectrumPrinter
from modules.mprint.report import mReportPrinter

from modules.mexport.document import mDocumentExporter
from modules.mexport.spectrum import mSpectrumExporter
from modules.mexport.peaklist import mPeaklistExporter
from modules.mexport.image import mImageExporter


class docWindow(wx.SplitterWindow):
    """ Document window. """

    # ----
    def __init__(self, parent):

        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docFrame = parent

        # initialize window slitter
        wx.SplitterWindow.__init__(self, parent, -1, style=wx.CLIP_CHILDREN|wx.SP_NOBORDER)
        self.SetMinimumPaneSize(170)
        if wx.Platform == "__WXMAC__":
            self.SetSashSize(8)
        self.Show(False)

        # initialize document data structure
        self.docData = docData()
        self.docData.setMassParam('masstype', self.config.cfg['common']['masstype'])
        self.docData.setMassParam('errortype', self.config.cfg['common']['errortype'])
        self.docData.setMassParam('tolerance', self.config.cfg['common']['tolerance'])

        # initialize main modules
        self.mPeak = mPeak(self, self.docData)
        self.mSpec = mSpec(self, self.docData)
        self.mSpec.Show(False)
        self.mSeq = mSeq(self, self.docData)
        self.mSeq.Show(False)
        self.mCut = mCut(self, self.docData)
        self.mCut.Show(False)
        self.mFrag = mFrag(self, self.docData)
        self.mFrag.Show(False)
        self.mDiff = mDiff(self, self.docData)
        self.mDiff.Show(False)
        self.mComp = mComp(self, self.docData)
        self.mComp.Show(False)
        self.mProc = mProc(self, self.docData)

        # split window to peaklist and module
        self.moduleFrame = self.mSpec
        if self.config.cfg['mpeak']['splitting'] == 'h':
            self.SplitHorizontally(self.moduleFrame, self.mPeak, -170)
            self.SetSashGravity(1.0)
            self.sashPosition = -170
        else:
            self.SplitVertically(self.mPeak, self.moduleFrame, 170)
            self.SetSashGravity(0.0)
            self.sashPosition = 170
        if not self.config.cfg['mpeak']['showpeaklist']:
            self.Unsplit(self.mPeak)

        # initialize printer
        self.mSpectrumPrinter = mSpectrumPrinter(self)
        self.mReportPrinter = mReportPrinter(self)

        # show document window
        self.Show(True)
        self.docFrame.SendSizeEvent()
        wx.SafeYield()
    # ----


    # ----
    def setDocumentData(self, data):
        """ Set data to document. """

        # set data to document
        if data:
            self.docData.setPath(data['path'])
            self.docData.setDocType(data['docType'])
            self.docData.setScanID(data['scanID'])
            self.docData.setSpectrum(data['spectrum'])
            self.docData.setPeaklist(data['peaklist'], undo=False)
            self.docData.setSequence(data['sequence'])
            self.docData.setModifications(data['modifications'])
            self.docData.setSequenceTitle(data['seqtitle'])
            self.docData.setDescription('operator', data['operator'])
            self.docData.setDescription('contact', data['contact'])
            self.docData.setDescription('institution', data['institution'])
            self.docData.setDescription('date', data['date'])
            self.docData.setDescription('instrument', data['instrument'])
            self.docData.setDescription('notes', data['notes'])

            # check applied modifications
            if data['modifications'] and not self.checkAppliedModifications():
                message = "Some of the used modifications are not available and were removed from the sequence!\nCheck sequence editor please."
                dlg = wx.MessageDialog(self.docFrame, message, "Unknown Modification Found", wx.OK|wx.ICON_EXCLAMATION)
                dlg.ShowModal()
                dlg.Destroy()

            # update all modules
            self.mSpec.resetModule()
            self.mPeak.resetModule()
            self.mSeq.resetModule()

            # revert change status
            self.docData.setChangeStatus(False)
    # ----


    # ----
    def onFilePrintSpectrum(self):
        """ Print spectrum. """

        # get data
        title = self.docData.getPath()
        title = os.path.split(title)[1]
        pageSetup = self.mSpectrumPrinter.getPageSetup()
        printout = self.mSpec.getPrintOut(pageSetup, title)

        # print spectrum
        self.mSpectrumPrinter.onPrint(printout)
    # ----


    # ----
    def onFilePrintReport(self):
        """ Print sample report. """
        self.mReportPrinter.onPrint(self.docData)
    # ----


    # ----
    def onFilePageSetup(self):
        """ Setup page before printing. """

        self.mSpectrumPrinter.onPageSetup()
        pageSetup = self.mSpectrumPrinter.getPageSetup()
        self.mReportPrinter.setPageSetup(pageSetup)
    # ----


    # ----
    def onFileSave(self, saveAs=False):
        """ Save document data to file. """

        # set application working
        self.docMonitor('setAppStatus', "Saving data...")

        # init exporter and save document data to file
        lastDir = self.config.cfg['common']['lastspecdir']
        exporter = mDocumentExporter(self.docFrame)
        path = exporter.exportData(self.docData, lastDir=lastDir, saveAs=saveAs)
        wx.SafeYield()

        # check if file was saved as mMass document (path is received for mMass docs only)
        if path:
            self.docData.setDocType('mSD')
            self.docData.setPath(path)
            self.docData.setChangeStatus(False)

            # update application
            self.docMonitor('onDocumentSaved')
            self.docMonitor('setAppStatus', 0)
            return path

        else:
            # set application ready
            self.docMonitor('setAppStatus', 0)
            return False
    # ----


    # ----
    def onFileInfo(self):
        """ Show information about current document. """

        # raise info dialog
        dlg = mDocInfo(self.docFrame, self.docData)

        # get dialog data
        if dlg.ShowModal() == wx.ID_OK:
            data = dlg.getDialogData()
            dlg.Destroy()

            # set new data to document
            self.docData.setDescription('date', data['date'])
            self.docData.setDescription('operator', data['operator'])
            self.docData.setDescription('contact', data['contact'])
            self.docData.setDescription('institution', data['institution'])
            self.docData.setDescription('instrument', data['instrument'])
            self.docData.setDescription('notes', data['notes'])

            # update application
            self.docMonitor('onDocumentChanged')

        else:
            dlg.Destroy()
    # ----


    # ----
    def onFileImportPeaklist(self, data):
        """ Import peaklist data to current document. """

        self.docData.setPeaklist(data)
        self.docMonitor('onPeaklistChanged', 'import')
    # ----


    # ----
    def onFileImportSequence(self, data):
        """ Import sequence data to current document. """

        # ask to rewrite current data if any
        if self.docData.getDataStatus('mSeq'):
            dlg = wx.MessageDialog(self, "Rewrite current sequence?", "Import Sequence", wx.YES_NO|wx.ICON_QUESTION|wx.NO_DEFAULT)
            if dlg.ShowModal() != wx.ID_YES:
                dlg.Destroy()
                self.docMonitor('setAppStatus', 0)
                return

        # update document
        self.docData.setSequenceTitle(data['title'])
        self.docData.setSequence(data['sequence'])
        self.docData.setModifications(data['modifications'])

        # check modifications
        if data['modifications'] and not self.checkAppliedModifications():
            message = "Some of the used modifications are not available and were removed from the sequence!\nCheck the sequence editor please."
            dlg = wx.MessageDialog(self, message, "Unknown Modification Found", wx.OK|wx.ICON_EXCLAMATION)
            dlg.ShowModal()
            dlg.Destroy()

        # update application
        self.docMonitor('setAppStatus', 0)
        self.docMonitor('onSequenceChanged')
        self.onToolSwitch('mSeq')

        # parse sequence and update info
        self.mSeq.resetModule()
    # ----


    # ----
    def onFileExportSpectrumData(self):
        """ Export spectrum to ASCII file. """

        # set application working
        self.docMonitor('setAppStatus', "Exporting spectrum data...")

        # init exporter and save data
        lastDir = self.config.cfg['common']['lastspecdir']
        exporter = mSpectrumExporter(self.docFrame)
        exporter.exportData(self.docData, lastDir=lastDir)

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onFileExportSpectrumImage(self):
        """ Export spectrum to image file. """

        # set application working
        self.docMonitor('setAppStatus', "Exporting spectrum image...")

        # init exporter and save data
        lastDir = self.config.cfg['common']['lastspecdir']
        exporter = mImageExporter(self)
        exporter.exportData(self.docData, self.mSpec, lastDir=lastDir)

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onFileExportPeaklist(self):
        """ Export peaklist to ASCII file. """

        # set application working
        self.docMonitor('setAppStatus', "Exporting peaklist data...")

        # init exporter and save data
        lastDir = self.config.cfg['common']['lastspecdir']
        exporter = mPeaklistExporter(self.docFrame, self.config.cfg['mpeak'])
        exporter.exportData(self.docData, lastDir=lastDir)

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onFileClose(self):
        """ Ask to save unsaved data and close current document. """

        # check 'changed status' and ask to close current document
        if self.docData.getChangeStatus():

            message = "Document contains unsaved data. Do you want to save the document?"
            dlg = wx.MessageDialog(self, message, "Save Document", wx.YES_NO|wx.CANCEL|wx.ICON_QUESTION|wx.YES_DEFAULT)
            button = dlg.ShowModal()
            dlg.Destroy()

            # save data
            if button == wx.ID_YES:
                if not self.onFileSave():
                    return False
            elif button == wx.ID_NO:
                return True
            else:
                return False

        return True
    # ----


    # ----
    def onPeaklistUndo(self):
        """ Step back in peaklist history. """

        self.docData.undoPeaklist()
        self.docMonitor('onPeaklistChanged', 'undo')
    # ----


    # ----
    def onPeaklistRedo(self):
        """ Step forward in peaklist history. """

        self.docData.redoPeaklist()
        self.docMonitor('onPeaklistChanged', 'redo')
    # ----


    # ----
    def onPeaklistAddPeak(self):
        """ Add new peak to peaklist. """
        self.mPeak.onAddPeak()
    # ----


    # ----
    def onPeaklistClearAllAnnots(self):
        """ Clear all annotations in peaklist. """
        self.mPeak.onClearAllAnnots()
    # ----


    # ----
    def onPeaklistDeleteAllPeaks(self):
        """ Delete all peaks in peaklist. """
        self.mPeak.onDeleteAllPeaks()
    # ----


    # ----
    def onPeaklistSelectAllPeaks(self):
        """ Select all peaks in peaklist. """
        self.mPeak.onSelectAllPeaks()
    # ----


    # ----
    def onPeaklistSelectAnnotatedPeaks(self):
        """ Select all annotated peaks in peaklist. """
        self.mPeak.onSelectAnnotatedPeaks()
    # ----


    # ----
    def onPeaklistSelectByTreshold(self):
        """ Select peaks by intensity treshold. """
        self.mPeak.onSelectByTreshold()
    # ----


    # ----
    def onPeaklistInvertSelection(self):
        """ Invert current peaklist selection. """
        self.mPeak.onInvertSelection()
    # ----


    # ----
    def onPeaklistCopyToClipboard(self):
        """ Copy peaklist data to clipboard. """
        self.mPeak.onCopyToClipboard()
    # ----


    # ----
    def onPeaklistRefresh(self):
        """ Redraw peaklist. """
        self.mPeak.onRefresh()
    # ----


    # ----
    def onSpectrumSetMouseFce(self, fce):
        """ Set mouse fce in spectrum module. """

        self.mSpec.setMouseFce(fce)
        self.docMonitor('onSpectrumFceChanged', fce)
    # ----


    # ----
    def onSpectrumAddToCompare(self):
        """ Add spectrum to compare with current. """

        self.mSpec.onCompareAdd()
        self.onToolSwitch('mSpec')
    # ----


    # ----
    def onSpectrumCopyToClipboard(self):
        """ Copy current spectrum view to clipboard. """

        # set application working
        self.docMonitor('setAppStatus', "Copying spectrum to clipboard...")

        self.mSpec.copySpectrumToClipboard()

        # set application ready
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onSpectrumSetRange(self):
        """ Set spectrum range. """
        self.mSpec.setRange()
    # ----


    # ----
    def onSpectrumViewOptions(self):
        """ Show spectrum view options. """
        self.mSpec.onViewOptions()
    # ----


    # ----
    def onSpectrumRefresh(self):
        """ Set view params for spectrum and redraw spectrum. """
        self.mSpec.refreshCanvas()
    # ----


    # ----
    def onSpectrumHighlightPoint(self, x, zoom=False):
        """ Ensure X point visible and show point arrow. """

        if self.moduleFrame != self.mSpec:
            self.onToolSwitch('mSpec')
            wx.Yield()

        self.mSpec.highlightPoint(x, zoom)
    # ----


    # ----
    def onProcessingUndo(self):
        """ Revert spectrum and peaklist to init data. """

        message = "Do you want to revert peaklist and spectrum data\nto the state before last processing applied?\nThere is no undo for this operation!"
        dlg = wx.MessageDialog(self, message, "Revert processing?", wx.OK|wx.CANCEL|wx.ICON_EXCLAMATION|wx.NO_DEFAULT)
        if dlg.ShowModal() == wx.ID_OK:
            dlg.Destroy()
            self.mProc.undo()
        else:
            dlg.Destroy()
    # ----


    # ----
    def onProcessingCalibration(self, statType):
        """ Data calibration. """
        self.mProc.calibrate(statType)
    # ----


    # ----
    def onProcessingBaseline(self):
        """ Do spectrum baseline correction. """
        self.mProc.baseline()
    # ----


    # ----
    def onProcessingSmooth(self):
        """ Smooth spectrum by Savitzky-Golay filter. """
        self.mProc.smooth()
    # ----


    # ----
    def onProcessingCrop(self):
        """ Crop spectrum and peaklist data to selected range. """
        self.mProc.crop()
    # ----


    # ----
    def onMatchingMatch(self):
        """ Match module data to peaklist. """

        # set application working
        self.docMonitor('setAppStatus', "Matching data to peaklist...")

        # match data to peaklist
        matchStatus = self.moduleFrame.matchDataToPeaklist()

        # update application
        self.docMonitor('setAppStatus', 0)
        self.docMonitor('onPeaklistMatched')

        # show no-match-alert
        if not matchStatus and self.config.cfg['common']['nomatchalert']:
            message = "No match found!\nCheck error type in statusbar or increase mass tolerance."
            dlg = wx.MessageDialog(self, message, "No Match", wx.OK|wx.ICON_INFORMATION)
            dlg.ShowModal()
            dlg.Destroy()

        # show match info
        elif matchStatus and self.config.cfg['common']['showmatchinfo']:
            self.onMatchingInfo()
    # ----


    # ----
    def onMatchingInfo(self):
        """ Show info about last match. """

        # get match info from module
        data = self.moduleFrame.getMatchInfo()

        # raise match-info dialog
        dlg = mMatchInfo(self, data)
        dlg.ShowModal()
        dlg.Destroy()
    # ----


    # ----
    def onMatchingAnnotatePeaklist(self):
        """ Annotate matched peaks in peaklist. """

        # set application working
        self.docMonitor('setAppStatus', "Annotating peaklist...")

        # annotate document peaklist
        self.moduleFrame.annotatePeaklist()

        # update application
        self.docMonitor('onPeaklistChanged', 'annotate')
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onSequenceEditMod(self):
        """ Rise dialog for sequence modifications. """

        self.onToolSwitch('mSeq')
        self.mSeq.onEditModifications()
    # ----


    # ----
    def onSequenceSearchMass(self):
        """ Search for specified mass in sequence. """
        self.mSeq.onSearchMass()
    # ----


    # ----
    def onSequenceSearchSequence(self):
        """ Search for specified sub-sequence in sequence. """
        self.mSeq.onSearchSequence()
    # ----


    # ----
    def onToolSwitch(self, tool):
        """ Switch current module. """

        # get selected module
        if tool == 'mSpec':
            module = self.mSpec
        elif tool == 'mSeq':
            module = self.mSeq
        elif tool == 'mCut':
            module = self.mCut
        elif tool == 'mFrag':
            module = self.mFrag
        elif tool == 'mDiff':
            module = self.mDiff
        elif tool == 'mComp':
            module = self.mComp

        # switch module
        if self.moduleFrame != module:
            self.moduleFrame.Show(False)
            self.ReplaceWindow(self.moduleFrame, module)
            self.moduleFrame = module
            self.moduleFrame.onShow()

        # update controls
        self.docMonitor('onModuleChanged', tool)
    # ----


    # ----
    def onToolMascotSearch(self, tool):
        """ Send peaklist data to Mascot server. """

        # init module
        mascot = mMascotSearch(self, self.docData)

        # run search
        mascot.search(tool)
    # ----


    # ----
    def onToolSeqConverter(self):
        """ Show sequence converter tool. """

        # set current module to mSeq
        self.onToolSwitch('mSeq')

        # raise converter dialog
        converter = mConv(self.docFrame)

        # paste sequence to document
        if converter.ShowModal() == wx.ID_OK:
            sequence = converter.data
            converter.Destroy()

            # check if sequence in current document and ask to rewrite
            if self.docData.getDataStatus('mSeq'):
                dlg = wx.MessageDialog(self, "Rewrite current sequence?", "Paste Sequence", wx.YES_NO|wx.ICON_EXCLAMATION|wx.NO_DEFAULT)
                if dlg.ShowModal() != wx.ID_YES:
                    dlg.Destroy()
                    return
                else:
                    dlg.Destroy()

            # paste sequence to current document
            self.docData.setSequence(sequence)
            self.mSeq.resetModule()

        else:
            converter.Destroy()
    # ----


    # ----
    def onConfigMassType(self, massType):
        """ Set actual type of mass (average/monoisotopic). """

        self.docData.setMassParam('masstype', massType)
        self.docMonitor('onMassTypeChanged', massType)
    # ----


    # ----
    def onConfigTolerance(self):
        """ Raise dialog to set tolerance value. """

        # get tolerance and errortype
        tolerance = self.docData.getMassParam('tolerance')
        errortype = self.docData.getMassParam('errortype')

        # rise dialog
        dlg = dlgSetTolerance(self, tolerance, errortype)
        if dlg.ShowModal() == wx.ID_OK:
            tolerance, errortype = dlg.getData()
            dlg.Destroy()
        else:
            dlg.Destroy()
            return

        # update document
        self.docData.setMassParam('tolerance', tolerance)
        self.docData.setMassParam('errortype', errortype)

        # update application
        tolerance = '%s %s' % (tolerance, errortype)
        self.docMonitor('onToleranceChanged', tolerance)
    # ----


    # ----
    def onConfigShowPeaklist(self, splitting=False):
        """ Show/hide peaklist panel. """

        if self.IsSplit():
            self.sashPosition = self.GetSashPosition()
        if splitting == 'h':
            self.sashPosition = -170
        elif splitting == 'v':
            self.sashPosition = 170

        # hide panel
        if self.IsSplit():
            self.Unsplit(self.mPeak)
        elif not splitting and self.GetSplitMode() == 1:
            splitting = 'h'
        elif not splitting and self.GetSplitMode() == 2:
            splitting = 'v'

        # show panel
        if splitting:
            if splitting == 'h':
                self.SplitHorizontally(self.moduleFrame, self.mPeak, self.sashPosition)
                self.SetSashGravity(1.0)
            else:
                self.SplitVertically(self.mPeak, self.moduleFrame, self.sashPosition)
                self.SetSashGravity(0.0)

        # update splitting button and menu items
        self.docMonitor('onPeaklistView', splitting)
    # ----


    # ----
    def onConfigSplitting(self):
        """ Set current splitting - peaklist position. """

        # get splitting param
        if self.GetSplitMode() == 1:
            splitting = 'v'
        elif self.GetSplitMode() == 2:
            splitting = 'h'

        # set new splitting mode
        self.onConfigShowPeaklist(splitting=splitting)
    # ----


    # ----
    def getCurrentModule(self):
        """ Get name of current module. """

        if self.moduleFrame == self.mSpec:
            return 'mSpec'
        elif self.moduleFrame == self.mSeq:
            return 'mSeq'
        elif self.moduleFrame == self.mCut:
            return 'mCut'
        elif self.moduleFrame == self.mFrag:
            return 'mFrag'
        elif self.moduleFrame == self.mDiff:
            return 'mDiff'
        elif self.moduleFrame == self.mComp:
            return 'mComp'
    # ----


    # ----
    def checkAppliedModifications(self):
        """ Check applied sequence modifications. """
        return self.mSeq.checkAppliedModifications()
    # ----
