# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Dialog with all view options for spectrum.

# load libs
import wx

# load modules
from nucleus import mwx

class dlgViewOptions(wx.Dialog):
    """ Dialog with all view options for spectrum. """

    # ----
    def __init__(self, parent, data):
        wx.Dialog.__init__(self, parent, -1, "Spectrum View Options", style=wx.DEFAULT_DIALOG_STYLE)

        self.parent = parent
        self.data = data

        # make items
        labelsBox = self.makeLabelsBox()
        spectrumBox = self.makeSpectrumBox()
        cursorBox = self.makeCursorBox()
        buttonBox = self.makeButtonBox()

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(labelsBox, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 10)
            mainSizer.Add(spectrumBox, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 10)
            mainSizer.Add(cursorBox, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 10)
            mainSizer.Add(buttonBox, 0, wx.ALL|wx.ALIGN_CENTER, 20)
        else:
            mainSizer.Add(labelsBox, 0, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(spectrumBox, 0, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(cursorBox, 0, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(buttonBox, 0, wx.ALL|wx.ALIGN_CENTER, 5)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makeLabelsBox(self):
        """ Make box for label view options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Label options"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        self.showTicks_check = wx.CheckBox(self, -1, "Show label ticks")
        self.showLabels_check = wx.CheckBox(self, -1, "Show labels")
        self.showAnnots_check = wx.CheckBox(self, -1, "Show annotations")

        labelAngle_choices = ['0', '45', '90']
        labelAngle_label = wx.StaticText(self, -1, "Label angle: ")
        self.labelAngle_combo = wx.ComboBox(self, -1, str(self.data['labelangle']), size=(60, -1), choices=labelAngle_choices, style=wx.CB_READONLY)
        
        # pack items
        grid.Add(labelAngle_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.labelAngle_combo, (0, 1))

        mainBox.Add(self.showTicks_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showLabels_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showAnnots_check, 0, wx.ALL, 5)
        mainBox.Add(grid, 0, wx.ALL, 5)

        # set defaults
        self.showTicks_check.SetValue(self.data['showticks'])
        self.showLabels_check.SetValue(self.data['showlabels'])
        self.showAnnots_check.SetValue(self.data['showannots'])

        return mainBox
    # ----


    # ----
    def makeSpectrumBox(self):
        """ Make box for spectrum options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Spectrum options"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        self.showGrid_check = wx.CheckBox(self, -1, "Show grid")
        self.showLegend_check = wx.CheckBox(self, -1, "Show legend")
        self.showPosBar_check = wx.CheckBox(self, -1, "Show position bar")
        self.showGelView_check = wx.CheckBox(self, -1, "Show gel view")

        gelViewHeight_label = wx.StaticText(self, -1, "Gel height: ")
        gelViewHeight_unit = wx.StaticText(self, -1, " px")
        self.gelViewHeight_spin = wx.SpinCtrl(self, -1, str(self.data['gelviewheight']), size=(83, -1), min=5, max=100, style=wx.SP_WRAP|wx.SP_ARROW_KEYS)
        self.gelViewHeight_spin.SetToolTip(wx.ToolTip("Use odd numbers only"))

        # pack items
        grid.Add(gelViewHeight_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.gelViewHeight_spin, (0, 1))
        grid.Add(gelViewHeight_unit, (0, 2), flag=wx.ALIGN_CENTER_VERTICAL)

        mainBox.Add(self.showGrid_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showLegend_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showPosBar_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showGelView_check, 0, wx.ALL, 5)
        mainBox.Add(grid, 0, wx.ALL, 5)

        # set defaults
        self.showGrid_check.SetValue(self.data['showgrid'])
        self.showLegend_check.SetValue(self.data['showlegend'])
        self.showPosBar_check.SetValue(self.data['showposbar'])
        self.showGelView_check.SetValue(self.data['showgelview'])

        return mainBox
    # ----


    # ----
    def makeCursorBox(self):
        """ Make box for cursor options. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Cursor options"), wx.VERTICAL)

        self.showTracker_check = wx.CheckBox(self, -1, "Show cursor tracker")
        self.showPosition_check = wx.CheckBox(self, -1, "Show cursor position")
        self.showDistance_check = wx.CheckBox(self, -1, "Show cursor distance")

        # pack items
        mainBox.Add(self.showTracker_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showPosition_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(self.showDistance_check, 0, wx.ALL, 5)

        # set defaults
        self.showTracker_check.SetValue(self.data['showtracker'])
        self.showPosition_check.SetValue(self.data['showposition'])
        self.showDistance_check.SetValue(self.data['showdistance'])

        return mainBox
    # ----


    # ----
    def makeButtonBox(self):
        """ Make box for main buttons. """

        # make items
        OK_button = wx.Button(self, wx.ID_OK, "OK")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        mainBox = wx.BoxSizer(wx.HORIZONTAL)
        mainBox.Add(OK_button, 0, wx.ALL, 5)
        mainBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set defaults
        OK_button.SetDefault()

        return mainBox
    # ----


    # ----
    def getData(self):
        """ Check and get data from page. """

        self.data['importpeaklist'] = 0
        self.data['showlabels'] = 0
        self.data['showannots'] = 0
        self.data['showticks'] = 0
        self.data['showtracker'] = 0
        self.data['showposition'] = 0
        self.data['showdistance'] = 0
        self.data['showgrid'] = 0
        self.data['showlegend'] = 0
        self.data['showposbar'] = 0
        self.data['showgelview'] = 0

        if self.showLabels_check.IsChecked():
            self.data['showlabels'] = 1
        if self.showAnnots_check.IsChecked():
            self.data['showannots'] = 1
        if self.showTicks_check.IsChecked():
            self.data['showticks'] = 1
        if self.showTracker_check.IsChecked():
            self.data['showtracker'] = 1
        if self.showPosition_check.IsChecked():
            self.data['showposition'] = 1
        if self.showDistance_check.IsChecked():
            self.data['showdistance'] = 1
        if self.showGrid_check.IsChecked():
            self.data['showgrid'] = 1
        if self.showLegend_check.IsChecked():
            self.data['showlegend'] = 1
        if self.showPosBar_check.IsChecked():
            self.data['showposbar'] = 1
        if self.showGelView_check.IsChecked():
            self.data['showgelview'] = 1

        labelAngle = self.labelAngle_combo.GetValue()
        gelViewHeight = self.gelViewHeight_spin.GetValue()
        try:
            self.data['labelangle'] = int(labelAngle)
            self.data['gelviewheight'] = int(gelViewHeight)
        except ValueError:
            pass

        return self.data
    # ----
