/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: simple.c,v 1.3 2004/06/09 02:10:33 kos Exp $
 *
 * Password cracking, simple methods.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <time.h>

#include <openssl/md5.h>
#include <openssl/des.h>
#include <openssl/sha.h>
#include <openssl/md2.h>

#include "method.h"
#include "keyspace.h"
#include "common.h"
#include "method.h"
#include "array.h"
#include "xmalloc.h"
#include "simple.h"
typedef unsigned char BYTE;
#define CHAR_MAX 256

const BYTE S_BOX[] = {
  0xBD,0x56,0xEA,0xF2,0xA2,0xF1,0xAC,0x2A,
  0xB0,0x93,0xD1,0x9C,0x1B,0x33,0xFD,0xD0,
  0x30,0x04,0xB6,0xDC,0x7D,0xDF,0x32,0x4B,
  0xF7,0xCB,0x45,0x9B,0x31,0xBB,0x21,0x5A,
  0x41,0x9F,0xE1,0xD9,0x4A,0x4D,0x9E,0xDA,
  0xA0,0x68,0x2C,0xC3,0x27,0x5F,0x80,0x36,
  0x3E,0xEE,0xFB,0x95,0x1A,0xFE,0xCE,0xA8,
  0x34,0xA9,0x13,0xF0,0xA6,0x3F,0xD8,0x0C,
  0x78,0x24,0xAF,0x23,0x52,0xC1,0x67,0x17,
  0xF5,0x66,0x90,0xE7,0xE8,0x07,0xB8,0x60,
  0x48,0xE6,0x1E,0x53,0xF3,0x92,0xA4,0x72,
  0x8C,0x08,0x15,0x6E,0x86,0x00,0x84,0xFA,
  0xF4,0x7F,0x8A,0x42,0x19,0xF6,0xDB,0xCD,
  0x14,0x8D,0x50,0x12,0xBA,0x3C,0x06,0x4E,
  0xEC,0xB3,0x35,0x11,0xA1,0x88,0x8E,0x2B,
  0x94,0x99,0xB7,0x71,0x74,0xD3,0xE4,0xBF,
  0x3A,0xDE,0x96,0x0E,0xBC,0x0A,0xED,0x77,
  0xFC,0x37,0x6B,0x03,0x79,0x89,0x62,0xC6,
  0xD7,0xC0,0xD2,0x7C,0x6A,0x8B,0x22,0xA3,
  0x5B,0x05,0x5D,0x02,0x75,0xD5,0x61,0xE3,
  0x18,0x8F,0x55,0x51,0xAD,0x1F,0x0B,0x5E,
  0x85,0xE5,0xC2,0x57,0x63,0xCA,0x3D,0x6C,
  0xB4,0xC5,0xCC,0x70,0xB2,0x91,0x59,0x0D,
  0x47,0x20,0xC8,0x4F,0x58,0xE0,0x01,0xE2,
  0x16,0x38,0xC4,0x6F,0x3B,0x0F,0x65,0x46,
  0xBE,0x7E,0x2D,0x7B,0x82,0xF9,0x40,0xB5,
  0x1D,0x73,0xF8,0xEB,0x26,0xC7,0x87,0x97,
  0x25,0x54,0xB1,0x28,0xAA,0x98,0x9D,0xA5,
  0x64,0x6D,0x7A,0xD4,0x10,0x81,0x44,0xEF,
  0x49,0xD6,0xAE,0x2E,0xDD,0x76,0x5C,0x2F,
  0xA7,0x1C,0xC9,0x09,0x69,0x9A,0x83,0xCF,
  0x29,0x39,0xB9,0xE9,0x4C,0xFF,0x43,0xAB
  , /* repeated again to avoid (& 0xff) */
  0xBD,0x56,0xEA,0xF2,0xA2,0xF1,0xAC,0x2A,
  0xB0,0x93,0xD1,0x9C,0x1B,0x33,0xFD,0xD0,
  0x30,0x04,0xB6,0xDC,0x7D,0xDF,0x32,0x4B,
  0xF7,0xCB,0x45,0x9B,0x31,0xBB,0x21,0x5A,
  0x41,0x9F,0xE1,0xD9,0x4A,0x4D,0x9E,0xDA,
  0xA0,0x68,0x2C,0xC3,0x27,0x5F,0x80,0x36,
  0x3E,0xEE,0xFB,0x95,0x1A,0xFE,0xCE,0xA8,
  0x34,0xA9,0x13,0xF0,0xA6,0x3F,0xD8,0x0C,
  0x78,0x24,0xAF,0x23,0x52,0xC1,0x67,0x17,
  0xF5,0x66,0x90,0xE7,0xE8,0x07,0xB8,0x60,
  0x48,0xE6,0x1E,0x53,0xF3,0x92,0xA4,0x72,
  0x8C,0x08,0x15,0x6E,0x86,0x00,0x84,0xFA,
  0xF4,0x7F,0x8A,0x42,0x19,0xF6,0xDB,0xCD,
  0x14,0x8D,0x50,0x12,0xBA,0x3C,0x06,0x4E,
  0xEC,0xB3,0x35,0x11,0xA1,0x88,0x8E,0x2B,
  0x94,0x99,0xB7,0x71,0x74,0xD3,0xE4,0xBF,
  0x3A,0xDE,0x96,0x0E,0xBC,0x0A,0xED,0x77,
  0xFC,0x37,0x6B,0x03,0x79,0x89,0x62,0xC6,
  0xD7,0xC0,0xD2,0x7C,0x6A,0x8B,0x22,0xA3,
  0x5B,0x05,0x5D,0x02,0x75,0xD5,0x61,0xE3,
  0x18,0x8F,0x55,0x51,0xAD,0x1F,0x0B,0x5E,
  0x85,0xE5,0xC2,0x57,0x63,0xCA,0x3D,0x6C,
  0xB4,0xC5,0xCC,0x70,0xB2,0x91,0x59,0x0D,
  0x47,0x20,0xC8,0x4F,0x58,0xE0,0x01,0xE2,
  0x16,0x38,0xC4,0x6F,0x3B,0x0F,0x65,0x46,
  0xBE,0x7E,0x2D,0x7B,0x82,0xF9,0x40,0xB5,
  0x1D,0x73,0xF8,0xEB,0x26,0xC7,0x87,0x97,
  0x25,0x54,0xB1,0x28,0xAA,0x98,0x9D,0xA5,
  0x64,0x6D,0x7A,0xD4,0x10,0x81,0x44,0xEF,
  0x49,0xD6,0xAE,0x2E,0xDD,0x76,0x5C,0x2F,
  0xA7,0x1C,0xC9,0x09,0x69,0x9A,0x83,0xCF,
  0x29,0x39,0xB9,0xE9,0x4C,0xFF,0x43,0xAB
};

static void RC4_MixTable1(BYTE * const BLK);
static void RC4_MixTable2(BYTE * const BLK);
BYTE * domino_algo(char *passwd, int len);
/*
 * freebsd/linux/cisco md5 implementation
 * liberated from libcrypt1.0/crypt_md5.c/Poul-Henning Kamp's implementation
 *
 */
static unsigned char md5_itoa64[] = "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

static void
md5_to64(char *s, unsigned long v, int n) {
  while (--n >= 0) {
    *s++ = md5_itoa64[v&0x3f];
    v >>= 6;
  }
}

md5_t *
md5_init(char *params) {
  md5_t *u;

  u = xcalloc(1, sizeof(md5_t));
  u->pool = array_init(sizeof(md5_key_t));
  u->found = 0;
  return u;
}

int
md5_test(md5_t *u, uchar *hash, int len) {
  if(len < 4)
    return 1;
  if(strncmp(hash, MD5_MAGIC, MD5_MAGIC_LEN))
    return 2;
  return 0;
}

int
md5_crypt(md5_t *u) {
  int i;
  md5_key_t *uk;
  MD5_CTX ctx;
  MD5_CTX ctx1;
  uchar final[16];
  int j, k, pl, m;
  uchar attempt[METH_LINELEN];
  char *p;
  unsigned long l;
  
  for(i=0; (uk=array_get(u->pool, i)); i++) {

    MD5_Init(&ctx);    
    MD5_Update(&ctx, uk->key, uk->len);
    MD5_Update(&ctx, MD5_MAGIC, MD5_MAGIC_LEN);
    MD5_Update(&ctx, u->salt, strlen(u->salt));

    MD5_Init(&ctx1);
    MD5_Update(&ctx1, uk->key, uk->len);
    MD5_Update(&ctx1, u->salt, strlen(u->salt));
    MD5_Update(&ctx1, uk->key, uk->len);
    MD5_Final(final, &ctx1);

    for(pl=uk->len; pl>0; pl-=16)
      MD5_Update(&ctx, final, pl>16?16:pl);

    memset(final,0,sizeof final);

    for(j=0,k=uk->len; k; k>>=1)
      if(k&1)
	MD5_Update(&ctx, final+j, 1);
      else
	MD5_Update(&ctx, uk->key+j, 1);
    MD5_Final(final, &ctx);

    for(m=0; m<1000; m++) {
      MD5_Init(&ctx1);
      if(m&1)
	MD5_Update(&ctx1, uk->key, uk->len);
      else
	MD5_Update(&ctx1, final, 16);
      if(m%3)
	MD5_Update(&ctx1, u->salt, strlen(u->salt));
      if(m%7)
	MD5_Update(&ctx1, uk->key, uk->len);
      if(m&1)
	MD5_Update(&ctx1, final, 16);
      else
	MD5_Update(&ctx1, uk->key, uk->len);
      MD5_Final(final, &ctx1);
    }

    p = attempt;
    l = (final[ 0]<<16) | (final[ 6]<<8) | final[12]; md5_to64(p,l,4); p += 4;
    l = (final[ 1]<<16) | (final[ 7]<<8) | final[13]; md5_to64(p,l,4); p += 4;
    l = (final[ 2]<<16) | (final[ 8]<<8) | final[14]; md5_to64(p,l,4); p += 4;
    l = (final[ 3]<<16) | (final[ 9]<<8) | final[15]; md5_to64(p,l,4); p += 4;
    l = (final[ 4]<<16) | (final[10]<<8) | final[ 5]; md5_to64(p,l,4); p += 4;
    l =                    final[11]                ; md5_to64(p,l,2); p += 2;
    *p = '\0';

    if(!strcmp(u->hash, attempt)) {
      u->found = 1;
      strncpy(u->key, uk->key, MIN(MD5_MAX_KEYLEN, METH_LINELEN));
      u->len = uk->len;
      u->index = i;
      array_zero(u->pool);
      return 1;
    }
  }
  array_zero(u->pool);
  return 0;
}

int
md5_add(md5_t *u, uchar *key, int len) {
  md5_key_t k;

  u->found = 0;
  memset(k.key, '\0', MD5_MAX_KEYLEN);
  memcpy(k.key, key, len>MD5_MAX_KEYLEN?MD5_MAX_KEYLEN:len);
  k.len = len;
  array_add(u->pool, &k);
  return 0;
}

int
md5_siz(md5_t *u) {
  return 256*1024;
}

int
md5_destroy(md5_t *u) {
  array_nuke(u->pool);
  xfree(u);
  return 0;
}

int
md5_sethash(md5_t *u, uchar *hash, int len) {
  char lhash[METH_LINELEN];
  char lsalt[METH_LINELEN];
  char *p;
  
  strncpy(lsalt, hash+3, MIN(len, METH_LINELEN));
  if(!(p = index(lsalt, '$')))
    return 1;
  *p++ = '\0';
  strncpy(lhash, p, MIN(len, METH_LINELEN));	  
  memset(u->hash, 0, METH_LINELEN);
  memset(u->salt, 0, METH_LINELEN);
  strncpy(u->hash, lhash, MIN(len, METH_LINELEN));
  strncpy(u->salt, lsalt, MIN(len, METH_LINELEN));
  return 0;
}

int
md5_setsalt(md5_t *u, uchar *salt, int len) {
  /* comes from sethash */
  return 0;
}

int
md5_getkey(md5_t *u, uchar *buf, int maxlen, int *len, int *relindex) {

  if(!u->found)
    return 1;
  memcpy(buf, u->key, MIN(u->len, maxlen));
  *len = u->len;
  *relindex = u->index;
  return 0;
}




/*
 * single md5
 */

md5raw_t *
md5raw_init(char *params) {
  md5raw_t *u;

  u = xcalloc(1, sizeof(md5raw_t));
  u->pool = array_init(sizeof(md5raw_key_t));
  u->found = 0;
  return u;
}

int
md5raw_test(md5raw_t *u, uchar *hash, int len) {
  if(len != MD5_HASHLEN)
    return 1;
  return 0;
}

int
md5raw_crypt(md5raw_t *u) {
  int i;
  md5raw_key_t *uk;
  uchar attempt[MD5_HASHLEN];
  MD5_CTX context;

  for(i=0; (uk=array_get(u->pool, i)); i++) {
    MD5_Init(&context);
    MD5_Update(&context, uk->key, uk->len);
    MD5_Final(attempt, &context);
    if(!memcmp(u->hash, attempt, MD5_HASHLEN)) {
      u->found = 1;
      memcpy(u->key, uk->key, MD5_HASHLEN);
      u->len = uk->len;
      u->index = i;
      array_zero(u->pool);
      return 1;
    }
  }
  array_zero(u->pool);
  return 0;
}

int
md5raw_add(md5raw_t *u, uchar *key, int len) {
  md5raw_key_t k;

  u->found = 0;
  memset(k.key, '\0', MD5_MAX_KEYLEN);
  memcpy(k.key, key, len>MD5_MAX_KEYLEN?MD5_MAX_KEYLEN:len);
  k.len = len;
  array_add(u->pool, &k);
  return 0;
}

int
md5raw_siz(md5raw_t *u) {
  return 256*1024;
}

int
md5raw_destroy(md5raw_t *u) {
  array_nuke(u->pool);
  xfree(u);
  return 0;
}

int
md5raw_sethash(md5raw_t *u, uchar *hash, int len) {
  memset(u->hash, 0, METH_LINELEN);
  if(len != 16)
    return 1;
  memcpy(u->hash, hash, len);
  return 0;
}

int
md5raw_setsalt(md5raw_t *u, uchar *salt, int len) {
  /* No such thing for this method */
  return 0;
}

int
md5raw_getkey(md5raw_t *u, uchar *buf, int maxlen, int *len, int *relindex) {
  if(!u->found)
    return 1;
  memcpy(buf, u->key, MIN(u->len, maxlen));
  *len = u->len;
  *relindex = u->index;
  return 0;
}

/*
 * single md2
 */

md2raw_t *
md2raw_init(char *params) {
  md2raw_t *u;

  u = xcalloc(1, sizeof(md2raw_t));
  u->pool = array_init(sizeof(md2raw_key_t));
  u->found = 0;
  return u;
}

int
md2raw_test(md2raw_t *u, uchar *hash, int len) {
  if(len != MD2_HASHLEN)
    return 1;
  return 0;
}

int
md2raw_crypt(md2raw_t *u) {
  int i;
  md2raw_key_t *uk;
  uchar attempt[MD2_HASHLEN];
  MD2_CTX context;

  for(i=0; (uk=array_get(u->pool, i)); i++) {
    MD2_Init(&context);
    MD2_Update(&context, uk->key, uk->len);
    MD2_Final(attempt, &context);
    if(!memcmp(u->hash, attempt, MD2_HASHLEN)) {
      u->found = 1;
      memcpy(u->key, uk->key, MD2_HASHLEN);
      u->len = uk->len;
      u->index = i;
      array_zero(u->pool);
      return 1;
    }
  }
  array_zero(u->pool);
  return 0;
}

int
md2raw_add(md2raw_t *u, uchar *key, int len) {
  md2raw_key_t k;

  u->found = 0;
  memset(k.key, '\0', MD2_MAX_KEYLEN);
  memcpy(k.key, key, len>MD2_MAX_KEYLEN?MD2_MAX_KEYLEN:len);
  k.len = len;
  array_add(u->pool, &k);
  return 0;
}

int
md2raw_siz(md2raw_t *u) {
  return 256*1024;
}

int
md2raw_destroy(md2raw_t *u) {
  array_nuke(u->pool);
  xfree(u);
  return 0;
}

int
md2raw_sethash(md2raw_t *u, uchar *hash, int len) {
  memset(u->hash, 0, METH_LINELEN);
  if(len != 16)
    return 1;
  memcpy(u->hash, hash, len);
  return 0;
}

int
md2raw_setsalt(md2raw_t *u, uchar *salt, int len) {
  /* No such thing for this method */
  return 0;
}

int
md2raw_getkey(md2raw_t *u, uchar *buf, int maxlen, int *len, int *relindex) {
  if(!u->found)
    return 1;
  memcpy(buf, u->key, MIN(u->len, maxlen));
  *len = u->len;
  *relindex = u->index;
  return 0;
}

/*
 * single sha1. Either plain or netscape ldap server version (unsalted)
 */

sharaw_t *
sharaw_init(char *params) {
  sharaw_t *u;

  u = xcalloc(1, sizeof(sharaw_t));
  u->pool = array_init(sizeof(sharaw_key_t));
  u->found = 0;
  return u;
}

int
sharaw_test(sharaw_t *u, uchar *hash, int len) {
  
  if(!strncmp(hash, SHA_NS_MAGIC, SHA_NS_MAGIC_LEN) || len == SHA_HASHLEN)
    return 0;
  return 1;
}

int
sharaw_crypt(sharaw_t *u) {
  int i;
  sharaw_key_t *uk;
  uchar attempt[SHA_HASHLEN];
  SHA_CTX context;

  for(i=0; (uk=array_get(u->pool, i)); i++) {
    SHA1_Init(&context);
    SHA1_Update(&context, uk->key, uk->len);
    SHA1_Final(attempt, &context);
    if(!memcmp(u->hash, attempt, SHA_HASHLEN)) {
      u->found = 1;
      memcpy(u->key, uk->key, uk->len);
      u->len = uk->len;
      u->index = i;
      array_zero(u->pool);
      return 1;
    }
  }
  array_zero(u->pool);
  return 0;
}

int
sharaw_add(sharaw_t *u, uchar *key, int len) {
  sharaw_key_t k;

  u->found = 0;
  memset(k.key, '\0', SHA_MAX_KEYLEN);
  memcpy(k.key, key, len>SHA_MAX_KEYLEN?SHA_MAX_KEYLEN:len);
  k.len = len;
  array_add(u->pool, &k);
  return 0;
}

int
sharaw_siz(sharaw_t *u) {
  return 64*1024;
}

int
sharaw_destroy(sharaw_t *u) {
  array_nuke(u->pool);
  xfree(u);
  return 0;
}

int
sharaw_sethash(sharaw_t *u, uchar *hash, int len) {

  memset(u->hash, '\0', METH_LINELEN);
  if(!memcmp(hash, SHA_NS_MAGIC, SHA_NS_MAGIC_LEN))
    base64_decode(hash+SHA_NS_MAGIC_LEN, len-SHA_NS_MAGIC_LEN, u->hash);
  else
    memcpy(u->hash, hash, SHA_HASHLEN);
  return 0;
}

int
sharaw_setsalt(sharaw_t *u, uchar *salt, int len) {
  /* XXX: Support the salted version on netscape ldap 4 */
  return 0;
}

int
sharaw_getkey(sharaw_t *u, uchar *buf, int maxlen, int *len, int *relindex) {
  if(!u->found)
    return 1;
  memcpy(buf, u->key, MIN(u->len, maxlen));
  *len = u->len;
  *relindex = u->index;
  return 0;
}

method_impl_t sharaw_impl = {
  "sharaw",
  (meth_init)sharaw_init,
  (meth_test)sharaw_test,
  (meth_crypt)sharaw_crypt,
  (meth_add)sharaw_add,
  (meth_siz)sharaw_siz,
  (meth_sethash)sharaw_sethash,
  (meth_sethash)sharaw_setsalt,
  (meth_getkey)sharaw_getkey,
  (meth_destroy)sharaw_destroy
};

/*
 * des-based crypt
 */

unixcrypt_t *
unixcrypt_init(char *params) {
  unixcrypt_t *u;

  u = xcalloc(1, sizeof(unixcrypt_t));
  u->pool = array_init(sizeof(unixcrypt_key_t));
  return u;
}

int
unixcrypt_test(unixcrypt_t *u, uchar *hash, int len) {
  int i;

  if(len != UNIXCRYPT_HASHLEN)
    return 1;
  for(i=0; i<UNIXCRYPT_HASHLEN; i++) {
    if(!isalnum(hash[i]) || hash[i] != '.' || hash[i] != '/')
      return 2;
  }
  return 0;
}

int
unixcrypt_crypt(unixcrypt_t *u) {
  int i;
  unixcrypt_key_t *uk;
  char attempt[24];

  for(i=0; (uk=array_get(u->pool, i)); i++) {
    des_fcrypt(uk->key, u->salt, attempt);

 //   printf("Attempt : %s\nHash    : %s\nClear : %s\n\n", attempt, u->hash , u->key);

    if(!memcmp(u->hash, attempt+2, 11)) {
      strcpy(u->key, uk->key);
      u->index = i;
      array_zero(u->pool);
      return 1;
    }
  }
  array_zero(u->pool);
  return 0;
}

int
unixcrypt_add(unixcrypt_t *u, uchar *key, int len) {
  unixcrypt_key_t k;

  u->key[0] = '\0';
  strncpy(k.key, key, 9);
  array_add(u->pool, &k);
  return 0;
}

int
unixcrypt_siz(unixcrypt_t *u) {
  return 512*1024;
}

int
unixcrypt_destroy(unixcrypt_t *u) {
  array_nuke(u->pool);
  xfree(u);
  return 0;
}

int
unixcrypt_sethash(unixcrypt_t *u, uchar *hash, int len) {
  memset(u->hash, 0, METH_LINELEN);
  memset(u->salt, 0, 4);
  strncpy(u->salt, hash, 2);
  strncpy(u->hash, hash+2, METH_LINELEN);
  return 0;
}

int
unixcrypt_setsalt(unixcrypt_t *u, unsigned char *salt, int len) {
  /* comes from sethash */
  return 0;
}

int
unixcrypt_getkey(unixcrypt_t *u, uchar *buf, int maxlen, int *len, int *relindex) {
  int keylen;

  if(u->key[0] == '\0')
    return 1;
  keylen = strlen(u->key);
  strncpy(buf, u->key, MIN(maxlen, keylen));
  *len = keylen;
  *relindex = u->index;
  return 0;
}

 method_impl_t unixcrypt_impl = {
  "unixcrypt",
  (meth_init)unixcrypt_init,
  (meth_test)unixcrypt_test,
  (meth_crypt)unixcrypt_crypt,
  (meth_add)unixcrypt_add,
  (meth_siz)unixcrypt_siz,
  (meth_sethash)unixcrypt_sethash,
  (meth_sethash)unixcrypt_setsalt,
  (meth_getkey)unixcrypt_getkey,
  (meth_destroy)unixcrypt_destroy
};


method_impl_t md5_impl = {
  "md5",
  (meth_init)md5_init,
  (meth_test)md5_test,
  (meth_crypt)md5_crypt,
  (meth_add)md5_add,
  (meth_siz)md5_siz,
  (meth_sethash)md5_sethash,
  (meth_sethash)md5_setsalt,
  (meth_getkey)md5_getkey,
  (meth_destroy)md5_destroy
};

method_impl_t md5raw_impl = {
  "md5raw",
  (meth_init)md5raw_init,
  (meth_test)md5raw_test,
  (meth_crypt)md5raw_crypt,
  (meth_add)md5raw_add,
  (meth_siz)md5raw_siz,
  (meth_sethash)md5raw_sethash,
  (meth_sethash)md5raw_setsalt,
  (meth_getkey)md5raw_getkey,
  (meth_destroy)md5raw_destroy
};

method_impl_t md2raw_impl = {
  "md2raw",
  (meth_init)md2raw_init,
  (meth_test)md2raw_test,
  (meth_crypt)md2raw_crypt,
  (meth_add)md2raw_add,
  (meth_siz)md2raw_siz,
  (meth_sethash)md2raw_sethash,
  (meth_sethash)md2raw_setsalt,
  (meth_getkey)md2raw_getkey,
  (meth_destroy)md2raw_destroy
};


/*
 * Lotus Domino crypt   by G0ur0u
 */

domino_t *
domino_init(char *params) {
  domino_t *u;

  u = xcalloc(1, sizeof(domino_t));
  u->pool = array_init(sizeof(domino_t));
  return u;
}

int
domino_test(domino_t *u, uchar *hash, int len) {
  int i;

  if(len != DOMINO_HASHLEN)
    return 1;
  for(i=0; i<DOMINO_HASHLEN; i++) {
    if(!isalnum(hash[i]) || hash[i] != '.' || hash[i] != '/')
      return 2;
  }
  return 0;
}

int
domino_crypt(domino_t *u) {
  int i;
  domino_key_t *uk;
  char attempt[48];
  BYTE * attemp ;

  for(i=0; (uk=array_get(u->pool, i)); i++) {

    attemp = domino_algo(uk->key, uk->len) ;
    strncpy(attempt,attemp, 16)  ;

 /*    printf("Attempt : ")  ; for(i = 0; i < 16; i ++) printf("%X",attempt[i]);
     printf("\nHash    : "); for(i = 0; i < 16; i ++) printf("%X",u->hash[i]);
     printf("\nClear : %s : len : %d\n\n",uk->key, uk->len)   ; */


    if(!memcmp(u->hash, attempt, 16)) {
      strcpy(u->key, uk->key);
      u->len=uk->len;
      u->index = i;
      array_zero(u->pool);
      return 1;
    }
  }
  array_zero(u->pool);
  return 0;
}

int
domino_add(domino_t *u, uchar *key, int len) {
  domino_key_t k;
  int i=0;
  int len2=0;

  while( (*(key+i)!='\0') && len2<DOMINO_MAX_KEYLEN ) {i++; len2++;}

  memcpy(k.key, key, strlen(key));
  k.len = strlen(key) ;
  array_add(u->pool, &k);
  return 0;

}

int
domino_siz(domino_t *u) {
  return 512*1024;
}

int
domino_destroy(domino_t *u) {
  array_nuke(u->pool);
  xfree(u);
  return 0;
}

signed int hex(char x) {
  if((x >= '0') && (x <= '9'))
    return (x - '0');
  else if((x >= 'A') && (x <= 'F'))
    return (x - 'A' + 10);
  else if((x >= 'a') && (x <= 'f'))
    return (x - 'a' + 10);

  return -1;
}

int
domino_sethash(domino_t *u, uchar *hash, int len) {
  int i,j,len2;
  uchar hash2[32];
  memset(u->hash, 0, METH_LINELEN);
  memset(u->salt, 0, 4);
  strncpy(u->salt, hash, 2);

  for(len2 = 0, i = 0; i < len-1; i += 2, len2 ++) {
     if((j = hex(hash[i])) < 0) printf("domino_sethash: %s: bad hexadecimal character\n", hash[i]);;
     hash2[len2] = ((j & 0xf) << 4);

     if((j = hex(hash[i+1])) < 0) printf("domino_sethash: %s: bad hexadecimal character\n", hash[i]);;
     hash2[len2] += (j & 0x0f);     }
  
  strncpy(u->hash, hash2, METH_LINELEN);


  return 0;
}

int
domino_setsalt(domino_t *u, unsigned char *salt, int len) {
  /* comes from sethash */
  return 0;
}

int
domino_getkey(domino_t *u, uchar *buf, int maxlen, int *len, int *relindex) {

  if(u->key[0] == '\0')
    return 1;
  strncpy(buf, u->key, MIN(maxlen, strlen(u->key)));
  *len = u->len;
  *relindex = u->index;
  return 0;
}






method_impl_t domino_impl = {
  "domino",
  (meth_init)domino_init,
  (meth_test)domino_test,
  (meth_crypt)domino_crypt,
  (meth_add)domino_add,
  (meth_siz)domino_siz,
  (meth_sethash)domino_sethash,
  (meth_sethash)domino_setsalt,
  (meth_getkey)domino_getkey,
  (meth_destroy)domino_destroy
};




BYTE * domino_algo(char *passwd, int len) {
  static BYTE RC4_Block[48];
  unsigned long *RC4_ptr = (unsigned long *)RC4_Block;
  BYTE RC4_Key[16];
  BYTE RC4_Seed;
  unsigned int i,j;

  *(RC4_ptr + 0) = 0;
  *(RC4_ptr + 1) = 0;
  *(RC4_ptr + 2) = 0;
  *(RC4_ptr + 3) = 0;
  /*for (i=0; i<len; i++) printf("%c.",*(passwd+i)); printf(" & len : %d\n",len);        */

  RC4_Seed = 0;
  
  for(j = 0; j < len; j ++) {
    BYTE X = (int)passwd[j];

    RC4_Block[16 + j] = RC4_Block[32 + j] = X;
    RC4_Seed = RC4_Key[j] = S_BOX[(RC4_Seed ^ X)];
  }

  for(; j < 16; j ++) {
    RC4_Block[16 + j] = RC4_Block[32 + j] = (16 - len);
    RC4_Seed = RC4_Key[j] = S_BOX[(RC4_Seed ^ (16 - len))];
  }

  RC4_MixTable1(RC4_Block);
  for(j = 0; j < 16; j ++) {
    BYTE X = RC4_Key[j];

    RC4_Block[16 + j] = X;
    RC4_Block[32 + j] = (RC4_Block[j] ^ X);
  }
  RC4_MixTable2(RC4_Block);
  /*printf("4       : "); for(i = 0; i < 32; i ++) printf("%X",RC4_Block[i]); printf("!!!\n"); */
  return RC4_Block;
}

static void RC4_MixTable1(BYTE * const BLK) {
  register int i;
  register BYTE X = 0;
  register const BYTE *S;

  S = &S_BOX[48];
  for(i =  0; i < 16; i ++, S --)  X = (BLK[i]  = S[X]);
  for(i = 16; i < 48; i ++, S --)  X = (BLK[i] ^= S[X]);

  for(i = 17; i > 0; i --) {
    register int j;

    S = &S_BOX[48];
    for(j = 0; j < 48; j ++, S --) X = (BLK[j] ^= S[X]);
  }
}

static void RC4_MixTable2(BYTE * const BLK) {
  register int i;
  register BYTE X = 0;
  register const BYTE *S;

  for(i = 17; i > 0; i --) {
    register int j;

    S = &S_BOX[48];
    for(j = 0; j < 48; j ++, S --) X = (BLK[j] ^= S[X]);
  }

  S = &S_BOX[48];
  for(i = 0; i < 16; i ++, S --)   X = (BLK[i] ^= S[X]);
}

