""" The base class for all editors. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.debug.api import log_called_from
from enthought.envisage.api import Application
from enthought.traits.api import Any, Bool, HasTraits, Instance, List, Property
from enthought.traits.api import Str


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class Editor(HasTraits):
    """ The base class for all editors. """

    #### 'Editor' interface ###################################################

    # The application that the editor is part of. This is a convenience
    # property and is equivalent to 'self.window.application'.
    application = Property(Instance(Application))

    # The toolkit-specific control that represents the editor.
    #
    # The framework sets this to the value returned by 'create_control'.
    control = Any

    # Does the editor currently have the focus?
    has_focus = Bool(False)

    # Is the resource that the editor is editing 'dirty' i.e., has it been
    # modified but not saved?
    dirty = Bool(False)

    # The editor's globally unique identifier.
    id = Str

    # The editor's name.
    name = Str

    # The resource that the editor is editing.
    #
    # The framework sets this when the editor is created.
    resource = Any

    # The current selection within the editor.
    selection = List

    # The workbench window that the editor is in.
    #
    # The framework sets this when the editor is created.
    window = Instance('enthought.envisage.workbench.Window')

    ###########################################################################
    # 'Editor' interface.
    ###########################################################################

    #### Trait Defaults ######################################################

    def _id_default(self):
        return 'Editor_%d' % id(self.resource)

    #### Properties ###########################################################

    def _get_application(self):
        """ Property getter. """

        return self.window.application

    #### Methods ##############################################################

    def close(self):
        """ Closes the editor.

        This method is not currently called by the framework itself as the user
        is normally in control of the editor lifecycle. Call this if you want
        to control the editor lifecycle programmatically.

        """

        if self.control is not None:
            logger.debug('closing editor [%s]', self)
            self.window.close_editor(self)

        else:
            logger.error('editor [%s] is not open', self)

        return

    def create_control(self, parent):
        """ Creates the toolkit-specific control that represents the editor.

        The parameter *parent* is the toolkit-specific control that is the
        editor's parent.

        Editor implementors should override this!

        """

        # fixme: We only call the 'protected' method '_create_contents' here
        # to make editors backward compatible with the editors in the old UI
        # plugin! Editor implementors should really override *this* method.
        logger.warn('EDITOR [%s] FAILED TO OVERRIDE "create_control"', self)
        control = self._create_contents(parent)

        return control

    def destroy_control(self):
        """ Destroys the toolkit-specific control that represents the editor.

        """

        if self.control is not None:
            logger.debug('destroying control for editor [%s]', self)
            self.control.Destroy()

        else:
            logger.debug('no control to destroy for editor [%s]', self)

        return

    def set_focus(self):
        """ Sets the focus to the appropriate control in the editor.

        By default we set the focus to be the editor's top-level control.
        Override this method if you need to give focus to some other child
        control.

        """

        if self.control is not None:
            self.control.SetFocus()

        return

    # Convenience methods for common operations on the service registry.
    def get_service(self, interface, query=None):
        """ Returns a service that implements the specified interface.

        Raises a **SystemError** if no such service is found.

        """

        logger.warn('DEPRECATED: Use self.application.get_service')
        log_called_from()

        return self.application.get_service(interface, query)

    def get_services(self, interface, query=None):
        """ Returns all services that match the specified query.

        If no services match the query, then an empty list is returned.

        """

        logger.warn('DEPRECATED: Use self.application.get_services')
        log_called_from()

        return self.application.get_services(interface, query)

    ###########################################################################
    # DEPRECATED Protected 'Editor' interface.
    ###########################################################################

    def _create_contents(self, parent):
        """ Create the toolkit-specific control that represents the editor.

        'parent' is the toolkit-specific control that is the editor's parent.

        DEPRECATED: Override 'create_control' instead.

        """

        logger.warn('EDITOR [%s] FAILED TO OVERRIDE "_create_contents"', self)

        # By default we create a yellow panel!
        import wx

        panel = wx.Panel(parent, -1)
        panel.SetBackgroundColour("yellow")
        panel.SetSize((100, 200))

        return panel

#### EOF ######################################################################
