#-------------------------------------------------------------------------------
#
#  (c) Copyright 2006-2007 by Enthought, Inc.
#  All rights reserved.
#
#  The repository plugin.
#
#-------------------------------------------------------------------------------

""" The repository plugin. """


#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import logging

from enthought.envisage \
    import Plugin

from repository \
    import Repository

from repository_plugin_definition \
    import RepositoryRootFactory


logger = logging.getLogger(__name__)


#-------------------------------------------------------------------------------
#  Constants:
#-------------------------------------------------------------------------------

# The repository service id:
IREPOSITORY = 'enthought.envisage.repository.IRepository'

#-------------------------------------------------------------------------------
#  'RepositoryPlugin' class:
#-------------------------------------------------------------------------------

class RepositoryPlugin ( Plugin ):
    """ The repository plugin. """

    #---------------------------------------------------------------------------
    # 'Plugin' interface.
    #---------------------------------------------------------------------------

    def start ( self, application ):
        """ Starts the plugin.
        """
        # Create and register the repository as a service:
        self.register_service( IREPOSITORY,
                               self._create_repository( application ) )

    def save_preferences ( self ):
        """ Saves the plugin's user preferences.
        """
        # Save the current set of repository roots:
        self.preferences.set( 'roots',
                            [ rr for rr in self.get_service( IREPOSITORY ).roots
                              if rr.persistent ] )

        # Save them:
        super( RepositoryPlugin, self ).save_preferences()

    #---------------------------------------------------------------------------
    # Private interface.
    #---------------------------------------------------------------------------

    def _create_repository ( self, application ):
        """ Creates the repository.
        """
        # Create a new repository:
        repository = Repository( application = application )

        roots = []
        # Get all the RepositoryRootFactory extensions that apply to us:
        for rrf in self.get_extensions( RepositoryRootFactory ):
            if rrf.id == IREPOSITORY:
                factory = self.import_symbol( rrf.class_name )

                # Add the roots that the factory defines:
                roots.extend( factory( plugin = self ).roots() )

        # Sort the roots by their priority:
        roots.sort( lambda l, r: cmp( l.priority, r.priority ) )

        # Log out the identified roots
        msg = ', '.join(['"%s"=%s' % (r.name, r.path) for r in roots])
        logger.info('Using repository roots: %s', msg)

        # Add the roots to the repository:
        repository.roots = roots

        return repository

