""" A wizard for creating a new resource. """


# Enthought library imports.
from enthought.naming.api import Binding, Context
from enthought.pyface.wizard.api import SimpleWizard
from enthought.traits.api import Any, Instance

# Local imports.
from new_other_wizard_controller import NewOtherWizardController
from new_resource_wizard import NewResourceWizard
from resource_factory import ResourceFactory
from wizard_selection_page import WizardSelectionPage


class GenericResourceFactory(ResourceFactory):
    """ A factory that creates generic resources. """

    # The wizard used to create the resource.
    wizard = Any

    ###########################################################################
    # 'ResourceFactory' interface.
    ###########################################################################

    def create_resource(self):
        """ Creates the resource. """

        return self.wizard.factory.create_resource()


class NewOtherWizard(NewResourceWizard):
    """ A wizard for creating a new resource. """

    #### 'Window' interface ###################################################

    # The window title.
    title = 'New Resource'

    #### 'Wizard' interface ###################################################

    # The wizard controller provides the pages displayed in the wizard, and
    # determines when the wizard is complete etc.
    controller = Instance(NewOtherWizardController, ())

    #### 'NewResourceWizard' interface ########################################

    # The resource factory that the wizard is configuring.
    factory = Instance(GenericResourceFactory, ())

    #### 'NewOtherWizard' interface ###########################################

    # The default folder in which to create the resource.
    default_folder = Instance(Binding)

    # The root of the wizard hierarchy to be displayed in the tree.
    root = Instance(Context)

    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, **traits):
        """ Creates a new wizard. """
        # Base class constructor.
        super(NewOtherWizard, self).__init__(**traits)

        self.factory.on_trait_change(self._on_wizard_changed, 'wizard')

        return

    ###########################################################################
    # 'NewResourceWizard' interface.
    ###########################################################################

    def reset(self):
        """ Reset the wizard to the initial state. """

        # This wizard contains a single page that allows the user to select the
        # folder in which to create the resource, and to give the new resource
        # a name.
        self.pages = [
            WizardSelectionPage(id='first', root=self.root, obj=self.factory)
        ]

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    #### Trait event handlers #################################################

    #### Static ####

    def _factory_changed(self, old, new):
        """ Called when the factory is changed. """

        if old is not None:
            old.on_trait_change(self._on_wizard_changed, 'wizard', remove=True)

        if new is not None:
            new.on_trait_change(self._on_wizard_changed, 'wizard')

        return

    #### Dynamic ####

    def _on_wizard_changed(self, old, new):
        """ Called when the actual resource wizard is changed. """

        if new is not None:
            self.next_wizard = new
            new.factory.parent_folder = self.default_folder

        return

#### EOF ######################################################################
