/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
*****/
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <locale.h>
#include <dirent.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include "magpie.h"

/*+
Recursively remove all files under the specified root.
+*/
int rmrf (const char *basename)
{
	DIR *dirp;
	struct dirent *dp;
	struct stat buf;
	char pathname[256];

	dirp = opendir (basename);
	if (dirp) {
		while ((dp = readdir (dirp)) != NULL) {
			if (strcmp (dp->d_name, ".") == 0)
				continue;
			if (strcmp (dp->d_name, "..") == 0)
				continue;

			sprintf (pathname, "%s/%s", basename, dp->d_name);
			stat (pathname, &buf);
			if (S_ISDIR (buf.st_mode) && !S_ISLNK (buf.st_mode)) {
				rmrf (pathname);
				rmdir (pathname);
			}
			unlink (pathname);
		}
		closedir (dirp);
	}
	return 0;
}


/*+
Compress the specified file.
+*/
int gzip (const char *filename)
{
	char pathname[256];
	pid_t pid;
	int status;

	sprintf (pathname, "%s.gz", filename);
	if (access (pathname, R_OK) == 0)
		return 0;

	pid = fork();

	if (pid == 0) {
		/* surrender any hidden root permission first! */
		setuid (geteuid ());
		if (execl ("/bin/gzip", "gzip", "-9", filename, NULL) == -1) {
			printf ("unable to compress '%s'\n", filename);
		}
		/* should never get here */
		exit (0);
	}
	else if (pid == -1)
		return -1;

	waitpid (pid, &status, 0);
	return status; 
}


/*+
Check the MD5 checksums.
I want to do this with fork/exec, but the usual code never seems
to see the EOF.  Obviously I'm missing some key system call.
We change our fsuid to root so we can open the files, but don't
open the rest of the risks of running a subshell as root.
+*/
FILE *md5sum (const char *filename)
{
	char pathname[512];
	FILE *fp;
	uid_t fsuid;

	sprintf (pathname,
		"/bin/cat %s | (cd /; /usr/bin/md5sum -c) 2>&1", filename);
	fsuid = setfsuid (0);
	fp = popen (pathname, "r");
	setfsuid (fsuid);
	return fp;

#if 0
	int fildes[2];
	pid_t pid;

	if (pipe (fildes) == -1) {
		perror ("pipe");
		return -1;
	}
	*fp = fdopen (fildes[0], "r");

	pid = fork ();
	if (pid == 0) {
		freopen (filename, "r", stdin);
		close (1);
		dup2 (fildes[1], 1);
		close (2);
		dup2 (fildes[1], 2);

		seteuid (0);
		chdir ("/");
		execl ("/usr/bin/md5sum", "md5sum", "-c", NULL);
		exit (0);
	}
	return pid;
#endif
}
