import os, sys, re, getopt

class CLParser:
    def __init__(self):
        self.arguments = {}
        self._opts = None
        self._args = None

    def parse_args(self):
        try:
            self._opts, self._args = getopt.getopt(sys.argv[1:], \
                'c:C:dDf:hiIl:L:m:nopPqsSt:Tu:vV', ['config=', 'checks=', \
                'debug', 'disable-local', 'format=', 'help', 'info', \
                'informational', 'lab-root=', 'list-checks=', \
                'more-overrides=', 'no-cull', 'show-overridden', \
                'print-overrides', 'profiling', 'quiet', 'show-tag', \
                'seteuid', 'types=', 'traceback', 'unpack=', 'verbose', \
                'version'])
        except getopt.GetoptError:
            self.usage(exitstatus=7)
        self.parse()
        self.set_defaults()
        self.add_files()
        
    def parse(self):
        for o, a in self._opts:
            if o in ('-c', '--config'):
                if os.path.exists(os.path.abspath(a)):
                    self.arguments['config'] = os.path.abspath(a)
                else:
                    raise CLParsingException, "Config file doesn't exist"
            elif o in ('-C', '--checks'):
                self.arguments['checks'] = a
            elif o in ('-d', '--debug'):
                if self.arguments.has_key('debug'):
                    self.arguments['debug'] += 1
                else:
                    self.arguments['debug'] = 1
            elif o in ('-D', '--disable-local'):
                self.arguments['disable-local'] = 1
            elif o in ('-f', '--format'):
                self.arguments['format'] = a
            elif o in ('-h', '--help'):
                self.usage()
            elif o in ('-i', '--info'):
                self.arguments['info'] = 1
            elif o in ('-I', '--informational'):
                print _("N: Informational is deprecated, use -t E,W,I")
            elif o in ('-l', '--lab'):
                if os.path.isdir(os.path.abspath(a)):
                    self.arguments['lab-root'] = os.path.abspath(a)
                else:
                    raise CLParsingException, "Lab dir doesn't exist"
            elif o in ('-L', '--list-checks'):
                if a == 'all':
                    self.arguments['list-checks'] = (('binary', 'source'), \
                        (1, 2))
                elif a in ('bin1', 'bin2', 'src1', 'src2'):
                    mapping = {'bin': 'binary', 'src': 'source'}
                    self.arguments['list-checks'] = ((mapping[a[:-1]],), \
                        (int(a[-1]),))
                else:
                    raise CLParsingException, \
                        "%s is unknown arg to list checks" % a
            elif o in ('-m', '--more-overrides'):
                if os.path.isfile(os.path.abspath(a)):
                    self.arguments['more-overrides'] = os.path.abspath(a)
                else:
                    raise CLParsingException, "Overrides file doesn't exist"
            elif o in ('-n', '--no-cull'):
                self.arguments['no-cull'] = 1
            elif o in ('-o', '--show-overridden'):
                self.arguments['show-overridden'] = 1
            elif o in ('-p', '--print-overrides'):
                self.arguments['print-overrides'] = 1
            elif o in ('-P', '--profiling'):
                self.arguments['profiling'] = 1
            elif o in ('-q', '--quiet'):
                self.arguments['quiet'] = 1
            elif o in ('-s', '--show-tag'):
                self.arguments['show-tag'] = 1
            elif o in ('-S', '--seteuid'):
                self.arguments['seteuid'] = 1
            elif o in ('-t', '--types'):
                if not self.arguments.has_key('types'):
                    self.arguments['types'] = []
                for type in re.split(r', ?', a):
                    if type in ('E', 'W', 'X', 'I'):
                        self.arguments['types'].append(type)
                        if type == 'X' and self.arguments.has_key('quiet'):
                            print _("N: eXperimental type enabled.")
                    else:
                        raise CLParsingException, \
                            "'%s' isn't a valid type" % type
            elif o in ('-T', '--traceback'):
                self.arguments['traceback'] = 1
            elif o in ('-u', '--unpack'):
                if a in ('1', '2'):
                    self.arguments['unpack'] = int(a)
                    self.arguments['no-cull'] = 1
                else:
                    raise CLParsingException, "Unpack level must be 1 or 2"
            elif o in ('-v', '--verbose'):
                if self.arguments.has_key('verbose'):
                    self.arguments['verbose'] += 1
                else:
                    self.arguments['verbose'] = 1
            elif o in ('-V', '--version'):
                print _("Linda, version %s") % "VERSION"
                sys.exit(0)

    def add_files(self):
        usable_files = filter(os.path.exists, self._args)
        self.arguments['files'] = []
        for i in usable_files:
            extension = os.path.splitext(i)[1]
            if extension in ('.deb', '.udeb', '.dsc', '.changes'):
                self.arguments['files'].append(os.path.abspath(i))

    def set_defaults(self):
        for key in ('debug', 'disable-local', 'info', 'no-cull', \
            'show-overridden', 'print-overrides', 'profiling', 'quiet', \
            'seteuid', 'traceback', 'verbose'):
            if self[key] == '':
                self.arguments[key] = 0
        if not self.arguments.has_key('types'):
            self.arguments['types'] = ['E', 'W']

    def __getitem__(self, key):
        if self.arguments.has_key(key):
            return self.arguments[key]
        else:
            return ''

    def __setitem__(self, key, data):
        self.arguments[key] = data

    def usage(self, exitstatus=0):
        print \
_("""Linda, a Debian package checker.

Usage: linda [options] [.dsc|.deb|.udeb|.changes] ...
-c CONFIG, --config=CONFIG := Specify config directory
-C CHECK_STRING, --checks=CHECK_STRING := Specify checks to run
-d, --debug := Set debug
-D, --disable-local := Don't register local checks
-f FORMAT, --format=FORMAT := Set output format, -f help lists them
-h, --help := Show this help, and exit
-i, --info := Show more information about errors and warnings
-l LAB_ROOT, --lab=LAB_ROOT := Set the directory the lab should be in
-L LIST_CHECKS, --list-checks=LIST_CHECKS := List checks and exit
-m MORE_OVERRIDES, --more-overrides=MORE_OVERRIDES := Also parse overrides from
this file
-n, --no-cull := Don't delete the lab after tests are completed
-o, --show-overridden := Display errors that are overridden
-p, --print-overrides := Print parsed overrides and exit
-P, --profiling := Print some profiling stats at exit
-q, --quiet := Quieten me a little bit
-s, --show-tag := Also print out the tag when printing errors
-S, --seteuid := Don't seteuid() to nobody when running as root
-t TYPES_STRING, --types=TYPES_STRING := Specify the types to show. Default is
E,W,X
-T, --traceback := Show a traceback from any exception
-u UNPACK_LEVEL, --unpack=UNPACK_LEVEL := Set unpack level, implies -n
-v, --verbose := Set verbose
-V, --version := Display version
""")
        sys.exit(exitstatus)

class CLParsingException(Exception):
    pass

