// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Andrea Parrella (anandrea@iname.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviURL"

#define _KVI_URL_CPP_

#include <qcheckbox.h>
#include <qfile.h>
#include <qlayout.h>

#ifdef QT_NO_ASCII_CAST
	#undef QT_NO_ASCII_CAST
#endif

#define __KVIRC_PLUGIN__

#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_mdi_manager.h"
#include "kvi_process.h"
#include "kvi_url.h"
#include "kvi_url_banframe.h"
#include "kvi_url_configdialog.h"
#include "kvi_url_dialog.h"
#include "kvi_url_toolbar.h"
#include "kvi_winproperties.h"
#include "kvirc_plugin.h"

const char *g_pUrlListFilename = "/list.kviurl";
const char *g_pBanListFilename = "/list.kviban";

void saveUrlList()
{
	KviStr urllist;
	g_pApp->getLocalKVIrcDirectory(urllist, KviApp::ConfigPlugins);
	urllist += g_pUrlListFilename;
	QFile file;
	file.setName(urllist.ptr());
	file.open(IO_WriteOnly);

	QTextStream stream(&file);

	stream << g_pList->count() << endl;

	for(KviUrl *tmp = g_pList->first(); tmp; tmp = g_pList->next() ) {
		stream << tmp->url.utf8()      << endl;
		stream << tmp->window.utf8()   << endl;
		stream << tmp->count           << endl;
		stream << tmp->timestamp.ptr() << endl;
	}
	file.flush();
	file.close();
}

void loadUrlList()
{
	KviStr urllist;
	g_pApp->getLocalKVIrcDirectory(urllist, KviApp::ConfigPlugins);
	urllist += g_pUrlListFilename;
	QFile file;
	file.setName(urllist.ptr());
	if( !file.exists() ) {
		KviStr s;
		s.sprintf("touch %s", urllist.ptr());
		KviProcess proc;
		if( !proc.run(s.ptr()) )
			g_pApp->warningBox(_i18n_("Could not execute %s"), s.ptr());
	}
	if( !file.open(IO_ReadOnly) ) return;
	QTextStream stream(&file);

	g_pList->clear();

	for( KviUrlDlgList *tmpitem = g_pUrlDlgList->first(); tmpitem; tmpitem = g_pUrlDlgList->next() ) {
		if( tmpitem->dlg ) tmpitem->dlg->m_pUrlList->clear();
	}
	KviUrl *tmp;
	int i = 0;
	int num = stream.readLine().toInt();
	while( (!stream.eof()) && (i<num) ) {
		tmp = new KviUrl();
		tmp->url       = stream.readLine();
		tmp->window    = stream.readLine();
		tmp->count     = stream.readLine().toInt();
		tmp->timestamp = stream.readLine();

		g_pList->append(tmp);

		for( KviUrlDlgList *tmpitem = g_pUrlDlgList->first(); tmpitem; tmpitem = g_pUrlDlgList->next() ) {
			if( !tmpitem->dlg ) continue;
			QString tmpCount;
			tmpCount.setNum(tmp->count);
			tmpitem->dlg->addUrl(tmp->url, tmp->window, tmpCount, QString(tmp->timestamp.ptr()));
		}
		i++;
	}
	file.close();
}

void saveBanList()
{
	KviStr banlist;
	g_pApp->getLocalKVIrcDirectory(banlist, KviApp::ConfigPlugins);
	banlist += g_pBanListFilename;
	QFile file;
	file.setName(banlist.ptr());
	file.open(IO_WriteOnly);

	QTextStream stream(&file);

	stream << g_pBanList->count() << endl;
	for(QString *tmp = g_pBanList->first(); tmp; tmp = g_pBanList->next() )
		stream << tmp->utf8() << endl;
	file.flush();
	file.close();
}

void loadBanList()
{
	KviStr banlist;
	g_pApp->getLocalKVIrcDirectory(banlist, KviApp::ConfigPlugins);
	banlist += g_pBanListFilename;
	QFile file;
	file.setName(banlist.ptr());
	if( !file.exists() ) {
		KviStr s;
		s.sprintf("touch %s", banlist.ptr());
		KviProcess proc;
		if( !proc.run(s.ptr()) )
			g_pApp->warningBox(_i18n_("Could not execute %s"), s.ptr());
	}
	if( !file.open(IO_ReadOnly) ) return;
	QTextStream stream(&file);

	g_pBanList->clear();

	int i = 0;
	int num = stream.readLine().toInt();
	while( (!stream.eof()) && (i < num) ) {
		QString *tmp = new QString(QString::fromUtf8(stream.readLine()));
		g_pBanList->append(tmp);
		i++;
	}
	file.close();
}

bool urllist_command(KviPluginCommandStruct *cmd)
{
	urllist(cmd->frame);
	return true;
}

KviUrlDlgList *findFrame(KviFrame *frame)
{
	KviUrlDlgList *tmpitem;
	for( tmpitem = g_pUrlDlgList->first(); tmpitem; tmpitem = g_pUrlDlgList->next() ) {
		if( frame == tmpitem->frm ) break;
	}
	if( !tmpitem ) {
		KviUrlDlgList *udl = new KviUrlDlgList();
		udl->frm     = frame;
		udl->dlg     = 0;
		udl->toolbar = 0;
		g_pUrlDlgList->append(udl);
		tmpitem = g_pUrlDlgList->current();
	}
	return tmpitem;
}

bool urllist(KviFrame *frame)
{
	KviUrlDlgList *tmpitem = findFrame(frame);
	if( tmpitem->dlg ) return false;

	tmpitem->dlg = new KviUrlDialog(g_pList, tmpitem->frm);
	tmpitem->frm->addWindow(tmpitem->dlg);

	for(KviUrl *tmp = g_pList->first(); tmp; tmp = g_pList->next() ) {
		QString tmpCount;
		tmpCount.setNum(tmp->count);
		tmpitem->dlg->addUrl(tmp->url, tmp->window, tmpCount, QString(tmp->timestamp.ptr()));
	}
	return true;
}

bool urllistconfig(KviPluginCommandStruct *cmd)
{
	if( !g_pConfigDialog ) g_pConfigDialog = new KviUrlConfigDialog();
	return true;
}

/**
 * Return 0 if no occurrences of the url were found.
 */
int check_url(KviPluginCommandStruct *cmd)
{
	int tmp = 0;

	for( QString *tmpitem = g_pBanList->first(); tmpitem; tmpitem = g_pBanList->next() ) {
		if( cmd->params->at(1)->contains(tmpitem->utf8()) ) tmp++;
	}
	if( tmp > 0 ) return tmp;

	for( KviUrl *u = g_pList->first(); u; u = g_pList->next() ) {
		if( u->url == cmd->params->at(1)->ptr() ) {
			u->window = cmd->window->caption();
			u->count++;
			tmp++;
		}
	}

	for( KviUrlDlgList *tmpitem = g_pUrlDlgList->first(); tmpitem; tmpitem = g_pUrlDlgList->next() ) {
		if( tmpitem->dlg ) {
			KviListViewItemIterator lvi(tmpitem->dlg->m_pUrlList);
			for( ; lvi.current(); ++lvi ) {
				if( lvi.current()->text(0) == cmd->params->at(1)->ptr() ) {
					int tmpCount = lvi.current()->text(2).toInt();
					tmpCount++;
					QString tmpStr;
					tmpStr.setNum(tmpCount);
					lvi.current()->setText(2, tmpStr);
					lvi.current()->setText(1, cmd->window->caption());
				}
			}
		}
	}
	return tmp;
}

bool urllist_plugin_hook_onUrl(KviPluginCommandStruct *cmd)
{
	if( check_url(cmd) != 0 ) return true;

	KviUrl *tmp = new KviUrl;
	KviStr tmpTimestamp;
	QDate d = QDate::currentDate();
	KviStr date(KviStr::Format, "%d-%d%d-%d%d", d.year(), d.month() / 10, d.month() % 10, d.day() / 10, d.day() % 10);
	tmpTimestamp = "[" + date + "]" + " [";
	tmpTimestamp += QTime::currentTime().toString() + "]";

	tmp->url = kvirc_plugin_param(cmd, 1);
	tmp->window = cmd->window->caption();
	tmp->count = 1;
	tmp->timestamp = tmpTimestamp;

	g_pList->append(tmp);

	for( KviUrlDlgList *tmpitem = g_pUrlDlgList->first(); tmpitem; tmpitem = g_pUrlDlgList->next() ) {
		if( tmpitem->dlg ) {
			QString tmpCount;
			tmpCount.setNum(tmp->count);
			tmpitem->dlg->addUrl(tmp->url, tmp->window, tmpCount, QString(tmp->timestamp.ptr()));
			tmpitem->dlg->highlight();
		} else {
			if( tmpitem->toolbar )
				tmpitem->toolbar->setButtonIcon(tmpitem->toolbar->m_urlButton, "urlhigh.png");
		}
	}
	return true;
}

bool urllist_plugin_hook_onStartup(KviPluginCommandStruct *cmd)
{
	KviConfig cfg(g_configPath.ptr());
	cfg.setGroup("ConfigDialog");
	if( cfg.readBoolEntry("LoadUrlToolbarOnStartup", false) ) {
		KviUrlDlgList *tmpitem = findFrame(cmd->frame);
		tmpitem->toolbar = new KviUrlToolBar(cmd->frame);
	}
	return false;
}

bool url_plugin_init(KviPluginCommandStruct *cmd)
{
	g_pList = new QPtrList<KviUrl>;
	g_pList->setAutoDelete(true);

	g_pUrlDlgList = new QPtrList<KviUrlDlgList>;
	g_pUrlDlgList->setAutoDelete(true);

	g_pBanList = new QPtrList<QString>;
	g_pBanList->setAutoDelete(true);

	kvirc_plugin_register_command(cmd->handle, "URLLIST",       urllist_command);
	kvirc_plugin_register_command(cmd->handle, "URLLISTCONFIG", urllistconfig);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnUrl,     urllist_plugin_hook_onUrl);
	kvirc_plugin_add_hook(cmd->handle, KviEvent_OnStartup, urllist_plugin_hook_onStartup);

	g_pApp->getLocalKVIrcDirectory(g_configPath, KviApp::ConfigPlugins);
	g_configPath += "/kvi.url.conf";

	loadUrlList();
	loadBanList();

	return true;
}

void url_plugin_cleanup()
{
	KviConfig cfg(g_configPath.ptr());
	cfg.setGroup("ConfigDialog");
	if( cfg.readBoolEntry("SaveUrlListOnUnload", false) == true )
		saveUrlList();
	for( KviUrlDlgList *tmpitem = g_pUrlDlgList->first(); tmpitem; tmpitem = g_pUrlDlgList->next() ) {
		if( tmpitem->toolbar ) {
			delete tmpitem->toolbar;
			tmpitem->toolbar = 0;
		}
		if( tmpitem->dlg ) {
			tmpitem->dlg->close();
			tmpitem->dlg = 0;
		}
	}
	delete g_pList;    g_pList    = 0;
	delete g_pBanList; g_pBanList = 0;
}

void url_plugin_config()
{
	if( !g_pConfigDialog ) g_pConfigDialog = new KviUrlConfigDialog();
}

/**
	@document: doc_plugin_url.kvihelp
	@title: The Url list plugin
		This plugin keeps track of all urls shown in KVIrc windows.
		<H3>Exported commands:</H3>
		<B>/urllist</B>: this command opens a window containing the url list.
		In the list there is the following information:<BR>
		<U>Window</U> : window where the url has been shown last<BR>
		<U>Count</U> : number of urls shown<BR>
		<U>Timestamp</U> : date/time when the url has been shown first<BR>
		Right-clicking on the url column of the list pops up a menu, through which
		you can remove the selected item, find the url in the window it appeared last, and
		say it to: <I>@Console, Channels, Queries, DCC Chats</I> and <I>User windows</I>.<BR>
		The list is saved to file when you click on the menu item or when you unload the plugin
		on condition that you have checked the relevant checkbox in the configuration dialog.<BR>
		You can also open the url in your web browser by double clicking on it in the url list window.<BR><BR>

		<B>/urllistconfig</B> : this command opens a configuration window where it is possible
		to setup the plugin's parameters. You can also open this window by
		using the popup menu in the url list window or by clicking on the "configure plugin" button
		in the plugins options.<BR><BR>
		<H3>Configure dialog options:</H3>
		There is an option which loads a small <U>toolbar</U> when the KVIrc onStartup event is fired.<BR>
		In the toolbar you will find two icons:<BR>
		<UL><LI>The first toggles the url list window and when the onUrl event is fired the button is highlighted;</LI>
		<LI>The second opens the configure dialog.</LI></UL>
		There is also a ban list widget, which sports a list of words that plugin should not catch.<BR><BR>
		<I>E.g.<BR>
		<blockquote>if the word "ftp" is inserted in the ban list and if in a window there is an output like "ftp.kvirc.net",
		the url will not be caught.</blockquote></I>
		<HR>
		Mail me if you have any suggestion or you want to report a bug.<BR>
		<B>Andrea 'YaP' Parrella</B> &lt;anandrea@iname.com&gt;<BR><BR>
		Thanks to:<BR>
		<B>Szymon Stefanek</B> and <B>Till Bush</B> for their help.<BR>
		<B>Ted Podgurski</B> for toolbar's icons.
*/
void url_plugin_help()
{
	kvirc_plugin_show_help_page("doc_plugin_url.kvihelp");
}

KviPlugin kvirc_plugin =
{
	"URL",
	"URL list plugin",
	"0.1",
	"Andrea Parrella <anandrea@iname.com>",
	"<html>"
		"This plugin implements a list of URLs<br>"
		"shown in KVIrc windows.<br><br>"
		"Commands it exports are:<br>"
		"<b>/URLLIST</b><br>"
		"<b>/URLLISTCONFIG</b>"
		"</html>",
	url_plugin_init,
	url_plugin_cleanup,
	url_plugin_config,
	url_plugin_help
};
