// =============================================================================
//
//      --- kvi_taskbar.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviTaskbar"

#define _KVI_TASKBAR_CPP_

#include <qpixmap.h>

#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_frame.h"
#include "kvi_mdi_manager.h"
#include "kvi_taskbar.h"
#include "kvi_taskbarbutton.h"
#include "kvi_transparentwidget.h"

#define KVI_TASKBAR_BUTTON_MINIMUM_WIDTH 80

/*
	@quickhelp: KviTaskBar
	@widget: Taskbar
		This window lists the currently open windows.<br>
		Each button corresponds to a single MDI (child) window.<br>
		The button is enabled (clickable) when the window is docked, and can be
		pressed to bring it to the top of the other docked windows.<br>
		The button text becomes red when new output is shown in the window and it is not the active one.<br>
		The button text becomes blue when new highlighted output is shown in the window and it is not the active one.<br>
		By right-clicking on the handle you will be able to dock it in
		one of the four sides of the main window.
*/

/**
 * KviTaskBar
 */
KviTaskBar::KviTaskBar(KviFrame *parent)
	: KviToolBar("TaskBar", parent, parent, Qt::DockBottom, true, "taskbar_toolbar")
{
	m_pFrm  = parent;
	m_pBase = new KviTransparentWidget(this, "taskbar_base");

	recalcButtonHeight();

	m_pMemBuffer = new QPixmap(KVI_TASKBAR_MINIMUM_WIDTH, m_iButtonHeight);

	setStretchableWidget(m_pBase);

	m_pButtonList = new QPtrList<KviTaskBarButton>;
	m_pButtonList->setAutoDelete(true);

	setVerticalStretchable(true);
	setHorizontalStretchable(true);
	setBackgroundMode(PaletteBackground);
	connect(this, SIGNAL(orientationChanged(Orientation)), this, SLOT(setupMinimumSizes(Orientation)));
}

KviTaskBar::~KviTaskBar()
{
	if( m_pButtonList ) {
		delete m_pButtonList;
		m_pButtonList = 0;
	}
	if( m_pMemBuffer ) {
		delete m_pMemBuffer;
		m_pMemBuffer  = 0;
	}
}

void KviTaskBar::buttonLeftClicked(KviWindow *win_ptr, KviTaskBarButton *btn)
{
	m_pFrm->taskbarButtonLeftClicked(win_ptr, btn);
}

void KviTaskBar::buttonRightClicked(KviWindow *win_ptr, KviTaskBarButton *btn)
{
	m_pFrm->taskbarButtonRightClicked(win_ptr, btn);
}

KviTaskBarButton *KviTaskBar::addWinButton(KviWindow *win_ptr)
{
	__range_valid(win_ptr);
	KviTaskBarButton *b = new KviTaskBarButton(m_pBase, this, win_ptr, m_pMemBuffer);
	m_pButtonList->append(b);
	b->show();
	layoutButtons();
	return b;
}

void KviTaskBar::removeWinButton(KviWindow *win_ptr)
{
	__range_valid(win_ptr);
	KviTaskBarButton *b = getButton(win_ptr);
	__range_valid(b);
	if( b ) {
		m_pButtonList->removeRef(b);
		layoutButtons();
	}
}

void KviTaskBar::windowAttached(KviWindow *win_ptr, bool bAttached)
{
	KviTaskBarButton *b = getButton(win_ptr);
	b->setAttachState(bAttached);
}

KviTaskBarButton *KviTaskBar::getButton(KviWindow *win_ptr)
{
	for( KviTaskBarButton *b = m_pButtonList->first(); b; b = m_pButtonList->next() ) {
		if( b->m_pWindow == win_ptr )
			return b;
	}
	__debug("WARNING: button not found!");
	return 0;
}

KviTaskBarButton *KviTaskBar::getNextWindowButton(bool bRight, KviWindow *win_ptr)
{
	if( bRight ) {
		for( KviTaskBarButton *b = m_pButtonList->first(); b; b = m_pButtonList->next() ) {
			if( b->m_pWindow == win_ptr ) {
				b = m_pButtonList->next();
				if( !b )
					b = m_pButtonList->first();
				if( win_ptr != b->m_pWindow )
					return b;
				else
					return 0;
			}
		}
	} else {
		for( KviTaskBarButton *b = m_pButtonList->first(); b; b = m_pButtonList->next() ) {
			if( b->m_pWindow == win_ptr ) {
				b = m_pButtonList->prev();
				if( !b )
					b = m_pButtonList->last();
				if( win_ptr != b->m_pWindow )
					return b;
				else
					return 0;
			}
		}
	}
	return 0;
}

void KviTaskBar::setActiveButton(KviWindow *win_ptr)
{
	for( KviTaskBarButton *b = m_pButtonList->first(); b; b = m_pButtonList->next() ) {
		b->setOn(b->m_pWindow == win_ptr);
	}
}

void KviTaskBar::layoutButtons()
{
	int btnCount = m_pButtonList->count();
	int curPos   = 0;
	if( btnCount == 0 )
		return;

	if( orientation() == Qt::Horizontal ) {
		// Horizontal layout
		setupMinimumSizes(Qt::Horizontal);

		int btnWidth = (m_pBase->width() / btnCount) - 1;
		if( btnWidth < KVI_TASKBAR_BUTTON_MINIMUM_WIDTH )
			btnWidth = KVI_TASKBAR_BUTTON_MINIMUM_WIDTH;

		m_pMemBuffer->resize(btnWidth, m_pBase->height());
		int theY = 0;
		for( KviTaskBarButton *b = m_pButtonList->first(); b; b = m_pButtonList->next() ) {
			b->setGeometry(curPos, theY, btnWidth, m_iButtonHeight);
			curPos += btnWidth;
			if( (curPos + btnWidth) >= m_pBase->width() ) {
				curPos = 0;
				theY  += m_iButtonHeight;
			}
		}
	} else {
		// Vertical layout
		setupMinimumSizes(Qt::Vertical);
		int btnWidth = m_pBase->width();
		m_pMemBuffer->resize(btnWidth, m_iButtonHeight);
		for( KviTaskBarButton *b = m_pButtonList->first(); b; b = m_pButtonList->next() ) {
			b->setGeometry(0, curPos, btnWidth, m_iButtonHeight);
			curPos += m_iButtonHeight;
		}
	}
}

void KviTaskBar::recalcButtonHeight()
{
	QFontMetrics fm(font());
	m_iButtonHeight = fm.height() + KVI_TASKBAR_BUTTON_DOUBLE_BORDER;
	if( m_iButtonHeight < 18 )
		m_iButtonHeight = 18;
	m_pBase->setMinimumSize(QSize(KVI_TASKBAR_MINIMUM_WIDTH, m_iButtonHeight));
}

void KviTaskBar::fontChange(const QFont &oldFont)
{
	recalcButtonHeight();
}

void KviTaskBar::resizeEvent(QResizeEvent *e)
{
	QToolBar::resizeEvent(e);
	layoutButtons();
}

void KviTaskBar::setupMinimumSizes(Orientation o)
{
	if( o == Qt::Horizontal ) {
		int btnCount = m_pButtonList->count();
		if( btnCount == 0 )
			btnCount = 1;
		int btnWidth = (m_pBase->width() / btnCount) - 1;
		int nRows = 1;
		if( btnWidth < KVI_TASKBAR_BUTTON_MINIMUM_WIDTH ) {
			int btnPerRow = m_pBase->width() / KVI_TASKBAR_BUTTON_MINIMUM_WIDTH;
			if( btnPerRow == 0 )
				btnPerRow = 1; // Absurd, but still...
			nRows = (m_pButtonList->count() / btnPerRow);
			if( ((unsigned int) (btnPerRow * nRows)) < m_pButtonList->count() )
				nRows++;
		}
		m_pBase->setMinimumSize(QSize(KVI_TASKBAR_MINIMUM_WIDTH, m_iButtonHeight * nRows));
	} else {
		int hght = m_iButtonHeight * m_pButtonList->count();
		if( hght > m_pFrm->m_pMdi->height() - 18 )
			hght = m_pFrm->m_pMdi->height();
		m_pBase->setMinimumSize(QSize(KVI_TASKBAR_MINIMUM_WIDTH, hght));
	}
}

QSizePolicy KviTaskBar::sizePolicy()
{
	return QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
}

#include "m_kvi_taskbar.moc"
