//
//   File : kvi_mdi.cpp (/usr/build/NEW_kvirc/kvirc/src/kvilib/kvi_mdi.cpp)
//   Last major modification : Tue Jul 6 1999 16:01:04 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Opaque move & resize code by Falk Brettschneider (gigafalk@geocities.com)
//

#define _KVI_MDI_CPP_

//#define _KVI_DEBUG_CLASS_NAME_ "KviMdiClasses"
//#define _KVI_DEBUG_CHECK_RANGE_

//#gnome on irc.mint.net 16.09.99
//[03:44:30] <Zaphod> Does KVirc still use evil MDI? :-)

#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_locale.h"
#include "kvi_mdibutton.h"
#include "kvi_mdicaption.h"
#include "kvi_settings.h"
#include "kvi_mdi.h"

#include <unistd.h>
#include <qpainter.h>
#include <qapplication.h>

#include "kvi_xutils.h"

#include <qcursor.h>
#include <qnamespace.h>
#include <qtooltip.h>

#ifdef HAVE_MATH_H
	#include <math.h>
#endif


//////////////////////////////////////////////////////////////////////////////
// KviMdiChild
//////////////////////////////////////////////////////////////////////////////

#define KVI_MDI_NORESIZE 0
#define KVI_MDI_RESIZE_TOP 1
#define KVI_MDI_RESIZE_LEFT 2
#define KVI_MDI_RESIZE_RIGHT 4
#define KVI_MDI_RESIZE_BOTTOM 8
#define KVI_MDI_RESIZE_TOPLEFT (1|2)
#define KVI_MDI_RESIZE_TOPRIGHT (1|4)
#define KVI_MDI_RESIZE_BOTTOMLEFT (8|2)
#define KVI_MDI_RESIZE_BOTTOMRIGHT (8|4)


//============ KviMdiChild ============//

KviMdiChild::KviMdiChild(KviMdiManager *parent,const char * name)
:QFrame(parent,name)
{
	m_pClient   = 0;
	m_pCaption  = new KviMdiCaption(this,"mdi_caption");
	m_pManager  = parent;

	m_pMinimize = new KviMdiCaptionButton(m_pCaption,KviMdiCaptionButton::Minimize,"minimize_button");
	connect(m_pMinimize,SIGNAL(buttonPressed()),this,SLOT(minimizePressed()));
	QToolTip::add(m_pMinimize,_CHAR_2_QSTRING(__tr("Minimize")));

	m_pMaximize = new KviMdiCaptionButton(m_pCaption,KviMdiCaptionButton::Maximize,"maximize_button");
	connect(m_pMaximize,SIGNAL(buttonPressed()),this,SLOT(maximizePressed()));
	QToolTip::add(m_pMaximize,_CHAR_2_QSTRING(__tr("Maximize")));

	m_pClose    = new KviMdiCaptionButton(m_pCaption,KviMdiCaptionButton::Close,"close_button");
	connect(m_pClose,SIGNAL(buttonPressed()),this,SLOT(closePressed()));
	QToolTip::add(m_pClose,_CHAR_2_QSTRING(__tr("Close")));

	setFrameStyle(QFrame::StyledPanel|QFrame::Raised);
	setFocusPolicy(ClickFocus);

	m_state             = Normal;

	m_iResizeCorner     = KVI_MDI_NORESIZE;
	m_iLastCursorCorner = KVI_MDI_NORESIZE;

	setMouseTracking(true);
	setMinimumSize(QSize(KVI_MDI_CHILD_MIN_WIDTH,KVI_MDI_CHILD_MIN_HEIGHT));
	m_bResizeMode = false;
}

//============ ~KviMdiChild ============//

KviMdiChild::~KviMdiChild()
{
}

//============ mousePressEvent =============//

void KviMdiChild::mousePressEvent(QMouseEvent *e)
{
//#ifdef _KVI_DEBUG_CLASS_NAME_
//	KviStr cl = m_pClient->className();
//	debug("mousePressEvent %s",cl.ptr());
//#endif
	m_iResizeCorner=getResizeCorner(e->pos().x(),e->pos().y());
	if(m_iResizeCorner != KVI_MDI_NORESIZE)
	{
		if(m_pManager->m_bOpaqueResize)
		{
			grabMouse(getResizeCursor(m_iResizeCorner));
			m_bResizeMode = true;
		} else resizeWindow(m_iResizeCorner);
	}
}

//============ mouseReleaseEvent ==============//

void KviMdiChild::mouseReleaseEvent(QMouseEvent *)
{
//#ifdef _KVI_DEBUG_CLASS_NAME_
//	KviStr cl = m_pClient->className();
//	debug("mouseReleaseEvent %s",cl.ptr());
//#endif
	m_iResizeCorner=KVI_MDI_NORESIZE;
	m_iLastCursorCorner=KVI_MDI_NORESIZE;
	if(m_bResizeMode)
	{
		m_bResizeMode = false;
		releaseMouse();
	}
	if(QApplication::overrideCursor())QApplication::restoreOverrideCursor();
}

//#warning "After a quick resize , releasing the button in another MDI child"
//#warning "this child loses focus and the other gains it."
//#warning "the problem is that the mouseReleaseEvent is sent AFTER focusOutEvent!"

//============= leaveEvent ===============//

void KviMdiChild::leaveEvent(QEvent *)
{
//#ifdef _KVI_DEBUG_CLASS_NAME_
//	KviStr cl = m_pClient->className();
//	debug("leaveEvent %s",cl.ptr());
//#endif
	if(!m_bResizeMode){
		m_iResizeCorner=KVI_MDI_NORESIZE;
		m_iLastCursorCorner=KVI_MDI_NORESIZE;
		if(QApplication::overrideCursor())QApplication::restoreOverrideCursor();
	} else {
		if(m_iResizeCorner != KVI_MDI_NORESIZE)resizeWindowOpaque(m_iResizeCorner);
	}
}

QCursor KviMdiChild::getResizeCursor(int resizeCorner)
{
	switch (resizeCorner) { 
		case KVI_MDI_RESIZE_LEFT: 
		case KVI_MDI_RESIZE_RIGHT: 
			return Qt::sizeHorCursor;
			break; 
		case KVI_MDI_RESIZE_TOP: 
		case KVI_MDI_RESIZE_BOTTOM: 
			return Qt::sizeVerCursor;
			break; 
		case KVI_MDI_RESIZE_TOPLEFT: 
		case KVI_MDI_RESIZE_BOTTOMRIGHT: 
			return Qt::sizeFDiagCursor;
			break; 
		case KVI_MDI_RESIZE_BOTTOMLEFT: 
		case KVI_MDI_RESIZE_TOPRIGHT: 
			return Qt::sizeBDiagCursor;
			break; 
		default:
			return Qt::arrowCursor;
			break;
	}
}

//============= setResizeCursor ===============//

void KviMdiChild::setResizeCursor(int resizeCorner)
{
	if(resizeCorner == m_iLastCursorCorner)return; //Don't do it twice
	m_iLastCursorCorner = resizeCorner;
	if(resizeCorner == KVI_MDI_NORESIZE){
		if(QApplication::overrideCursor())QApplication::restoreOverrideCursor(); 
	} else QApplication::setOverrideCursor(getResizeCursor(resizeCorner),true);
}

//============= mouseMoveEvent ===============//

void KviMdiChild::mouseMoveEvent(QMouseEvent *e)
{
//#ifdef _KVI_DEBUG_CLASS_NAME_
//	KviStr cl = m_pClient->className();
//	debug("mouseMoveEvent %s",cl.ptr());
//#endif
	if(e->state() & LeftButton){
		if(m_iResizeCorner){
			if(m_bResizeMode)resizeWindowOpaque(m_iResizeCorner);
			else resizeWindow(m_iResizeCorner);
		}
	} else setResizeCursor(getResizeCorner(e->pos().x(), e->pos().y()));
}

void KviMdiChild::calculateResizeRect(int resizeCorner,QPoint mousePos,QRect &resizeRect,int minWidth,int minHeight)
{
	switch (resizeCorner){
		case KVI_MDI_RESIZE_LEFT:
			resizeRect.setLeft(mousePos.x() - 1);
			if(resizeRect.width() < minWidth)resizeRect.setLeft(resizeRect.right() - minWidth);
			break; 
		case KVI_MDI_RESIZE_RIGHT: 
			resizeRect.setRight(mousePos.x() + 1);
			if(resizeRect.width() < minWidth)resizeRect.setRight(resizeRect.left() + minWidth);
			break; 
		case KVI_MDI_RESIZE_TOP:
			resizeRect.setTop(mousePos.y() - 1);
			if(resizeRect.height() < minHeight)resizeRect.setTop(resizeRect.bottom() - minHeight);
			break; 
		case KVI_MDI_RESIZE_BOTTOM: 
			resizeRect.setBottom(mousePos.y() + 1);
			if(resizeRect.height() < minHeight)resizeRect.setBottom(resizeRect.top() + minHeight);
			break; 
		case KVI_MDI_RESIZE_BOTTOMRIGHT:
			resizeRect.setBottom(mousePos.y() + 1);
			if(resizeRect.height() < minHeight)resizeRect.setBottom(resizeRect.top() + minHeight);
			resizeRect.setRight(mousePos.x() + 1);
			if(resizeRect.width() < minWidth)resizeRect.setRight(resizeRect.left() + minWidth);
			break;
		case KVI_MDI_RESIZE_TOPRIGHT:
			resizeRect.setTop(mousePos.y() - 1);
			if(resizeRect.height() < minHeight)resizeRect.setTop(resizeRect.bottom() - minHeight);
			resizeRect.setRight(mousePos.x() + 1);
			if(resizeRect.width() < minWidth)resizeRect.setRight(resizeRect.left() + minWidth);
			break;
		case KVI_MDI_RESIZE_BOTTOMLEFT:
			resizeRect.setBottom(mousePos.y() + 1);
			if(resizeRect.height() < minHeight)resizeRect.setBottom(resizeRect.top() + minHeight);
			resizeRect.setLeft(mousePos.x() - 1);			
			if(resizeRect.width() < minWidth)resizeRect.setLeft(resizeRect.right() - minWidth);
			break;
		case KVI_MDI_RESIZE_TOPLEFT:
			resizeRect.setTop(mousePos.y() - 1);
			if(resizeRect.height() < minHeight)resizeRect.setTop(resizeRect.bottom() - minHeight);
			resizeRect.setLeft(mousePos.x() - 1);
			if(resizeRect.width() < minWidth)resizeRect.setLeft(resizeRect.right() - minWidth);
			break;
	}
}

void KviMdiChild::calculateMinimumSize(int &minWidth,int &minHeight)
{
	if(m_pClient){
		minWidth  = m_pClient->minimumSize().width() + KVI_MDI_CHILD_DOUBLE_BORDER;
		minHeight = m_pClient->minimumSize().height()+ KVI_MDI_CHILD_DOUBLE_BORDER +
					m_pCaption->heightHint() + KVI_MDI_CHILD_SEPARATOR;
	}
	if(minWidth<KVI_MDI_CHILD_MIN_WIDTH)minWidth=KVI_MDI_CHILD_MIN_WIDTH;
	if(minHeight<KVI_MDI_CHILD_MIN_WIDTH)minHeight=KVI_MDI_CHILD_MIN_HEIGHT;
}

void KviMdiChild::resizeWindowOpaque(int resizeCorner)
{
	int minWidth=0;
	int minHeight=0;
	QRect resizeRect(x(),y(),width(),height());
	calculateMinimumSize(minWidth,minHeight);
	QPoint mousePos = m_pManager->mapFromGlobal(QCursor::pos());
	calculateResizeRect(resizeCorner,mousePos,resizeRect,minWidth,minHeight);
	setGeometry(resizeRect.x(),resizeRect.y(),resizeRect.width(),resizeRect.height());
	if(m_state==Maximized){
		m_state=Normal;
		m_pMaximize->setType(KviMdiCaptionButton::Maximize);
		m_pManager->childRestored(this,true);
	}
}

void KviMdiChild::resizeWindow(int resizeCorner)
{
	m_iResizeCorner = KVI_MDI_NORESIZE;
	QRect resizeRect(x(),y(),width(),height());

	int grab=XGrabPointer(qt_xdisplay(),m_pManager->winId(),true,(ButtonReleaseMask),
		GrabModeAsync,GrabModeAsync,m_pManager->winId(),None,CurrentTime);

	kvi_drawDragRectangle(resizeRect,m_pManager->handle());

	int minWidth=0;
	int minHeight=0;

	calculateMinimumSize(minWidth,minHeight);

	XEvent ev;
	QPoint oldMousePos = m_pManager->mapFromGlobal(QCursor::pos());

	while (!XCheckMaskEvent(qt_xdisplay(),ButtonReleaseMask, &ev)){
		Window dw1, dw2;
		int t1,t2,newX,newY;
		unsigned int t3;
		XQueryPointer(qt_xdisplay(), qt_xrootwin(), &dw1, &dw2, &newX, &newY, &t1, &t2, &t3);
		QPoint mousePos = m_pManager->mapFromGlobal(QPoint(newX, newY));
		if(oldMousePos != mousePos){
			kvi_drawDragRectangle(resizeRect,m_pManager->handle());
			calculateResizeRect(resizeCorner,mousePos,resizeRect,minWidth,minHeight);
			kvi_drawDragRectangle(resizeRect,m_pManager->handle());
			oldMousePos = mousePos;
		}	
	}
	kvi_drawDragRectangle(resizeRect,m_pManager->handle());

	if(grab==GrabSuccess)XUngrabPointer(qt_xdisplay(),CurrentTime);

	setGeometry(resizeRect.x(),resizeRect.y(),resizeRect.width(),resizeRect.height());
	if(QApplication::overrideCursor())QApplication::restoreOverrideCursor();

	if(m_state==Maximized){
		m_state=Normal;
		m_pMaximize->setType(KviMdiCaptionButton::Maximize);
		m_pManager->childRestored(this,true);
	}
	// Resync Qt
	ev.xany.window = winId();
	XSendEvent(ev.xany.display,ev.xany.window,false,ButtonReleaseMask,&ev);
}

void KviMdiChild::moveWindowOpaque(QPoint diff)
{
	if(m_state==Maximized){
		m_state=Normal;
		m_pMaximize->setType(KviMdiCaptionButton::Maximize);
	}
	setGeometry( x()+diff.x(), y()+diff.y(), width(), height());
//	move(x() + diff.x(), y()+diff.y());
}

//=========== moveWindow ==============//

void KviMdiChild::moveWindow()
{
	// This is called by the caption on mousePressEvent
	QRect absoluteWinRect(x(),y(),width(),height());
	// Grab the pointer if possible
	QApplication::setOverrideCursor(Qt::sizeAllCursor,true);
//	XGrabServer(qt_xdisplay());
	int grab=XGrabPointer(qt_xdisplay(),m_pManager->winId(),false,(ButtonReleaseMask),
		GrabModeAsync,GrabModeAsync,m_pManager->winId(),None,CurrentTime);

	kvi_drawDragRectangle(absoluteWinRect,m_pManager->handle());
	
	XEvent ev;
	Window dummyWin1, dummyWin2;
	int dummy1, dummy2;
	unsigned int dummy3;
	QPoint oldMousePos = QCursor::pos();
	int mouseNewX,mouseNewY;
	QPoint mousePos;
	
	while((!XCheckMaskEvent(qt_xdisplay(),(ButtonReleaseMask),&ev))){
		XQueryPointer(qt_xdisplay(),qt_xrootwin(),&dummyWin1,&dummyWin2,
			&mouseNewX,&mouseNewY,&dummy1,&dummy2,&dummy3);

		mousePos = QPoint(mouseNewX, mouseNewY);
		if (oldMousePos != mousePos){
			kvi_drawDragRectangle(absoluteWinRect,m_pManager->handle());
			int nX = absoluteWinRect.x() + (mousePos.x()-oldMousePos.x());
			nX = QMAX(nX, KVI_MDI_CHILD_MIN_VISIBLE_EDGE-width());
			nX = QMIN(nX, parentWidget()->width()-KVI_MDI_CHILD_MIN_VISIBLE_EDGE);
			int nY = absoluteWinRect.y() + (mousePos.y()-oldMousePos.y());
			nY = QMAX(nY, 0);
			nY = QMIN(nY, parentWidget()->height()-KVI_MDI_CHILD_MIN_VISIBLE_EDGE);
			absoluteWinRect = QRect(nX, nY,  absoluteWinRect.width(), absoluteWinRect.height());
			kvi_drawDragRectangle(absoluteWinRect,m_pManager->handle());
			oldMousePos = mousePos;
		}
	}

	kvi_drawDragRectangle(absoluteWinRect,m_pManager->handle());

	if(grab==GrabSuccess)XUngrabPointer(qt_xdisplay(),CurrentTime);
//	XUngrabServer(qt_xdisplay());

	if(m_state==Maximized){
		m_state=Normal;
		m_pMaximize->setType(KviMdiCaptionButton::Maximize);
	}
	setGeometry(absoluteWinRect.x(),absoluteWinRect.y(),width(),height());
	QApplication::restoreOverrideCursor();
	// Qt or X gets confused here , and mouseMoveEvents are not
	// sent to this window until a mouse button is pressed
	// So...resync Qt :)
//	XSendEvent(ev.xany.display,ev.xany.window,false,ButtonReleaseMask,&ev);
	ev.xany.window = winId();
	XSendEvent(ev.xany.display,ev.xany.window,false,ButtonReleaseMask,&ev);
}

//================= getResizeCorner =============//

int KviMdiChild::getResizeCorner(int ax,int ay)
{
//	debug("GET RESIZE CORNER");
	int ret = KVI_MDI_NORESIZE; 
	if((ax>0)&&(ax<(KVI_MDI_CHILD_BORDER+2))) ret |= KVI_MDI_RESIZE_LEFT; 
	if((ax<width())&&(ax>(width()-(KVI_MDI_CHILD_BORDER+2)))) ret |= KVI_MDI_RESIZE_RIGHT; 
	if((ay>0)&&(ay<(KVI_MDI_CHILD_BORDER+2))) ret |= KVI_MDI_RESIZE_TOP;
	if((ay<(height()))&&(ay>(height()-(KVI_MDI_CHILD_BORDER+2)))) ret |= KVI_MDI_RESIZE_BOTTOM; 
	return ret; 
}

//============= maximizePressed ============//

void KviMdiChild::maximizePressed()
{
	switch(m_state){
		case Maximized: setState(Normal);    break;
		case Normal:    setState(Maximized); break;
		case Minimized: setState(Maximized); break;
	}
}

//============= minimizePressed ============//

void KviMdiChild::minimizePressed()
{
	switch(m_state){
		case Minimized: setState(Normal);    break;
		case Normal:    setState(Minimized); break;
		case Maximized: setState(Minimized); break;
	}
}

//============= closePressed ============//

void KviMdiChild::closePressed()
{
	if(m_pClient)m_pClient->close();
}

//============ setState =================//

void KviMdiChild::setState(MdiWindowState state,bool bAnimate)
{
	if(m_state==Normal){ //save the current rect
		m_restoredRect=QRect(x() > 0 ? x() : 0,y() > 0 ? y() : 0,width(),height());
	}
	QRect begin(x(),y(),width(),height());
	QRect end=begin;
	switch(state){
		case Normal:
			switch(m_state){
				case Maximized: 
					if(bAnimate)m_pManager->animate(begin,m_restoredRect);
					m_pMaximize->setType(KviMdiCaptionButton::Maximize);
					setGeometry(m_restoredRect);
					m_state=state;
					m_pManager->childRestored(this,true);
					break;
				case Minimized:
					begin=QRect(x()+width()/2,y()+height()/2,1,1);
					if(bAnimate)m_pManager->animate(begin,end);
					m_pMinimize->setType(KviMdiCaptionButton::Minimize);
					show();
					m_state=state;
					m_pManager->childRestored(this,false);
					break;
				case Normal:
					break;
			}
			break;
		case Maximized:
			end=QRect(-KVI_MDI_CHILD_MAXIMIZED_HIDDEN_EDGE,
				-(KVI_MDI_CHILD_BORDER + m_pCaption->heightHint() + 2),
				m_pManager->width() + KVI_MDI_CHILD_DOUBLE_MAXIMIZED_HIDDEN_EDGE,
				m_pManager->height() + KVI_MDI_CHILD_BORDER + m_pCaption->heightHint() + KVI_MDI_CHILD_MAXIMIZED_HIDDEN_EDGE + 3);
			switch(m_state){
				case Minimized:
					m_state=state;
					begin=QRect(x()+width()/2,y()+height()/2,1,1);
					if(bAnimate)m_pManager->animate(begin,end);
					setGeometry(end); //m_pManager->rect());
					m_pMaximize->setType(KviMdiCaptionButton::Restore);
					m_pMinimize->setType(KviMdiCaptionButton::Minimize);
					show();
					m_pManager->childMaximized(this,true);
					break;
				case Normal:
					m_state=state;
					if(bAnimate)m_pManager->animate(begin,end);
					m_pMaximize->setType(KviMdiCaptionButton::Restore);
					setGeometry(end); //m_pManager->rect());
					m_pManager->childMaximized(this,false);
					break;
				case Maximized:
					// HACK , needed because of wrong rect in window properties
					setGeometry(end);
					m_pMaximize->setType(KviMdiCaptionButton::Restore);
					m_pMinimize->setType(KviMdiCaptionButton::Minimize);
					m_pManager->childMaximized(this,false);
					break;
			}
			break;
		case Minimized:
			end=QRect(x()+width()/2,y()+height()/2,1,1);
			switch(m_state){
				case Maximized:
					hide();
					if(bAnimate)m_pManager->animate(begin,end);
					setGeometry(m_restoredRect);
					m_state=state;
					m_pMinimize->setType(KviMdiCaptionButton::Restore);
					m_pMaximize->setType(KviMdiCaptionButton::Maximize);
					m_pManager->childMinimized(this,true);
					break;
				case Normal:
					hide();
					if(bAnimate)m_pManager->animate(begin,end);
					m_state=state;
					m_pMinimize->setType(KviMdiCaptionButton::Restore);
					m_pManager->childMinimized(this,false);
					break;
				case Minimized:
					break;
			}
			break;
	}
}

//============ setCaption ===============//

void KviMdiChild::setCaption(const char *text)
{
	m_pCaption->setCaption(text);
	m_pManager->fillWindowMenu();
}

//============ enableClose ==============//

void KviMdiChild::enableClose(bool bEnable)
{
	m_pClose->setEnabled(bEnable);
	m_pClose->repaint(false);
}

bool KviMdiChild::closeEnabled()
{
	return m_pClose->isEnabled();
}

//============ setIconPointer =============//

void KviMdiChild::setIconPointer(QPixmap *ptr)
{
	m_pCaption->m_pIcon=ptr;
	m_pCaption->repaint(false);
}

//============ setClient ============//

void KviMdiChild::setClient(QWidget *w)
{
	__range_valid(m_pClient==0);
	__range_valid(w!=0);
	m_pClient=w;
	//resize to match the client
	int clientYPos=m_pCaption->heightHint()+KVI_MDI_CHILD_SEPARATOR+KVI_MDI_CHILD_BORDER;
	resize(w->width()+KVI_MDI_CHILD_DOUBLE_BORDER,w->height()+KVI_MDI_CHILD_BORDER+clientYPos);
	//Reparent if needed
	if(w->parent()!=this){
		//reparent to this widget , no flags , point , show it
		QPoint pnt2(KVI_MDI_CHILD_BORDER,clientYPos);
		w->reparent(this,pnt2,true);
	} else w->move(KVI_MDI_CHILD_BORDER,clientYPos);
	setFocusProxy(w);
	m_pCaption->setFocusProxy(w);
	m_pMinimize->setFocusProxy(w);
	m_pMaximize->setFocusProxy(w);
	m_pClose->setFocusProxy(w);
	linkChildren(w);
	if(m_pClient->minimumSize().width() > KVI_MDI_CHILD_MIN_WIDTH && 
		m_pClient->minimumSize().height() > KVI_MDI_CHILD_MIN_HEIGHT){
		setMinimumWidth(m_pClient->minimumSize().width() + KVI_MDI_CHILD_DOUBLE_BORDER);
		setMinimumHeight(m_pClient->minimumSize().height()+ KVI_MDI_CHILD_DOUBLE_BORDER +
					m_pCaption->heightHint() + KVI_MDI_CHILD_SEPARATOR);
	}
	KviStr tmp(KviStr::Format,"mdi_child_%s",w->name());
	setName(tmp.ptr());
}

//============ unsetClient ============//

void KviMdiChild::unsetClient()
{
	__range_valid(m_pClient!=0);
	if(!m_pClient)return;
	//reparent to desktop widget , no flags , point , show it
	unlinkChildren(m_pClient);
	setFocusProxy(0); //remove the focus proxy...
	//Kewl...the reparent function has a small prob now..
	//the new toplelvel widgets gets not reenabled for dnd
	m_pClient->reparent(0,QPoint(0,0),true);
	m_pClient=0;
	setName("mdi_child");
}

//============== linkChildren =============//

void KviMdiChild::linkChildren(QWidget *w)
{
	//recursive : installs an event filter to all childrens of the widget w
	QList<QObject> *list = (QList<QObject> *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *o = list->at(i);
			if(o->inherits("QWidget"))linkChildren(((QWidget *)o));
		}
	}
	w->installEventFilter(this);
}

//============== unlinkChildren =============//

void KviMdiChild::unlinkChildren(QWidget *w)
{
	//recursive : installs an event filter to all childrens of the widget w
	QList<QObject> *list = (QList<QObject> *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *o = list->at(i);
			if(o->inherits("QWidget"))unlinkChildren(((QWidget *)o));
		}
	}
	w->removeEventFilter(this);
}

//============== resizeEvent ===============//

void KviMdiChild::resizeEvent(QResizeEvent *)
{
	//Resize the caption
	int captionHeight=m_pCaption->heightHint();
	int captionWidth=width()-KVI_MDI_CHILD_DOUBLE_BORDER;
	m_pCaption->setGeometry(
		KVI_MDI_CHILD_BORDER,
		KVI_MDI_CHILD_BORDER,
		captionWidth,
		captionHeight
	);
	//The buttons are caption children
	m_pClose->setGeometry((captionWidth-captionHeight)+1,1,captionHeight-2,captionHeight-2);
	m_pMaximize->setGeometry((captionWidth-(captionHeight*2))+2,1,captionHeight-2,captionHeight-2);
	m_pMinimize->setGeometry((captionWidth-(captionHeight*3))+3,1,captionHeight-2,captionHeight-2);
	//Resize the client
	if(m_pClient){
		m_pClient->setGeometry(
			KVI_MDI_CHILD_BORDER,
			KVI_MDI_CHILD_BORDER+captionHeight+KVI_MDI_CHILD_SEPARATOR,
			captionWidth,
			height()-(KVI_MDI_CHILD_DOUBLE_BORDER+captionHeight+KVI_MDI_CHILD_SEPARATOR)
		);
	}
}

//============== evertFilter ==============//

#if defined(FocusIn) || defined(FocusOut) || defined(Enter) || defined(TrueColor)
	// Defined in X.h ... egcs may complain about it...
	// Heh...QEvent::FocusIn , if FocusIn is a define ?
	#undef FocusIn
	#undef FocusOut
	#undef Enter
	#undef TrueColor
#endif

#include <qapplication.h>

bool KviMdiChild::eventFilter(QObject *o,QEvent *e)
{
	switch(e->type()){
		case QEvent::FocusIn:
			__range_valid(isVisible()); //Still not sure...but seems a Qt bug. Even_FocusIn from a HIDDEN NoBackground widget!
			m_pCaption->setActive(true);
			m_pManager->setTopChild(this,false);
			return false;
		break;
		case QEvent::Enter:
			if(QApplication::overrideCursor())QApplication::restoreOverrideCursor();
		break;
		default:
			return false;
		break;
	}
	return false; //newer here
}

//============= focusInEvent ===============//

void KviMdiChild::focusInEvent(QFocusEvent *)
{
	// We gained focus by click , tab or from the caption label
	// Bring this child to top
	m_pCaption->setActive(true);
	m_pManager->setTopChild(this,false); //Do not focus by now...
	/*The client is our focusProxy ! it should be focused by Qt !*/
#ifdef _KVI_DEBUG_CLASS_NAME_
	__range_valid(focusProxy() == m_pClient);
#endif
}

///////////////////////////////////////////////////////////////////////////////
// KviMdiManager
///////////////////////////////////////////////////////////////////////////////

//============ KviMdiManager ============//

KviMdiManager::KviMdiManager(QWidget *parent,const char * name)
:QFrame(parent,name)
{
	setFrameStyle(QFrame::Panel|QFrame::Sunken);
	// Basic config
	m_captionFont              = QFont("helvetica",12);
	QFontMetrics fm(m_captionFont);
	m_captionFontLineSpacing   = fm.lineSpacing();
	m_captionActiveBackColor   = QColor(0,0,0);
	m_captionActiveForeColor   = QColor(80,255,0);
	m_captionInactiveBackColor = QColor(80,80,80);
	m_captionInactiveForeColor = QColor(200,200,200);
	m_bOpaqueMove              = true;
	m_bOpaqueResize            = true;

	m_pZ                       = new QList<KviMdiChild>; // Z order stack
	m_pZ->setAutoDelete(true);
	setFocusPolicy(ClickFocus);

	m_pWindowMenu = new QPopupMenu(this,"window_list_popup");
	connect(m_pWindowMenu,SIGNAL(activated(int)),this,SLOT(menuActivated(int)));
	fillWindowMenu();
}

void KviMdiManager::setMdiCaptionFont(const QFont &fnt)
{
	m_captionFont = fnt;
	QFontMetrics fm(m_captionFont);
	m_captionFontLineSpacing = fm.lineSpacing();
}
void KviMdiManager::setMdiCaptionActiveForeColor(const QColor &clr)
{ m_captionActiveForeColor = clr; }
void KviMdiManager::setMdiCaptionActiveBackColor(const QColor &clr)
{ m_captionActiveBackColor = clr; }
void KviMdiManager::setMdiCaptionInactiveForeColor(const QColor &clr)
{ m_captionInactiveForeColor = clr; }
void KviMdiManager::setMdiCaptionInactiveBackColor(const QColor &clr)
{ m_captionInactiveBackColor = clr; }

//============ ~KviMdiManager ============//

KviMdiManager::~KviMdiManager()
{
	delete m_pZ; //This will destroy all the widgets inside.
}

//============ manageChild ============//

void KviMdiManager::manageChild(KviMdiChild *lpC,bool bShow,bool bCascade)
{
	__range_valid(lpC);
	KviMdiChild * top=topChild();
	if(bShow)m_pZ->append(lpC); //visible -> first in the Z order
	else m_pZ->insert(0,lpC); //hidden -> last in the Z order
	if(bCascade)lpC->move(getCascadePoint(m_pZ->count()-1));
	if(bShow){
		if(top){ //Maximize if needed
			if(top->state() == KviMdiChild::Maximized){
				top->setState(KviMdiChild::Normal,false);
				lpC->setState(KviMdiChild::Maximized,false);
			}
		}
		lpC->show();
		lpC->raise();
		lpC->setFocus();
	}
	fillWindowMenu();
}

//============= focusInEvent ===============//

void KviMdiManager::focusInEvent(QFocusEvent *)
{
	focusTopChild();
}

//============ destroyChild ============//

void KviMdiManager::destroyChild(KviMdiChild *lpC,bool bFocusTopChild)
{
	bool bWasMaximized = lpC->state() == KviMdiChild::Maximized;
	disconnect(lpC);
	lpC->blockSignals(true);
#ifdef _KVI_DEBUG_CHECK_RANGE_
	//Report invalid results in a debug session
	__range_valid(m_pZ->removeRef(lpC));
#else
	m_pZ->removeRef(lpC);
#endif
	if(bWasMaximized){
		KviMdiChild * c=topChild();
		if(c)c->setState(KviMdiChild::Maximized,false);
		else {
			// SDI state change
			enterSDIMode(false);
		}
	}
	if(bFocusTopChild)focusTopChild();
	fillWindowMenu();
}

//============= setTopChlid ============//

void KviMdiManager::setTopChild(KviMdiChild *lpC,bool bSetFocus)
{
	__range_valid(lpC);
	__range_valid(lpC->isVisible());
	if(m_pZ->last() != lpC){
//		__debug_1arg("KviMdiManager::Moving child %s to top of the Z stack",lpC->caption());
		m_pZ->setAutoDelete(false);
#ifdef _KVI_DEBUG_CHECK_RANGE_
		//Report invalid results in a debug session
		__range_valid(m_pZ->removeRef(lpC));
#else
		m_pZ->removeRef(lpC);
#endif
		//disable the labels of all the other children
		for(KviMdiChild *pC=m_pZ->first();pC;pC=m_pZ->next()){
			pC->m_pCaption->setActive(false);
		}
		KviMdiChild *pMaximizedChild=m_pZ->last();
		if(pMaximizedChild->m_state != KviMdiChild::Maximized)pMaximizedChild=0;
		m_pZ->setAutoDelete(true);
		m_pZ->append(lpC);
		if(pMaximizedChild)lpC->setState(KviMdiChild::Maximized,false); //do not animate the change
		lpC->raise();
		if(pMaximizedChild){
			pMaximizedChild->setState(KviMdiChild::Normal,false);
		}
		if(bSetFocus){
			if(!lpC->hasFocus())lpC->setFocus();
		}
		fillWindowMenu();
	}
}

//========== aniumate ============//

void KviMdiManager::animate(QRect &start, QRect &finish)
{
	//Animation of movement from rect start to rect end
    QRect r;
	int times=10;
    int cX = start.x();
    int cY = start.y();
    int cW = start.width();
    int cH = start.height();
    XGrabServer(qt_xdisplay());
    for (int i=0; i<=times; i++)
    {
		if(cW < 4)cW = 4; //prevent negative widths later...
		if(cH < 4)cH = 4;
        r = QRect(cX, cY, cW, cH);
        kvi_drawDragRectangle(r,handle(),2);
        XFlush(qt_xdisplay());
        XSync(qt_xdisplay(), false);
		//loose some time...
#ifdef HAVE_USLEEP
        usleep(100);
#endif
        kvi_drawDragRectangle(r,handle(),2);
        cX += (finish.x()-start.x())/(times);
        cY +=  (finish.y()-start.y())/(times);
        cW += (finish.width()-start.width())/times;
        cH += (finish.height()-start.height())/times;
    }
    XUngrabServer(qt_xdisplay());
}

//============== drawDragRectangle ===========//
/*
void KviMdiManager::drawDragRectangle( QRect &rectToDraw,int style)
{
	//Draw a rectangle with the invert function.
	//Style != 0 means : draw it 'fat'
	if (!rectToDraw.isNull()){
		XGCValues gcvals;
		gcvals.foreground=black.pixel();
		gcvals.subwindow_mode=IncludeInferiors;
		gcvals.function=GXinvert;
        
		GC gc=XCreateGC(x11Display(),handle(),GCForeground|GCSubwindowMode|GCFunction,&gcvals);
		XDrawRectangle(x11Display(),handle(),gc,rectToDraw.x(),rectToDraw.y(),rectToDraw.width(),rectToDraw.height());
		if(style > 0){
			XDrawRectangle(x11Display(),handle(),gc,rectToDraw.x()+2,rectToDraw.y()+2,rectToDraw.width()-4,rectToDraw.height()-4);
		}
		XDrawRectangle(x11Display(),handle(),gc,rectToDraw.x()+1,rectToDraw.y()+1,rectToDraw.width()-2,rectToDraw.height()-2);
		XFreeGC(x11Display(),gc);
	}
}
*/

//============== resizeEvent ================//

void KviMdiManager::resizeEvent(QResizeEvent *e)
{
	//If we have a maximized children at the top , adjust its size
	KviMdiChild *lpC=m_pZ->last();
	if(lpC){
		if(lpC->m_state==KviMdiChild::Maximized)lpC->resize(width()+KVI_MDI_CHILD_DOUBLE_MAXIMIZED_HIDDEN_EDGE,
			height() + KVI_MDI_CHILD_BORDER + lpC->m_pCaption->heightHint() + KVI_MDI_CHILD_MAXIMIZED_HIDDEN_EDGE + 3);
	}
}

//=============== mousePressEvent =============//

void KviMdiManager::mousePressEvent(QMouseEvent *e)
{
	//Popup the window menu
	if(e->button() & RightButton)m_pWindowMenu->popup(mapToGlobal(e->pos()));
}

//=============== getCascadePoint ============//

QPoint KviMdiManager::getCascadePoint(int indexOfWindow)
{
	QPoint pnt(0,0);
	if(indexOfWindow==0)return pnt;
	KviMdiChild *lpC=m_pZ->first();
	int step=(lpC ? lpC->m_pCaption->heightHint()+KVI_MDI_CHILD_BORDER : 20);
	int availableHeight=height()-(lpC ? lpC->minimumSize().height() : KVI_MDI_CHILD_MIN_HEIGHT);
	int availableWidth=width()-(lpC ? lpC->minimumSize().width() : KVI_MDI_CHILD_MIN_WIDTH);
	int ax=0;
	int ay=0;
	for(int i=0;i<indexOfWindow;i++){
		ax+=step;
		ay+=step;
		if(ax>availableWidth)ax=0;
		if(ay>availableHeight)ay=0;
	}
	pnt.setX(ax);
	pnt.setY(ay);
	return pnt;
}

//=============== fillWindowMenu ===============//

void KviMdiManager::fillWindowMenu()
{
	m_pWindowMenu->clear();
	int i=100;
	for(KviMdiChild *lpC=m_pZ->first();lpC;lpC=m_pZ->next()){
		KviStr szItem;
		szItem.setNum(((uint)i)-99);
		szItem+=". ";
		if(lpC->m_state==KviMdiChild::Minimized){
			szItem+="( ";
			szItem+=lpC->caption();
			szItem+=" )";
		} else szItem+=lpC->caption();
		m_pWindowMenu->insertItem(szItem.ptr(),i);
		m_pWindowMenu->setItemChecked(i,((uint)i)==(m_pZ->count()+99)); //Active item :)
		i++;
	}
	m_pWindowMenu->insertSeparator();
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("&Cascade Windows")),this,SLOT(cascadeWindows()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("Cascade &Maximized")),this,SLOT(cascadeMaximized()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("Expand &Vertical")),this,SLOT(expandVertical()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("Expand &Horizontal")),this,SLOT(expandHorizontal()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("&Anodine's Tile")),this,SLOT(tileAnodine()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("&Pragma's Tile (Vertical)")),this,SLOT(tilePragmaVertical()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("P&ragma's Tile (Horizontal)")),this,SLOT(tilePragmaHorizontal()));
}

//================ menuActivated ===============//

void KviMdiManager::menuActivated(int id)
{
	if(id<100)return;
	id-=100;
	__range_valid(((uint)id) < m_pZ->count());
	KviMdiChild *lpC=m_pZ->at(id);
	if(!lpC)return;
	if(lpC->m_state==KviMdiChild::Minimized)lpC->minimizePressed();
	if(lpC==m_pZ->last())return;
	setTopChild(lpC,true);
}

//================ childMinimized ===============//

void KviMdiManager::childMinimized(KviMdiChild *lpC,bool bWasMaximized)
{
	__range_valid(lpC);
	if(m_pZ->findRef(lpC) == -1)return;
	if(m_pZ->count() > 1){
		m_pZ->setAutoDelete(false);
#ifdef _KVI_DEBUG_CHECK_RANGE_
		//Report invalid results in a debug session
		__range_valid(m_pZ->removeRef(lpC));
#else
		m_pZ->removeRef(lpC);
#endif
		m_pZ->setAutoDelete(true);
		m_pZ->insert(0,lpC);
		if(bWasMaximized){
			// Need to maximize the top child
			lpC = m_pZ->last();
			if(!lpC)return; //??
			if(lpC->m_state==KviMdiChild::Minimized){
				if(bWasMaximized)emit enterSDIMode(false);
				return;
			}
			lpC->setState(KviMdiChild::Maximized,false); //do not animate the change
		}
		focusTopChild();
	} else {
		// Unique window minimized...it won't loose the focus...!!
//#warning "Unique window minimized : don't loose the focus"
		setFocus(); //Remove focus from the child
		if(bWasMaximized)emit enterSDIMode(false);
	}
	fillWindowMenu();
}

void KviMdiManager::childMaximized(KviMdiChild * lpC,bool bWasMinimized)
{
	if(lpC == m_pZ->last())emit enterSDIMode(true);
}

void KviMdiManager::childRestored(KviMdiChild * lpC,bool bWasMaximized)
{
	if((lpC == m_pZ->last()) && bWasMaximized)emit enterSDIMode(false);
}

//============= focusTopChild ===============//

void KviMdiManager::focusTopChild()
{
	KviMdiChild *lpC=m_pZ->last();
	if(!lpC)return;
	if(lpC->m_state==KviMdiChild::Minimized)return;
	//disable the labels of all the other children
	for(KviMdiChild *pC=m_pZ->first();pC;pC=m_pZ->next()){
		if(pC != lpC)pC->m_pCaption->setActive(false);
	}
	lpC->raise();
	if(!lpC->hasFocus())lpC->setFocus();
}

void KviMdiManager::minimizeTopChild()
{
	KviMdiChild * lpC = m_pZ->last();
	if(!lpC)return;
	if(lpC->state() != KviMdiChild::Minimized)lpC->minimizePressed();
}

void KviMdiManager::restoreTopChild()
{
	KviMdiChild * lpC = m_pZ->last();
	if(!lpC)return;
	if(lpC->state() == KviMdiChild::Maximized)lpC->maximizePressed();
}

void KviMdiManager::closeTopChild()
{
	KviMdiChild * lpC = m_pZ->last();
	if(!lpC)return;
	lpC->closePressed();
}

//============= cascadeWindows ===============//

void KviMdiManager::cascadeWindows()
{
	int idx=0;
	QList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->m_state != KviMdiChild::Minimized){
			if(lpC->m_state==KviMdiChild::Maximized)lpC->setState(KviMdiChild::Normal,false);
			lpC->move(getCascadePoint(idx));
			//lpC->resize(lpC->minimumSize());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

//============= cascadeMaximized ===============//

void KviMdiManager::cascadeMaximized()
{
	int idx=0;
	QList<KviMdiChild> list(*m_pZ);

	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->m_state != KviMdiChild::Minimized){
			if(lpC->m_state==KviMdiChild::Maximized)lpC->setState(KviMdiChild::Normal,false);
			QPoint pnt(getCascadePoint(idx));
			lpC->move(pnt);
			QSize curSize(width()-pnt.x(),height()-pnt.y());
			if((lpC->minimumSize().width() > curSize.width()) ||
				(lpC->minimumSize().height() > curSize.height()))lpC->resize(lpC->minimumSize());
			else lpC->resize(curSize);
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::expandVertical()
{
	int idx=0;
	QList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->m_state != KviMdiChild::Minimized){
			if(lpC->m_state==KviMdiChild::Maximized)lpC->setState(KviMdiChild::Normal,false);
			lpC->setGeometry(lpC->x(),0,lpC->width(),height());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::expandHorizontal()
{
	int idx=0;
	QList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->m_state != KviMdiChild::Minimized){
			if(lpC->m_state==KviMdiChild::Maximized)lpC->setState(KviMdiChild::Normal,false);
			lpC->setGeometry(0,lpC->y(),width(),lpC->height());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::expand()
{
	int idx=0;
	QList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->m_state != KviMdiChild::Minimized){
			if(lpC->m_state==KviMdiChild::Maximized)lpC->setState(KviMdiChild::Normal,false);
			lpC->setGeometry(0,0,width(),height());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}
//============= getVisibleChildCount =============//

int KviMdiManager::getVisibleChildCount()
{
	int cnt=0;
	for(KviMdiChild *lpC=m_pZ->first();lpC;lpC=m_pZ->next()){
		if(lpC->m_state != KviMdiChild::Minimized)cnt++;
	}
	return cnt;
}

//============ tilePragma ============//

void KviMdiManager::tilePragmaHorizontal()
{
	tileAllInternal(9,true);
}

void KviMdiManager::tilePragmaVertical()
{
	tileAllInternal(9,false);
}

//============ tileAllInternal ============//

void KviMdiManager::tileAllInternal(int maxWnds,bool bHorizontal)
{
	//NUM WINDOWS =           1,2,3,4,5,6,7,8,9
	static int colstable[9]={ 1,1,1,2,2,2,3,3,3 }; //num columns
	static int rowstable[9]={ 1,2,3,2,3,3,3,3,3 }; //num rows
	static int lastwindw[9]={ 1,1,1,1,2,1,3,2,1 }; //last window multiplier
	static int colrecall[9]={ 0,0,0,3,3,3,6,6,6 }; //adjust self
	static int rowrecall[9]={ 0,0,0,0,4,4,4,4,4 }; //adjust self

	int * pColstable = bHorizontal ? colstable : rowstable;
	int * pRowstable = bHorizontal ? rowstable : colstable;
	int * pColrecall = bHorizontal ? colrecall : rowrecall;
	int * pRowrecall = bHorizontal ? rowrecall : colrecall;

	KviMdiChild *lpTop=topChild();
	int numVisible=getVisibleChildCount();
	if(numVisible<1)return;
	int numToHandle=((numVisible > maxWnds) ? maxWnds : numVisible);
	int xQuantum=width()/pColstable[numToHandle-1];
	if(xQuantum < ((lpTop->minimumSize().width() > KVI_MDI_CHILD_MIN_WIDTH) ? lpTop->minimumSize().width() : KVI_MDI_CHILD_MIN_WIDTH)){
		if(pColrecall[numToHandle-1]==0)debug(__tr("Tile: Not enouh space"));
		else tileAllInternal(pColrecall[numToHandle-1],bHorizontal);
		return;
	}
	int yQuantum=height()/pRowstable[numToHandle-1];
	if(yQuantum < ((lpTop->minimumSize().height() > KVI_MDI_CHILD_MIN_HEIGHT) ? lpTop->minimumSize().height() : KVI_MDI_CHILD_MIN_HEIGHT)){
		if(pRowrecall[numToHandle-1]==0)debug(__tr("Tile: Not enough space"));
		else tileAllInternal(pRowrecall[numToHandle-1],bHorizontal);
		return;
	}
	int curX=0;
	int curY=0;
	int curRow=1;
	int curCol=1;
	int curWin=1;
	for(KviMdiChild *lpC=m_pZ->first();lpC;lpC=m_pZ->next()){
		if(lpC->m_state!=KviMdiChild::Minimized){
			//restore the window
			if(lpC->m_state==KviMdiChild::Maximized)lpC->setState(KviMdiChild::Normal,false);
			if((curWin%numToHandle)==0)lpC->setGeometry(curX,curY,xQuantum * lastwindw[numToHandle-1],yQuantum);
			else lpC->setGeometry(curX,curY,xQuantum,yQuantum);
			//example : 12 windows : 3 cols 3 rows
			if(curCol<pColstable[numToHandle-1]){ //curCol<3
				curX+=xQuantum; //add a column in the same row
				curCol++;       //increase current column
			} else {
				curX=0;         //new row
				curCol=1;       //column 1
				if(curRow<pRowstable[numToHandle-1]){ //curRow<3
					curY+=yQuantum; //add a row
					curRow++;       //
				} else {
					curY=0;         //restart from beginning
					curRow=1;       //
				}
			}
			curWin++;
		}
	}
	if(lpTop)lpTop->setFocus();
}

//============ tileAnodine ============//
void KviMdiManager::tileAnodine()
{
	KviMdiChild *lpTop=topChild();
	int numVisible=getVisibleChildCount(); // count visible windows
	if(numVisible<1)return;
	int numCols=int(sqrt(numVisible)); // set columns to square root of visible count
	// create an array to form grid layout
	int *numRows=new int[numCols];
	int numCurCol=0;
	while(numCurCol<numCols){
		numRows[numCurCol]=numCols; // create primary grid values
		numCurCol++;
	}
	int numDiff=numVisible-(numCols*numCols); // count extra rows
	int numCurDiffCol=numCols; // set column limiting for grid updates
	while(numDiff>0){
		numCurDiffCol--;
		numRows[numCurDiffCol]++; // add extra rows to column grid
		if(numCurDiffCol<1)numCurDiffCol=numCols; // rotate through the grid
		numDiff--;
	}
	numCurCol=0;
	int numCurRow=0;
	int curX=0;
	int curY=0;
	// the following code will size everything based on my grid above
	// there is no limit to the number of windows it will handle
	// it's great when a kick-ass theory works!!!                      // Pragma :)
	int xQuantum=width()/numCols;
	int yQuantum=height()/numRows[numCurCol];
	for(KviMdiChild *lpC=m_pZ->first();lpC;lpC=m_pZ->next()){
		if(lpC->m_state != KviMdiChild::Minimized){
			if(lpC->m_state==KviMdiChild::Maximized)lpC->setState(KviMdiChild::Normal,false);
			lpC->setGeometry(curX,curY,xQuantum,yQuantum);
			numCurRow++;
			curY+=yQuantum;
			if(numCurRow==numRows[numCurCol]){
				numCurRow=0;
				numCurCol++;
				curY=0;
				curX+=xQuantum;
				if(numCurCol!=numCols)yQuantum=height()/numRows[numCurCol];
			}
		}
	}
	delete[] numRows;
	if(lpTop)lpTop->setFocus();
}

#include "m_kvi_mdi.moc"
