//
//   File : kvi_config.cpp (/usr/build/NEW_kvirc/kvirc/kvilib/kvi_config.cpp)
//   Last major modification : Thu Jan 14 1999 18:03:59 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

// #define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#define _KVI_CONFIG_CPP_
#include "kvi_config.h"
#include "kvi_defines.h" // for KVI_CONFIG_DEFAULT_GROUP
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "config.h"


KviConfig::KviConfig(const char *filename)
{
	__range_valid(filename);
	m_szFileName = filename;
	m_pDict      = new QDict<KviStrDict>(17,false);
	m_pDict->setAutoDelete(true);
	m_bDirty     = false;
	m_szGroup    = KVI_CONFIG_DEFAULT_GROUP;
	load();
}

KviConfig::~KviConfig()
{
	if(m_bDirty)save();
	delete m_pDict;
}

void KviConfig::clear()
{
	delete m_pDict;
	m_pDict      = new QDict<KviStrDict>(17,false);
	m_pDict->setAutoDelete(true);
	m_bDirty     = false;
	m_szGroup    = KVI_CONFIG_DEFAULT_GROUP;
}

void KviConfig::clearGroup(const char *szGroup)
{

	m_pDict->remove(szGroup);
	__range_invalid(m_pDict->find(szGroup));
	if(!m_pDict->find(m_szGroup.ptr()))m_szGroup = KVI_CONFIG_DEFAULT_GROUP; //removed the current one
}

void KviConfig::clearKey(const char *szKey)
{
	KviStrDict * p_group = getCurrentGroup();
	p_group->remove(szKey);
	if(p_group->count() == 0)clearGroup(m_szGroup.ptr());
}

void KviConfig::getContentsString(KviStr &buffer)
{
	buffer = __tr("Contents of config file ");
	buffer.append(m_szFileName.ptr());
	buffer.append('\n');
	int sections = 0;
	int keys     = 0;
	QDictIterator<KviStrDict> it(*m_pDict);
	while(it.current()){
		buffer.append(" Section [");
		buffer.append(it.currentKey());
		buffer.append("]\n");
		int sectionKeys = 0;
		QDictIterator<KviStr> it2(*it.current());
		while(it2.current()){
			buffer.append("  Key [");
			buffer.append(it2.currentKey());
			buffer.append("] : ");
			buffer.append(it2.current()->ptr());
			buffer.append('\n');
			++it2;
			++sectionKeys;
			++keys;
		}
		KviStr tmp(KviStr::Format,__tr("  Total: %d keys"),sectionKeys);
		buffer.append(tmp);
		buffer.append('\n');
		++it;
		++sections;
	}
	KviStr tmp(KviStr::Format,__tr("Total: %d keys in %d sections"),keys,sections);
	buffer.append(tmp);
}

bool KviConfig::load()
{
	QFile f(m_szFileName.ptr());
	if(!f.open(IO_ReadOnly))return false;

	__range_valid(m_pDict->isEmpty());

	KviStrDict * p_group = new KviStrDict(17,false);
	p_group->setAutoDelete(true);

	m_pDict->insert(KVI_CONFIG_DEFAULT_GROUP,p_group);

	KviStr dataLine;
	bool bContinue;

	do{
		bContinue = kvi_readFirstNonEmptyStrippedLine(&f,dataLine);
		if(dataLine.hasData()){
			if(dataLine.at(0)=='['){
				//set the group
				dataLine.cutLeft(1);
				dataLine.cutRight(1);
				p_group = (KviStrDict *)m_pDict->find(dataLine.ptr());
				if(!p_group){
					p_group = new KviStrDict(17,false);
					p_group->setAutoDelete(true);
					m_pDict->insert(dataLine.ptr(),p_group);
				}
			} else {
				//data entry...split in two...
				KviStr name=dataLine.getToken('=');
				name.stripWhiteSpace();
				dataLine.stripWhiteSpace();

				if(name.hasData() && dataLine.hasData()){
					//insert (replace items if needed)
					KviStr *p_data=new KviStr(dataLine);
					p_group->replace(name.ptr(),p_data);
				}
			}
		}
	} while (bContinue);

	f.close();
	return true;
}

bool KviConfig::save()
{
	QFile f(m_szFileName.ptr());
	if(!f.open(IO_WriteOnly | IO_Truncate))return false;
	QDictIterator<KviStrDict> it(*m_pDict);
	while (it.current()){
		if(it.current()->count() != 0){
			KviStr key(it.currentKey());
			KviStr str(KviStr::Format,"[%s]",key.ptr());
			if(!kvi_writeLine(&f,str.ptr()))return false;
			KviStrDict * dict = (KviStrDict *)it.current();
			QDictIterator<KviStr> it2(*dict);
			while(it2.current()){
				key=it2.currentKey();
				KviStr *p_str = it2.current();
				str.sprintf("%s=%s",key.ptr(),p_str->ptr());
				if(!kvi_writeLine(&f,str.ptr()))return false;
				++it2;
			}
		}
		++it;
	}
	f.close();
	m_bDirty = false;
	return true;
}

bool KviConfig::hasKey(const char *szKey)
{
	KviStrDict * p_group = getCurrentGroup();
	return (p_group->find(szKey) != 0);
}

KviStrDict * KviConfig::getCurrentGroup()
{
	if(m_szGroup.isEmpty())m_szGroup = KVI_CONFIG_DEFAULT_GROUP;
	KviStrDict * p_group = m_pDict->find(m_szGroup.ptr());
	if(!p_group){
		//create the group
		p_group = new KviStrDict(17,false);
		p_group->setAutoDelete(true);
		m_pDict->insert(m_szGroup.ptr(),p_group);
	}
	return p_group;
}

void KviConfig::writeEntry(const char *szKey,const char *szValue)
{
	__range_valid(szKey);
	__range_valid(szValue);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data=new KviStr(szValue);
	p_group->replace(szKey,p_data);
}

const char * KviConfig::readEntry(const char *szKey,const char *szDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	m_szStrBuffer = p_str ? p_str->ptr() : szDefault;
	return m_szStrBuffer.ptr();
}

void KviConfig::writeEntry(const char *szKey,const QColor &clr)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr(KviStr::Format,"%d,%d,%d",clr.red(),clr.green(),clr.blue());
	p_group->replace(szKey,p_data);
}

QColor KviConfig::readColorEntry(const char *szKey,const QColor &clr)
{
	KviStrDict * p_group = getCurrentGroup();
	QColor color(clr);
	KviStr * pointer_that_IS_initialized = p_group->find(szKey);


	if(pointer_that_IS_initialized){

		KviStr str(*pointer_that_IS_initialized);
		str.stripLeftWhiteSpace();

		KviStr red,green,blue;

		str.getToken(red,',');
		str.getToken(green,',');
		str.getToken(blue,',');

		if((red.isUnsignedNum())&&(green.isUnsignedNum())&&(blue.isUnsignedNum())){
			bool bOk;
			int r = red.toInt(&bOk) % 256;
			int g = green.toInt(&bOk) % 256;
			int b = blue.toInt(&bOk) % 256;
			if(r < 0)r = -r;
			if(g < 0)g = -g;
			if(b < 0)b = -b;
			color.setRgb(r,g,b);
		}
	}
	return color;
}

void KviConfig::getFontProperties(KviStr & buffer,QFont *fnt)
{

//#warning "may be not so portable between versions ?"
	KviStr family(fnt->family());
#ifdef COMPILE_WITH_QT_3
	buffer.sprintf("%s,%d,%d,0,%d",
		family.ptr(),fnt->pointSize(),fnt->styleHint(),fnt->weight());
#else
	buffer.sprintf("%s,%d,%d,%d,%d",
		family.ptr(),fnt->pointSize(),fnt->styleHint(),fnt->charSet(),fnt->weight());
#endif
	if(fnt->italic())buffer.append(",bold");
	if(fnt->underline())buffer.append(",underline");
	if(fnt->strikeOut())buffer.append(",strikeout");
	if(fnt->fixedPitch())buffer.append(",fixed");
	if(fnt->rawMode())buffer.append(",raw");
}

void KviConfig::writeEntry(const char *szKey,QFont &fnt)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr tmp;
	getFontProperties(tmp,&fnt);
	KviStr *p_data = new KviStr(tmp);
	p_group->replace(szKey,p_data);
}

void KviConfig::setFontProperties(KviStr & str,QFont *fnt)
{
	KviStr family,pointSize,styleHint,charSet,weight,options;
	str.getToken(family,',');
	str.getToken(pointSize,',');
	str.getToken(styleHint,',');
	str.getToken(charSet,',');
	str.getToken(weight,','); 
	str.getToken(options,'\n');
	if(family.hasData())fnt->setFamily(family.ptr());
	if(pointSize.isUnsignedNum())fnt->setPointSize(pointSize.toInt());
	if(styleHint.isUnsignedNum())fnt->setStyleHint((QFont::StyleHint)styleHint.toUInt());
#ifndef COMPILE_WITH_QT_3
	if(charSet.isUnsignedNum())fnt->setCharSet((QFont::CharSet)charSet.toUInt());
#endif
	if(weight.isUnsignedNum())fnt->setWeight(weight.toUInt());
	if(options.hasData()){
		fnt->setItalic(options.contains("italic"));
		fnt->setUnderline(options.contains("underline"));
		fnt->setStrikeOut(options.contains("strikeout"));
		fnt->setFixedPitch(options.contains("fixed"));
		fnt->setRawMode(options.contains("raw"));
	}
}

QFont KviConfig::readFontEntry(const char *szKey,QFont &fnt)
{
	QFont font(fnt);
//	bool bOk;
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str       = p_group->find(szKey);
	if(p_str){
		//FontEntry=Arial,12,9,0,100,italic,underline,strikeout,
		KviStr str(*p_str);
		str.stripLeftWhiteSpace();
		setFontProperties(str,&font);
	}
	return font;
}

void KviConfig::writeEntry(const char *szKey,bool bTrue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr(bTrue ? "true" : "false");
	p_group->replace(szKey,p_data);
}

bool KviConfig::readBoolEntry(const char *szKey,bool bTrue)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return bTrue;
	return (*p_str == "true");
}

void KviConfig::writeEntry(const char *szKey,const QRect &rct)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_data = new KviStr(KviStr::Format,"%d,%d,%d,%d",rct.left(),rct.top(),rct.right(),rct.bottom());
	p_group->replace(szKey,p_data);
}

QRect KviConfig::readRectEntry(const char *szKey,const QRect &rct)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * another_pointer_that_IS_initialized = p_group->find(szKey);
	if(!another_pointer_that_IS_initialized)return rct;
	KviStr tmp;
	QRect ret;
	const char *aux = kvi_extractToken(tmp,another_pointer_that_IS_initialized->ptr(),',');
	bool bOk;
	ret.setLeft(tmp.toInt(&bOk));
	if(!bOk)return rct;
	aux = kvi_extractToken(tmp,aux,',');
	ret.setTop(tmp.toInt(&bOk));
	if(!bOk)return rct;
	aux = kvi_extractToken(tmp,aux,',');
	ret.setRight(tmp.toInt(&bOk));
	if(!bOk)return rct;
	aux = kvi_extractToken(tmp,aux,',');
	ret.setBottom(tmp.toInt(&bOk));
	if(!bOk)return rct;
	return ret;
}


void KviConfig::writeEntry(const char *szKey,unsigned short usValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(usValue);
	p_group->replace(szKey,p_data);
}

unsigned short int KviConfig::readUShortEntry(const char *szKey,unsigned short int usDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return usDefault;
	bool bOk;
	unsigned short int usVal=p_str->toUShort(&bOk);
	return bOk ? usVal : usDefault;
}
/*
void KviConfig::writeEntry(const char *szKey,unsigned long lValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(lValue);
	p_group->replace(szKey,p_data);
}

unsigned long KviConfig::readULongEntry(const char *szKey,unsigned long lDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return lDefault;
	bool bOk;
	unsigned long lVal=p_str->toULong(&bOk);
	return bOk ? lVal : lDefault;
}
*/
void KviConfig::writeEntry(const char *szKey,int iValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(iValue);
	p_group->replace(szKey,p_data);
}

int KviConfig::readIntEntry(const char *szKey,int iDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return iDefault;
	bool bOk;
	int iVal=p_str->toInt(&bOk);
	return bOk ? iVal : iDefault;
}
void KviConfig::writeEntry(const char *szKey,unsigned int iValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(iValue);
	p_group->replace(szKey,p_data);
}

unsigned int KviConfig::readUIntEntry(const char *szKey,unsigned int iDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return iDefault;
	bool bOk;
	unsigned int iVal=p_str->toUInt(&bOk);
	return bOk ? iVal : iDefault;
}

void KviConfig::writeEntry(const char *szKey,char iValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(iValue);
	p_group->replace(szKey,p_data);
}

char KviConfig::readCharEntry(const char *szKey,char iDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return iDefault;
	bool bOk;
	char iVal=p_str->toChar(&bOk);
	return bOk ? iVal : iDefault;
}

void KviConfig::writeEntry(const char *szKey,unsigned char iValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(iValue);
	p_group->replace(szKey,p_data);
}

unsigned char KviConfig::readUCharEntry(const char *szKey,unsigned char iDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return iDefault;
	bool bOk;
	unsigned char iVal=p_str->toUChar(&bOk);
	return bOk ? iVal : iDefault;
}

